
/**************************************************************************
 *
 *  $Id: xdevfeat.c 1.1.1.28.1.12 2017/03/07 13:45:34 thomas-b TEST $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    Meinberg API functions to check extended device features.
 *
 *    See notes near "defgroup xdevfeat_chk_supp_fncs" in xdevfeat.h.
 *
 * -----------------------------------------------------------------------
 *  $Log: xdevfeat.c $
 *  Revision 1.1.1.28.1.12  2017/03/07 13:45:34  thomas-b
 *  Check receiver info feature instead of builtin feature for ignore lock
 *  Revision 1.1.1.28.1.11  2017/02/22 11:32:47  martin
 *  Fixed warnings due to preliminary, unfinished code.
 *  Revision 1.1.1.28.1.10  2017/02/07 14:24:09  thomas-b
 *  Added function xdevfeat_has_monitoring, which checks if the extended feature MBG_XFEATURE_MONITORING is set
 *  Revision 1.1.1.28.1.9  2017/02/07 14:10:36  daniel
 *  Check license feature types
 *  Revision 1.1.1.28.1.8  2016/12/06 10:43:37  thomas-b
 *  Added function xdevfeat_has_sv_info
 *  Revision 1.1.1.28.1.7  2016/12/01 13:47:49  philipp
 *  Moved helper function check_byte_array_bit from xdevfeat.c to xtiocomm.c and make it public
 *  Revision 1.1.1.28.1.6  2016/11/30 16:12:26  thomas-b
 *  Added function xdevfeat_has_scu_stat
 *  Revision 1.1.1.28.1.5  2016/11/22 10:33:16  philipp
 *  Implemented I/O ports
 *  Revision 1.1.1.28.1.4  2016/11/02 12:16:18  thomas-b
 *  Fixed documentation
 *  Revision 1.1.1.28.1.3  2016/11/02 12:08:21  thomas-b
 *  Added function to check for new MBG_XFEATURE_REQ_TTM
 *  Revision 1.1.1.28.1.2  2016/11/01 09:34:43  martin
 *  Doxygen fixes.
 *  Revision 1.1.1.28.1.1  2016/10/25 08:07:54  martin
 *  Merged changes from 1.1.1.23.1.x.
 *  Revision 1.1.1.28  2016/10/20 14:46:04  thomas-b
 *  Added function to check if XBP is supported
 *  Revision 1.1.1.27  2016/10/20 10:43:36  thomas-b
 *  Added function to check if a device is a bus level device
 *  Revision 1.1.1.26  2016/10/14 11:09:41  thomas-b
 *  Added function to check MBG_XFEATURE_UCAP_NET
 *  Revision 1.1.1.25  2016/09/29 14:35:35  thomas-b
 *  Added function to check reboot feature
 *  Revision 1.1.1.24  2016/09/29 12:19:59  thomas-b
 *  Added function to check transactions feature
 *  Revision 1.1.1.23  2016/07/07 15:04:11  martin
 *  Renamed functions due to renamed library symbols.
 *  Revision 1.1.1.22  2016/06/15 10:15:51  thomas-b
 *  Added functions which check if the user capture feature is supported by a device
 *  Revision 1.1.1.21  2016/06/14 10:33:23  thomas-b
 *  Added functions which check if the event log feature is supported by a device
 *  Revision 1.1.1.20  2016/06/07 07:43:08  philipp
 *  New function to check for Irig Rx feature
 *  Revision 1.1.1.19  2016/06/02 10:15:40  philipp
 *  Renaming all MBG_EXT_REV_INFO related stuff to MBG_EXT_SYS_INFO.
 *  Revision 1.1.1.18  2016/05/30 08:10:47  thomas-b
 *  Added functions to check several builtin features
 *  Revision 1.1.1.17  2016/05/20 09:41:38  thomas-b
 *  Removed functions which check for a specific IMS card
 *  Revision 1.1.1.16  2016/05/20 08:50:46  thomas-b
 *  Added functions xdevfeat_has_time_scale and xdevfeat_has_tzcode
 *  Revision 1.1.1.15  2016/05/17 06:33:12  philipp
 *  New function to check whether a device has serial outputs
 *  Revision 1.1.1.14  2016/05/12 10:41:40  philipp
 *  New functions to check bpe card and IRIG Tx feature
 *  Revision 1.1.1.13  2016/05/10 06:15:32  philipp
 *  New function to check whether a device has programmable pulses
 *  Revision 1.1.1.12  2016/04/26 06:52:46  thomas-b
 *  Added functions to check if NET_CFG and LAN_IP4 APIs are supported
 *  Revision 1.1.1.11  2016/04/22 10:54:01  philipp
 *  New function to check whether device is LIU
 *  Revision 1.1.1.10  2016/04/20 13:21:06  thomas-b
 *  Added function to check if a device supports NTP
 *  Revision 1.1.1.9  2016/04/20 09:26:03  philipp
 *  Moved all HPS-PTP related structures to gpspriv.h and removed related extended feature bit from gpsdefs.h.
 *  Also removed functions from mbgextio and xdevfeat since HPS-PTP handling needs a redesign concerning structures.
 *  Thus, handle everything explicitly for now!
 *  -> Redesing this A.S.A.P.!!!
 *  Revision 1.1.1.8  2016/04/15 08:17:27  philipp
 *  New feature MBG_XFEATURE_EXT_PTP
 *  Revision 1.1.1.7  2016/04/13 07:01:22  philipp
 *  New function to check whether device is HPS
 *  Revision 1.1.1.6  2016/04/12 13:27:42  philipp
 *  Several new functions to check for device models and device features
 *  Revision 1.1.1.5  2016/04/11 13:57:13  thomas-b
 *  Added function xdevfeat_has_xmulti_ref
 *  Revision 1.1.1.4  2016/04/07 12:36:37  martin
 *  New function xdevfeat_has_ext_rev_info().
 *  Revision 1.1.1.3  2016/04/04 15:31:25  martin
 *  New function xdevfeat_is_vsg().
 *  Revision 1.1.1.2  2016/03/24 14:08:52  martin
 *  *** empty log message ***
 *  Revision 1.1.1.1  2016/03/18 10:48:10  martin
 *  *** empty log message ***
 *  Revision 1.1  2016/03/16 14:32:52  martin
 *  Initial revision.
 *
 **************************************************************************/

#define _XDEVFEAT
  #include <xdevfeat.h>
#undef _XDEVFEAT

#include <mbgerror.h>
#include <xtiocomm.h>



typedef uint32_t BUILTIN_FEATURE_MASK;


/**
 * @brief Entry for a table to specify model-depending built-in features
 */
typedef struct
{
  /// Model code according to ::RECEIVER_INFO::model_code, see ::GPS_MODEL_CODES
  uint16_t model_code;

  /// A combination of @ref GPS_FEATURE_MASKS bit masks specifying the
  /// built-in features supported by the specified device model.
  BUILTIN_FEATURE_MASK feature_mask;

} BUILTIN_FEATURE_TABLE_ENTRY;


/**
 * @brief A static table of builtin features
 *
 * Used to lookup the builtin features of a particular device model.
 */
static BUILTIN_FEATURE_TABLE_ENTRY builtin_feature_table[] = GPS_MODEL_BUILTIN_FEATURES;



static /*HDR*/
/**
 * @brief Check if a device supports a specific built-in feature or API
 *
 * Some features or API calls are implicitly supported by particular devices.
 * This function uses the ::RECEIVER_INFO::model_code to look up the specific
 * device in a table of ::BUILTIN_FEATURE_TABLE_ENTRY entries, and checks if
 * the requested feature bits are set for this device model.
 * If the model code of the specific device can't be found in the table then
 * then ::MBG_ERR_DEV_NOT_SUPP is returned and the source code files probably
 * need to be updated to support this device.
 *
 * @param[in] msk   One of the @ref GPS_FEATURE_MASKS bit masks
 * @param[in] p_ri  A ::RECEIVER_INFO structure read from the device before
 *
 * @return ::MBG_SUCCESS if all specified mask bits are set in ::RECEIVER_INFO::features,
 *         ::MBG_ERR_NOT_SUPP_BY_DEV if the bits are not set, and ::MBG_ERR_DEV_NOT_SUPP
 *         if the model code can't be found in the table.
 */
int check_builtin_feature( BUILTIN_FEATURE_MASK msk, const RECEIVER_INFO *p_ri )
{
  BUILTIN_FEATURE_TABLE_ENTRY *p;

  //### TODO Implement a kind of cache so we don't have to search the whole
  // table if several features of the same device are checked after each other.

  for ( p = builtin_feature_table; p->model_code || p->feature_mask; p++ )
    if ( p->model_code == p_ri->model_code )
      return ( ( p->feature_mask & msk ) == msk ) ? MBG_SUCCESS : MBG_ERR_NOT_SUPP_BY_DEV;

  return MBG_ERR_DEV_NOT_SUPP;

}  // check_builtin_feature



static /*HDR*/
/**
 * @brief Check if a device supports a specific feature or API
 *
 * This API call checks if a specific feature or API is supported
 * according to the ::RECEIVER_INFO::features mask read from the device.
 *
 * @param[in] msk   One of the @ref GPS_FEATURE_MASKS bit masks
 * @param[in] p_ri  A ::RECEIVER_INFO structure read from the device before
 *
 * @return ::MBG_SUCCESS if all specified mask bits are set in ::RECEIVER_INFO::features,
 *         else ::MBG_ERR_NOT_SUPP_BY_DEV
 */
int check_ri_feature( RI_FEATURES msk, const RECEIVER_INFO *p_ri )
{
  return ( ( p_ri->features & msk ) == msk ) ? MBG_SUCCESS : MBG_ERR_NOT_SUPP_BY_DEV;

}  // check_ri_feature



static /*HDR*/
/**
 * @brief Check if a specific extended feature is supported
 *
 * This API call checks if a specific extended feature or API is supported
 * according to the ::MBG_XFEATURE_BUFFER read from the device.
 *
 * @param[in] xf_bit   One of the ::MBG_XFEATURE_BITS
 * @param[in] xv_buf   Pointer to a ::MBG_XFEATURE_BUFFER read from the device before
 *
 * @return ::MBG_SUCCESS if the specified feature bit is set, else ::MBG_ERR_NOT_SUPP_BY_DEV or ::MBG_ERR_RANGE
 *
 * @see ::check_byte_array_bit
 */
int check_xfeature( int xf_bit, const MBG_XFEATURE_BUFFER *xv_buf )
{
  return check_byte_array_bit( xf_bit, xv_buf->b, sizeof( xv_buf->b ) );

}  // check_xfeature



#if defined( _PRELIMINARY_CODE )  //### TODO do we need this?

static /*HDR*/
/**
 * @brief Check if a specific bit is set in TLV's byte array.
 *
 * This API call checks if a specific TLV feature is supported according to
 * the ::MBG_TLV_INFO read from the device.
 *
 * @param[in] tlv_feat_bit   One of the ::MBG_TLV_FEAT_TYPES
 * @param[in] tlv_info       Pointer to a ::MBG_TLV_INFO read from the device before
 *
 * @return ::MBG_SUCCESS if the specified feature bit is set, else ::MBG_ERR_NOT_SUPP_BY_DEV or ::MBG_ERR_RANGE
 *
 * @see ::check_byte_array_bit
 */
int check_tlv_feat_supp( int tlv_feat_bit, const MBG_TLV_INFO *tlv_info )
{
  return check_byte_array_bit( tlv_feat_bit, tlv_info->supp_tlv_feat.b, sizeof( tlv_info->supp_tlv_feat.b ) );

}  // check_tlv_feat_supp

#endif  // defined( _PRELIMINARY_CODE )



/*HDR*/
/**
 * @brief Check if a device can receive the GPS satellite system
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::mbgextio_dev_is_gps
 * @see ::mbg_chk_dev_is_gps
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_is_gps( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_IS_GPS, &p_xdf->receiver_info );

}  // xdevfeat_is_gps

XDEVFEAT_CHK_SUPP_FNC xdevfeat_is_gps;



/*HDR*/
/**
 * @brief Check if a device supports the GNSS API
 *
 * This is usually supported by devices which can receive signals
 * from different satellite systems, e.g. GPS, GLONASS, ...
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::mbgextio_dev_is_gnss
 * @see ::mbg_chk_dev_is_gnss
 * @see ::MBG_GNSS_TYPES
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_is_gnss( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_IS_GNSS, &p_xdf->receiver_info );

}  // xdevfeat_is_gnss

XDEVFEAT_CHK_SUPP_FNC xdevfeat_is_gnss;



/*HDR*/
/**
 * @brief Check if a device is a time code receiver (IRIG or similar)
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::mbgextio_dev_is_tcr  //### TODO
 * @see ::mbg_chk_dev_is_tcr   //### TODO
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_is_tcr( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_IS_TCR, &p_xdf->receiver_info );

}  // xdevfeat_is_tcr

XDEVFEAT_CHK_SUPP_FNC xdevfeat_is_tcr;



/*HDR*/
/**
 * @brief Check if a device is a DCF77 AM receiver
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::mbgextio_dev_is_dcf  //### TODO
 * @see ::mbg_chk_dev_is_dcf
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_is_dcf( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_IS_DCF_AM, &p_xdf->receiver_info );

}  // xdevfeat_is_dcf_am

XDEVFEAT_CHK_SUPP_FNC xdevfeat_is_dcf;



/*HDR*/
/**
 * @brief Check if a device can receive DCF77 PZF
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::mbgextio_dev_has_pzf  //### TODO
 * @see ::mbg_chk_dev_has_pzf
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_pzf( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_IS_DCF_PZF, &p_xdf->receiver_info );

}  // xdevfeat_has_pzf

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_pzf;



/*HDR*/
/**
 * @brief Check if a device is an MSF receiver
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_is_msf( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_IS_MSF, &p_xdf->receiver_info );

}  // xdevfeat_is_msf

XDEVFEAT_CHK_SUPP_FNC xdevfeat_is_msf;



/*HDR*/
/**
 * @brief Check if a device is a JJY receiver
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_is_jjy( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_IS_JJY, &p_xdf->receiver_info );

}  // xdevfeat_is_jjy

XDEVFEAT_CHK_SUPP_FNC xdevfeat_is_jjy;



/*HDR*/
/**
 * @brief Check if a device is a WWVB receiver
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_is_wwvb( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_IS_WWVB, &p_xdf->receiver_info );

}  // xdevfeat_is_wwvb

XDEVFEAT_CHK_SUPP_FNC xdevfeat_is_wwvb;



/*HDR*/
/**
 * @brief Check if a device is a bus level device
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_is_bus_lvl_dev( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_IS_BUS_LVL_DEV, &p_xdf->receiver_info );

}  // xdevfeat_is_bus_lvl_dev

XDEVFEAT_CHK_SUPP_FNC xdevfeat_is_bus_lvl_dev;



/*HDR*/
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_ims( const MBG_XDEV_FEATURES *p_xdf )
{
  if ( mbg_rc_is_success( check_ri_feature( GPS_HAS_IMS, &p_xdf->receiver_info ) ) )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // xdevfeat_has_ims

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_ims;



/*HDR*/
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_gpio( const MBG_XDEV_FEATURES *p_xdf )
{
  if ( mbg_rc_is_success( check_ri_feature( GPS_HAS_GPIO, &p_xdf->receiver_info ) ) )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // xdevfeat_has_gpio

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_gpio;



/*HDR*/
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_synth( const MBG_XDEV_FEATURES *p_xdf )
{
  if ( mbg_rc_is_success( check_ri_feature( GPS_HAS_SYNTH, &p_xdf->receiver_info ) ) )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // xdevfeat_has_synth

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_synth;



/*HDR*/
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_prog_pulses( const MBG_XDEV_FEATURES *p_xdf )
{
  const RECEIVER_INFO *ri = &p_xdf->receiver_info;

  if ( ri->n_prg_out > 0 )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // xdevfeat_has_prog_pulses

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_prog_pulses;



/*HDR*/
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_irig_tx( const MBG_XDEV_FEATURES *p_xdf )
{
  if ( mbg_rc_is_success( check_ri_feature( GPS_HAS_IRIG_TX, &p_xdf->receiver_info ) ) )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // xdevfeat_has_irig_tx

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_irig_tx;



/*HDR*/
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_irig_rx( const MBG_XDEV_FEATURES *p_xdf )
{
  if ( mbg_rc_is_success( check_ri_feature( GPS_HAS_IRIG_RX, &p_xdf->receiver_info ) ) )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // xdevfeat_has_irig_rx

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_irig_rx;



/*HDR*/
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_serouts( const MBG_XDEV_FEATURES *p_xdf )
{
  const RECEIVER_INFO *ri = &p_xdf->receiver_info;

  if ( ri->n_com_ports > 0 )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // xdevfeat_has_serouts

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_serouts;



/*HDR*/
/**
 * @brief Check if a device supports the ::BVAR_STAT structure and API
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_bvar_stat( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_HAS_BVAR_STAT, &p_xdf->receiver_info );

}  // xdevfeat_has_bvar_stat

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_bvar_stat;



/*HDR*/
/**
 * @brief Check if a device supports reading the position as ::XYZ array
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_pos_xyz( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_HAS_POS_XYZ, &p_xdf->receiver_info );

}  // xdevfeat_has_pos_xyz

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_pos_xyz;



/*HDR*/
/**
 * @brief Check if a device supports reading the position as ::LLA array
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_pos_lla( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_HAS_POS_LLA, &p_xdf->receiver_info );

}  // xdevfeat_has_pos_lla

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_pos_lla;



/*HDR*/
/**
 * @brief Check if the device supports the builtin feature TIME
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_time_ttm( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_HAS_TIME_TTM, &p_xdf->receiver_info );

} // xdevfeat_has_time_ttm



/*HDR*/
/**
 * @brief Check if a device supports the ::MBG_TIME_SCALE_INFO structure and API
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_ri_feature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::mbgextio_get_time_scale_info
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_time_scale( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_ri_feature( GPS_HAS_TIME_SCALE, &p_xdf->receiver_info );

}  // xdevfeat_has_time_scale

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_time_scale;



/*HDR*/
/**
 * @brief Check if a device supports the ::TZDL structure and API
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_tzdl( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_HAS_TZDL, &p_xdf->receiver_info );

}  // xdevfeat_has_tzdl

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_tzdl;



/*HDR*/
/**
 * @brief Check if a device supports the ::TZCODE API
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_tzcode( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_HAS_TZCODE, &p_xdf->receiver_info );

} // xdevfeat_has_tzcode

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_tzcode;



/*HDR*/
/**
 * @brief Check if a device supports the ::ANT_INFO structure and API
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_ant_info( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_HAS_ANT_INFO, &p_xdf->receiver_info );

}  // xdevfeat_has_ant_info

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_ant_info;



/*HDR*/
/**
 * @brief Check if a device supports the ::ENABLE_FLAGS structure and API
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_enable_flags( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_HAS_ENABLE_FLAGS, &p_xdf->receiver_info );

}  // xdevfeat_has_enable_flags

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_enable_flags;



/*HDR*/
/**
 * @brief Check if a device supports the ::STAT_INFO structure and API
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_gps_stat_info( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_HAS_STAT_INFO, &p_xdf->receiver_info );

}  // xdevfeat_has_gps_stat_info

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_gps_stat_info;



/*HDR*/
/**
 * @brief Check if a device supports the ::ANT_CABLE_LEN structure and API
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_ant_cable_length( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_HAS_ANT_CABLE_LEN, &p_xdf->receiver_info );

}  // xdevfeat_has_ant_cable_length

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_ant_cable_length;



/*HDR*/
/**
 * @brief Check if a device supports the ::IGNORE_LOCK structure and API
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_gps_ignore_lock( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_ri_feature( GPS_HAS_IGNORE_LOCK, &p_xdf->receiver_info );

}  // xdevfeat_has_gps_ignore_lock

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_gps_ignore_lock;



#if 0 //### TODO

 /*HDR*/
/**
 * @brief Check if a device supports the :: structure and API
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_HAS_, &p_xdf->receiver_info );

}  // xdevfeat_has_

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_;



 /*HDR*/
/**
 * @brief Check if a device supports the :: structure and API
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_HAS_, &p_xdf->receiver_info );

}  // xdevfeat_has_

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_;

#endif  // ###


/*HDR*/
/**
 * @brief Check if the device supports the SCU_STAT structures
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 * @see @ref group_scu
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_scu_stat( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_HAS_SCU_STAT, &p_xdf->receiver_info );

} // xdevfeat_has_scu_stat

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_scu_stat;


/*HDR*/
/**
 * @brief Check if the device supports the SV_INFO structures
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_sv_info( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_builtin_feature( GPS_MODEL_HAS_SV_INFO, &p_xdf->receiver_info );

} // xdevfeat_has_sv_info

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_sv_info;


/*HDR*/
/**
 * @brief Check if a timecode receiver provides ::MBG_RAW_IRIG_DATA
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_ri_feature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::mbgextio_get_raw_irig_data
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_raw_irig_data( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_ri_feature( GPS_HAS_RAW_IRIG_DATA, &p_xdf->receiver_info );

}  // xdevfeat_has_raw_irig_data

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_raw_irig_data;



/*HDR*/
/**
 * @brief Check if a device supports the old LAN_IP4 API
 *
 * The LAN_IP4 API provides structures and functions to configure
 * parts of the networking of a device and is superseded by the
 * NET_CFG API. Some devices combine NET_CFG and LAN_IP4.
 * Therefore, ::mbgextio_get_all_net_cfg_info should be used
 * preferably to read the network configuration.
 * It will translate the old structures into the new ones.
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_ri_feature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::mbgextio_get_all_net_cfg_info
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_lan_ip4( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_ri_feature( GPS_HAS_LAN_IP4, &p_xdf->receiver_info );

}  // xdevfeat_has_lan_ip4

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_lan_ip4;



/*HDR*/
/**
 * @brief Check if a device supports the new NET_CFG API
 *
 * The NET_CFG API provides structures and functions to configure
 * the complete networking part of a device and supersedes the
 * LAN_IP4 API. Not all devices support the whole feature set
 * of the NET_CFG API or combine NET_CFG and LAN_IP4.
 * Therefore, ::mbgextio_get_all_net_cfg_info should be used
 * preferably to read the network configuration.
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_ri_feature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::mbgextio_get_all_net_cfg_info
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_net_cfg( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_ri_feature( GPS_HAS_NET_CFG, &p_xdf->receiver_info );

}  // xdevfeat_has_net_cfg

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_net_cfg;



/*HDR*/
/**
 * @brief Check if a device supports the PTP API
 *
 * The PTP API consists of different calls and associated structures
 * which * have evolved over time. Not all devices support every call,
 * so ::mbgextio_get_all_ptp_cfg_info takes care to check which parts are
 * supported and thus should be used preferably to read PTP information.
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_ri_feature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::mbgextio_get_all_ptp_cfg_info
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_ptp( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_ri_feature( GPS_HAS_PTP, &p_xdf->receiver_info );

}  // xdevfeat_has_ptp

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_ptp;


/*HDR*/
/**
 * @brief Check if a device supports the NTP API
 *
 * The NTP API consists of different calls and associated structures
 * which have evolved over time. Not all devices support every call,
 * so ::mbgextio_get_all_ntp_cfg_info takes care to check which parts are
 * supported and thus should be used preferably to read NTP information.
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_ri_feature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::mbgextio_get_all_ntp_cfg_info
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_ntp( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_ri_feature( GPS_HAS_NTP, &p_xdf->receiver_info );

} // xdevfeat_has_ntp

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_ntp;



/*HDR*/
/**
 * @brief Check if a device supports the event log API
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_ri_feature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_evt_log( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_ri_feature( GPS_HAS_EVT_LOG, &p_xdf->receiver_info );

} // xdevfeat_has_evt_log

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_evt_log;


/*HDR*/
/**
 * @brief Check if a device supports the user capture API
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_ri_feature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_ucap( const MBG_XDEV_FEATURES *p_xdf )
{
  const RECEIVER_INFO *ri = &p_xdf->receiver_info;

  if ( ri->n_ucaps > 0 )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

} // xdevfeat_has_ucap

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_ucap;


/*HDR*/
/**
 * @brief Check if a device supports the user capture via network feature
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_xfeature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref group_ucap_net
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_ucap_net( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_xfeature( MBG_XFEATURE_UCAP_NET, &p_xdf->xfeature_buffer );

} // xdevfeat_has_ucap_net

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_ucap_net;



/*HDR*/
/**
 * @brief Check if a device supports the TLV API
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_xfeature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref group_tlv_api
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_tlv_api( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_xfeature( MBG_XFEATURE_TLV_API, &p_xdf->xfeature_buffer );

}  // xdevfeat_has_ptp

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_tlv_api;



/*HDR*/
/**
 * @brief Check if a device supports a firmware update via TLV
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_tlv_feat_supp
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::mbgextio_xmt_fw_update
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_supp_tlv_fw_update( const MBG_XDEV_FEATURES *p_xdf )
{
#if defined( _PRELIMINARY_CODE )
  return check_tlv_feat_supp( MBG_TLV_FEAT_TYPE_FW_UPDATE, &p_xdf->tlv_info );
#else
  return MBG_ERR_NOT_SUPP_BY_DEV;
#endif

}  // xdevfeat_supp_tlv_fw_update

XDEVFEAT_CHK_SUPP_FNC xdevfeat_supp_tlv_fw_update;



/*HDR*/
/**
 * @brief Check if a device supports creating / sending a diagnostics file via TLV
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_tlv_feat_supp
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::TODO  //refer to get diag function
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_supp_tlv_diag_file( const MBG_XDEV_FEATURES *p_xdf )
{
#if defined( _PRELIMINARY_CODE )
  return check_tlv_feat_supp( MBG_TLV_FEAT_TYPE_DIAG_FILE, &p_xdf->tlv_info );
#else
  return MBG_ERR_NOT_SUPP_BY_DEV;
#endif

}  // xdevfeat_supp_tlv_diag_file

XDEVFEAT_CHK_SUPP_FNC xdevfeat_supp_tlv_diag_file;



/*HDR*/
/**
 * @brief Check if a device supports PTPv2 license infos
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_tlv_feat_supp
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_supp_tlv_ptpv2_license( const MBG_XDEV_FEATURES *p_xdf )
{
#if defined( _PRELIMINARY_CODE )
  return check_tlv_feat_supp( MBG_TLV_FEAT_TYPE_LICENSE_PTPV2_IDX, &p_xdf->tlv_info );
#else
  return MBG_ERR_NOT_SUPP_BY_DEV;
#endif

}  // xdevfeat_supp_tlv_ptpv2_license

XDEVFEAT_CHK_SUPP_FNC xdevfeat_supp_tlv_ptpv2_license;



/*HDR*/
/**
 * @brief Check if a device supports NTP license infos via TLV
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_tlv_feat_supp
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_supp_tlv_ntp_license( const MBG_XDEV_FEATURES *p_xdf )
{
#if defined( _PRELIMINARY_CODE )
  return check_tlv_feat_supp( MBG_TLV_FEAT_TYPE_LICENSE_NTP_IDX, &p_xdf->tlv_info );
#else
  return MBG_ERR_NOT_SUPP_BY_DEV;
#endif

}  // xdevfeat_supp_tlv_ntp_license

XDEVFEAT_CHK_SUPP_FNC xdevfeat_supp_tlv_ntp_license;



/*HDR*/
/**
 * @brief Check if a device supports PTPv1 License Infos via TLV
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_tlv_feat_supp
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_supp_tlv_ptpv1_license( const MBG_XDEV_FEATURES *p_xdf )
{
#if defined( _PRELIMINARY_CODE )
  return check_tlv_feat_supp( MBG_TLV_FEAT_TYPE_LICENSE_PTPV1_IDX, &p_xdf->tlv_info );
#else
  return MBG_ERR_NOT_SUPP_BY_DEV;
#endif

}  // xdevfeat_supp_tlv_ptpv1_license

XDEVFEAT_CHK_SUPP_FNC xdevfeat_supp_tlv_ptpv1_license;



/*HDR*/
/**
 * @brief Check if a device supports Time Monitor License infos via TLV
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_tlv_feat_supp
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_supp_tlv_time_monitor_license( const MBG_XDEV_FEATURES *p_xdf )
{
#if defined( _PRELIMINARY_CODE )
  return check_tlv_feat_supp( MBG_TLV_FEAT_TYPE_LICENSE_TIME_MONITOR_IDX, &p_xdf->tlv_info );
#else
  return MBG_ERR_NOT_SUPP_BY_DEV;
#endif

}  // xdevfeat_supp_tlv_time_monitor_license

XDEVFEAT_CHK_SUPP_FNC xdevfeat_supp_tlv_time_monitor_license;



/*HDR*/
/**
 * @brief Check if a device supports the ::GPS_SAVE_CFG command
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_xfeature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::mbgextio_cmd_save_cfg
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_cmd_save_cfg( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_xfeature( MBG_XFEATURE_SAVE_CFG, &p_xdf->xfeature_buffer );

}  // xdevfeat_has_cmd_save_cfg

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_cmd_save_cfg;



/*HDR*/
/**
 * @brief Check if a device supports the extended feature monitoring
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_xfeature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_monitoring( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_xfeature( MBG_XFEATURE_MONITORING, &p_xdf->xfeature_buffer );

}  // xdevfeat_has_monitoring

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_monitoring;



/*HDR*/
/**
 * @brief Check if a device supports the LED API
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_xfeature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::TODO ###
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_led_api( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_xfeature( MBG_XFEATURE_LED_API, &p_xdf->xfeature_buffer );

}  // xdevfeat_has_led_api

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_led_api;



/*HDR*/
/**
 * @brief Check if a device supports the LNE API
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_xfeature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::TODO ###
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_lne_api( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_xfeature( MBG_XFEATURE_LNE_API, &p_xdf->xfeature_buffer );

}  // xdevfeat_has_lne_api

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_lne_api;



/*HDR*/
/**
 * @brief Check if a device supports the power control API
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_xfeature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::TODO ###
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_pwr_ctl_api( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_xfeature( MBG_XFEATURE_PWR_CTL_API, &p_xdf->xfeature_buffer );

}  // xdevfeat_has_pwr_ctl_api

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_pwr_ctl_api;



/*HDR*/
/**
 * @brief Check if a device supports the ::MBG_EXT_SYS_INFO command
 *
 * @param[in,out] p_xdf  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_xfeature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::TODO ###
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_ext_sys_info( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_xfeature( MBG_XFEATURE_EXT_SYS_INFO, &p_xdf->xfeature_buffer );

}  // xdevfeat_has_ext_sys_info

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_ext_sys_info;



/*HDR*/
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_io_ports( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_xfeature( MBG_XFEATURE_IO_PORTS, &p_xdf->xfeature_buffer );

}  // xdevfeat_has_io_ports

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_io_ports;



/*HDR*/
/**
 * @brief Check if a device has ::MBG_XFEATURE_TRANSACTIONS
 *
 * @param[in,out] p_xdf  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_xfeature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::TODO ###
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_transactions( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_xfeature( MBG_XFEATURE_TRANSACTIONS, &p_xdf->xfeature_buffer );

}  // xdevfeat_has_transactions

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_transactions;



/*HDR*/
/**
 * @brief Check if a device has ::MBG_XFEATURE_REBOOT
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_xfeature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_reboot( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_xfeature( MBG_XFEATURE_REBOOT, &p_xdf->xfeature_buffer );

}  // xdevfeat_has_reboot

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_reboot;



/*HDR*/
/**
 * @brief Check if a device has ::MBG_XFEATURE_REQ_TTM
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_xfeature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 * @see mbgextio_get_time
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_req_ttm( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_xfeature( MBG_XFEATURE_REQ_TTM, &p_xdf->xfeature_buffer );

} // xdevfeat_has_req_ttm

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_req_ttm;



/*HDR*/
/**
 * @brief Check if a device supports the extended multi ref features including multi instances
 *
 * The different multi ref feature and its appropriate flags have evolved over time.
 * This function only checks the currently up-to-date GPS_HAS_XMRS_MULT_INSTC flag.
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_ri_feature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see ::mbgextio_get_all_xmulti_ref_info
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_xmulti_ref( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_ri_feature( GPS_HAS_XMRS_MULT_INSTC, &p_xdf->receiver_info );

}  // xdevfeat_has_xmulti_ref

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_xmulti_ref;



/*HDR*/
/**
 * @brief Check if a device supports the extended binary protocol (XBP) feature
 *
 * @param[in,out] p_xdf  Pointer to a ::MBG_XDEV_FEATURES buffer associated with the device
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_ri_feature
 *
 * @ingroup xdevfeat_chk_supp_fncs
 * @see @ref xdevfeat_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _NO_MBG_API xdevfeat_has_xbp( const MBG_XDEV_FEATURES *p_xdf )
{
  return check_ri_feature( GPS_HAS_XBP, &p_xdf->receiver_info );

}  // xdevfeat_has_xbp

XDEVFEAT_CHK_SUPP_FNC xdevfeat_has_xbp;
