
/**************************************************************************
 *
 *  $Id: str_util.h 1.3 2016/12/14 16:22:24 martin TEST $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    Definitions and prototypes for str_util.c
 *
 * -----------------------------------------------------------------------
 *  $Log: str_util.h $
 *  Revision 1.3  2016/12/14 16:22:24  martin
 *  Added macro _sn_cpy_str_safe() to simplify calls.
 *  Revision 1.2  2016/08/05 12:33:17  martin
 *  Moved string trim functions from cfg_util module here.
 *  Added variable str_not_avail.
 *  Fixed some spelling.
 *  Updated function prototypes.
 *  Revision 1.1  2015/08/25 15:57:43  martin
 *  Initial revision.
 *
 **************************************************************************/

#ifndef _STR_UTIL_H
#define _STR_UTIL_H

/* Other headers to be included */

#include <words.h>  // implicitly includes mbg_tgt.h for non-firmware projects

#include <stdlib.h>
#include <stdarg.h>


#ifdef _STR_UTIL
 #define _ext
 #define _DO_INIT
#else
 #define _ext extern
#endif


/* Start of header body */

#ifdef __cplusplus
extern "C" {
#endif


_ext const char *str_not_avail
#ifdef _DO_INIT
  = "N/A"
#endif
;

#define _sn_cpy_str_safe( _dst, _src )  sn_cpy_str_safe( _dst, sizeof( _dst ), _src )


/* function prototypes: */

/* ----- function prototypes begin ----- */

/* This section was generated automatically */
/* by MAKEHDR, do not remove the comments. */

 /**
 * @brief A portable, safe implementation of vsnprintf()
 *
 * Unfortunately the behaviour of vsnprintf() and thus snprintf()
 * differs in detail across various build environments and run time
 * libraries.
 *
 * If the output exceeds the buffer size and thus is truncated then:<br>
 *
 * - Under Windows a negative value is returned and eventually *no*
 *   terminating 0 is written to the output buffer, so the output string
 *   may not be terminated properly.
 *
 * - Some versions of glibc return the number of bytes that *would*
 *   have been written to the buffer *if* the buffer would have been
 *   large enough, instead of the true number of characters that have
 *   been written to the buffer.
 *
 * So subsequent calls like
 *
 *   n = snprintf( s, max_len, ... );
 *   n += snprintf( &s[n], max_len - n, ... );
 *
 * may always work properly, or fail with buffer overruns or stack
 * corruption depending on the build environment.
 * This wrapper function takes care that strings are always terminated
 * properly, and that the returned value always matches the number of
 * characters really written to the string buffer, excluding the
 * terminating 0
 *
 * @note The "size_t" type parameter used to specify the buffer size
 * can be larger (e.g. "unsigned long") than the "int" type returned
 * by mostly all functions of the printf() family. So if a very large
 * buffer is specified, and a large number of characters (more than
 * MAXINT) are written to that buffer then how can an "int" type
 * return the large number of characters written to the buffer?
 * We also try to workaround this here.
 *
 * @param[out] s        The string buffer to be filled
 * @param[in]  max_len  Size of the output buffer for 0-terminated string
 * @param[in]  fmt      Format string according to subsequent parameters
 * @param[in]  ap       Variable argument list in va_list format
 *
 * @return Number of characters written to the output buffer, except the terminating 0
 *
 * @see ::snprintf_safe
 * @see ::strncpy_safe
 * @see ::sn_cpy_str_safe
 * @see ::sn_cpy_char_safe
 */
 size_t __attribute__( ( format( printf, 3, 0 ) ) ) vsnprintf_safe( char *s, size_t max_len, const char *fmt, va_list ap ) ;

 /**
 * @brief A portable, safe implementation of snprintf()
 *
 * For a detailed description see ::vsnprintf_safe
 *
 * @param[out] s        The string buffer to be filled
 * @param[in]  max_len  Size of the output buffer for 0-terminated string
 * @param[in]  fmt      Format string according to subsequent parameters
 * @param[in]  ...      Variable argument list according to the format string
 *
 * @return Number of characters written to the output buffer, except the terminating 0
 *
 * @see ::vsnprintf_safe
 * @see ::strncpy_safe
 * @see ::sn_cpy_str_safe
 * @see ::sn_cpy_char_safe
 */
 size_t __attribute__( ( format( printf, 3, 4 ) ) ) snprintf_safe( char *s, size_t max_len, const char * fmt, ... ) ;

 /**
 * @brief A portable, safe implementation of strncpy()
 *
 * In the original implementation of strncpy(), if the length of the
 * string to be copied into the destination buffer exceeds the specified
 * buffer length then the string in the output buffer is not 0-terminated.
 *
 * Our implementation always forces a proper termination by 0, but unlike
 * the original implementation of strncpy() it does *not* fill the whole
 * remaining buffer space with '0' characters.
 *
 * @param[out] dst      Pointer to the output buffer
 * @param[in]  src      Pointer to the input buffer
 * @param[in]  max_len  Size of the output buffer for 0-terminated string
 *
 * @return Pointer to the destination buffer
 *
 * @see ::vsnprintf_safe
 * @see ::snprintf_safe
 * @see ::sn_cpy_str_safe
 * @see ::sn_cpy_char_safe
 */
 char *strncpy_safe( char *dst, const char *src, size_t max_len ) ;

 /**
 * @brief A function to copy a string safely, returning the number of characters copied
 *
 * This basically works like ::strncpy_safe but instead of a pointer to
 * the destination buffer it returns the number of characters copied
 * to the destination buffer.
 *
 * @param[out] dst      Pointer to the output buffer
 * @param[in]  max_len  Size of the output buffer for 0-terminated string
 * @param[in]  src      Pointer to the input buffer
 *
 * @return Number of characters copied to the destination buffer
 *
 * @see ::vsnprintf_safe
 * @see ::snprintf_safe
 * @see ::strncpy_safe
 * @see ::sn_cpy_char_safe
 */
 size_t sn_cpy_str_safe( char *dst, size_t max_len, const char *src ) ;

 /**
 * @brief A function to copy a character safely to a string buffer
 *
 * This basically works like ::sn_cpy_str_safe but expects a character
 * to be copied to the destination buffer. Appends a terminating 0 to
 * the string buffer and returns the number of characters copied to
 * the destination buffer, usually 0 or 1.
 *
 * @param[out] dst      Pointer to the output buffer
 * @param[in]  max_len  Size of the output buffer for 0-terminated string
 * @param[in]  c        Character to be copied to the destination buffer
 *
 * @return Number of characters copied to the destination buffer, without the terminating 0
 *
 * @see ::vsnprintf_safe
 * @see ::snprintf_safe
 * @see ::strncpy_safe
 * @see ::sn_cpy_str_safe
 */
 size_t sn_cpy_char_safe( char *dst, size_t max_len, char c ) ;

 /**
 * @brief Trim whitespace at the end of a string
 *
 * @param[in,out] s  The string to be trimmed
 */
 void trim_trailing_whitespace( char *s ) ;

 /**
 * @brief Trim whitespace at the beginning of a string
 *
 * @param[in,out] s  The string to be trimmed
 */
 void trim_leading_whitespace( char *s ) ;

 /**
 * @brief Trim both leading and trailing whitespace from a string
 *
 * @param[in,out] s  The string to be trimmed
 */
 void trim_whitespace( char *s ) ;

 /**
 * @brief Copy array of bytes starting at beginning of buffer
 *
 * Can be used if the destination address is in the same buffer
 * in front of the source address. Even though you would expect
 * that memcpy() would also work for this properly, we have seen
 * cases where it didn't, and only memmove() worked correctly.
 * Anyway, we try to avoid the overhead of memmove().
 *
 * @param[out] dst      Destination address behind the source address
 * @param[in]  src      Source address
 * @param[in]  n_bytes  Number of bytes to copy
 *
 * @see ::mbg_memcpy_reversed
 */
 void mbg_memcpy( void *dst, const void *src, size_t n_bytes ) ;

 /**
 * @brief Copy an array of bytes in reversed order, starting at end of buffer
 *
 * Can be used if the destination address is in the same buffer
 * behind the source address, so the source address would be
 * overwritten by a normal memcpy().
 *
 * @param[out] dst      Destination address behind the source address
 * @param[in]  src      Source address
 * @param[in]  n_bytes  Number of bytes to copy
 *
 * @see ::mbg_memcpy
 */
 void mbg_memcpy_reversed( void *dst, const void *src, size_t n_bytes ) ;


/* ----- function prototypes end ----- */

#ifdef __cplusplus
}
#endif


/* End of header body */

#undef _ext
#undef _DO_INIT

#endif  /* _STR_UTIL_H */
