
/**************************************************************************
 *
 *  $Id: mbgextio.c 1.20.1.33.1.160 2017/04/12 08:34:53 martin TEST $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    Meinberg extended I/O functions for the binary data protocol
 *    via serial communication, network socket I/O, or direct USB I/O.
 *
 *    These functions can *not* be used to access LANTIME NTP servers,
 *    or the modules assembled within a LANTIME since LANTIMEs are using
 *    this kind of communication only internally.
 *
 *    Also, standalone USB devices are usually handled by the driver
 *    software package for the given operating system and thus can be
 *    accessed in the same way as PCI cards, using the API functions
 *    provided by the mbgdevio library.
 *
 *    These functions can be used, however, with standalone devices
 *    which are accessible either directly via a serial port, or via a
 *    special serial-to-LAN converter like the meinberg LAN_XPT module.
 *
 * -----------------------------------------------------------------------
 *  $Log: mbgextio.c $
 *  Revision 1.20.1.33.1.160  2017/04/12 08:34:53  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.159  2017/04/12 07:34:20  martin
 *  Fixed build with VC6.
 *  Revision 1.20.1.33.1.158  2017/04/11 14:28:26Z  martin
 *  Revision 1.20.1.33.1.157  2017/04/11 14:12:49Z  martin
 *  Fixed DOS build.
 *  Revision 1.20.1.33.1.156  2017/04/11 13:19:41Z  martin
 *  Made mbgextio_setup_receiver_info() more fault tolerant.
 *  Use function call to retrieve RECEIVER_INFO address
 *  from message control block.
 *  Revision 1.20.1.33.1.155  2017/04/10 13:41:55Z  martin
 *  Fixed some compiler warnings.
 *  Revision 1.20.1.33.1.154  2017/04/10 07:49:07Z  martin
 *  Fixed a compiler warning.
 *  Revision 1.20.1.33.1.153  2017/04/05 16:06:38  martin
 *  Added mbgextio_dev_has_bvar_stat().
 *  Revision 1.20.1.33.1.152  2017/03/27 10:38:40  thomas-b
 *  Added functions to read and write PTPv1 common datasets
 *  Revision 1.20.1.33.1.151  2017/03/20 17:11:07  martin
 *  Replaced dummy swab...() macro calls by real macros.
 *  Revision 1.20.1.33.1.150  2017/03/20 10:10:03  martin
 *  Fixed build without _PRELIMINARY_CODE.
 *  Revision 1.20.1.33.1.149  2017/03/07 13:45:55  thomas-b
 *  Added functions to get and set ignore lock
 *  Revision 1.20.1.33.1.148  2017/03/03 06:46:04  thomas-b
 *  Added functions to read/write NTP key, server, refclock and orphan mode settings
 *  Revision 1.20.1.33.1.147  2017/02/23 14:32:08  thomas-b
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.146  2017/02/22 08:31:13  thomas-b
 *  Added functions to read/write event info and read monitoring status and event status
 *  Revision 1.20.1.33.1.145  2017/02/16 13:01:52  thomas-b
 *  Changed get and set functions for PTPv2 port dataset to use idx structure
 *  Revision 1.20.1.33.1.144  2017/02/16 09:27:59  thomas-b
 *  Fixed doxygen docu
 *  Revision 1.20.1.33.1.143  2017/02/16 09:24:32  thomas-b
 *  Added functions to get and set PTPv2 common datasets
 *  Revision 1.20.1.33.1.142  2017/02/15 16:17:37  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.141  2017/02/08 13:12:36  thomas-b
 *  Added functions to get and set SNMP configuration
 *  Revision 1.20.1.33.1.140  2017/02/07 14:24:52  thomas-b
 *  Added function mbgextio_dev_has_monitoring, which checks if the extended feature MBG_XFEATURE_MONITORING is set
 *  Revision 1.20.1.33.1.139  2017/02/07 14:10:19  daniel
 *  Added calls to check for new License TLV feature types
 *  Revision 1.20.1.33.1.138  2016/12/07 09:33:58  martin
 *  Renamed mbgextio_get_utc_param() to mbgextio_get_utc_parm()
 *  and mbgextio_set_utc_param() to mbgextio_set_utc_parm()
 *  for consisten naming with related functions.
 *  Check if index of received data set matches requested index, and
 *  keep receiving if not. Only in mbgextio_get_xmr_status_idx() for now.
 *  Revision 1.20.1.33.1.137  2016/12/01 10:17:27  thomas-b
 *  Temporarily removed ACK request from GPS_SCU_STAT set command
 *  Revision 1.20.1.33.1.136  2016/11/30 16:12:54  thomas-b
 *  Added functions to get/set SCU_STAT_INFO/SCU_STAT_SETTINGS
 *  Revision 1.20.1.33.1.135  2016/11/24 07:06:34  philipp
 *  Honour big endian system when swapping bytes
 *  Revision 1.20.1.33.1.134  2016/11/22 10:33:15  philipp
 *  Implemented I/O ports
 *  Revision 1.20.1.33.1.133  2016/11/22 10:28:04  gregoire.diehl
 *  chk index added in mbgextio_get_xmr_status_idx
 *  Revision 1.20.1.33.1.132  2016/11/15 15:44:12Z  martin
 *  Account for modified mbgserio functions.
 *  Revision 1.20.1.33.1.131  2016/11/02 12:15:17  thomas-b
 *  Added function to check if a device has MBG_XFEATURE_REQ_TTM and adapted the documentation for mbgextio_get_time
 *  Revision 1.20.1.33.1.130  2016/11/01 07:39:02  thomas-b
 *  Fixed reception of XBP packets, copy std_msg_data from xbp_msg_data
 *  Revision 1.20.1.33.1.129  2016/10/31 09:28:06  martin
 *  Doxygen fixes.
 *  Revision 1.20.1.33.1.128  2016/10/25 09:11:17  thomas-b
 *  Added functions to get CFGH, ALM and IONO structures from a device
 *  Revision 1.20.1.33.1.127  2016/10/24 13:59:42  thomas-b
 *  Removed needless space from comment
 *  Fixed mbgextio_rcv_msg_unlocked for reception of XBP packets
 *  Revision 1.20.1.33.1.126  2016/10/20 14:46:03  thomas-b
 *  Added function to check if XBP is supported
 *  Revision 1.20.1.33.1.125  2016/10/20 11:29:10  martin
 *  Added missing const qualifier in mbgextio_get_utc_param().
 *  Revision 1.20.1.33.1.124  2016/10/20 10:43:34  thomas-b
 *  Added function to check if a device is a bus level device
 *  Revision 1.20.1.33.1.123  2016/10/19 14:33:43  thomas-b
 *  Added functions to read and write ucap via network configuration
 *  Revision 1.20.1.33.1.122  2016/10/14 11:09:40  thomas-b
 *  Added function to check MBG_XFEATURE_UCAP_NET
 *  Revision 1.20.1.33.1.121  2016/10/14 08:18:54  thomas-b
 *  Decreased poll timeout
 *  Revision 1.20.1.33.1.120  2016/10/12 08:13:27  thomas-b
 *  Fixed check_init_winsock to return MBG_SUCCESS if WSAStartup returns 0
 *  Revision 1.20.1.33.1.119  2016/09/30 05:36:53  thomas-b
 *  Increased serial poll timeout
 *  Revision 1.20.1.33.1.118  2016/09/29 14:35:12  thomas-b
 *  Moved new get and set functions for NET_CFG_API stage 2 to mbgextio
 *  Revision 1.20.1.33.1.117  2016/09/29 12:19:57  thomas-b
 *  Added function to check transactions feature
 *  Revision 1.20.1.33.1.116  2016/09/20 14:18:27  martin
 *  Fixed a compiler warning.
 *  Revision 1.20.1.33.1.115  2016/09/20 13:52:38  martin
 *  Moved GPS_REQACK preprocessor handling to the header file.
 *  Revision 1.20.1.33.1.114  2016/09/16 08:00:20  thomas-b
 *  Always try to use high speed baud in mbgextio_open_serial_force_default
 *  Revision 1.20.1.33.1.113  2016/09/15 10:15:12  philipp
 *  Fixed compiler warning (uninitialized variable may be used)
 *  Revision 1.20.1.33.1.112  2016/09/12 15:26:24  martin
 *  Conditionally try to retrieve serial number from very old devices.
 *  Revision 1.20.1.33.1.111  2016/09/12 13:42:35Z  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.110  2016/08/23 15:56:59  martin
 *  Set up default receiver info for very old devices
 *  which don't provide it.
 *  Revision 1.20.1.33.1.109  2016/08/23 11:02:59Z  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.108  2016/08/23 09:25:17  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.107  2016/08/23 08:33:25  udo
 *  renamed some Time Mon functions
 *  Revision 1.20.1.33.1.106  2016/08/22 14:14:47  gregoire.diehl
 *  new call: mbgextio_get_corr_info
 *  Revision 1.20.1.33.1.105  2016/08/15 09:36:23Z  udo
 *  added mbgextio_time_mon_get_target_ext_data_set_idx()
 *  Revision 1.20.1.33.1.104  2016/08/11 10:25:53  martin
 *  Started to support time_mon.
 *  Revision 1.20.1.33.1.103  2016/08/09 16:00:36  martin
 *  Removed obsolete string variable.
 *  Revision 1.20.1.33.1.102  2016/08/09 07:12:26  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.101  2016/08/01 10:32:53  daniel
 *  Add functions to get ptp statistics
 *  Revision 1.20.1.33.1.100  2016/07/07 15:04:11  martin
 *  Renamed functions due to renamed library symbols.
 *  Revision 1.20.1.33.1.99  2016/06/29 11:59:11  philipp
 *  Extended socket API by TCP client
 *  Revision 1.20.1.33.1.98  2016/06/15 10:15:49  thomas-b
 *  Added functions which check if the user capture feature is supported by a device
 *  Revision 1.20.1.33.1.97  2016/06/14 10:33:21  thomas-b
 *  Added functions which check if the event log feature is supported by a device
 *  Revision 1.20.1.33.1.96  2016/06/08 09:55:12  thomas-b
 *  Added function mbgextio_open_serial_force_default
 *  Revision 1.20.1.33.1.95  2016/06/07 14:50:33  udo
 *  Reset Meinberg USB device in case if the open_usb function return MBG_ERR_BUSY
 *  Revision 1.20.1.33.1.94  2016/06/07 07:43:06  philipp
 *  New function to check for Irig Rx feature
 *  Revision 1.20.1.33.1.93  2016/06/03 11:15:49  thomas-b
 *  Fixed Windows compatibility issues in get_sock_error and mbgextio_find_lan_devices
 *  Revision 1.20.1.33.1.92  2016/06/02 10:15:38  philipp
 *  Renaming all MBG_EXT_REV_INFO related stuff to MBG_EXT_SYS_INFO.
 *  Revision 1.20.1.33.1.91  2016/05/30 08:10:44  thomas-b
 *  Added functions to check several builtin features
 *  Revision 1.20.1.33.1.90  2016/05/27 05:18:17  philipp
 *  New functions mbgextio_get_xmr_ext_source_stats_idx and mbgextio_get_xmr_ext_source_metrics_idx
 *  Revision 1.20.1.33.1.89  2016/05/25 09:22:58  philipp
 *  New function mbgextio_get_xmr_ext_source_stats_idx
 *  Revision 1.20.1.33.1.88  2016/05/20 13:37:53  philipp
 *  New function mbgextio_set_gpio_settings_idx
 *  Revision 1.20.1.33.1.87  2016/05/20 09:41:33  thomas-b
 *  Removed functions which check for a specific IMS card
 *  Revision 1.20.1.33.1.86  2016/05/20 08:51:19  thomas-b
 *  Added functions mbgextio_dev_has_time_scale and mbgextio_dev_has_tzcode
 *  Revision 1.20.1.33.1.85  2016/05/20 07:51:32  thomas-b
 *  Added function mbgextio_dev_has_tzdl
 *  Revision 1.20.1.33.1.84  2016/05/20 06:42:06  thomas-b
 *  Added function mbgextio_dev_has_enable_flags
 *  Revision 1.20.1.33.1.83  2016/05/18 13:15:40  philipp
 *  Fixed wrong GPS command in mbgextio_cmd_save_cfg
 *  Revision 1.20.1.33.1.82  2016/05/17 13:29:35  philipp
 *  New function to check whether device is LNO
 *  Revision 1.20.1.33.1.81  2016/05/17 06:33:07  philipp
 *  New function to check whether a device has serial outputs
 *  Revision 1.20.1.33.1.80  2016/05/12 10:41:35  philipp
 *  New functions to check bpe card and IRIG Tx feature
 *  Revision 1.20.1.33.1.79  2016/05/10 06:15:26  philipp
 *  New function to check whether a device has programmable pulses
 *  Revision 1.20.1.33.1.78  2016/05/04 13:04:55  thomas-b
 *  Fixed double mutex acquisition in dns set functions (NULL pointer)
 *  Revision 1.20.1.33.1.77  2016/04/26 06:55:45  thomas-b
 *  Added functions to check if NET_CFG and LAN_IP4 APIs are supported
 *  Revision 1.20.1.33.1.76  2016/04/25 14:47:13  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.75  2016/04/22 10:53:58  philipp
 *  New function to check whether device is LIU
 *  Revision 1.20.1.33.1.74  2016/04/20 13:20:25  thomas-b
 *  Added function to check if a device supports NTP
 *  Revision 1.20.1.33.1.73  2016/04/20 09:26:11  philipp
 *  Moved all HPS-PTP related structures to gpspriv.h and removed related extended feature bit from gpsdefs.h.
 *  Also removed functions from mbgextio and xdevfeat since HPS-PTP handling needs a redesign concerning structures.
 *  Thus, handle everything explicitly for now!
 *  -> Redesing this A.S.A.P.!!!
 *  Revision 1.20.1.33.1.72  2016/04/15 08:17:25  philipp
 *  New feature MBG_XFEATURE_EXT_PTP
 *  Revision 1.20.1.33.1.71  2016/04/13 07:01:20  philipp
 *  New function to check whether device is HPS
 *  Revision 1.20.1.33.1.70  2016/04/12 13:27:40  philipp
 *  Several new functions to check for device models and device features
 *  Revision 1.20.1.33.1.69  2016/04/11 13:56:47  thomas-b
 *  Added function mbgextio_dev_has_xmulti_ref
 *  Revision 1.20.1.33.1.68  2016/04/07 14:08:33  philipp
 *  Added function mbgextio_get_ext_rev_info
 *  Revision 1.20.1.33.1.67  2016/04/07 12:45:43  martin
 *  New function mbgextio_dev_has_ext_rev_info().
 *  Revision 1.20.1.33.1.66  2016/03/24 14:08:50  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.65  2016/03/23 13:49:44  thomas-b
 *  Added case for overflow error after check_transfer
 *  Revision 1.20.1.33.1.64  2016/03/23 10:12:52  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.63  2016/03/23 09:48:51  thomas-b
 *  Stop mbgextio_rcv_msg_unlocked after poll_timeout expires
 *  Several changes in mbgextio_find_lan_devices
 *  Revision 1.20.1.33.1.62  2016/03/22 14:52:37  thomas-b
 *  Shutdown socket before close in mbgextio_find_lan_devices
 *  Revision 1.20.1.33.1.61  2016/03/22 14:24:26  thomas-b
 *  Several fixes in mbgextio_find_lan_devices
 *  Revision 1.20.1.33.1.60  2016/03/22 12:53:30  thomas-b
 *  Added functions to find and free Meinberg LAN devices
 *  Revision 1.20.1.33.1.59  2016/03/22 08:39:42  thomas-b
 *  Removed debug output
 *  Revision 1.20.1.33.1.58  2016/03/22 08:16:16  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.57  2016/03/21 13:27:33  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.56  2016/03/18 11:21:55  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.55  2016/03/18 10:48:32  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.54  2016/03/17 11:33:10  philipp
 *  Added XMR functions
 *  Revision 1.20.1.33.1.53  2016/03/16 15:08:50  martin
 *  Moved some low level functions to new module xtiocomm.c.
 *  Revision 1.20.1.33.1.52  2016/03/14 11:53:15  martin
 *  Moved some low level feature check functions to new
 *  module xdevfeat.c, and use those functions.
 *  Removed function mbgextio_get_xfeature_buffer().
 *  Revision 1.20.1.33.1.51  2016/03/11 11:48:18  thomas-b
 *  Check MBG_SUCCESS with macro mbg_rc_is_success
 *  Revision 1.20.1.33.1.50  2016/03/11 10:29:03  thomas-b
 *  Decreased default serial message timeout
 *  Read more than one byte in each call to serial read
 *  Increase timeout if header has been completely received
 *  Revision 1.20.1.33.1.49  2016/03/11 08:20:13  marvin
 *  Added function to save PTP unicast master settings.
 *  Revision 1.20.1.33.1.48  2016/03/02 16:31:40Z  marvin
 *  Do not close socket if opened successfully.
 *  Revision 1.20.1.33.1.47  2016/02/25 12:02:54Z  udo
 *  used mbgextio_req_data_idx instead of mbgextio_req_data for TIME_MON_TARGET_STATUS_IDX
 *  Revision 1.20.1.33.1.46  2016/02/17 15:03:16  udo
 *  used MBG_TIME_MON_TARGET_STATUS_IDX
 *  Revision 1.20.1.33.1.45  2016/02/15 08:28:29  gregoire
 *  typo fixed
 *  Revision 1.20.1.33.1.44  2016/02/15 08:23:51Z  gregoire
 *  mbgextio_get_xfeature_buffer_addr added
 *  Revision 1.20.1.33.1.43  2016/02/11 10:18:47Z  thomas-b
 *  Added function mbgextio_set_fdm_tdev
 *  Revision 1.20.1.33.1.42  2016/02/03 09:16:48  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.41  2016/02/02 15:58:42  martin
 *  Added mbgextio_get_raw_irig_data().
 *  Revision 1.20.1.33.1.40  2016/01/19 14:13:53  udo
 *  improve time monitor functions
 *  Revision 1.20.1.33.1.39  2016/01/18 08:53:34  udo
 *  added support for PTP Time Monitoring on TSU/HPS100
 *  Revision 1.20.1.33.1.38  2016/01/04 15:55:07  martin
 *  New function mbgextio_open_usb_ldev which takes a libusb_device as parameter.
 *  Revision 1.20.1.33.1.37  2015/12/10 16:30:16  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.36  2015/12/10 11:55:14  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.35  2015/12/10 11:10:39  martin
 *  Don't try to read RECEIVER_INFO etc. from legacy USB devices.
 *  Revision 1.20.1.33.1.34  2015/12/09 17:35:01  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.33  2015/12/09 10:45:34  martin
 *  Revision 1.20.1.33.1.32  2015/12/09 10:21:22Z  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.31  2015/12/07 16:34:54  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.30  2015/12/03 16:00:20  martin
 *  Also mbgextio_xmt_cmd() and mbgextio_xmt_cmd() now wait for ACK or NACK
 *  if the transmitted command code has been or'ed with GPS_REQACK.
 *  Doxygen stuff.
 *  Revision 1.20.1.33.1.29  2015/12/01 11:54:49  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.28  2015/12/01 11:36:01  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.27  2015/11/30 17:00:49  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.26  2015/11/30 08:16:14  philipp
 *  Added (wrapper) functions to test for TLV features
 *  Revision 1.20.1.33.1.25  2015/11/26 16:19:26  martin
 *  Reworked check feature API causing some other API calls to be
 *  simplified/changed since receiver info is now stored inside
 *  the message control block.
 *  Revision 1.20.1.33.1.24  2015/11/26 09:57:10  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.23  2015/11/23 12:55:25  philipp
 *  Removed TLV functions
 *  Revision 1.20.1.33.1.21  2015/11/20 10:47:12  daniel
 *  Removed debug messages
 *  Revision 1.20.1.33.1.20  2015/11/20 08:14:36  philipp
 *  Added id member to struct MBG_TLV_ANNOUNCE
 *  Revision 1.20.1.33.1.19  2015/11/20 07:31:19  philipp
 *  Added _mbg_swab_tlv macro in function mbgextio_send_tlv_chunk
 *  Revision 1.20.1.33.1.18  2015/11/20 07:27:37  philipp
 *  Adjusted functions and prototypes to match new TLV structures from gpsdefs.h
 *  Revision 1.20.1.33.1.17  2015/11/19 16:31:14  daniel
 *  Added debug messages
 *  Revision 1.20.1.33.1.16  2015/11/19 13:15:39  philipp
 *  Added TLV functionality
 *  Revision 1.20.1.33.1.15  2015/11/02 10:12:27  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.14  2015/10/19 16:42:20  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.13  2015/10/19 09:35:10  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.12  2015/10/19 09:16:44  martin
 *  Fixed some spelling.
 *  Revision 1.20.1.33.1.11  2015/10/15 14:00:57  martin
 *  Doxygen comments.
 *  Revision 1.20.1.33.1.10  2015/10/09 11:09:16  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.9  2015/10/06 14:19:06  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.8  2015/10/02 08:57:44  thomas-b
 *  Added several functions to retrieve and set parameter structures of the new FDM API
 *  Revision 1.20.1.33.1.7  2015/10/02 08:18:52  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.6  2015/10/01 09:15:56  philipp
 *  _USE_USB_DIRECT_IO: Return MBG_ERR_TIMEOUT and MBG_ERR_DISCONN if poll times out or holds POLLHUP event after reading data returns
 *  Revision 1.20.1.33.1.5  2015/09/15 13:25:55  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.4  2015/09/14 07:23:32  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.3  2015/09/11 12:06:16  martin
 *  *** empty log message ***
 *  Revision 1.20.1.33.1.2  2015/09/10 14:45:16  martin
 *  Unified timeout handling.
 *  If a NACK is received then accept and return an optional
 *  error code which may have been appended by the device.
 *  Revision 1.20.1.33.1.1  2015/09/09 16:00:08  martin
 *  Debug timing.
 *  Revision 1.20.1.33  2015/09/09 15:26:13  martin
 *  Revision 1.20.1.32  2015/09/09 08:41:01  martin
 *  Revision 1.20.1.31  2015/09/09 08:26:27  martin
 *  Revision 1.20.1.30  2015/09/09 06:57:51  daniel
 *  Revision 1.20.1.29  2015/09/08 15:25:48  martin
 *  Copy received data to destination buffer before
 *  device mutex is release.
 *  Revision 1.20.1.28  2015/09/08 14:22:32  martin
 *  Added some some missing mutex code.
 *  Use new inline functions mbg_rc_is_error() and mbg_rc_is_success().
 *  Revision 1.20.1.27  2015/09/08 12:15:02  martin
 *  Changed xmt_mutex to dev_mutex.
 *  Revision 1.20.1.26  2015/09/08 08:38:22  daniel
 *  Use mutex in mbgextio_req_data()
 *  Revision 1.20.1.25  2015/09/04 09:20:17  daniel
 *  Added fuctions to exchange refclock state and tsu version with TSU
 *  Revision 1.20.1.24  2015/09/02 16:42:20  martin
 *  Preliminary code which is only included if a preprocessor symbol
 *  _PRELIMINARY_CODE is defined in the project Makefile.
 *  Revision 1.20.1.23  2015/08/27 16:21:30  martin
 *  Revision 1.20.1.22  2015/08/25 15:33:56  martin
 *  Use safe string copy function from str_util.c.
 *  Revision 1.20.1.21  2015/08/20 14:50:05  martin
 *  Revision 1.20.1.20  2015/08/12 15:53:57  martin
 *  Revision 1.20.1.19  2015/07/22 16:02:07  martin
 *  Started to support variable USB endpoint numbers.
 *  Revision 1.20.1.18  2015/07/14 15:08:43  martin
 *  Unified parameter naming and updated doxygen comments.
 *  Revision 1.20.1.17  2015/07/14 14:05:55  martin
 *  Support XBP addressing.
 *  Added functions to read/send UTC parameters.
 *  Support for USB_DIRECT_IO
 *  Fixed dealloc_msg_ctl() where not all memory was freed.
 *  Reworked mbgextio_force_conn_serial_ftdi().
 *  Revision 1.20.1.16  2014/10/30 16:03:05  martin
 *  Doxygen fixes.
 *  Revision 1.20.1.15  2014/10/30 14:45:55  martin
 *  Generally return Meinberg error codes only.
 *  Do not use GPS_REQACK for secu_settings.
 *  Added FTDI support functions.
 *  Reworked "force connection" functions, also supporting high speed now.
 *  Updated libusb support.
 *  Many new API functions.
 *  Revision 1.20.1.14  2013/11/28 15:51:45Z  marvin
 *  Added mbgextio_get_ntp_peer_state_idx.
 *  Revision 1.20.1.13  2013/11/26 16:06:43Z  marvin
 *  Added mbgextio_get_ntp_sys_state.
 *  Revision 1.20.1.12  2013/11/21 07:46:44Z  marvin
 *  Added mbgextio_xmt_secu_settings.
 *  Cleanup for socket port.
 *  Revision 1.20.1.11  2013/11/15 12:17:30Z  marvin
 *  Added error return for TGT_LINUX if socket_opt_error detected.
 *  Revision 1.20.1.10  2013/11/15 11:49:12Z  marvin
 *  Fixed: return 0 if non_blocking_socket is OK.
 *  check if received GPS_NACK for encryption mode (socket connection).
 *  Revision 1.20.1.9  2013/11/13 16:30:10Z  martin
 *  Revision 1.20.1.8  2013/11/13 16:13:26  martin
 *  Revision 1.20.1.7  2013/11/13 16:05:52  marvin
 *  Revision 1.20.1.6  2013/11/13 15:14:17Z  martin
 *  Cleaned up socket connection code.
 *  Revision 1.20.1.5  2013/11/12 12:12:02  marvin
 *  Changed calls for NTP info and settings.
 *  Revision 1.20.1.4  2013/11/11 11:30:27Z  marvin
 *  Changed socket connection.
 *  Revision 1.20.1.3  2013/11/05 15:53:11Z  marvin
 *  Changed connecting via socket.
 *  Revision 1.20.1.2  2013/10/14 08:54:19Z  marvin
 *  Added mbgextio_set_ntp_clnt_mode_cfg.
 *  Revision 1.20.1.1  2013/09/25 11:09:53Z  marvin
 *  Added NTP support and new support for PTP.
 *  Revision 1.20  2013/09/10 08:56:35Z  marvin
 *  Changed Doxygen comments.
 *  Revision 1.19  2013/09/05 15:10:39Z  marvin
 *  Added support for LAN interface setup
 *  Added support for PTP state
 *  Added support for XMR.
 *  Revision 1.18  2013/09/02 15:16:48Z  marvin
 *  Added support for XMR (get multi ref info, set multi ref settings)
 *  Revision 1.17  2013/08/28 11:01:41Z  marvin
 *  Added read and set tr_distance and gnss_mode.
 *  Revision 1.16.1.1  2013/06/05 09:17:54Z  marvin
 *  Changed close_connection for socket.
 *  Revision 1.16  2013/04/11 14:18:33Z  Gregoire
 *  new calls for sending havequick rx and tx settings to clock
 *  Revision 1.15  2013/02/14 14:42:13Z  martin
 *  Fixed syntax err due to unintentionally pasted word.
 *  Revision 1.14  2013/02/06 15:43:54  martin
 *  Updated doxygen comments.
 *  Revision 1.13  2013/02/01 15:58:43  martin
 *  In mbgextio_force_connection() for Windows wait until all chars
 *  must have been sent since flushing output buffer doesn't work reliably.
 *  Added a number of new functions.
 *  Added doxygen comments.
 *  Revision 1.12  2012/10/30 16:17:30  martin
 *  Started to migrate to opaque stuctures.
 *  Conditionally let xmt routines request for ACK packet,
 *  though this is by default disabled.
 *  Support receiving NACK status.
 *  Support big endian target platforms.
 *  Merged and adapted Daniel's USB support functions.
 *  Support event log entries.
 *  Syntax workaround which is required until this module becomes a DLL.
 *  Added some new functions.
 *  Updated doxygen comments.
 *  Huge cleanup.
 *  Revision 1.11  2011/04/15 13:17:14  martin
 *  Use common mutex support macros from mbgmutex.h.
 *  Revision 1.10  2011/04/08 11:28:24  martin
 *  Modified mbgextio_get_ucap() to account for different device behaviour.
 *  Added missing braces.
 *  Revision 1.9  2009/10/02 14:19:05  martin
 *  Added a bunch of missing functions.
 *  Revision 1.8  2009/10/01 11:10:51  martin
 *  Added functions to set/retrieve char and msg rcv timeout.
 *  Revision 1.7  2009/09/01 10:44:58  martin
 *  Cleanup for CVI.
 *  Use new portable timeout functions from mbg_tmo.h.
 *  Timeouts are now specified in milliseconds.
 *  Distinguish between character timeout and message timeout.
 *  Only fetch one character at a time to prevent received characters
 *  from being discarded after the end of one message.
 *  Revision 1.6  2009/03/10 17:02:08Z  martin
 *  Added support for configurable time scales.
 *  Added mbgextio_get_time() call.
 *  Fixed some compiler warnings.
 *  Revision 1.5  2008/09/04 14:35:50Z  martin
 *  Fixed opening COM port under CVI.
 *  Moved generic serial I/O stuff to mbgserio.c and mbgserio.h.
 *  Restart reception if received msg does not match expected cmd code.
 *  Fixed timeout value for Windows.
 *  New symbol _MBGEXTIO_DIRECT_RC controls whether the return code of the
 *  mbgextio_set_...() functions is evaluated or returned as-is.
 *  New functions mbgextio_set_time(), mbgextio_set_tzdl().
 *  Added mbgextio_get_ucap(). This may not work with older firmware,
 *  see the comments in mbgextio_get_ucap().
 *  Conditionally support checking of time strings.
 *  Revision 1.4  2007/02/27 10:35:19Z  martin
 *  Added mutex for transmit buffer to make transmission thread-safe.
 *  Fixed timeout handling for serial reception.
 *  Renamed mbgextio_get_data() to mbgextio_rcv_msg().
 *  Added some new functions.
 *  Temp. changes for parameter setting functions.
 *  Added comments on POSIX flags used when opening serial port.
 *  Revision 1.3  2006/12/21 10:56:17  martin
 *  Added function mbgextio_set_port_parm.
 *  Revision 1.2  2006/10/25 12:18:31  martin
 *  Support serial I/O under Windows.
 *  Revision 1.1  2006/08/24 12:40:37Z  martin
 *  Initial revision.
 *
 **************************************************************************/

#define _MBGEXTIO
  #include <mbgextio.h>
#undef _MBGEXTIO

#include <xtiocomm.h>
#include <xdevfeat.h>
#include <mbgserio.h>
#include <cfg_hlp.h>
#include <mbg_arch.h>
#include <mbgerror.h>
#include <gpsutils.h>
#include <str_util.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>


#if !defined( _TRY_IDENT_DECODE )
  #define _TRY_IDENT_DECODE   0
#endif

#if _TRY_IDENT_DECODE
  #include <identdec.h>
#endif

#if _USE_SOCKET_IO
  #if defined( MBG_TGT_POSIX )
    #include <ifaddrs.h>
  #elif defined( MBG_TGT_WIN32 )
    #if defined( _MSC_VER ) && ( _MSC_VER <= 1200 )  // at least up to VC6
      #define VS_MISSING_IPHLPAPI
    #endif

    #if !defined( VS_MISSING_IPHLPAPI )
      #include <IPHlpApi.h>
    #endif
  #endif
#endif

#if defined( MBG_TGT_POSIX )
  #include <fcntl.h>
  #include <errno.h>
#endif

#if defined( MBG_TGT_DOS )
  #include <dos.h>
#endif

#if _USE_USB_IO
  #include <mbgusbio.h>
#endif

#if _USE_USB_DIRECT_IO
  #include <poll.h>
#endif

#if defined( DEBUG )
  #define DEBUG_SOCK_IO     1
  #define DEBUG_FORCE_CONN  1
  #define DEBUG_INIT_CONN   1
#else
  #define DEBUG_SOCK_IO     0
  #define DEBUG_FORCE_CONN  0
  #define DEBUG_INIT_CONN   0
#endif

#if defined( MBG_TGT_WIN32 ) && _USE_SOCKET_IO
  #define _USE_WIN32_CONSOLE_CONTROL_HANDLER   1
#else
  #define _USE_WIN32_CONSOLE_CONTROL_HANDLER   0
#endif

// default serial message timeout
#if !defined( MBGEXTIO_MSG_TIMEOUT_SERIAL )
  #define MBGEXTIO_MSG_TIMEOUT_SERIAL          2000  // [ms]
#endif

// default serial single character timeout
#if !defined( MBGEXTIO_POLL_TIMEOUT_SERIAL )
  #define MBGEXTIO_POLL_TIMEOUT_SERIAL         200  // [ms]
#endif


// default USB message timeout
#if !defined( MBGEXTIO_MSG_TIMEOUT_USB )
  #define MBGEXTIO_MSG_TIMEOUT_USB             100  // [ms]
#endif

// default USB character timeout, same as message timeout
#if !defined( MBGEXTIO_POLL_TIMEOUT_USB )
  #define MBGEXTIO_POLL_TIMEOUT_USB            MBGEXTIO_MSG_TIMEOUT_USB
#endif



static const char force_conn_cmd_str[] = MBG_FORCE_CONN_CMD_STR;
static const char force_conn_hs_cmd_str[] = MBG_FORCE_CONN_HS_CMD_STR;



/**
 * @brief Binary message control device flags
 *
 * Used to define ::MSG_CTL_DEVICE_FLAG_MSKS
 *
 * @see ::MSG_CTL_DEVICE_FLAG_MSKS
 */
enum MSG_CTL_DEVICE_FLAGS
{
  MSG_CTL_DEVICE_FLAG_LEGACY,   ///< legacy device with proprietary protocol, see ::mbgextio_dev_is_legacy
  N_MSG_CTL_DEVICE_FLAGS
};



/**
 * @brief Binary message control device flag masks
 *
 * Used with ::MBG_MSG_CTL::device_flags
 */
enum MSG_CTL_DEVICE_FLAG_MSKS
{
  MSG_CTL_DEVICE_FLAG_MSK_LEGACY = ( 1UL << MSG_CTL_DEVICE_FLAG_LEGACY )  ///< see ::MSG_CTL_DEVICE_FLAG_LEGACY
};



//### TODO proper header / prototype
/*HDR*/
const char *mbgextio_get_cmd_name( GPS_CMD cmd_code )
{
  return xtiocomm_get_cmd_name( cmd_code );

}  // mbgextio_get_cmd_name



#if _USE_WIN32_CONSOLE_CONTROL_HANDLER

static /*HDR*/
BOOL WINAPI mbgextio_on_console_event( DWORD dwCtrlType )
{
  switch ( dwCtrlType )
  {
    case CTRL_BREAK_EVENT:
    case CTRL_C_EVENT:
    case CTRL_CLOSE_EVENT:
    case CTRL_SHUTDOWN_EVENT:
      exit( 0 );

  }  // switch

  return FALSE;

}  // mbgextio_on_console_event



static /*HDR*/
void mbgextio_set_console_control_handler( void )
{
  static int has_been_set;

  if ( !has_been_set )
  {
    SetConsoleCtrlHandler( mbgextio_on_console_event, TRUE );
    has_been_set = 1;
  }

}  // mbgextio_set_console_control_handler

#endif  // _USE_WIN32_CONSOLE_CONTROL_HANDLER



#if _USE_USB_IO

static /*HDR*/
/**
 * @brief Check if a USB device is a legacy device which doesn't support the binary protocol
 *
 * There are some legacy USB devices which only support a proprietary protocol
 * instead of the standard binary protocol.
 * We try to find this out based on the USB vendor and device IDs, and we need
 * to check this *before* we call any of the standard API functions, since otherwise
 * a binary message sent to the device may have unpredictable side-effects.
 *
 * @param[in]  mdev   An ::MBGUSBIO_DEV handle of the device
 * @param[out] p_ri   Pointer to a ::RECEIVER_INFO to be set up, or NULL
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
int check_setup_legacy_usb_device( const MBG_USB_DEV_INFO *mdev_info, RECEIVER_INFO *p_ri )
{
  int model_code = GPS_MODEL_UNKNOWN;
  const char *model_name = str_unknown;

  if ( mdev_info->ven_id == USB_VENDOR_MEINBERG )
  {
    switch ( mdev_info->prod_id )
    {
      case USB_DEV_CPC_01:
        model_code = GPS_MODEL_CPC_01;
        model_name = GPS_MODEL_NAME_CPC_01;
        break;

      case USB_DEV_CPC180:
        model_code = GPS_MODEL_CPC180;
        model_name = GPS_MODEL_NAME_CPC180;
        break;

      case USB_DEV_TSU_01:
        model_code = GPS_MODEL_TSU_01;
        model_name = GPS_MODEL_NAME_TSU_01;
        break;

      case USB_DEV_CMC:
        model_code = GPS_MODEL_CMC_01;
        model_name = GPS_MODEL_NAME_CMC_01;
        break;

      case USB_DEV_SCU_USB:
        model_code = GPS_MODEL_SCU_01;
        model_name = GPS_MODEL_NAME_SCU_01;
        break;

      case USB_DEV_FCU_01:
        model_code = GPS_MODEL_FCU_01;
        model_name = GPS_MODEL_NAME_FCU_01;
        break;

    #if 0  //    //### TODO check these device, they probably do support binary
      case USB_DEV_SDI_01:
        model_code = GPS_MODEL_UNKNOWN;
        model_name =  str_unknown;
        break;

      case USB_DEV_MDU300:
        model_code = GPS_MODEL_MDU300;
        model_name =  str_unknown;
        break;
    #endif
    }
  }

  #if DEBUG
    fprintf( stderr, "Checking for legacy USB device %04X:%04X: %s, code %i\n",
             mdev_info->ven_id, mdev_info->ven_id, model_name, model_code );
  #endif

  if ( model_code == GPS_MODEL_UNKNOWN )
    return MBG_ERR_DEV_NOT_SUPP;

  if ( p_ri )
  {
    memset( p_ri, 0, sizeof( *p_ri ) );

    p_ri->model_code = model_code;
    sn_cpy_str_safe( p_ri->model_name, sizeof( p_ri->model_name ), model_name );
    p_ri->ticks_per_sec = DEFAULT_GPS_TICKS_PER_SEC;
  }

  return MBG_SUCCESS;

}  // check_setup_legacy_usb_device

#endif  // _USE_USB_IO



static /*HDR*/
/**
 * @brief Set up some extended device info for a device which has just been opened
 *
 * This function should be after a device has been opened successfully to read
 * some extended information which is stored in (and can be retrieved from)
 * the message control structure.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
int setup_extended_device_info( MBG_MSG_CTL *pmctl )
{
  MBG_XDEV_FEATURES *p_xdf = &pmctl->xdev_features;
  int rc;

  memset( p_xdf, 0, sizeof( *p_xdf ) );

  // Get the receiver info first.
   rc = mbgextio_setup_receiver_info( pmctl, NULL, &p_xdf->receiver_info );

  if ( mbg_rc_is_error( rc ) )
    goto out;

  // If the device supports extended features then read the
  // extended feature buffer from the device, else clear extended features.
  if ( p_xdf->receiver_info.features & GPS_HAS_XFEATURE )
  {
    rc = mbgextio_req_data( pmctl, NULL, GPS_XFEATURES, &p_xdf->xfeature_buffer, sizeof( p_xdf->xfeature_buffer ) );
    //### TODO use special API call?
  }

#if defined( _PRELIMINARY_CODE )

  // If the device supports TLVs then read the TLV info and supported TLV
  // features from the device, else clear the structure.
  if ( mbg_rc_is_success( xdevfeat_has_tlv_api( p_xdf ) ) )
  {
    rc = mbgextio_req_data( pmctl, NULL, GPS_TLV_INFO, &p_xdf->tlv_info, sizeof( p_xdf->tlv_info ) );
    //### TODO use special API call?
  }

#endif  // defined( _PRELIMINARY_CODE )

out:
  return rc;

}  // setup_extended_device_info



static /*HDR*/
//### TODO
int dev_open_finish_setup( MBG_MSG_CTL **ppmctl, MBG_MSG_CTL **caller_ppmctl )
{
  int rc;

  #if _USE_WIN32_CONSOLE_CONTROL_HANDLER
    mbgextio_set_console_control_handler();
  #endif

  #if _USE_MUTEX
    _mbg_mutex_init( &(*ppmctl)->dev_mutex );
  #endif

  rc = setup_extended_device_info( *ppmctl );

  if ( mbg_rc_is_error( rc ) )
    return dev_open_fail_close( rc, ppmctl, caller_ppmctl );

  return dev_open_finish( rc, *ppmctl, caller_ppmctl );

}  // dev_open_finish_setup



#if _USE_SOCKET_IO

static /*HDR*/
/**
 * @brief Set the blocking mode of a network socket
 *
 * @param[in] sock_fd   The socket descriptor
 * @param[in] blocking  Blocking mode, 0: non-blocking, else blocking
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
int set_socket_blocking_mode( MBG_SOCK_FD sock_fd, int blocking )
{
  int rc = MBG_ERR_UNSPEC;

  if ( sock_fd == MBG_INVALID_SOCK_FD )
  {
    rc = MBG_ERR_INV_SOCK_FD;
    goto out;
  }

  #if defined( MBG_TGT_WIN32 )
  {
    u_long mode = blocking ? 0 : 1;  // ioctlsocket expects an (u_long *)

    rc = ioctlsocket( sock_fd, FIONBIO, &mode );  // returns 0 on success

    if ( rc != 0 )  // error
    {
      rc = mbg_get_last_socket_error( "ioctlsocket(FIONBIO) failed in set_socket_blocking_mode" );
      goto out;
    }
  }
  #elif defined( MBG_TGT_POSIX )
  {
    int val = fcntl( sock_fd, F_GETFL, 0 );  // returns -1 on error

    if ( val == -1 )
    {
      rc = mbg_get_last_socket_error( "fcntl(F_GETFL) failed in set_socket_blocking_mode" );
      goto out;
    }

    if ( blocking )
      val &= ~O_NONBLOCK;
    else
      val |= O_NONBLOCK;

    rc = fcntl( sock_fd, F_SETFL, val );  // returns -1 on error

    if ( rc == -1 )
    {
      rc = mbg_get_last_socket_error( "fcntl(F_SETFL) failed in set_socket_blocking_mode" );
      goto out;
    }
  }
  #else
  {
    rc = MBG_ERR_NOT_SUPP_ON_OS;
    goto out;
  }
  #endif

  rc = MBG_SUCCESS;

out:
  return rc;

}  // set_socket_blocking_mode



static /*HDR*/
/**
 * @brief Get last socket error code converted to @ref MBG_RETURN_CODES
 *
 * @param[in] sock_fd  The socket descriptor
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
int get_sock_error( MBG_SOCK_FD sock_fd )
{
  int so_err = 0;
  socklen_t so_err_sz = sizeof( so_err );
  int rc = getsockopt( sock_fd, SOL_SOCKET, SO_ERROR, (char *) &so_err, &so_err_sz );  // returns 0 on success

  if ( rc != 0 )  // error
  {
    rc = mbg_get_last_socket_error( "getsockopt SO_ERROR failed in get_sock_error" );
    goto out;
  }

  if ( so_err_sz != sizeof( so_err ) )  // might have been modified by getsockopt()
  {
    fprintf( stderr, "Warning: getsockopt option size changed in get_sock_error: %li -> %li",
             (long) sizeof( so_err ), (long) so_err_sz );
    rc = MBG_ERR_UNSPEC;
    goto out;
  }

  if ( so_err != 0 )  // error
  {
    #if defined( MBG_TGT_WIN32 )
      rc = mbg_win32_wsa_err_to_mbg( so_err, "wait failed in wait_nonblocking_socket_ready" );
    #elif defined( MBG_TGT_POSIX )
      rc = mbg_posix_errno_to_mbg( so_err, "wait failed in wait_nonblocking_socket_ready" );
    #else
      #error This function is not supported for this target.
    #endif
    goto out;
  }

  rc = MBG_SUCCESS;

out:
  return rc;

}  // get_sock_error



static /*HDR*/
/**
 * @brief Wait for a non-blocking socket to get ready after a connect() call
 *
 * A connect() call to a socket which is not available can fail
 * after a pretty long timeout which usually can't be changed.
 *
 * A workaround is to switch the socket to non-blocking mode,
 * then call connect(), then do a select() with specified timeout,
 * and finally switch the socket back to blocking mode.
 *
 * @param[in] sock_fd  The socket descriptor
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
int wait_nonblocking_socket_ready( MBG_SOCK_FD sock_fd, long timeout_msec )
{
  // At least under Windows a select call can set except_fds if
  // the non-blocking attempt to connect fails.
  // See: http://msdn.microsoft.com/en-us/library/windows/desktop/ms740141(v=vs.85).aspx
  // Linux and FreeBSD suggest to use write_fds only.
  #define USE_EXCEPT_FDS_WITH_CONNECT ( 1 || defined( MBG_TGT_WIN32 ) )  //##+++++++++++++++++++++++++++

  fd_set write_fds;
  #if USE_EXCEPT_FDS_WITH_CONNECT
    fd_set except_fds;
  #endif
  int max_fd;
  struct timeval tv_timeout;
  int rc = MBG_ERR_UNSPEC;

  mbg_msec_to_timeval( timeout_msec, &tv_timeout );

  FD_ZERO( &write_fds );
  FD_SET( sock_fd, &write_fds );

  #if defined( MBG_TGT_WIN32 )
    // Under Windows an fd is a handle which can't simply
    // be converted to an int, but the first argument of
    // select() is ignored under Windows anyway, so we just
    // set max_fd to 0.
    max_fd = 0;
  #else
    max_fd = sock_fd;
  #endif

  #if USE_EXCEPT_FDS_WITH_CONNECT
    FD_ZERO( &except_fds );
    FD_SET( sock_fd, &except_fds );
    rc = select( max_fd + 1, NULL, &write_fds, &except_fds, &tv_timeout );
  #else
    rc = select( max_fd + 1, NULL, &write_fds, NULL, &tv_timeout );
  #endif

  if ( rc == MBG_SOCKET_ERR_RETVAL )  // < 0, error
  {
    rc = mbg_get_last_socket_error( "select failed in wait_nonblocking_socket_ready" );
    goto out;
  }

  if ( rc == 0 )  // timeout
  {
    #if DEBUG_SOCK_IO
      fprintf( stderr, "select timed out in wait_nonblocking_socket_ready" );
    #endif
    rc = MBG_ERR_NBLOCK_WAIT_SLCT;
    goto out;
  }

  #if USE_EXCEPT_FDS_WITH_CONNECT
    // The except_fds might be set if no connection could be
    // established if e.g. the target socket is already busy.
    if ( FD_ISSET( sock_fd, &except_fds ) )
    {
      rc = get_sock_error( sock_fd );
      goto out;
    }
  #endif

  // Usually the write_fds are checked to see if the
  // non-blocking connect() call completed successfully,
  // in which case the associated descriptor is set.
  if ( !FD_ISSET( sock_fd, &write_fds ) )
  {
    #if DEBUG_SOCK_IO
      fprintf( stderr, "write_fd is not set after select in wait_nonblocking_socket_ready" );
    #endif
    rc = MBG_ERR_NBLOCK_WAIT_WR_FD;
    goto out;
  }

  #if defined( MBG_TGT_LINUX )
    // From the connect(2) man page:
    // After select(2) indicates writability, use getsockopt(2) to read
    // the SO_ERROR option at level SOL_SOCKET to determine whether connect()
    // completed successfully (SO_ERROR is zero) or unsuccessfully (SO_ERROR
    // is one of the usual error codes, explaining the reason for the failure).
    rc = get_sock_error( sock_fd );
    goto out;
  #endif

  rc = MBG_SUCCESS;

out:
  return rc;

}  // wait_nonblocking_socket_ready


#if defined( MBG_TGT_WIN32 )

/**
 * @brief Cleans up the winsock lib by calling WSACleanup
 *
 * This function is automatically called at the exit of a program.
 * It is registered in ::check_init_winsock
 *
 * @see ::check_init_winsock
 */
void deinit_winsock( void )
{
  WSACleanup( );
}


/**
 * @brief Checks if the winsock lib has been initialized before and initializes it if not
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
int check_init_winsock( void )
{
  static bool wsa_init;
  int rc = MBG_SUCCESS;

  if( !wsa_init )
  {
    WORD wVersionRequested = MAKEWORD( 2, 2 );
    WSADATA wsaData;

    rc = WSAStartup( wVersionRequested, &wsaData );
    if ( rc != 0 )
      rc = mbg_win32_wsa_err_to_mbg( rc , NULL );
    else rc = MBG_SUCCESS;

    if( mbg_rc_is_success( rc ) )
    {
      wsa_init = true;
      atexit( deinit_winsock );
    }

  }

  return rc;
}

#endif // defined( MBG_TGT_WIN32 )


/*HDR*/
/**
 * @brief Sets up a list of all available Meinberg LAN devices in the network
 *
 * List entries will be allocated and have to be freed by calling ::mbgextio_free_lan_devices
 *
 * @param[out]    list        Pointer to the list
 * @param[in]     timeout_ms  Timeout in ms for each network interface
 *
 * @return One of the @ref MBG_RETURN_CODES or the number of found devices on success
 *
 * @see ::mbgextio_free_lan_devices
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_find_lan_devices( MBG_LAN_DEV_LIST **list, uint16_t timeout_ms )
{
#if defined( VS_MISSING_IPHLPAPI )
  return MBG_ERR_NOT_SUPP_ON_OS;
#else
  struct sockaddr_in src_addr = { 0 };
  struct sockaddr_in bcast_addr = { 0 };
  struct sockaddr_in global_bcast_addr = { 0 };
  MBG_LAN_DEV_LIST *list_head = NULL;
  MBG_LAN_DEV_LIST *curr_entry = NULL;
  MBG_LAN_DEV_LIST *tmp_entry;
  const uint32_t query = htonl( MBG_LAN_REQUEST_CONFIG );
  socklen_t addr_len = sizeof ( struct sockaddr_in );
  struct in_addr src_in_addr = { 0 };
  struct in_addr broad_in_addr = { 0 };
  struct timeval select_timeout;
  int opt = 1;
  int rc = MBG_SUCCESS;
  MBG_SOCK_FD sock = MBG_INVALID_SOCK_FD;
  MBG_TMO_TIME t_end;
  MBG_TMO_TIME t_now;
  MBG_LAN_DEV_CFG buf;
  fd_set socks;
  bool found;

#if defined( MBG_TGT_POSIX )

  struct ifaddrs *ifaddr = NULL, *ifa;

#elif defined( MBG_TGT_WIN32 )

  PIP_ADAPTER_ADDRESSES pAdptAddrs, pAdptAddr;
  PIP_ADAPTER_UNICAST_ADDRESS pUnicastAddr;
  ULONG flags = GAA_FLAG_SKIP_ANYCAST | GAA_FLAG_SKIP_MULTICAST | GAA_FLAG_SKIP_DNS_SERVER;
  unsigned long adptAddrsBuflen = sizeof( IP_ADAPTER_ADDRESSES );
  uint8_t prefix;

#endif

  global_bcast_addr.sin_family = AF_INET;
  global_bcast_addr.sin_addr.s_addr = htonl( INADDR_BROADCAST );
  global_bcast_addr.sin_port = htons( MBG_LAN_UDP_BROADCAST_PORT );

  bcast_addr.sin_family = AF_INET;
  bcast_addr.sin_port = htons( MBG_LAN_UDP_BROADCAST_PORT );

#if defined( MBG_TGT_POSIX )

  if ( getifaddrs( &ifaddr ) == 0 )
  {
    for ( ifa = ifaddr; ifa != NULL; ifa = ifa->ifa_next )
    {
      if ( ifa->ifa_addr && ( ifa->ifa_addr->sa_family == AF_INET ) )
      {
        src_in_addr.s_addr = ( (struct sockaddr_in *) ifa->ifa_addr )->sin_addr.s_addr;

        if ( ifa->ifa_broadaddr )
          broad_in_addr = ( (struct sockaddr_in *) ifa->ifa_broadaddr )->sin_addr;

#elif defined( MBG_TGT_WIN32 )

  rc = check_init_winsock();
  if( mbg_rc_is_error( rc ) )
    return rc;

  pAdptAddrs = (PIP_ADAPTER_ADDRESSES) malloc( sizeof( IP_ADAPTER_ADDRESSES ) );

  if ( !pAdptAddrs )
    return MBG_ERR_NO_MEM;

  rc = GetAdaptersAddresses( AF_INET, flags, NULL, pAdptAddrs, &adptAddrsBuflen );
  if ( rc == ERROR_BUFFER_OVERFLOW )
  {
    free( pAdptAddrs );
    pAdptAddrs = (PIP_ADAPTER_ADDRESSES) malloc( adptAddrsBuflen );
    if ( !pAdptAddrs )
      return MBG_ERR_NO_MEM;
    rc = GetAdaptersAddresses( AF_INET, flags, NULL, pAdptAddrs, &adptAddrsBuflen );
  }

  if ( rc == NO_ERROR )
  {
    for ( pAdptAddr = pAdptAddrs; pAdptAddr != NULL; pAdptAddr = pAdptAddr->Next )
    {
      if ( pAdptAddr->OperStatus != IfOperStatusUp )
        continue;

      for ( pUnicastAddr = pAdptAddr->FirstUnicastAddress; pUnicastAddr != NULL; pUnicastAddr = pUnicastAddr->Next )
      {
        if ( pUnicastAddr->Address.lpSockaddr->sa_family != AF_INET )
          continue;

        src_in_addr.s_addr = ( (struct sockaddr_in *) pUnicastAddr->Address.lpSockaddr )->sin_addr.s_addr;

        broad_in_addr = src_in_addr;
        for( prefix = pUnicastAddr->OnLinkPrefixLength; prefix < 32; prefix++ )
          broad_in_addr.s_addr |= ( 1UL << prefix );

#endif

        if ( ntohl( src_in_addr.s_addr ) == INADDR_LOOPBACK )
          continue;

        sock = socket( AF_INET, SOCK_DGRAM, IPPROTO_UDP );

        if ( sock >= 0 )
        {
          mbg_tmo_get_time( &t_end );
          mbg_tmo_add_ms( &t_end, timeout_ms );

          src_addr.sin_family = AF_INET;
          src_addr.sin_addr.s_addr = src_in_addr.s_addr;

          bcast_addr.sin_addr.s_addr = broad_in_addr.s_addr;

          if ( ( setsockopt( sock, SOL_SOCKET, SO_BROADCAST, (const char*) &opt, sizeof( opt ) ) == 0 ) &&
              ( bind( sock, (const struct sockaddr *) &src_addr, addr_len ) == 0 ) )
          {
            if ( ( sendto( sock, (char *) &query, sizeof( query ), 0, (struct sockaddr *) &global_bcast_addr, addr_len ) == sizeof( query ) ) &&
                 ( sendto( sock, (char *) &query, sizeof( query ), 0, (struct sockaddr *) &bcast_addr, addr_len ) == sizeof( query ) ) )
            {
              while ( 1 )
              {
                FD_ZERO( &socks );
                FD_SET( sock, &socks );

                mbg_tmo_get_time( &t_now );
                if ( mbg_tmo_time_is_after( &t_now, &t_end ) )
                  break;
                mbg_msec_to_timeval( mbg_tmo_time_diff_ms( &t_end, &t_now ), &select_timeout );

                addr_len = sizeof( struct sockaddr_in );

                // The first parameter for select() is always an int, but sock
                // is not an int in Windows, but always in POSIX. Since the parameter
                // is ignored in windows anyway, it's safe to cast it to an int
                // to avoid compiler warnings.
                if( select( (int) ( sock + 1 ), &socks, NULL, NULL, &select_timeout ) > 0 )
                {
                  if( recvfrom( sock, (char *) &buf, sizeof( buf ), 0, (struct sockaddr *) &src_addr, &addr_len ) >= sizeof( buf ) )
                  {
                    if( ( ntohl( buf.command ) == MBG_LAN_RESPOND_CONFIG ) &&
                        ( ( ntohs( buf.channel_0.own_tcp_port ) == LAN_XPT_PORT ) ||
                          ( buf.channel_0.own_tcp_port == LAN_XPT_PORT ) ) )
                    {
                      found = false;

                      if ( list_head )
                      {
                        tmp_entry = list_head;
                        while( tmp_entry && !found )
                        {
                          if( tmp_entry->config.ip_address == ntohl( src_addr.sin_addr.s_addr ) )
                            found = true;
                          tmp_entry = tmp_entry->next;
                        }
                      }

                      if ( !found )
                      {
                        if ( curr_entry )
                        {
                          curr_entry->next = (MBG_LAN_DEV_LIST *) malloc( sizeof( *curr_entry->next ) );
                          if ( !curr_entry->next )
                          {
                            rc = MBG_ERR_NO_MEM;
                            goto cleanup_and_return;
                          }
                          curr_entry = curr_entry->next;
                        }
                        else
                        {
                          curr_entry = (MBG_LAN_DEV_LIST *) malloc( sizeof( *curr_entry ) );
                          if( !curr_entry )
                          {
                            rc = MBG_ERR_NO_MEM;
                            goto cleanup_and_return;
                          }
                          list_head = curr_entry;
                        }

                        memset( curr_entry, 0, sizeof( *curr_entry ) );
                        curr_entry->config = buf;
                        curr_entry->config.ip_address = ntohl( src_addr.sin_addr.s_addr );
                        rc++;
                      }
                    }
                  }
                }
                else
                  break;
              }
            }
            else
            {
              rc = MBG_ERR_NBYTES;
              goto cleanup_and_return;
            }
          }
          else
          {
            rc = MBG_ERR_INV_SOCK_FD;
            goto cleanup_and_return;
          }

#if defined( MBG_TGT_POSIX )

          shutdown( sock, SHUT_RDWR );
          close( sock );

#elif defined( MBG_TGT_WIN32 )

          shutdown( sock, SD_BOTH );
          closesocket( sock );

#endif
          sock = -1;
        }
        else
        {
          rc = MBG_ERR_INV_SOCK_FD;
          goto cleanup_and_return;
        }
      }
    }
  }
  else
    rc = MBG_ERR_GENERIC;

cleanup_and_return:

#if defined( MBG_TGT_POSIX )

  if ( ifaddr )
    freeifaddrs( ifaddr );

  if ( sock >= 0 )
  {
    shutdown( sock, SHUT_RDWR );
    close( sock );
  }

#elif defined( MBG_TGT_WIN32 )

  if ( pAdptAddrs )
    free( pAdptAddrs );

  if ( sock >= 0 )
  {
    shutdown( sock, SD_BOTH );
    closesocket( sock );
  }

#endif

  if ( rc <= 0 )
  {
    mbgextio_free_lan_devices( list_head );
    list_head = NULL;
  }

  *list = list_head;

  return rc;
#endif
}


/*HDR*/
/**
 * @brief Frees the ::MBG_LAN_DEV_LIST allocated by ::mbgextio_find_lan_devices
 *
 * @param[in]     list      The list that will be freed
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_find_lan_devices
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_free_lan_devices( MBG_LAN_DEV_LIST *list )
{
  if( list )
  {
    MBG_LAN_DEV_LIST *next_entry = list;
    while( next_entry )
    {
      list = next_entry;
      next_entry = list->next;
      free( list );
    }
    return MBG_SUCCESS;
  }
  return MBG_ERR_INV_PARM;
}


static /*HDR*/
/**
 * @brief Initialize a socket connection to a specified host
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     host   Host name or IP address of the target system
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
int socket_init( MBG_MSG_CTL *pmctl, const char *host )
{
  struct hostent *hp;
  struct sockaddr_in *paddr;
  const struct sockaddr *p;
  int sz;
  int rc = MBG_ERR_UNSPEC;

#if defined( MBG_TGT_WIN32 )
  rc = check_init_winsock( );

  if( !mbg_rc_is_success( rc ) )
    goto out;
#endif

  //##++++++++ should use getaddrinfo() preferably
  hp = gethostbyname( host );

  if ( hp == NULL )  // error
  {
    rc = mbg_get_gethostbyname_error( "gethostbyname failed in socket_init" );
    goto out;
  }

  // Create socket on which to send.
  pmctl->st.sockio.sockfd = socket( PF_INET, SOCK_STREAM, 0 );

  if ( pmctl->st.sockio.sockfd == MBG_INVALID_SOCK_FD )
  {
    rc = mbg_get_last_socket_error( "failed to create socket in socket_init" );
    goto out;
  }

  // Set the socket to nonblocking mode to be able to
  // reduce the timeout when connecting to a socket
  // which is offline.
  rc = set_socket_blocking_mode( pmctl->st.sockio.sockfd, 0 );

  if ( mbg_rc_is_error( rc ) )
    goto out_close;


  // Now try to connect to the socket.
  paddr = &pmctl->st.sockio._addr;
  memset( paddr, 0, sizeof( *paddr ) );

  memcpy( &paddr->sin_addr, hp->h_addr, hp->h_length );
  paddr->sin_family = AF_INET;
  paddr->sin_port = htons( LAN_XPT_PORT );

  p = (const struct sockaddr *) paddr;
  sz = sizeof( *paddr );

  rc = connect( pmctl->st.sockio.sockfd, p, sz );  // returns 0 on success on Windows and Linux

  // In nonblocking mode connect() might complete successfully
  // if the destination socket is on the same machine (localhost)
  // However, for remote connections the call usually returns
  // immediately with WSAEWOULDBLOCK on Windows, or EINPROGRESS or
  // EALREADY on POSIX systems. In the latter cases we need to wait
  // and see if the connection can be established successfully.

  if ( rc != 0 )  // eventually not (yet) completed in non-blocking mode
  {
    #if defined( MBG_TGT_WIN32 )
      DWORD wsa_err = WSAGetLastError();

      // WSAEWOULDBLOCK is actually not an error in non-blocking mode
      if ( wsa_err != WSAEWOULDBLOCK )
      {
        rc = mbg_win32_wsa_err_to_mbg( wsa_err, "connect failed in socket_init" );
        goto out_close;
      }
    #else  // POSIX ...
      int posix_errno = errno;

      // EINPROGRESS and EALREADY are actually not errors in non-blocking mode
      if ( ( posix_errno != EINPROGRESS ) && ( posix_errno != EALREADY ) )
      {
        rc = mbg_posix_errno_to_mbg( posix_errno, "connect failed in socket_init" );
        goto out_close;
      }
    #endif

    rc = wait_nonblocking_socket_ready( pmctl->st.sockio.sockfd, 1500 );  //##++++++++++++

    if ( mbg_rc_is_error( rc ) )
      goto out_close;
  }

  // Set the socket back to blocking mode.
  rc = set_socket_blocking_mode( pmctl->st.sockio.sockfd, 1 );

  pmctl->st.sockio.p_addr = &pmctl->st.sockio._addr;
  pmctl->st.sockio.addrlen = sizeof( pmctl->st.sockio._addr );

  if ( mbg_rc_is_success( rc ) )
    goto out;

  // error, fall through and close socket

out_close:
  socket_close( pmctl );

out:
  return rc;

}  // socket_init



/*HDR*/
/**
 * @brief Send security settings for the socket connection
 *
 * If new_passwd is not a NULL pointer, the old_passwd
 * will be overwritten by the new_passwd.
 *
 * @param[in,out] pmctl       Pointer to a valid message control structure
 * @param[in]     p_addr      Pointer to an XBP address specifier, or NULL
 * @param[in]     old_passwd  Pointer to the current LAN port password of the device
 * @param[in]     new_passwd  Pointer to the new LAN port password for the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_open_socket
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_xmt_secu_settings( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                const char *old_passwd, const char *new_passwd )
{
  MBG_MSG_BUFF *pmb;
  SECU_SETTINGS *pss = &pmctl->secu_settings;
  GPS_CMD cmd = GPS_SECU_SETTINGS;  // GPS_REQACK is not supported by LAN_XPT, so never use it
  int rc = MBG_ERR_UNSPEC;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  memset( pss, 0, sizeof *pss );
  strncpy_safe( pss->password, old_passwd, sizeof( pss->password ) );

  set_encryption_mode( pmctl, MBG_XFER_MODE_ENCRYPTED, pss->password );

  if ( new_passwd )
  {
    if ( strlen( new_passwd ) > 0 )
      strncpy_safe( pss->new_password, new_passwd, sizeof( pss->new_password ) );

    pss->flags |= MSK_FLAG_CHANGE_PASSWORD;
  }

  pmb = pmctl->xmt.pmb;
  pmb->u.msg_data.secu_settings = *pss;
  pmb->hdr.cmd = cmd;
  pmb->hdr.len = sizeof( pmb->u.msg_data.secu_settings );

  rc = xmt_tbuff( pmctl, NULL );    //### TODO Use mbgextio_req_data instead? Is NULL OK?

  if ( mbg_rc_is_error( rc ) )
    goto out;

  rc = mbgextio_rcv_msg_unlocked( pmctl, NULL, cmd, NULL, 0 );    //### TODO Is XBP_ADDR NULL OK?

  // With some devices a timeout may also occur
  // if a wrong password has been used.
  if ( mbg_rc_is_error( rc ) )
    goto out;

  cmd = pmctl->rcv.pmb->hdr.cmd;

  // If we either received a reply with GPS_NACK, or
  // without GPS_ACK, the password was definitely wrong.
  if ( ( cmd & GPS_NACK ) || !( cmd & GPS_ACK ) )
  {
    rc = MBG_ERR_AUTH;
    goto out;
  }

  rc = MBG_SUCCESS;

out:
  #if _USE_MUTEX
    _mbg_mutex_release( &pmctl->dev_mutex );
  #endif

  return rc;

}  // mbgextio_xmt_secu_settings



/*HDR*/
/**
 * @brief Open a binary communication channel via a LAN/socket connection
 *
 * @param[in]  host    DNS name or IP address of the target device
 * @param[out] ppmctl  Address of a pointer to a ::MBG_MSG_CTL control structure
 *                     allocated and set up by this call.
 *                     Pointer is set to NULL on error.
 * @param[in]  passwd  Password string for the encrypted communication
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_open_serial
 * @see ::mbgextio_open_serial_ftdi
 * @see ::mbgextio_open_usb
 * @see ::mbgextio_open_usb_direct_io
 * @see ::mbgextio_close_connection
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_open_socket( const char *host,
                              MBG_MSG_CTL **ppmctl, const char *passwd )
{
  MBG_MSG_CTL *pmctl;
  int rc = alloc_msg_ctl( &pmctl );

  if ( mbg_rc_is_error( rc ) )
    return dev_open_finish( rc, pmctl, ppmctl );

  rc = socket_init( pmctl, host );

  if ( mbg_rc_is_error( rc ) )
    return dev_open_fail_free( rc, &pmctl, ppmctl );

  dev_open_init( pmctl, MBG_CONN_TYPE_SOCKET, MBGEXTIO_MSG_TIMEOUT_SOCKET,
                 MBGEXTIO_POLL_TIMEOUT_SOCKET );

  rc = mbgextio_xmt_secu_settings( pmctl, NULL, passwd, NULL );

  if ( mbg_rc_is_error( rc ) )
    return dev_open_fail_close( rc, &pmctl, ppmctl );

  return dev_open_finish_setup( &pmctl, ppmctl );

}  // mbgextio_open_socket

#endif  // _USE_SOCKET_IO



#if _USE_SERIAL_IO

/*HDR*/
int mbgextio_open_serial_raw( const char *dev, MBG_MSG_CTL **ppmctl,
                              uint32_t baud_rate, const char *framing )
{
  MBG_MSG_CTL *pmctl;
  int rc = alloc_msg_ctl( &pmctl );

  if ( mbg_rc_is_error( rc ) )
    return dev_open_finish( rc, pmctl, ppmctl );

  rc = mbgserio_open( &pmctl->st.p_serio, dev );

  if ( mbg_rc_is_error( rc ) )
    return dev_open_fail_free( rc, &pmctl, ppmctl );

  dev_open_init( pmctl, MBG_CONN_TYPE_SERIAL, MBGEXTIO_MSG_TIMEOUT_SERIAL,
                 MBGEXTIO_POLL_TIMEOUT_SERIAL );

  rc = mbgserio_set_parms( pmctl->st.p_serio, baud_rate, framing );

  if ( mbg_rc_is_error( rc ) )
    return dev_open_fail_close( rc, &pmctl, ppmctl );

  return dev_open_finish( rc, pmctl, ppmctl );

}  // mbgextio_open_serial_raw



/*HDR*/
/**
 * @brief Open a binary communication channel using direct serial I/O
 *
 * Commonly used serial parameters are 19200/8N1, see
 * ::MBG_DEFAULT_BAUDRATE and ::MBG_DEFAULT_FRAMING.
 * Some newer devices may also support ::MBG_DEFAULT_BAUDRATE_HS.
 *
 * @param[in]  dev        Name of the serial port to which the device is connected,
 *                        depending on the naming conventions of the host system.
 * @param[out] ppmctl     Address of a pointer to a ::MBG_MSG_CTL control structure
 *                        allocated and set up by this call. Set to NULL on error.
 * @param[in]  baud_rate  Baud rate used for serial communication
 * @param[in]  framing    Framing used for serial communication
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_force_conn_serial
 * @see ::mbgextio_open_serial_ftdi
 * @see ::mbgextio_force_conn_serial_ftdi
 * @see ::mbgextio_open_socket
 * @see ::mbgextio_open_usb
 * @see ::mbgextio_open_usb_direct_io
 * @see ::mbgextio_close_connection
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_open_serial( const char *dev, MBG_MSG_CTL **ppmctl,
                              uint32_t baud_rate, const char *framing )
{
  MBG_MSG_CTL *pmctl;
  int rc = alloc_msg_ctl( &pmctl );

  if ( mbg_rc_is_error( rc ) )
    return dev_open_finish( rc, pmctl, ppmctl );

  rc = mbgserio_open( &pmctl->st.p_serio, dev );

  if ( mbg_rc_is_error( rc ) )
    return dev_open_fail_free( rc, &pmctl, ppmctl );

  dev_open_init( pmctl, MBG_CONN_TYPE_SERIAL, MBGEXTIO_MSG_TIMEOUT_SERIAL,
                 MBGEXTIO_POLL_TIMEOUT_SERIAL );

  rc = mbgserio_set_parms( pmctl->st.p_serio, baud_rate, framing );

  if ( mbg_rc_is_error( rc ) )
    return dev_open_fail_close( rc, &pmctl, ppmctl );

  return dev_open_finish_setup( &pmctl, ppmctl );

}  // mbgextio_open_serial


/*HDR*/
/**
 * @brief Open a binary communication channel forcing default serial parameters
 *
 * If current serial paramaters (baud rate and framing) do not match the default
 * parameters for binary communication then the serial device is forced into
 * binary communication mode.
 *
 * @param[in]  dev        Name of the serial port to which the device is connected,
 *                        depending on the naming conventions of the host system.
 * @param[out] ppmctl     Address of a pointer to a ::MBG_MSG_CTL control structure
 *                        allocated and set up by this call. Set to NULL on error.
 * @param[in]  baud_rate  Baud rate used for serial communication
 * @param[in]  framing    Framing used for serial communication
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_open_serial
 * @see ::mbgextio_open_serial_raw
 * @see ::mbgserio_write
 * @see ::mbgextio_close_connection
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_open_serial_force_default( const char *dev, MBG_MSG_CTL **ppmctl, uint32_t baud_rate, const char *framing )
{
  const char* cmd_str = force_conn_hs_cmd_str;
  int len = (int) strlen( cmd_str );
  MBG_MSG_CTL *pmctl = NULL;
  int rc;
  int high = 1;

  if ( ( baud_rate == MBG_DEFAULT_BAUDRATE_HS ) && ( strcmp( framing, MBG_DEFAULT_FRAMING ) == 0 ) )
    return mbgextio_open_serial( dev, ppmctl, baud_rate, framing );

retry:
  rc = mbgextio_open_serial_raw( dev, &pmctl, baud_rate, framing );

  if ( mbg_rc_is_success( rc ) )
  {
    mbgserio_write( pmctl->st.p_serio, cmd_str, len );

    #ifdef MBG_TGT_WIN32
      // Flushing the output when the serial port is closed doesn't
      // always work correctly under Windows, so we insert a delay
      // here to make sure the string has been set.
      // The required delay depends on the number of characters to
      // send, and on the transmission speed (baud rate).
      Sleep( ( ( 10 * 1000 * len ) / baud_rate ) + 1 );
    #endif

    mbgextio_close_connection( &pmctl );

    // Sleep for 10ms, so the device has time to switch to the new baudrate
    #if defined( MBG_TGT_WIN32 )
      Sleep( 10 );
    #elif defined( MBG_TGT_DOS )
      delay( 10 );
    #else
      usleep( 10000 );
    #endif

    rc = mbgextio_open_serial( dev, &pmctl, high ? MBG_DEFAULT_BAUDRATE_HS : MBG_DEFAULT_BAUDRATE, MBG_DEFAULT_FRAMING );

    if ( mbg_rc_is_error( rc ) && high )
    {
      // If baud_rate is default baudrate and highspeed baud_rate did not work, simply open the port with default baudrate
      if( ( baud_rate == MBG_DEFAULT_BAUDRATE ) && ( strcmp( framing, MBG_DEFAULT_FRAMING ) == 0 ) )
        return mbgextio_open_serial( dev, ppmctl, baud_rate, framing );

      high = 0;
      cmd_str = force_conn_cmd_str;
      len = ( int )strlen( cmd_str );
      goto retry;
    }
  }

  if ( mbg_rc_is_success( rc ) )
    *ppmctl = pmctl;

  return rc;

} // mbgextio_open_serial_force_default

#endif // _USE_SERIAL_IO



#if _USE_SERIAL_IO_FTDI

/*HDR*/
/**
 * @brief Get the port handle of a serial FTDI device
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return the port handle retrieved from the message control structure
 */
_NO_MBG_API_ATTR FT_HANDLE _MBG_API mbgextio_get_serial_ftdi_port_handle( MBG_MSG_CTL *pmctl )
{
  return pmctl->st.ftdi.port_handle;

}  // mbgextio_get_serial_ftdi_port_handle



/*HDR*/
/**
 * @brief Open a binary communication channel using serial FTDI D2xx port
 *
 * Commonly used serial parameters are 19200/8N1, see
 * ::MBG_DEFAULT_BAUDRATE and ::MBG_DEFAULT_FRAMING.
 * Some newer devices may also support ::MBG_DEFAULT_BAUDRATE_HS.
 *
 * @param[in]  device_num  device number from a list set up by FT_ListDevices()
 * @param[out] ppmctl      Address of a pointer to a ::MBG_MSG_CTL control structure
 *                         allocated and set up by this call. Set to NULL on error.
 * @param[in] baud_rate    Baud rate used for serial communication
 * @param[in] framing      Framing used for serial communication
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_force_conn_serial_ftdi
 * @see ::mbgextio_open_serial
 * @see ::mbgextio_force_conn_serial
 * @see ::mbgextio_open_socket
 * @see ::mbgextio_open_usb
 * @see ::mbgextio_open_usb_direct_io
 * @see ::mbgextio_close_connection
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_open_serial_ftdi( int device_num,
                 MBG_MSG_CTL **ppmctl, uint32_t baud_rate, const char *framing )
{
  FT_STATUS status;
  ULONG ft_baud;
  UCHAR ft_data_bits;
  UCHAR ft_stopbits;
  UCHAR ft_parity;
  const char *cp;
  MBG_MSG_CTL *pmctl;
  int rc = alloc_msg_ctl( &pmctl );

  if ( mbg_rc_is_error( rc ) )
    return dev_open_finish( rc, pmctl, ppmctl );

  status = FT_Open( device_num, &pmctl->st.ftdi.port_handle );

  if ( status != FT_OK )
    return dev_open_fail_free( mbg_ftdi_ft_status_to_mbg( status ), &pmctl, ppmctl );

  dev_open_init( pmctl, MBG_CONN_TYPE_SERIAL_FTDI, MBGEXTIO_MSG_TIMEOUT_SERIAL,
                 MBGEXTIO_POLL_TIMEOUT_SERIAL );

  // setup transmission speed
  switch ( baud_rate )
  {
    case 300:    ft_baud = FT_BAUD_300;    break;
    case 600:    ft_baud = FT_BAUD_600;    break;
    case 1200:   ft_baud = FT_BAUD_1200;   break;
    case 2400:   ft_baud = FT_BAUD_2400;   break;
    case 4800:   ft_baud = FT_BAUD_4800;   break;
    case 9600:   ft_baud = FT_BAUD_9600;   break;
    case 14400:  ft_baud = FT_BAUD_14400;  break;
    case 19200:  ft_baud = FT_BAUD_19200;  break;
    case 38400:  ft_baud = FT_BAUD_38400;  break;
    case 57600:  ft_baud = FT_BAUD_57600;  break;
    case 115200: ft_baud = FT_BAUD_115200; break;
    case 230400: ft_baud = FT_BAUD_230400; break;
    case 460800: ft_baud = FT_BAUD_460800; break;
    case 921600: ft_baud = FT_BAUD_921600; break;

    default:
      return dev_open_fail_close( MBG_ERR_INV_PARM, &pmctl, ppmctl );
  }


  // setup framing
  for ( cp = framing; *cp; cp++ )
  {
    char c = toupper( *cp );

    switch ( c )
    {
      case '7':  ft_data_bits = FT_BITS_7;      break;
      case '8':  ft_data_bits = FT_BITS_8;      break;

      case 'N':  ft_parity = FT_PARITY_NONE;    break;
      case 'E':  ft_parity = FT_PARITY_EVEN;    break;
      case 'O':  ft_parity = FT_PARITY_ODD;     break;
      // FT_PARITY_MARK and FT_PARITY_SPACE not supp. by Meinberg API

      case '1':  ft_stopbits = FT_STOP_BITS_1;  break;
      case '2':  ft_stopbits = FT_STOP_BITS_2;  break;

      default:
        return dev_open_fail_close( MBG_ERR_INV_PARM, &pmctl, ppmctl );
    }
  }


  status = FT_SetBaudRate( pmctl->st.ftdi.port_handle, ft_baud );

  if ( status != FT_OK )
    return dev_open_fail_close( mbg_ftdi_ft_status_to_mbg( status ), &pmctl, ppmctl );

  status = FT_SetDataCharacteristics( pmctl->st.ftdi.port_handle,
                              ft_data_bits, ft_stopbits, ft_parity );

  if ( status != FT_OK )
    return dev_open_fail_close( mbg_ftdi_ft_status_to_mbg( status ), &pmctl, ppmctl );

  return dev_open_finish_setup( &pmctl, ppmctl );

}  // mbgextio_open_serial_ftdi

#endif



#if _USE_USB_IO

/*HDR*/
/**
 * @brief Open a binary communication channel using direct USB I/O
 *
 * @param[in]  mdev_info  The USB device to communicate with.
 * @param[out] ppmctl     Address of a pointer to a ::MBG_MSG_CTL control structure
 *                        allocated and set up by this call. Set to NULL on error.
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_open_socket
 * @see ::mbgextio_open_serial
 * @see ::mbgextio_open_serial_ftdi
 * @see ::mbgextio_close_connection
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_open_usb( const MBG_USB_DEV_INFO *mdev_info,
                              MBG_MSG_CTL **ppmctl )
{
  MBG_MSG_CTL *pmctl;
  int rc = alloc_msg_ctl( &pmctl );

  if ( mbg_rc_is_error( rc ) )
    return dev_open_finish( rc, pmctl, ppmctl );

  rc = mbgusbio_open( &pmctl->st.usbio, mdev_info );

  if ( mbg_rc_is_error( rc ) )
    return dev_open_fail_free( rc, &pmctl, ppmctl );

  dev_open_init( pmctl, MBG_CONN_TYPE_USB, MBGEXTIO_MSG_TIMEOUT_USB,
                 MBGEXTIO_POLL_TIMEOUT_USB );

  return dev_open_finish_setup( &pmctl, ppmctl );

}  // mbgextio_open_usb



/*HDR*/
/**
 * @brief Reset an USB device in case that the open_usb function return MBG_ERR_BUSY
 *
 * @param[in]  mdev_info  The USB device to communicate with.
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_open_socket
 * @see ::mbgextio_open_serial
 * @see ::mbgextio_open_serial_ftdi
 * @see ::mbgextio_close_connection
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_reset_usb( const MBG_USB_DEV_INFO *mdev_info )
{
  return mbgusbio_reset( mdev_info );

}  // mbgextio_reset_usb



/*HDR*/
/**
 * @brief Open a binary communication channel using direct USB I/O
 *
 * //### TODO group opening functions, ref to group only
 *
 * @param[in]  ldev    A libusb_device to communicate with.
 * @param[out] ppmctl  Address of a pointer to a ::MBG_MSG_CTL control structure
 *                     allocated and set up by this call. Set to NULL on error.
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_open_socket
 * @see ::mbgextio_open_serial
 * @see ::mbgextio_open_serial_ftdi
 * @see ::mbgextio_close_connection
 */
int mbgextio_open_usb_ldev( libusb_device *ldev, MBG_MSG_CTL **ppmctl )
{
  MBG_MSG_CTL *pmctl;
  int rc = alloc_msg_ctl( &pmctl );

  if ( mbg_rc_is_error( rc ) )
    return dev_open_finish( rc, pmctl, ppmctl );

  rc = mbgusbio_open_specific_device( &pmctl->st.usbio, ldev );

  if ( mbg_rc_is_error( rc ) )
    return dev_open_fail_free( rc, &pmctl, ppmctl );

  dev_open_init( pmctl, MBG_CONN_TYPE_USB, MBGEXTIO_MSG_TIMEOUT_USB,
                 MBGEXTIO_POLL_TIMEOUT_USB );

  return dev_open_finish_setup( &pmctl, ppmctl );

}  // mbgextio_open_usb_ldev

#endif



#if _USE_USB_DIRECT_IO

/*HDR*/
/**
 * @brief Open a binary communication channel using direct USB I/O
 *
 * Currently it is only supported by Linux systems since a file like
 * USB device (e.g. "/dev/mbgims") is required and needs to support
 * basic I/O operations like write, read, etc...
 *
 * @param[in]   dev     Path to file like USB device
 * @param[in]   flags   Bitwise OR flags of access modes like W,RW,RO,etc.
 * @param[out]  ppmctl  Address of a pointer to a ::MBG_MSG_CTL control structure
 *                      allocated and set up by this call. Set to NULL on error.
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_open_socket
 * @see ::mbgextio_open_serial
 * @see ::mbgextio_open_serial_ftdi
 * @see ::mbgextio_open_usb
 * @see ::mbgextio_open_usb_direct_io
 * @see ::mbgextio_close_connection
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_open_usb_direct_io( const char *dev,
                              int flags, MBG_MSG_CTL **ppmctl )
{
  MBG_MSG_CTL *pmctl;
  int rc = alloc_msg_ctl( &pmctl );

  if ( mbg_rc_is_error( rc ) )
    return dev_open_finish( rc, pmctl, ppmctl );

  pmctl->st.usbdio.usbdiofd = open( dev, flags );

  if ( pmctl->st.usbdio.usbdiofd == MBG_USB_DIRECT_IO_INVALID_FD )
    return dev_open_fail_free( mbg_get_last_error( "failed to open direct USB I/O device" ),
                               &pmctl, ppmctl );

  dev_open_init( pmctl, MBG_CONN_TYPE_USB_DIRECT_IO, MBGEXTIO_MSG_TIMEOUT_USB_DIRECT_IO,
                 MBGEXTIO_POLL_TIMEOUT_USB_DIRECT_IO );

  return dev_open_finish_setup( &pmctl, ppmctl );

} // mbgextio_open_usb_direct_io

#endif



/*HDR*/
/**
 * @brief Close a binary communication channel and release resources
 *
 * Closes a binary communication channel which has been opened by one
 * of the mbgextio_open_...() functions and releases the buffers which
 * have been allocated when the channel was opened.
 *
 * The pointer to the message control structure passed by address is set
 * to NULL after the channel has been closed and the resources have
 * been released.
 *
 * @param[in,out] ppmctl Address of a pointer to a message control structure
 *                       created when the communication channel was opened
 *                       by one of the mbgextio_open_...() calls.
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_open_socket
 * @see ::mbgextio_open_serial
 * @see ::mbgextio_open_serial_ftdi
 * @see ::mbgextio_open_usb
 * @see ::mbgextio_open_usb_direct_io
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_close_connection( MBG_MSG_CTL **ppmctl )
{
  MBG_MSG_CTL *pmctl = *ppmctl;
  int rc = MBG_ERR_UNSPEC;

  switch ( pmctl->conn_type )
  {
    #if _USE_SERIAL_IO
      case MBG_CONN_TYPE_SERIAL:
        rc = mbgserio_close( &pmctl->st.p_serio );
        break;
    #endif  // _USE_SERIAL_IO

    #if _USE_SERIAL_IO_FTDI
      case MBG_CONN_TYPE_SERIAL_FTDI:
      {
        FT_STATUS status = FT_Close( pmctl->st.ftdi.port_handle );
        pmctl->st.ftdi.port_handle = NULL;  //### TODO FT_Handle is a PVOID
        rc = mbg_ftdi_ft_status_to_mbg( status );
      } break;
    #endif  // _USE_SERIAL_IO_FTDI

    #if _USE_SOCKET_IO
      case MBG_CONN_TYPE_SOCKET:
        rc = socket_close( pmctl );
        break;
    #endif  // _USE_SOCKET_IO

    #if _USE_USB_IO
      case MBG_CONN_TYPE_USB:
        rc = mbgusbio_close( &pmctl->st.usbio );
        break;
    #endif  // _USE_USB_IO

    #if _USE_USB_DIRECT_IO
      case MBG_CONN_TYPE_USB_DIRECT_IO:
        if ( pmctl->st.usbdio.usbdiofd == MBG_USB_DIRECT_IO_INVALID_FD )
        {
          rc = MBG_SUCCESS;
          break;
        }

        rc = close( pmctl->st.usbdio.usbdiofd );

        if ( rc < 0 )
          rc = mbg_get_last_error( "failed to close direct USB I/O device" );
        else
          rc = MBG_SUCCESS;
    #endif // _USE_USB_DIRECT_IO

    default:
      rc = MBG_ERR_CONN_TYPE;

  }  // switch

  #if _USE_MUTEX
    _mbg_mutex_destroy( &pmctl->dev_mutex );
  #endif

  dealloc_msg_ctl( ppmctl );

  return rc;

}  // mbgextio_close_connection



#if _USE_SERIAL_IO

static /*HDR*/
int do_force_conn_serial( const char *dev, int high_speed )
{
  MBG_MSG_CTL *pmctl;
  const char *cmd_str = high_speed ? force_conn_hs_cmd_str : force_conn_cmd_str;
  BAUD_RATE expected_baudrate = high_speed ? MBG_DEFAULT_BAUDRATE_HS : MBG_DEFAULT_BAUDRATE;
  #if DEBUG_FORCE_CONN
    const char *info = high_speed ? " (hs)" : "";
  #endif
  int len = (int) strlen( cmd_str );
  int baud_idx;
  int frame_idx;
  int rc = MBG_ERR_UNSPEC;

  for ( baud_idx = 0; baud_idx < N_MBG_BAUD_RATES; baud_idx++ )
  {
    for ( frame_idx = 0; frame_idx < N_MBG_FRAMINGS; frame_idx++ )
    {
      uint32_t baud_rate = mbg_baud_rates[baud_idx];
      const char *framing = mbg_framing_strs[frame_idx];

      rc = mbgextio_open_serial_raw( dev, &pmctl, baud_rate, framing );

      if ( mbg_rc_is_error( rc ) )
      {
        if ( rc == MBG_ERR_NO_ENTITY )  //### TODO duplicate code
        {
          #if DEBUG_FORCE_CONN
            fprintf( stderr, "force conn%s probing failed to open port %s: %s: aborting\n",
                     info, dev, mbg_strerror( rc ) );
          #endif
          goto out;
        }

        if ( rc == MBG_ERR_IO )  //### TODO duplicate code
        {
          #if DEBUG_FORCE_CONN
            fprintf( stderr, "force conn%s failed for port %s with %li/%s: %s: aborting\n",
                     info, dev, (long) baud_rate, framing, mbg_strerror( rc ) );
          #endif
          goto out;
        }

        #if DEBUG_FORCE_CONN
          fprintf( stderr, "force conn%s failed for port %s with %li/%s: %s: skipping\n",
                   info, dev, (long) baud_rate, framing, mbg_strerror( rc ) );
        #endif
        continue;
      }

      mbgserio_write( pmctl->st.p_serio, cmd_str, len );
      // should we check rc here or just continue?

      #ifdef MBG_TGT_WIN32
        // Flushing the output when the serial port is closed doesn't
        // always work correctly under Windows, so we insert a delay
        // here to make sure the string has been set.
        // The required delay depends on the number of characters to
        // send, and on the transmission speed (baud rate).
        Sleep( ( ( 10 * 1000 * len ) / baud_rate ) + 1 );
      #endif

      mbgextio_close_connection( &pmctl );
    }
  }

  rc = mbgextio_open_serial_raw( dev, &pmctl, expected_baudrate, MBG_DEFAULT_FRAMING );

  if ( mbg_rc_is_error( rc ) )
    goto out;

  rc = mbgextio_get_receiver_info( pmctl, NULL, NULL );

  if ( mbg_rc_is_success( rc ) )
    rc = expected_baudrate;

  mbgextio_close_connection( &pmctl);

out:
  return rc;

}  // do_force_conn_serial



/*HDR*/
/**
 * @brief Try to force a serial connection to a device
 *
 * A device's serial port may have been configured to work by default
 * in a way which is not appropriate for binary communication, e.g. using
 * a low communication speed, or some framing like "7E2" which messes up
 * binary data since the parity bit overrides a data bit.
 *
 * This function sends a special ASCII string to a device which lets the device
 * temporarily switch to a specific baud rate and 8N1 framing which is usually
 * used for binary communication, see ::MBG_DEFAULT_BAUDRATE,
 * ::MBG_DEFAULT_BAUDRATE_HS, and ::MBG_DEFAULT_FRAMING.
 *
 * Since the current settings of the device's serial port are unknown this
 * this command is sent in all common combinations of baud rates and
 * framings.
 *
 * If the connected device supports this it should be possible to open
 * the serial communication channel using the default parameters after
 * this function has finished.
 *
 * @param[in] dev  Name of the serial port to which the device is connected,
 *                 depending on the naming conventions of the target system
 *
 * @return One of the negative @ref MBG_ERROR_CODES on error,
 *         else the determined baud rate.
 *
 * @see ::mbgextio_open_serial
 * @see ::mbgextio_open_serial_ftdi
 */
_NO_MBG_API_ATTR long _MBG_API mbgextio_force_conn_serial( const char *dev )
{
  long rc = do_force_conn_serial( dev, 1 );

  if ( mbg_rc_is_success( rc ) )
    goto out;

  // If we failed because the specified port is not available
  // then it makes no sense to continue.
  if ( ( rc == MBG_ERR_NO_ENTITY ) || ( rc == MBG_ERR_IO ) )  //### TODO duplicate code
    goto out;

  rc = do_force_conn_serial( dev, 0 );

out:
  return rc;

}  // mbgextio_force_conn_serial

#endif  // _USE_SERIAL_IO



#if _USE_SERIAL_IO_FTDI

static /*HDR*/
int do_force_conn_serial_ftdi( int device_num, const char *cmd_str, BAUD_RATE expected_baudrate )
{
  int i;
  int j;
  MBG_MSG_CTL *pmctl;
  int len = (int) strlen( force_conn_cmd_str );
  long rc = MBG_ERR_UNSPEC;

  for ( i = 0; i < N_MBG_BAUD_RATES; i++ )
  {
    for ( j = 0; j < N_MBG_FRAMINGS; j++ )
    {
      uint32_t baud_rate = mbg_baud_rates[i];
      const char *framing = mbg_framing_strs[j];
      DWORD bytes_written;
      FT_HANDLE port_handle;
      FT_STATUS status;

      rc = mbgextio_open_serial_ftdi( device_num, &pmctl, baud_rate, framing );

      if ( mbg_rc_is_error( rc ) )
        goto out;

      port_handle = pmctl->st.ftdi.port_handle;

      status = FT_Write( port_handle, (LPVOID) force_conn_cmd_str, len, &bytes_written );

    #if 0  //### TODO  we need some error checking here
      if ( status != FT_OK )
      {
        rc = mbg_ftdi_ft_status_to_mbg( status );
        goto out;
      }

      if ( bytes_written != sizeof( len ) )
        goto out_write_failed;
    #endif

      #ifdef MBG_TGT_WIN32
        #if 1
          // TODO Check if this works as expected
          status = FT_Purge( port_handle, FT_PURGE_TX );
        #else

          // Flushing the output when the serial port is closed doesn't
          // always work correctly under Windows, so we insert a delay
          // here to make sure the string has been set.
          // The required delay depends on the number of characters to
          // send, and on the transmission speed (baud rate).
          Sleep( ( ( 10 * 1000 * len ) / baud_rate ) + 1 );
        #endif
      #endif

      mbgextio_close_connection( &pmctl );
    }
  }

  rc = mbgextio_open_serial_ftdi( device_num, &pmctl, expected_baudrate, MBG_DEFAULT_FRAMING );

  if ( mbg_rc_is_error( rc ) )
    goto out;

  rc = mbgextio_get_receiver_info( pmctl, NULL, NULL );

  mbgextio_close_connection( &pmctl );

out:
  return rc;

}  // do_force_conn_serial_ftdi



/*HDR*/
/**
 * @brief Try to force a serial connection to a device via the FTDI API
 *
 * A device's serial port may have been configured to work by default
 * in a way which is not appropriate for binary communication, e.g. using
 * a low communication speed, or some framing like "7E2" which messes up
 * binary data since the parity bit overrides a data bit.
 *
 * This function sends a special ASCII string to a device which lets the device
 * temporarily switch to a specific baud rate and 8N1 framing which is usually
 * used for binary communication, see ::MBG_DEFAULT_BAUDRATE,
 * ::MBG_DEFAULT_BAUDRATE_HS, and ::MBG_DEFAULT_FRAMING.
 *
 * Since the current settings of the device's serial port are unknown this
 * this command is sent in all common combinations of baud rates and
 * framings.
 *
 * If the connected device supports this it should be possible to open
 * the serial communication channel using the default parameters after
 * this function has finished.
 *
 * @param[in] device_num  device number from a list set up by FT_ListDevices()
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_open_serial_ftdi
 */
_NO_MBG_API_ATTR long _MBG_API mbgextio_force_conn_serial_ftdi( int device_num )
{
  long rc = do_force_conn_serial_ftdi( device_num, force_conn_hs_cmd_str, MBG_DEFAULT_BAUDRATE_HS );

  if ( mbg_rc_is_success( rc ) )
    return MBG_DEFAULT_BAUDRATE_HS;

  rc = do_force_conn_serial_ftdi( device_num, force_conn_cmd_str, MBG_DEFAULT_BAUDRATE );

  if ( mbg_rc_is_success( rc ) )
    return MBG_DEFAULT_BAUDRATE;

  return rc;

}  // mbgextio_force_conn_serial_ftdi

#endif  // _USE_SERIAL_IO_FTDI



/*HDR*/
/**
 * @brief Set a message callback function
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     fnc    Address of the callback function to be registered
 *
 * @return Address of the allocated receive buffer
 *
 * @see ::mbgextio_get_rcv_buffer_size
 * @see ::mbgextio_get_xmt_buffer_addr
 * @see ::mbgextio_get_xmt_buffer_size
 */
_NO_MBG_API_ATTR void _MBG_API mbgextio_register_msg_callback( MBG_MSG_CTL *pmctl, MBG_MSG_HANDLER *fnc )
{
  pmctl->msg_handler_fnc = *fnc;

}  // mbgextio_register_msg_callback



/*HDR*/
/**
 * @brief Retrieve address of the ::RECEIVER_INFO read from the device
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return Address of the ::RECEIVER_INFO read from the device when the device was opened
 */
_NO_MBG_API_ATTR RECEIVER_INFO * _MBG_API mbgextio_get_receiver_info_addr( MBG_MSG_CTL *pmctl )
{
  if ( pmctl == NULL )
    return NULL;

  return &pmctl->xdev_features.receiver_info;

}  // mbgextio_get_receiver_info_addr



/*HDR*/
/**
 * @brief Retrieve address of the allocated receive buffer
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return Address of the allocated receive buffer
 *
 * @see ::mbgextio_get_rcv_buffer_size
 * @see ::mbgextio_get_xmt_buffer_addr
 * @see ::mbgextio_get_xmt_buffer_size
 */
_NO_MBG_API_ATTR MBG_MSG_BUFF * _MBG_API mbgextio_get_rcv_buffer_addr( MBG_MSG_CTL *pmctl )
{
  if ( pmctl == NULL )
    return NULL;

  return pmctl->rcv.pmb;

}  // mbgextio_get_rcv_buffer_addr



/*HDR*/
/**
 * @brief Retrieve size of the allocated receive buffer
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return Size of the allocated receive buffer
 *
 * @see ::mbgextio_get_rcv_buffer_addr
 * @see ::mbgextio_get_xmt_buffer_addr
 * @see ::mbgextio_get_xmt_buffer_size
 */
_NO_MBG_API_ATTR size_t _MBG_API mbgextio_get_rcv_buffer_size( MBG_MSG_CTL *pmctl )
{
  if ( pmctl == NULL )
    return 0;

  return sizeof( *pmctl->rcv.pmb );

}  // mbgextio_get_rcv_buffer_size



/*HDR*/
/**
 * @brief Retrieve address of the allocated transmit buffer
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return Address of the allocated transmit buffer
 *
 * @see ::mbgextio_get_rcv_buffer_addr
 * @see ::mbgextio_get_rcv_buffer_size
 * @see ::mbgextio_get_xmt_buffer_size
 */
_NO_MBG_API_ATTR MBG_MSG_BUFF * _MBG_API mbgextio_get_xmt_buffer_addr( MBG_MSG_CTL *pmctl )
{
  if ( pmctl == NULL )
    return NULL;

  return pmctl->xmt.pmb;

}  // mbgextio_get_xmt_buffer_addr



/*HDR*/
/**
 * @brief Retrieve size of the allocated transmit buffer
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return Size of the allocated transmit buffer
 *
 * @see ::mbgextio_get_rcv_buffer_addr
 * @see ::mbgextio_get_rcv_buffer_size
 * @see ::mbgextio_get_xmt_buffer_addr
 */
_NO_MBG_API_ATTR size_t _MBG_API mbgextio_get_xmt_buffer_size( MBG_MSG_CTL *pmctl )
{
  if ( pmctl == NULL )
    return 0;

  return sizeof( *pmctl->xmt.pmb );

}  // mbgextio_get_xmt_buffer_size



/*HDR*/
/**
 * @brief Set device poll timeout
 *
 * @param[in,out] pmctl        Pointer to a valid message control structure
 * @param[in]     new_timeout  New poll timeout value [ms]
 *
 * @return Previous poll timeout value [ms]
 *
 * @see ::mbgextio_get_dev_poll_timeout
 * @see ::mbgextio_set_msg_rcv_timeout
 * @see ::mbgextio_get_msg_rcv_timeout
 */
_NO_MBG_API_ATTR ulong _MBG_API mbgextio_set_dev_poll_timeout( MBG_MSG_CTL *pmctl, ulong new_timeout )
{
  return xtiocomm_set_dev_poll_timeout( pmctl, new_timeout );

}  // mbgextio_set_dev_poll_timeout



/*HDR*/
/**
 * @brief Get device poll timeout
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return Current poll timeout value [ms]
 *
 * @see ::mbgextio_set_dev_poll_timeout
 * @see ::mbgextio_set_msg_rcv_timeout
 * @see ::mbgextio_get_msg_rcv_timeout
 */
_NO_MBG_API_ATTR ulong _MBG_API mbgextio_get_dev_poll_timeout( const MBG_MSG_CTL *pmctl )
{
  return xtiocomm_get_dev_poll_timeout( pmctl );

}  // mbgextio_get_dev_poll_timeout



/*HDR*/
/**
 * @brief Set message receive timeout value
 *
 * @param[in,out] pmctl        Pointer to a valid message control structure
 * @param[in]     new_timeout  New timeout value [ms]
 *
 * @see ::mbgextio_set_dev_poll_timeout
 * @see ::mbgextio_get_dev_poll_timeout
 * @see ::mbgextio_get_msg_rcv_timeout
 */
_NO_MBG_API_ATTR void _MBG_API mbgextio_set_msg_rcv_timeout( MBG_MSG_CTL *pmctl, ulong new_timeout )
{
  xtiocomm_set_msg_rcv_timeout( pmctl, new_timeout );

}  // mbgextio_set_msg_rcv_timeout



/*HDR*/
/**
 * @brief Get message receive timeout value
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return Current timeout value [ms]
 *
 * @see ::mbgextio_set_dev_poll_timeout
 * @see ::mbgextio_get_dev_poll_timeout
 * @see ::mbgextio_set_msg_rcv_timeout
 */
_NO_MBG_API_ATTR ulong _MBG_API mbgextio_get_msg_rcv_timeout( const MBG_MSG_CTL *pmctl )
{
  return xtiocomm_get_msg_rcv_timeout( pmctl );

}  // mbgextio_get_msg_rcv_timeout



/*HDR*/
/**
 * @brief Get last NACK error code
 *
 * If an API call has returned ::MBG_ERR_RCVD_NACK then this
 * function can be used to retrieve an associated error code
 * which eventually specifies the reason for the NACK message.
 *
 * A NACK message can be received from a device after the device
 * has received a command which it doesn't recognize, or a
 * parameter set which it doesn't accept, and the message can
 * optionally contain one of the @ref MBG_ERROR_CODES.
 *
 * If the NACK message contains an error code then that code is
 * saved, otherwise the saved code is set to ::MBG_ERR_UNSPEC.
 * A saved error code is available and can be retrieved until
 * the next API call starts to read some data from the device,
 * in which case the stored code is set to ::MBG_SUCCESS.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return The saved return code, i.e. one of the @ref MBG_RETURN_CODES.
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_last_nack_err_code( const MBG_MSG_CTL *pmctl )
{
  return pmctl->nack_err_code;

}  // mbgextio_get_last_nack_err_code



/*HDR*/
/**
 * @brief Check if a device is a legacy device
 *
 * Legacy devices don't support the standard binary protocol, so the
 * standard binary API functions must *not* be used with such a device.
 * Any attempt to access the device using the standard binary API
 * functions may have unpredictable side-effects for the device, but
 * the low level read/write functions can be used to access the device
 * via a proprietary protocol, depending on the device type.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_DEV_NOT_SUPP
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_is_legacy( const MBG_MSG_CTL *pmctl )
{
  return ( pmctl->device_flags & MSG_CTL_DEVICE_FLAG_MSK_LEGACY ) ? MBG_SUCCESS : MBG_ERR_DEV_NOT_SUPP;

}  // mbgextio_dev_is_legacy

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_is_legacy;



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_pos_xyz( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_pos_xyz( &pmctl->xdev_features );

}  // mbgextio_dev_has_pos_xyz

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_pos_xyz;



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_pos_lla( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_pos_lla( &pmctl->xdev_features );

}  // mbgextio_dev_has_pos_lla

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_pos_lla;



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_time_ttm( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_time_ttm( &pmctl->xdev_features );

}  // mbgextio_dev_has_time_ttm

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_time_ttm;



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_ant_info( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_ant_info( &pmctl->xdev_features );

}  // mbgextio_dev_has_ant_info

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_ant_info;



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_ant_cable_length( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_ant_cable_length( &pmctl->xdev_features );

}  // mbgextio_dev_has_ant_cable_length

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_ant_cable_length;



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_io_ports( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_io_ports( &pmctl->xdev_features );

}  // mbgextio_dev_has_io_ports

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_io_ports;



/*HDR*/
/**
 * @brief Check if a device supports the ::STAT_INFO structure and API
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_gps_stat_info( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_gps_stat_info( &pmctl->xdev_features );

}  // mbgextio_dev_has_gps_stat_info

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_gps_stat_info;



/*HDR*/
/**
 * @brief Check if a device can receive the GPS satellite system
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_is_gps( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_is_gps( &pmctl->xdev_features );

}  // mbgextio_dev_is_gps

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_is_gps;



/*HDR*/
/**
 * @brief Check if a device supports the GNSS API
 *
 * This is usually supported by devices which can receive signals
 * from different satellite systems, e.g. GPS, GLONASS, ...
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see ::mbgextio_chk_get_all_gnss_info
 * @see ::MBG_GNSS_TYPES
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_is_gnss( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_is_gnss( &pmctl->xdev_features );

}  // mbgextio_dev_is_gnss

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_is_gnss;



/*HDR*/
/**
 * @brief Check if a device is a bus level device
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_is_bus_lvl_dev( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_is_bus_lvl_dev( &pmctl->xdev_features );

}  // mbgextio_dev_is_bus_lvl_dev

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_is_bus_lvl_dev;



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_enable_flags( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_enable_flags( &pmctl->xdev_features );

}  // mbgextio_dev_has_enable_flags

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_enable_flags;


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_time_scale( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_time_scale( &pmctl->xdev_features );

}  // mbgextio_dev_has_time_scale

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_time_scale;


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_tzdl( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_tzdl( &pmctl->xdev_features );

}  // mbgextio_dev_has_tzdl

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_tzdl;


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_tzcode( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_tzcode( &pmctl->xdev_features );

}  // mbgextio_dev_has_tzcode

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_tzcode;


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_ims( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_ims( &pmctl->xdev_features );

}  // mbgextio_dev_has_ims

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_ims;



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_synth( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_synth( &pmctl->xdev_features );

}  // mbgextio_dev_has_synth

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_synth;



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_gpio( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_gpio( &pmctl->xdev_features );

}  // mbgextio_dev_has_gpio

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_gpio;



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_prog_pulses( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_prog_pulses( &pmctl->xdev_features );

}  // mbgextio_dev_has_prog_pulses

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_prog_pulses;



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_irig_tx( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_irig_tx( &pmctl->xdev_features );

}  // mbgextio_dev_has_irig_tx

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_irig_tx;



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_irig_rx( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_irig_rx( &pmctl->xdev_features );

}  // mbgextio_dev_has_irig_rx

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_irig_rx;



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_serouts( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_serouts( &pmctl->xdev_features );

}  // mbgextio_dev_has_serouts

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_serouts;



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_bvar_stat( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_bvar_stat( &pmctl->xdev_features );

}  // mbgextio_dev_has_serouts

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_bvar_stat;



/*HDR*/
/**
 * @brief Check if the device supports the SCU_STAT structures
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_ri_feature
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see @ref mbgextio_chk_supp_fncs
 * @see ::mbgextio_get_scu_stat_info
 * @see ::mbgextio_set_scu_stat_settings
 * @see @ref group_scu
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_scu_stat( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_scu_stat( &pmctl->xdev_features );

}  // mbgextio_dev_has_scu_stat

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_scu_stat;


/*HDR*/
/**
 * @brief Check if a timecode receiver provides ::MBG_RAW_IRIG_DATA
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_ri_feature
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see ::mbgextio_get_raw_irig_data
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_raw_irig_data( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_raw_irig_data( &pmctl->xdev_features );

}  // mbgextio_dev_has_raw_irig_data

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_raw_irig_data;



/*HDR*/
/**
 * @brief Check if a device supports the old LAN_IP4 API
 *
 * The LAN_IP4 API provides structures and functions to configure
 * parts of the networking of a device and is superseded by the
 * NET_CFG API. Some devices combine NET_CFG and LAN_IP4.
 * Therefore, ::mbgextio_get_all_net_cfg_info should be used
 * preferably to read the network configuration.
 * It will translate the old structures into the new ones.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_ri_feature
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see ::mbgextio_get_all_net_cfg_info
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_lan_ip4( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_lan_ip4( &pmctl->xdev_features );

}  // mbgextio_dev_has_lan_ip4

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_lan_ip4;



/*HDR*/
/**
 * @brief Check if a device supports the new NET_CFG API
 *
 * The NET_CFG API provides structures and functions to configure
 * the complete networking part of a device and supersedes the
 * LAN_IP4 API. Not all devices support the whole feature set
 * of the NET_CFG API or combine NET_CFG and LAN_IP4.
 * Therefore, ::mbgextio_get_all_net_cfg_info should be used
 * preferably to read the network configuration.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_ri_feature
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see ::mbgextio_get_all_net_cfg_info
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_net_cfg( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_net_cfg( &pmctl->xdev_features );

}  // mbgextio_dev_has_net_cfg

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_net_cfg;



/*HDR*/
/**
 * @brief Check if a device supports the PTP API
 *
 * The PTP API consists of different calls and associated structures
 * which * have evolved over time. Not all devices support every call,
 * so ::mbgextio_get_all_ptp_cfg_info takes care to check which parts are
 * supported and thus should be used preferably to read PTP information.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_ri_feature
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see ::mbgextio_get_all_ptp_cfg_info
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_ptp( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_ptp( &pmctl->xdev_features );

}  // mbgextio_dev_has_ptp

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_ptp;


/*HDR*/
/**
 * @brief Check if a device supports the NTP API
 *
 * The NTP API consists of different calls and associated structures
 * which have evolved over time. Not all devices support every call,
 * so ::mbgextio_get_all_ntp_cfg_info takes care to check which parts are
 * supported and thus should be used preferably to read NTP information.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_ri_feature
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see ::mbgextio_get_all_ntp_cfg_info
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_ntp( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_ntp( &pmctl->xdev_features );

} // mbgextio_dev_has_ntp

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_ntp;


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_evt_log( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_evt_log( &pmctl->xdev_features );

} // mbgextio_dev_has_evt_log

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_evt_log;


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_ucap( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_ucap( &pmctl->xdev_features );

} // mbgextio_dev_has_ucap

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_ucap;


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_ucap_net( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_ucap_net( &pmctl->xdev_features );

} // mbgextio_dev_has_ucap_net

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_ucap_net;


/*HDR*/
/**
 * @brief Check if a device supports the TLV API
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_xfeature
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see @ref group_tlv_api
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_tlv_api( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_tlv_api( &pmctl->xdev_features );

}  // mbgextio_dev_has_tlv_api

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_tlv_api;



/*HDR*/
/**
 * @brief Check if a device supports a firmware update via TLV
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_tlv_feat_supp
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see ::TODO  //refer to fw update function
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_supp_tlv_fw_update( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_supp_tlv_fw_update( &pmctl->xdev_features );

}  // mbgextio_dev_supp_tlv_fw_update

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_supp_tlv_fw_update;



/*HDR*/
/**
 * @brief Check if a device supports creating / sending a diagnostics file via TLV
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_tlv_feat_supp
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see ::TODO  //refer to get diag function
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_supp_tlv_diag_file( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_supp_tlv_diag_file( &pmctl->xdev_features );

}  // mbgextio_dev_supp_tlv_diag_file

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_supp_tlv_diag_file;

/*HDR*/
/**
 * @brief Check if a device supports upports PTPv2 license infos
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_tlv_feat_supp
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see ::TODO  //refer to get diag function
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_supp_tlv_ptpv2_license( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_supp_tlv_ptpv2_license( &pmctl->xdev_features );

}  // mbgextio_dev_supp_tlv_ptpv2_license

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_supp_tlv_ptpv2_license;

/*HDR*/
/**
 * @brief Check if a device supports supports PTPv1 License Infos via TLV
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_tlv_feat_supp
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see ::TODO  //refer to get diag function
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_supp_tlv_ptpv1_license( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_supp_tlv_ptpv1_license( &pmctl->xdev_features );

}  // mbgextio_dev_supp_tlv_ptpv1_license

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_supp_tlv_ptpv1_license;


/*HDR*/
/**
 * @brief Check if a device supports supports NTP license infos via TLV
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_tlv_feat_supp
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see ::TODO  //refer to get diag function
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_supp_tlv_ntp_license( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_supp_tlv_ntp_license( &pmctl->xdev_features );

}  // mbgextio_dev_supp_tlv_ntp_license

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_supp_tlv_ntp_license;


/*HDR*/
/**
 * @brief Check if a device supports supportsTime Monitor License infos via TLV
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_tlv_feat_supp
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see ::TODO  //refer to get diag function
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_supp_tlv_time_monitor_license( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_supp_tlv_time_monitor_license( &pmctl->xdev_features );

}  // mbgextio_dev_supp_tlv_time_monitor_license

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_supp_tlv_time_monitor_license;

/*HDR*/
/**
 * @brief Check if a device supports the ::GPS_SAVE_CFG command
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_xfeature
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see ::mbgextio_cmd_save_cfg
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_cmd_save_cfg( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_cmd_save_cfg( &pmctl->xdev_features );

}  // mbgextio_dev_has_cmd_save_cfg

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_cmd_save_cfg;



/*HDR*/
/**
 * @brief Check if a device supports the extended feature monitoring
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_xfeature
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see xdevfeat_has_monitoring
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_monitoring( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_monitoring( &pmctl->xdev_features );

}  // mbgextio_dev_has_monitoring

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_monitoring;



/*HDR*/
/**
 * @brief Check if a device supports the LED API
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_xfeature
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see ::TODO ###
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_led_api( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_led_api( &pmctl->xdev_features );

}  // mbgextio_dev_has_led_api

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_led_api;



/*HDR*/
/**
 * @brief Check if a device supports the LNE API
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_xfeature
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see ::TODO ###
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_lne_api( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_lne_api( &pmctl->xdev_features );

}  // mbgextio_dev_has_lne_api

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_lne_api;



/*HDR*/
/**
 * @brief Check if a device supports the power control API
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else error code from ::check_xfeature
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see ::TODO ###
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_pwr_ctl_api( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_pwr_ctl_api( &pmctl->xdev_features );

}  // mbgextio_dev_has_pwr_ctl_api

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_pwr_ctl_api;



/*HDR*/
/**
 * @brief Check if a device has MBG_XFEATURE_EXT_SYS_INFO
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_ext_sys_info( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_ext_sys_info( &pmctl->xdev_features );

}  // mbgextio_dev_has_ext_sys_info

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_ext_sys_info;



/*HDR*/
/**
 * @brief Check if a device has MBG_XFEATURE_TRANSACTIONS
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_transactions( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_transactions( &pmctl->xdev_features );

}  // mbgextio_dev_has_transactions

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_transactions;



/*HDR*/
/**
 * @brief Check if a device has MBG_XFEATURE_REBOOT
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_reboot( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_reboot( &pmctl->xdev_features );

}  // mbgextio_dev_has_reboot

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_reboot;



/*HDR*/
/**
 * @brief Check if a device has MBG_XFEATURE_REQ_TTM
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see @ref mbgextio_chk_supp_fncs
 * @see mbgextio_get_time
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_req_ttm( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_req_ttm( &pmctl->xdev_features );

}  // mbgextio_dev_has_req_ttm

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_req_ttm;



/*HDR*/
/**
 * @brief Check if a device supports the up-to-date extended multi ref API
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_xmulti_ref( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_xmulti_ref( &pmctl->xdev_features );

}  // mbgextio_dev_has_xmulti_ref

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_xmulti_ref;



/*HDR*/
/**
 * @brief Check if a device supports the extended binary protocl (XBP) feature
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return ::MBG_SUCCESS if supported, else ::MBG_ERR_NOT_SUPP_BY_DEV
 *         or ::MBG_ERR_DEV_NOT_SUPP (see @ref xdevfeat_chk_supp_fncs)
 *
 * @ingroup mbgextio_chk_supp_fncs
 * @see @ref mbgextio_chk_supp_fncs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_dev_has_xbp( const MBG_MSG_CTL *pmctl )
{
  return xdevfeat_has_xbp( &pmctl->xdev_features );

}  // mbgextio_dev_has_xbp

MBGEXTIO_CHK_SUPP_FNC mbgextio_dev_has_xbp;



/*HDR*/
/**
 * @brief Generic reception of a binary message
 *
 * @note A certain message type to be waited for can be specified by
 * passing one of the ::GPS_CMD_CODES.
 * If the special cmd code ::GPS_WILDCARD is specified the function returns
 * successfully after *any* type of binary message has been received.
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[in]     cmd       One of the command codes enumerated in ::GPS_CMD_CODES, or ::GPS_WILDCARD
 * @param[out]    buf       Pointer to a buffer to be filled with the requested data, or NULL
 * @param[in]     buf_size  Size of the buffer specified by buf
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_rcv_msg_unlocked
 * @see ::mbgextio_xmt_msg  //##++++
 * @see ::mbgextio_xmt_cmd
 * @see ::mbgextio_req_data
 * @see ::mbgextio_xmt_cmd_us
 * @see ::mbgextio_req_data_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_rcv_msg_unlocked( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                  GPS_CMD cmd, void *buf, size_t buf_size )
{
  MBG_MSG_RCV_CTL *prctl;
  MBG_MSG_BUFF *pmb;
  char buff[MBGEXTIO_READ_BUFFER_SIZE];
  ssize_t n_bytes = 0;
  int rc = MBG_ERR_UNSPEC;
  int i;
  #if _DEBUG_MSG_TIMING
    MBG_MSG_TIMES *pmt = &pmctl->mt;
  #else
    MBG_MSG_TIMES mt;
    MBG_MSG_TIMES *pmt = &mt;
  #endif

  init_transfer( &pmctl->rcv );
  pmctl->nack_err_code = MBG_SUCCESS;  // will be updated if a NACK msg is received
  mbg_tmo_get_time( &pmt->t_start );
  pmt->t_tmo = pmt->t_start;
  mbg_tmo_add_ms( &pmt->t_tmo, pmctl->msg_rcv_timeout );

  for (;;)  // loop until complete msg received
  {
    n_bytes = 0;

    mbg_tmo_get_time( &pmt->t_now );

    if ( mbg_tmo_time_is_after( &pmt->t_now, &pmt->t_tmo ) )
    {
      rc = MBG_ERR_TIMEOUT;
      goto out;
    }

    switch ( pmctl->conn_type )
    {
      #if _USE_SOCKET_IO
        case MBG_CONN_TYPE_SOCKET:
        {
          struct timeval tv_timeout;
          fd_set fds;
          int max_fd;

          mbg_msec_to_timeval( pmctl->st.sockio.poll_timeout, &tv_timeout );

          FD_ZERO( &fds );
          FD_SET( pmctl->st.sockio.sockfd, &fds );

          #if defined( MBG_TGT_WIN32 )
            // Under Windows an fd is a handle which can't simply
            // be converted to an int, but the first argument of
            // select() is ignored under Windows anyway, so we just
            // set max_fd to 0.
            max_fd = 0;
          #else
            max_fd = pmctl->st.sockio.sockfd;
          #endif

          rc = select( max_fd + 1, &fds, NULL, NULL, &tv_timeout );

          if ( rc == MBG_SOCKET_ERR_RETVAL )  // error, usually < 0, depending on build environment
          {
            rc = mbg_get_last_socket_error( "select failed in mbgextio_rcv_msg_unlocked" );
            goto out;
          }

          if ( rc == 0 )    // timeout
            continue;       // continue loop to check message timeout

          // data is available

          n_bytes = recv( pmctl->st.sockio.sockfd, buff, sizeof( buff ), 0 );

          /* 0 bytes mean remote site closed connection on TCP sockets */
          if ( n_bytes == 0 )
          {
            rc = MBG_ERR_DISCONN;
            goto out;
          }

          if ( n_bytes < 0 )
          {
            rc = mbg_get_last_socket_error( "recv failed in mbgextio_rcv_msg_unlocked" );
            goto out;
          }

        } break;
      #endif  // _USE_SOCKET_IO

      #if _USE_SERIAL_IO
        case MBG_CONN_TYPE_SERIAL:
        {
          rc = mbgserio_read_wait( pmctl->st.p_serio, &buff[0], sizeof( buff[0] ) );

          if ( mbg_rc_is_error( rc ) )
          {
            // Do not continue with reception after character timeout
            // if ( rc == MBG_ERR_TIMEOUT )  // just a character timeout
              // continue;                   // continue loop to check message timeout

            goto out;                     // non-timeout error: fatal, quit
          }

          // rc contains number of received bytes
          n_bytes = rc;

        } break;
      #endif  // _USE_SERIAL_IO

      #if _USE_SERIAL_IO_FTDI
        case MBG_CONN_TYPE_SERIAL_FTDI:
        {
          DWORD bytes_read;
          FT_STATUS status;

          //### TODO   pmctl->dev_poll_timeout
          status = FT_Read( pmctl->st.ftdi.port_handle, &buff[0],
                            sizeof( buff[0] ), &bytes_read );

          if ( status != FT_OK )
          {
            rc = mbg_ftdi_ft_status_to_mbg( status );
            goto out;
          }

          n_bytes = bytes_read;

        } break;
      #endif  // _USE_SERIAL_IO_FTDI

      #if _USE_USB_IO
        case MBG_CONN_TYPE_USB:
        {
          rc = mbgusbio_read( &pmctl->st.usbio, (uint8_t *) buff, sizeof( buff ) );

          if ( mbg_rc_is_error( rc ) )
          {
            if ( rc == MBG_ERR_TIMEOUT )  // just a character timeout
              continue;                   // continue loop to check message timeout

            goto out;                     // non-timeout error: fatal, quit
          }

          // rc contains number of received bytes
          n_bytes = rc;

        } break;
      #endif  // _USE_USB_IO

      #if _USE_USB_DIRECT_IO
        case MBG_CONN_TYPE_USB_DIRECT_IO:
        {
          struct pollfd pfd;

          pfd.fd = pmctl->st.usbdio.usbdiofd;
          pfd.events = POLLIN | POLLRDNORM | POLLRDBAND;
          pfd.revents = 0;

          rc = poll( &pfd, 1, pmctl->st.usbdio.poll_timeout );

          if ( rc == 0 )   // timeout
          {
            rc = MBG_ERR_TIMEOUT;
            goto out;
          }

          if ( rc < 0 )    // error
          {
            rc = mbg_get_last_error( "failed to poll direct USB I/O" );
            goto out;
          }

          if ( pfd.revents & POLLHUP )
          {
            rc = MBG_ERR_DISCONN;
            goto out;
          }

          // Other stuff?
          if ( pfd.revents & ( POLLERR | POLLNVAL ) )
          {
            rc = MBG_ERR_UNSPEC;
            goto out;
          }

          // Read data?
          if ( pfd.revents & ( POLLIN | POLLRDNORM | POLLRDBAND ) )
          {
            rc = read( pmctl->st.usbdio.usbdiofd, (void*)buff, sizeof(buff) );

            if ( rc < 0 )
            {
              rc = mbg_get_last_error( "failed to read direct USB I/O" );
              goto out;
            }

            n_bytes = rc;
          }

        } break;
      #endif

      default:
        rc = MBG_ERR_CONN_TYPE;
        goto out;

    }  // switch

    prctl = &pmctl->rcv;
    pmb = prctl->pmb;

    for ( i = 0; i < n_bytes; i++ )
    {
      char c = buff[i];

      /* check if the new char belongs to a data transfer sequence */
      rc = check_transfer( prctl, c );

      switch ( rc )
      {
        case TR_WAITING:      /* no data transfer sequence in progress */
          #if _USE_MBG_TSTR
            if ( prctl->tstr_rcv_fnc )  /* optionally handle normal, non-protocol data */
              prctl->tstr_rcv_fnc( c, prctl->tstr_rcv_arg );
          #endif

          // intentional fall-through

        case TR_RECEIVING:    /* data transfer sequence in progress, keep waiting */
          // Increase timeout if header has been completely received
          //if ( mbg_rc_is_success( chk_hdr_rcvd( prctl ) ) )
            //mbg_tmo_add_ms( &pmt->t_tmo, pmctl->st.serio.poll_timeout );
          continue;

        case TR_COMPLETE:
          {
            GPS_CMD rcvd_cmd;

            #if _USE_MBG_TSTR
              //### TODO
              // If a valid binary packet has been received then discard
              // a partial time string possibly received before.
            #endif

            rcvd_cmd = pmb->hdr.cmd & ~GPS_CTRL_MSK;

            if ( ( rcvd_cmd == GPS_XBP_PACKET ) && p_addr )
            {
              if ( memcmp( &pmb->u.xbp_msg_data.xbp_addr, p_addr, sizeof( *p_addr ) ) == 0 )
              {
                // If this packet is from the expected XBP node
                mbg_memcpy( pmb, &pmb->u.xbp_msg_data.std_msg, sizeof( pmb->u.xbp_msg_data.std_msg ) );
                rcvd_cmd = pmb->hdr.cmd & ~GPS_CTRL_MSK;
              }
            }

            if( rcvd_cmd == cmd )
            {
              /* the received packet is what we've been waiting for */
              if ( pmb->hdr.cmd & GPS_NACK )
              {
                // If the device has sent an explicit error code then we save that one,
                // otherwise we set the error code to "unspecified".
                // See also ::mbgextio_get_last_nack_err_code
                pmctl->nack_err_code = ( pmb->hdr.len == sizeof( pmb->u.msg_data.i16 ) ) ?
                                         pmb->u.msg_data.i16 : MBG_ERR_UNSPEC;

                rc = MBG_ERR_RCVD_NACK;
                goto out;
              }

              rc = MBG_SUCCESS;
              goto out;
            }

            #if _USE_ENCRYPTION
              // If an encrypted packet has been received then decrypt it.
              if ( rcvd_cmd == GPS_CRYPTED_PACKET )
              {
                rc = decrypt_message( pmctl );

                if ( rc < 0 )  /* decryption error */
                  goto out;

                rcvd_cmd = pmb->hdr.cmd & ~GPS_CTRL_MSK;

                if ( ( rcvd_cmd == GPS_XBP_PACKET ) && p_addr )
                {
                  if ( memcmp( &pmb->u.xbp_msg_data.xbp_addr, p_addr, sizeof( *p_addr ) ) == 0 )
                  {
                    // If this packet is from the expected XBP node
                    mbg_memcpy( pmb, &pmb->u.xbp_msg_data.std_msg, sizeof( pmb->u.xbp_msg_data.std_msg ) );
                    rcvd_cmd = pmb->hdr.cmd & ~GPS_CTRL_MSK;
                  }
                }

                if( rcvd_cmd == cmd )
                {
                  /* the received packet is what we've been waiting for */
                  if ( pmb->hdr.cmd & GPS_NACK )
                  {
                    // If the device has sent an explicit error code then we save that one,
                    // otherwise we set the error code to "unspecified".
                    // See also ::mbgextio_get_last_nack_err_code
                    pmctl->nack_err_code = ( pmb->hdr.len == sizeof( pmb->u.msg_data.i16 ) ) ?
                                             pmb->u.msg_data.i16 : MBG_ERR_UNSPEC;

                    rc = MBG_ERR_RCVD_NACK;
                    goto out;
                  }

                  rc = MBG_SUCCESS;
                  goto out;
                }
              }
            #endif

            // Not waiting for a specific message, so return if any message is complete
            if ( cmd == GPS_WILDCARD )
            {
              rc = MBG_SUCCESS;
              goto out;
            }

            // We are waiting for a specific message, but have just received
            // a message which does not match the expected code.
            // If a message handler callback function has been registered via
            // ::mbgextio_register_msg_callback then we call that function so
            // it can eventually evaluate the message, and the message doesn't
            // get lost. Otherwise the message is simply dropped.
            // See ::mbgextio_register_msg_callback.
            if ( pmctl->msg_handler_fnc )
              pmctl->msg_handler_fnc( pmctl );

            init_transfer( prctl );   // restart receiving
          }
          break;

        case TR_CSUM_HDR:
          rc = MBG_ERR_HDR_CSUM;
          goto out;

        case TR_CSUM_DATA:
          rc = MBG_ERR_DATA_CSUM;
          goto out;

        case TR_OVERFLOW:
          rc = MBG_ERR_OVERFLOW;
          goto out;

        default:    /* any error condition */
          rc = MBG_ERR_UNSPEC;    //##+++++++++++++++++++++ use detailed rc
          goto out;

      }  /* switch */
    }
  }

out:
  if ( mbg_rc_is_error( rc ) )
    goto out2;   //### TODO

  // Successfully received a binary message. If a destinaton buffer
  // has been specified then we copy the received data to that buffer
  // *before* the device mutex is released, so we can be sure the
  // received data isn't overwritten by a different thread.
  if ( buf )
  {
    MBG_MSG_BUFF *pmb = pmctl->rcv.pmb;

    if ( buf_size != pmb->hdr.len )
    {
      #if defined( DEBUG )
        const char *cp = mbgextio_get_cmd_name( cmd );

        fprintf( stderr, "received data size (%u) doesn't match buffer size (%llu) for cmd: %s\n",
                 pmb->hdr.len, (unsigned long long) buf_size, cp ? cp : "(unknown)" );
      #endif

      rc = MBG_ERR_DATA_SIZE;
      goto out2;
    }

    memcpy( buf, &pmb->u.msg_data, buf_size );
  }

out2:
  return rc;

}  // mbgextio_rcv_msg_unlocked



/*HDR*/
/**
 * @brief Generic reception of a binary message
 *
 * @note A certain message type to be waited for can be specified by
 * passing one of the ::GPS_CMD_CODES.
 * If the special cmd code ::GPS_WILDCARD is specified the function returns
 * successfully after *any* type of binary message has been received.
 *
 * //##++ TODO: callback function to handle asynchronous spontaneous messages
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[in]     cmd       One of the command codes enumerated in ::GPS_CMD_CODES, or ::GPS_WILDCARD
 * @param[out]    buf       Pointer to a buffer to be filled with the requested data, or NULL
 * @param[in]     buf_size  Size of the buffer specified by buf
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_rcv_msg_unlocked
 * @see ::mbgextio_xmt_msg  //##++++
 * @see ::mbgextio_xmt_cmd
 * @see ::mbgextio_req_data
 * @see ::mbgextio_xmt_cmd_us
 * @see ::mbgextio_req_data_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_rcv_msg( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                GPS_CMD cmd, void *buf, size_t buf_size )
{
  int rc;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  rc = mbgextio_rcv_msg_unlocked( pmctl, p_addr, cmd, buf, buf_size );   // wait for a message

  #if _USE_MUTEX
    _mbg_mutex_release( &pmctl->dev_mutex );
  #endif

  return rc;

}  // mbgextio_rcv_msg



static /*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_check_ack( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                  GPS_CMD cmd )
{
  int rc = MBG_SUCCESS;

  // Simply return if no acknowledge has been requested
  if ( !( cmd & GPS_REQACK ) )
    goto out;

  rc = mbgextio_rcv_msg_unlocked( pmctl, p_addr, (GPS_CMD) ( cmd &  ~GPS_CTRL_MSK ), NULL, 0 );

  if ( mbg_rc_is_error( rc ) )
    goto out;

  if ( pmctl->rcv.pmb->hdr.cmd & GPS_NACK )
  {
    rc = MBG_ERR_RCVD_NACK;
    goto out;
  }

  if ( !( pmctl->rcv.pmb->hdr.cmd & GPS_ACK ) )
    rc = MBG_ERR_RCVD_NO_ACK;

out:
  return rc;

}  // mbgextio_check_ack



/*HDR*/
/**
 * @brief Set up and send a generic binary message
 *
 * This function should preferably be used only as low level function
 * called from within more specific functions used to send specific data.
 *
 * If this function is called directly with a buffer p to be sent then the
 * device mutex is acquired by this function, and after this the specified
 * buffer p  is copied to the transmit buffer.
 *
 * However, other (higher level) API functions which set up the transmit
 * buffer directly have to acquire the device mutex by themselves *before*
 * they set up the transmit buffer, and then pass p as NULL pointer to
 * indicate the transmit buffer has already been set up. The correct number
 * of bytes to be sent (n_bytes) has to be specified anyway, though.
 *
 * @param[in,out] pmctl    Pointer to a valid message control structure
 * @param[in]     p_addr   Pointer to an XBP address specifier, or NULL
 * @param[in]     cmd      One of the command codes enumerated in ::GPS_CMD_CODES
 * @param[in]     p        Address of a data structure according to the specified cmd parameter, or NULL
 * @param[in]     n_bytes  Size of the data structure addressed by parameter p
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_xmt_msg( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                GPS_CMD cmd, const void *p, uint16_t n_bytes )
{
  MBG_MSG_BUFF *pmb;
  int rc = MBG_ERR_UNSPEC;

  if ( n_bytes > sizeof( pmb->u.msg_data ) )
  {
    rc = MBG_ERR_OVERFLOW;  // bytes to send exceed transmit buffer size

    // If p is not NULL then the transmit mutex has not yet
    // been acquired, and thus we can exit immediately.
    if ( p )
      goto out;

    // Otherwise we have to release the mutex before we exit.
    goto out_release;
  }


  pmb = pmctl->xmt.pmb;

  // If the buffer pointer p is NULL then we assume the transmit buffer
  // has already been set up by the caller, and thus the caller has already
  // acquired the device mutex.
  // However, if a data buffer has been specified (i.e., p != NULL)
  // then we have to acquire the device mutex before we start to
  // set up the transmit buffer.
  if ( p )
  {
    #if _USE_MUTEX
      _mbg_mutex_acquire( &pmctl->dev_mutex );
    #endif
    memcpy( pmb->u.bytes, p, n_bytes );
  }

  pmb->hdr.len = n_bytes;
  pmb->hdr.cmd = cmd;

  rc = xmt_tbuff( pmctl, p_addr );

  if ( mbg_rc_is_success( rc ) )
    rc = mbgextio_check_ack( pmctl, p_addr, cmd );

out_release:
  #if _USE_MUTEX
    _mbg_mutex_release( &pmctl->dev_mutex );
  #endif

out:
  return rc;

}  // mbgextio_xmt_msg



/*HDR*/
/**
 * @brief Transmit a command-only message without additional data.
 *
 * If the caller has or'ed the cmd code with ::GPS_REQACK then this
 * function expects an ACK or a NACK message to be replied by the device.
 *
 * Transmission is protected by a mutex which is acquired before
 * and released after the binary message has been sent.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     cmd    One of the command codes enumerated in ::GPS_CMD_CODES, optionally or'ed with ::GPS_REQACK
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_xmt_msg
 * @see ::mbgextio_rcv_msg_unlocked
 * @see ::mbgextio_req_data
 * @see ::mbgextio_xmt_cmd_us
 * @see ::mbgextio_req_data_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_xmt_cmd( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                GPS_CMD cmd )
{
  int rc;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  rc = xmt_cmd( pmctl, p_addr, cmd );

  // If the call above succeeded then this just means there
  // was no transmission error. It does *not* necessarily mean
  // the command has been received and understood by the device.
  // So if the cmd code has been or'ed with a flag requesting
  // an acknowledge then we wait for an ack msg from the device now.
  if ( mbg_rc_is_success( rc ) )
    rc = mbgextio_check_ack( pmctl, p_addr, cmd );

  #if _USE_MUTEX
    _mbg_mutex_release( &pmctl->dev_mutex );
  #endif

  return rc;

}  // mbgextio_xmt_cmd



/*HDR*/
/**
 * @brief Transmit a message with a single ushort (uint16_t) parameter
 *
 * The uint16_t parameter us is often used to send an index value when requesting
 * a certain element of an array of same data structures.
 *
 * Transmission is protected by a mutex which is acquired before
 * and released after the binary message has been sent.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     cmd    One of the command codes enumerated in ::GPS_CMD_CODES
 * @param[in]     us     The ushort parameter for the command code
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_xmt_msg
 * @see ::mbgextio_rcv_msg_unlocked
 * @see ::mbgextio_xmt_cmd
 * @see ::mbgextio_req_data
 * @see ::mbgextio_req_data_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_xmt_cmd_us( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                   GPS_CMD cmd, uint16_t us )
{
  int rc;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  rc = xmt_cmd_us( pmctl, p_addr, cmd, us );

  if ( mbg_rc_is_success( rc ) )
    rc = mbgextio_check_ack( pmctl, p_addr, cmd );

  #if _USE_MUTEX
    _mbg_mutex_release( &pmctl->dev_mutex );
  #endif

  return rc;

}  // mbgextio_xmt_cmd_us



/*HDR*/
/**
 * @brief Transmit a message without a single ushort (16 bit) parameter
 *
 * The ushort parameter is often used to send an index value when requesting
 * a specific element of an array of data structures.
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[in]     cmd       One of the command codes enumerated in ::GPS_CMD_CODES
 * @param[out]    buf       Pointer to a buffer to be filled with the requested data, or NULL
 * @param[in]     buf_size  Size of the buffer specified by buf
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_xmt_msg
 * @see ::mbgextio_rcv_msg_unlocked
 * @see ::mbgextio_xmt_cmd
 * @see ::mbgextio_xmt_cmd_us
 * @see ::mbgextio_req_data_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_req_data( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                 GPS_CMD cmd, void *buf, size_t buf_size )
{
  int rc;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  rc = xmt_cmd( pmctl, p_addr, cmd );                         // request data

  if ( mbg_rc_is_success( rc ) )
    rc = mbgextio_rcv_msg_unlocked( pmctl, p_addr, cmd, buf, buf_size );   // wait for the reply

  #if _USE_MUTEX
    _mbg_mutex_release( &pmctl->dev_mutex );
  #endif

  return rc;

}  // mbgextio_req_data



/*HDR*/
/**
 * @brief Read a specific element of an array of data structures
 *
 * This function dos not lock/unlock the device mutex, so this has to be done by
 * The type of data is implicitly associated with the cmd parameter.
 * The type of data is implicitly associated with the cmd parameter.
 * Usually the number of supported array elements has to be determined
 * by some other means, e.g. from a field in the ::RECEIVER_INFO structure.
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[in]     cmd       One of the command codes enumerated in ::GPS_CMD_CODES
 * @param[in]     idx       The index of the array element to read
 * @param[out]    buf       Pointer to a buffer to be filled with the requested data, or NULL
 * @param[in]     buf_size  Size of the buffer specified by buf
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_xmt_msg
 * @see ::mbgextio_rcv_msg_unlocked
 * @see ::mbgextio_xmt_cmd
 * @see ::mbgextio_req_data
 * @see ::mbgextio_xmt_cmd_us
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_req_data_idx_unlocked( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                              GPS_CMD cmd, uint16_t idx, void *buf, size_t buf_size )
{
  int rc;

  rc = xmt_cmd_us( pmctl, p_addr, cmd, idx );                 // request data

  if ( mbg_rc_is_success( rc ) )
    rc = mbgextio_rcv_msg_unlocked( pmctl, p_addr, cmd, buf, buf_size );   // wait for the reply

  return rc;

}  // mbgextio_req_data_idx_unlocked



/*HDR*/
/**
 * @brief Read a specific element of an array of data structures
 *
 * The type of data is implicitly associated with the cmd parameter.
 * Usually the number of supported array elements has to be determined
 * by some other means, e.g. from a field in the ::RECEIVER_INFO structure.
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[in]     cmd       One of the command codes enumerated in ::GPS_CMD_CODES
 * @param[in]     idx       The index of the array element to read
 * @param[out]    buf       Pointer to a buffer to be filled with the requested data, or NULL
 * @param[in]     buf_size  Size of the buffer specified by buf
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_xmt_msg
 * @see ::mbgextio_rcv_msg_unlocked
 * @see ::mbgextio_xmt_cmd
 * @see ::mbgextio_req_data
 * @see ::mbgextio_xmt_cmd_us
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_req_data_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                     GPS_CMD cmd, uint16_t idx, void *buf, size_t buf_size )
{
  int rc;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  rc = mbgextio_req_data_idx_unlocked( pmctl, p_addr, cmd, idx, buf, buf_size );

  #if _USE_MUTEX
    _mbg_mutex_release( &pmctl->dev_mutex );
  #endif

  return rc;

}  // mbgextio_req_data_idx



/*HDR*/
/**
 * @brief Read a receiver info structure
 *
 * The ::RECEIVER_INFO should be read at first to identify the connected
 * device and determine the basic features supported by the device.
 *
 * @note Some very old devices may not provide a ::RECEIVER_INFO,
 * so ::mbgextio_setup_receiver_info should be called preferably
 * to read the receiver info using this function, if supported,
 * or set up a default structure for devices which don't provide
 * a receiver info.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_setup_receiver_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_receiver_info( MBG_MSG_CTL *pmctl,
                                            const XBP_ADDR *p_addr, RECEIVER_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_RECEIVER_INFO, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_receiver_info( p );

  return rc;

}  // mbgextio_get_receiver_info



/*HDR*/
/**
 * @brief Read I/O port limits in ::MBG_IO_PORT_LIMITS format
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent by the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_io_port_settings_idx
 * @see ::mbgextio_get_io_port_info_idx
 * @see ::mbgextio_get_io_port_type_info_idx
 * @see ::mbgextio_get_io_port_status_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_io_port_limits( MBG_MSG_CTL *pmctl,
                                                           const XBP_ADDR *p_addr,
                                                           MBG_IO_PORT_LIMITS *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_IO_PORT_LIMITS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) )
    /* There is nothing to swaP */
    _mbg_swab_io_port_limits( p );

  return rc;

}  // mbgextio_get_io_port_limits



/*HDR*/
/**
 * @brief Send I/O port settings in ::MBG_IO_PORT_SETTINGS format
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of I/O port settings to be configured, 0..MBG_IO_PORT_LIMITS::num_ports - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_io_port_limits
 * @see ::mbgextio_get_io_port_info_idx
 * @see ::mbgextio_get_io_port_type_info_idx
 * @see ::mbgextio_get_io_port_status_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_io_port_settings_idx( MBG_MSG_CTL *pmctl,
                                                                 const XBP_ADDR *p_addr,
                                                                 const MBG_IO_PORT_SETTINGS *p,
                                                                 uint8_t idx )
{
  GPS_CMD cmd = GPS_IO_PORT_SETTINGS_IDX | OPT_GPS_ACK_CODE;
  MBG_IO_PORT_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.iop_settings_idx;
  uint16_t sizes[] = MBG_IO_PORT_SETTINGS_IDX_SIZES;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;

  _mbg_swab_io_port_settings_idx( p_data, 0 );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizes[p->type] );

}  // mbgextio_set_io_port_settings_idx



/*HDR*/
/**
 * @brief Read I/O port info in ::MBG_IO_PORT_INFO_IDX format
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be read from the device
 * @param[in]     idx    Index of I/O port info to be read, 0..MBG_IO_PORT_LIMITS::num_ports - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_io_port_limits
 * @see ::mbgextio_set_io_port_settings_idx
 * @see ::mbgextio_get_io_port_type_info_idx
 * @see ::mbgextio_get_io_port_status_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_io_port_info_idx( MBG_MSG_CTL *pmctl,
                                                             const XBP_ADDR *p_addr,
                                                             MBG_IO_PORT_INFO_IDX *p,
                                                             uint8_t idx )
{
  int rc;
  size_t len;
  MBG_MSG_BUFF *rbuf = mbgextio_get_rcv_buffer_addr(pmctl);
  MBG_IO_PORT_INFO_IDX *iopi_idx = &rbuf->u.msg_data.iop_info_idx;

  rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_IO_PORT_INFO_IDX, idx, NULL, 0 );
  if ( mbg_rc_is_error( rc ) )
    return rc;

  /* Check if at least header part is avail */
  if (rbuf->hdr.len < MBG_IO_PORT_INFO_IDX_MIN_SIZE)
    return MBG_ERR_NBYTES;

  _mbg_swab_io_port_info_idx( iopi_idx, 1 );

  len = sizeof(*p);
  if (len > rbuf->hdr.len)
    len = rbuf->hdr.len;

  memcpy(p, &rbuf->u.msg_data, len);

  return MBG_SUCCESS;

}  // mbgextio_get_io_port_info_idx



/*HDR*/
/**
 * @brief Read I/O port type info in ::MBG_IO_PORT_TYPE_INFO_IDX format
 *
 * @param[in,out] pmctl          Pointer to a valid message control structure
 * @param[in]     p_addr         Pointer to an XBP address specifier, or NULL
 * @param[out]    p              Pointer to the data structure to be read from the device
 * @param[in]     port_idx       Index of I/O port to be read, 0..MBG_IO_PORT_LIMITS::num_ports - 1
 * @param[in]     port_type_idx  Index of I/O port type info to be read, 0..MBG_IO_PORT_INFO::num_types - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_io_port_limits
 * @see ::mbgextio_set_io_port_settings_idx
 * @see ::mbgextio_get_io_port_info_idx
 * @see ::mbgextio_get_io_port_status_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_io_port_type_info_idx( MBG_MSG_CTL *pmctl,
                                                                  const XBP_ADDR *p_addr,
                                                                  MBG_IO_PORT_TYPE_INFO_IDX *p,
                                                                  uint8_t port_idx, uint8_t port_type_idx)
{
  int rc;
  size_t len;
  uint16_t idx = ((port_idx << 8) | port_type_idx);
  MBG_MSG_BUFF *rbuf = mbgextio_get_rcv_buffer_addr(pmctl);
  MBG_IO_PORT_TYPE_INFO_IDX *iopti_idx = &rbuf->u.msg_data.iop_type_info_idx;

  rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_IO_PORT_TYPE_INFO_IDX, idx, NULL, 0 );
  if ( mbg_rc_is_error( rc ) )
    return rc;

  /* Check if at least header part is avail */
  if (rbuf->hdr.len < MBG_IO_PORT_TYPE_INFO_IDX_MIN_SIZE)
    return MBG_ERR_NBYTES;

  _mbg_swab_io_port_type_info_idx( iopti_idx, 1 );

  len = sizeof(*p);
  if (len > rbuf->hdr.len)
    len = rbuf->hdr.len;

  memcpy(p, &rbuf->u.msg_data, len);

  return MBG_SUCCESS;

}  // mbgextio_get_io_port_type_info_idx



/*HDR*/
/**
 * @brief Read I/O port status in ::MBG_IO_PORT_STATUS_IDX format
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to be read from the device
 * @param[in]     idx       Index of I/O port to be read, 0..MBG_IO_PORT_LIMITS::num_ports - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_io_port_limits
 * @see ::mbgextio_set_io_port_settings_idx
 * @see ::mbgextio_get_io_port_info_idx
 * @see ::mbgextio_get_io_port_type_info_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_io_port_status_idx( MBG_MSG_CTL *pmctl,
                                                               const XBP_ADDR *p_addr,
                                                               MBG_IO_PORT_STATUS_IDX *p,
                                                               uint8_t idx)
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_IO_PORT_STATUS_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_io_port_status_idx( p );

  return rc;

}  // mbgextio_get_io_port_status_idx



static /*HDR*/
int chk_get_rcvr_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                       RECEIVER_INFO *p )
{
  #if DEBUG_INIT_CONN
    static int cnt;
  #endif

  // Try to read the RECEIVER_INFO structure from the device
  int rc = mbgextio_get_receiver_info( pmctl, p_addr, p );

  #if DEBUG_INIT_CONN
    printf( "get rcvr info %i: %i\n", ++cnt, rc );
  #endif

  return rc;

}  // chk_get_rcvr_info



/*HDR*/
/**
 * @brief Setup a receiver info structure
 *
 * The ::RECEIVER_INFO should be read at first to identify the connected
 * device and determine the basic features supported by the device.
 *
 * @note Some very old devices may not provide a ::RECEIVER_INFO.
 * This function tries to read the ::RECEIVER_INFO from the device,
 * and sets up a default structure if the device doesn't support this.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_receiver_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_setup_receiver_info( MBG_MSG_CTL *pmctl,
                                               const XBP_ADDR *p_addr, RECEIVER_INFO *p )
{
  int rc;

  #if _USE_USB_IO
    // If the device is a legacy USB device then it only supports a
    // proprietary protocol instead of the standard binary protocol.
    // We try to find this out based on the USB vendor and device IDs,
    // and we need to check this *before* we try to read a RECEIVER_INFO
    // structure from the device.
    // See also ::mbgextio_dev_is_legacy
    if ( pmctl->conn_type == MBG_CONN_TYPE_USB )
    {
      RECEIVER_INFO *p_ri = mbgextio_get_receiver_info_addr( pmctl );
      rc = check_setup_legacy_usb_device(&pmctl->st.usbio.mdev_info, p_ri );

      if ( mbg_rc_is_success( rc ) )
      {
        // A legacy device was detected, and an appropriate RECEIVER_INFO
        // for this device has been set up.
        pmctl->device_flags |= MSG_CTL_DEVICE_FLAG_MSK_LEGACY;
        goto out;
      }
    }
  #endif  // _USE_USB_IO

  // Try to read the RECEIVER_INFO structure from the device
  rc = chk_get_rcvr_info( pmctl, p_addr, p );

  if ( mbg_rc_is_success( rc ) )
    goto out;  // RECEIVER_INFO was read successfully

  // Failed, but eventually some synchronization is required,
  // e.g. if the baud rate was just changed, so retry once.
  rc = chk_get_rcvr_info( pmctl, p_addr, p );

  if ( mbg_rc_is_success( rc ) )
    goto out;  // RECEIVER_INFO was read successfully from the device.

  // In all error cases except MBG_ERR_TIMEOUT we return with error anyway.
  if ( rc != MBG_ERR_TIMEOUT )
    goto out;


  // There are a few very old devices which support the binary protocol but
  // don't support reading a RECEIVER_INFO, so we check if we can set up
  // a default structure, depending on a device type.
  //
  // Affected devices, at least:
  //   GPS166    (all versions)
  //   GPS167PC  (all versions)
  //   GPS167PCI < 4.22
  //   GPS167    < 4.17
  //   GPS167SV  < 4.17
  {
    SW_REV sw_rev = { 0 };
    IDENT ident = { { 0 } };

    rc = mbgextio_get_sw_rev( pmctl, p_addr, &sw_rev );

    if ( mbg_rc_is_error( rc ) )  // Failed to read SW revision, so
      goto out;                   // probably no device connected.

    // set up a default receiver info structure
    _setup_default_receiver_info_gps( p );

    // set up some generic device name
    sn_cpy_str_safe( p->model_name, sizeof( p->model_name ), "GPS16x" );

    // copy the software revision info we have read from the device
    p->sw_rev = sw_rev;
    trim_whitespace( p->sw_rev.name );

    #if _USE_MUTEX
      _mbg_mutex_acquire( &pmctl->dev_mutex );
    #endif

    xmt_cmd( pmctl, p_addr, 0x400 );
    xmt_cmd( pmctl, p_addr, 0x20F );
    xmt_cmd( pmctl, p_addr, 0x001 );
    rc = mbgextio_rcv_msg_unlocked( pmctl, p_addr, 0x20F, &ident, sizeof( ident ) );

    #if _USE_MUTEX
      _mbg_mutex_release( &pmctl->dev_mutex );
    #endif

    if ( mbg_rc_is_success( rc ) )
    {
      // Unfortunately there are older devices and firmware versions out there
      // where the serial number is saved word-wise, and others where it is
      // saved byte-wise, so it can be really tricky to get it right.
      // If we want to giver it a try we need to add some other library modules
      #if _TRY_IDENT_DECODE
        mbg_gps_ident_decode( p->sernum, &ident );
      #else
        sn_cpy_str_safe( p->sernum, sizeof( p->sernum ), ident.c );
      #endif
    }

    // Return successful in any case since we could
    // at least read the SW_REV structure.
    rc = MBG_SUCCESS;
  }

out:
  return rc;

}  // mbgextio_setup_receiver_info



/*HDR*/
/**
 * @brief Read the software revision
 *
 * @deprecated This function is deprecated since the ::SW_REV structure
 * is also a member of the ::RECEIVER_INFO. This call may be required,
 * though, for very old devices which don't support the ::RECEIVER_INFO.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_setup_receiver_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_sw_rev( MBG_MSG_CTL *pmctl,
                                               const XBP_ADDR *p_addr, SW_REV *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_SW_REV, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_sw_rev( p );

  return rc;

}  // mbgextio_get_sw_rev



/*HDR*/
/**
 * @brief Read the status of ignore lock
 *
 * @note Only supported if ::GPS_HAS_IGNORE_LOCK
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @see mbgextio_set_ignore_lock
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ignore_lock( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, IGNORE_LOCK *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_IGNORE_LOCK, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab16( p );

  return rc;

}  // mbgextio_get_ignore_lock



/*HDR*/
/**
 * @brief Send the ignore lock configuration
 *
 * @note Only supported if ::GPS_HAS_IGNORE_LOCK
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ignore_lock
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ignore_lock( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, const IGNORE_LOCK *p )
{
  GPS_CMD cmd = GPS_IGNORE_LOCK | OPT_GPS_ACK_CODE;
  IGNORE_LOCK *p_data = &pmctl->xmt.pmb->u.msg_data.ignore_lock;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab16( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ignore_lock



/*HDR*/
/**
 * @brief Read the status of buffered variables
 *
 * @note Only supported if ::GPS_MODEL_HAS_BVAR_STAT
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_bvar_stat( MBG_MSG_CTL *pmctl,
                                             const XBP_ADDR *p_addr, BVAR_STAT *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_BVAR_STAT, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_bvar_stat( p );

  return rc;

}  // mbgextio_get_bvar_stat



/*HDR*/
/**
 * @brief Read the current time as ::TTM strucure
 *
 * @note This function is only supported if the device has ::MBG_XFEATURE_REQ_TTM
 *
 * The returned time is not very accurate since the request can be sent at any time and
 * the response is sent immediately and not after a second change.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see mbgextio_dev_has_req_ttm
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_time( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, TTM *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_TIME, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ttm( p );

  return rc;

}  // mbgextio_get_time



/*HDR*/
/**
 * @brief Set the device's time by sending a ::TTM strucure
 *
 * @note The function is not supported by all devices. Time has to be
 * passed as local time according to the device's ::TZDL settings.
 * New time is only set with some tens of ms accuracy.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_time( MBG_MSG_CTL *pmctl,
                                         const XBP_ADDR *p_addr, const TTM *p )
{
  GPS_CMD cmd = GPS_TIME | OPT_GPS_ACK_CODE;
  TTM *p_data = &pmctl->xmt.pmb->u.msg_data.ttm;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ttm( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_time



/*HDR*/
/**
 * @brief Set the device's position by sending an ::LLA array
 *
 * @note This function is only supported by GPS receivers
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    lla    Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_pos_lla( MBG_MSG_CTL *pmctl,
                                           const XBP_ADDR *p_addr, const LLA lla )
{
  GPS_CMD cmd = GPS_POS_LLA | OPT_GPS_ACK_CODE;
  double *p_lla = pmctl->xmt.pmb->u.msg_data.lla;
  int i;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  for ( i = 0; i < N_LLA; i++ )
  {
    p_lla[i] = lla[i];
    swap_double( &p_lla[i] );
    _mbg_swab_double( &p_lla[i] );
  }

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( LLA ) );

}  // mbgextio_set_pos_lla



/*HDR*/
/**
 * @brief Read the current receiver position as ::LLA array
 *
 * @note This function is only supported by GPS receivers
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    lla    Pointer to the data array to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_pos_lla( MBG_MSG_CTL *pmctl,
                                                const XBP_ADDR *p_addr, LLA lla )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_POS_LLA, lla, sizeof( LLA ) );

  if ( mbg_rc_is_success( rc ) && lla )
  {
    int i;

    for ( i = 0; i < N_LLA; i++ )
      swap_double( &lla[i] );
  }

  return rc;

}  // mbgextio_get_pos_lla



/*HDR*/
/**
 * @brief Read the current receiver position as ::XYZ array
 *
 * @note This function is only supported by GPS receivers
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    xyz    Pointer to the data array to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_pos_xyz( MBG_MSG_CTL *pmctl,
                                                 const XBP_ADDR *p_addr, XYZ xyz )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_POS_XYZ, xyz, sizeof( XYZ ) );

  if ( mbg_rc_is_success( rc ) && xyz )
  {
    int i;

    for ( i = 0; i < N_XYZ; i++ )
      swap_double( &xyz[i] );
  }

  return rc;

}  // mbgextio_get_pos_xyz



/*HDR*/
/**
 * @brief Read the current receiver position as ::POS structure
 *
 * @note This function is only supported by GPS receivers
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p_pos  Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_pos( MBG_MSG_CTL *pmctl,
                                          const XBP_ADDR *p_addr, POS *p_pos )
{
  int rc = mbgextio_get_pos_xyz( pmctl, p_addr, p_pos->xyz );

  if ( mbg_rc_is_error( rc ) )
    goto out;

  rc = mbgextio_get_pos_lla( pmctl, p_addr, p_pos->lla );

  if ( mbg_rc_is_error( rc ) )
    goto out;

  lla_to_dms( p_pos );

out:
  return rc;

}  // mbgextio_get_pos



/*HDR*/
/**
 * @brief Read the local time conversion parameters in ::TZDL format
 *
 * @note Some devices may not support ::TZDL settings
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_tzdl( MBG_MSG_CTL *pmctl,
                                                const XBP_ADDR *p_addr, TZDL *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_TZDL, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_tzdl( p );

  return rc;

}  // mbgextio_get_tzdl



/*HDR*/
/**
 * @brief Set the local time conversion parameters in ::TZDL format
 *
 * @note Some devices may not support ::TZDL settings
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_tzdl( MBG_MSG_CTL *pmctl,
                                       const XBP_ADDR *p_addr, const TZDL *p )
{
  GPS_CMD cmd = GPS_TZDL | OPT_GPS_ACK_CODE;
  TZDL *p_data = &pmctl->xmt.pmb->u.msg_data.tzdl;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_tzdl( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_tzdl



/*HDR*/
/**
 * @brief Read serial port parameters in ::PORT_PARM format
 *
 * @deprecated This function is deprecated since the ::PORT_PARM structure
 * supports only 2 serial ports, and does not not support configuration
 * of a string type. The function ::mbgextio_get_serial_settings should
 * be used instead.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_serial_settings
 * @see ::mbgextio_save_serial_settings
 * @see ::mbgextio_set_port_parm
 * @see ::mbgextio_setup_receiver_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_port_parm( MBG_MSG_CTL *pmctl,
                                               const XBP_ADDR *p_addr, PORT_PARM *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_PORT_PARM, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_port_parm( p );

  return rc;

}  // mbgextio_get_port_parm



/*HDR*/
/**
 * @brief Send serial port parameters in ::PORT_PARM format
 *
 * @deprecated This function is deprecated since the ::PORT_PARM structure
 * supports only 2 serial ports, and does not not support configuration
 * of a string type. The function ::mbgextio_save_serial_settings should
 * be used instead.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_serial_settings
 * @see ::mbgextio_save_serial_settings
 * @see ::mbgextio_get_port_parm
 * @see ::mbgextio_setup_receiver_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_port_parm( MBG_MSG_CTL *pmctl,
                                         const XBP_ADDR *p_addr, const PORT_PARM *p )
{
  GPS_CMD cmd = GPS_PORT_PARM | OPT_GPS_ACK_CODE;
  PORT_PARM *p_data = &pmctl->xmt.pmb->u.msg_data.port_parm;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_port_parm( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_port_parm



/*HDR*/
/**
 * @brief Read the frequency synthesizer settings
 *
 * @note Some devices may not provide a frequency synthesizer
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_synth
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_synth( MBG_MSG_CTL *pmctl,
                                             const XBP_ADDR *p_addr, SYNTH *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_SYNTH, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_synth( p );

  return rc;

}  // mbgextio_get_synth



/*HDR*/
/**
 * @brief Write the frequency synthesizer settings
 *
 * @note Some devices may not provide a frequency synthesizer
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_synth
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_synth( MBG_MSG_CTL *pmctl,
                                       const XBP_ADDR *p_addr, const SYNTH *p )
{
  GPS_CMD cmd = GPS_SYNTH | OPT_GPS_ACK_CODE;
  SYNTH *p_data = &pmctl->xmt.pmb->u.msg_data.synth;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_synth( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_synth



/*HDR*/
/**
 * @brief Read the GPS antenna info structure
 *
 * @note This is only supported by GPS receivers.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ant_info( MBG_MSG_CTL *pmctl,
                                            const XBP_ADDR *p_addr, ANT_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_ANT_INFO, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ant_info( p );

  return rc;

}  // mbgextio_get_ant_info



/*HDR*/
/**
 * @brief Read a user capture event in ::TTM format
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_clr_ucap_buff
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ucap( MBG_MSG_CTL *pmctl,
                                               const XBP_ADDR *p_addr, TTM *p )
{
  int rc = xmt_cmd( pmctl, p_addr, GPS_UCAP );   /* request a set of data */

  if ( mbg_rc_is_error( rc ) )
    goto out;

  // Attention: Older firmware versions may reply with GPS_TIME
  // messages instead of GPS_UCAP messages, and may not send a reply
  // at all if no capture event is available in the on-board FIFO.
  for (;;)
  {
    rc = mbgextio_rcv_msg_unlocked( pmctl, p_addr, -1, NULL, 0 );

    if ( mbg_rc_is_error( rc ) )
      goto out;

    if ( pmctl->rcv.pmb->hdr.cmd == GPS_UCAP )
      break;

    if ( pmctl->rcv.pmb->hdr.cmd == GPS_TIME )
      if ( pmctl->rcv.pmb->hdr.len > 0 )
        if ( pmctl->rcv.pmb->u.msg_data.ttm.channel >= 0 )
          break;
  }

  if ( p )
  {
    // If the length of the msg header is 0 then the capture buffer
    // is empty. This is indicated with 0xFF in the seconds field of
    // the GPS time structure.
    if ( pmctl->rcv.pmb->hdr.len > 0 )
    {
      *p = pmctl->rcv.pmb->u.msg_data.ttm;
      _mbg_swab_ttm( p );
    }
    else
      _ttm_time_set_unavail( p );  // no capture event available
  }

out:
  return rc;

}  // mbgextio_get_ucap



/*HDR*/
/**
 * @brief Read the user capture network global info in ::MBG_UCAP_NET_GLB_INFO format
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_ucap_net
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ucap_net_glb_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                              MBG_UCAP_NET_GLB_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_UCAP_NET_GLB_INFO, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ucap_net_glb_info( p );

  return rc;

} // mbgextio_get_ucap_net_glb_info



/*HDR*/
/**
 * @brief Send the user capture network global configuration in ::MBG_UCAP_NET_GLB_SETTINGS format
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_ucap_net
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ucap_net_glb_settings( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                  const MBG_UCAP_NET_GLB_SETTINGS *p )
{
  GPS_CMD cmd = GPS_UCAP_NET_GLB_INFO | OPT_GPS_ACK_CODE;
  MBG_UCAP_NET_GLB_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.ucap_net_glb_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ucap_net_glb_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ucap_net_glb_settings



/*HDR*/
/**
 * @brief Read the user capture network receiver info with the given index
 * in ::MBG_UCAP_NET_RECV_INFO_IDX format
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received dat
 * @param[in]     idx    Index of the array element to be retrieved, 0..::MBG_UCAP_NET_GLB_INFO::settings::num_recvs
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_ucap_net
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ucap_net_recv_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                   MBG_UCAP_NET_RECV_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_UCAP_NET_RECV_INFO_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ucap_net_recv_info_idx( p );

  return rc;

} // mbgextio_get_ucap_net_recv_info_idx



/*HDR*/
/**
 * @brief Send the user capture network receiver configuration with the given index
 * in ::MBG_UCAP_NET_RECV_SETTINGS_IDX format
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the array element to be sent, 0..::MBG_UCAP_NET_GLB_INFO::settings::num_recvs
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_ucap_net
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ucap_net_recv_settings_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                       const MBG_UCAP_NET_RECV_SETTINGS_IDX *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_UCAP_NET_RECV_INFO_IDX | OPT_GPS_ACK_CODE;
  MBG_UCAP_NET_RECV_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.ucap_net_recv_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ucap_net_recv_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ucap_net_recv_settings_idx



/*HDR*/
/**
 * @brief Read the enable flags controlling when output signals are enabled
 *
 * @note Some devices may not support ::ENABLE_FLAGS
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_enable_flags
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_enable_flags( MBG_MSG_CTL *pmctl,
                                             const XBP_ADDR *p_addr, ENABLE_FLAGS *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_ENABLE_FLAGS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_enable_flags( p );

  return rc;

}  // mbgextio_get_enable_flags



/*HDR*/
/**
 * @brief Send the enable flags controlling when output signals are enabled
 *
 * @note Some devices may not support ::ENABLE_FLAGS
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_enable_flags
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_enable_flags( MBG_MSG_CTL *pmctl,
                                        const XBP_ADDR *p_addr, const ENABLE_FLAGS *p )
{
  GPS_CMD cmd = GPS_ENABLE_FLAGS | OPT_GPS_ACK_CODE;
  ENABLE_FLAGS *p_data = &pmctl->xmt.pmb->u.msg_data.enable_flags;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_enable_flags( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_enable_flags



/*HDR*/
/**
 * @brief Read GPS status info
 *
 * @note This is only supported by GPS receivers
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_gps_stat_info( MBG_MSG_CTL *pmctl,
                                              const XBP_ADDR *p_addr, STAT_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_STAT_INFO, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_stat_info( p );

  return rc;

}  // mbgextio_get_gps_stat_info



/*HDR*/
/**
 * @brief Read the configured length of the antenna cable
 *
 * This is only supported by GPS/GNSS receivers, check ::GPS_MODEL_HAS_ANT_CABLE_LEN
 *
 * @note Some older devices may not reply to this request unless
 * the application has registered itself as terminal application
 * (GPS_AUTO_ON is not sufficient).
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_ant_cable_len
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ant_cable_len( MBG_MSG_CTL *pmctl,
                                              const XBP_ADDR *p_addr, ANT_CABLE_LEN *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_ANT_CABLE_LENGTH, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ant_cable_len( p );

  return rc;

}  // mbgextio_get_ant_cable_len



/*HDR*/
/**
 * @brief Send the GPS antenna cable length configuration
 *
 * This is only supported by GPS/GNSS receivers, check ::GPS_MODEL_HAS_ANT_CABLE_LEN

 * @note Different devices may accept different maximum values, so the
 * written value should be re-read using ::mbgextio_get_ant_cable_len
 * to check if the parameter has been accepted.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ant_cable_len
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ant_cable_len( MBG_MSG_CTL *pmctl,
                                         const XBP_ADDR *p_addr, const ANT_CABLE_LEN *p )
{
  GPS_CMD cmd = GPS_ANT_CABLE_LENGTH | OPT_GPS_ACK_CODE;
  ANT_CABLE_LEN *p_data = &pmctl->xmt.pmb->u.msg_data.ant_cable_len;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ant_cable_len( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ant_cable_len



/*HDR*/
/**
 * @brief Read configuration info and supported features of the device's IRIG output
 *
 * @note This is only supported if ::GPS_HAS_IRIG_TX is set in ::RECEIVER_INFO::features
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_irig_tx_settings
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_irig_tx_info( MBG_MSG_CTL *pmctl,
                                               const XBP_ADDR *p_addr, IRIG_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_IRIG_TX_INFO, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_irig_info( p );

  return rc;

}  // mbgextio_get_irig_tx_info



/*HDR*/
/**
 * @brief Send new configuration settings for the device's IRIG output
 *
 * @note This is only supported if ::GPS_HAS_IRIG_TX is set in ::RECEIVER_INFO::features
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_irig_tx_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_irig_tx_settings( MBG_MSG_CTL *pmctl,
                                           const XBP_ADDR *p_addr, const IRIG_SETTINGS *p )
{
  GPS_CMD cmd = GPS_IRIG_TX_SETTINGS | OPT_GPS_ACK_CODE;
  IRIG_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.irig_tx_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_irig_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_irig_tx_settings



/*HDR*/
/**
 * @brief Read configuration info and supported features for the device's IRIG input
 *
 * @note This is only supported if ::GPS_HAS_IRIG_RX is set in ::RECEIVER_INFO::features
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_irig_rx_settings
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_irig_rx_info( MBG_MSG_CTL *pmctl,
                                                 const XBP_ADDR *p_addr, IRIG_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_IRIG_RX_INFO, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_irig_info( p );

  return rc;

}  // mbgextio_get_irig_rx_info



/*HDR*/
/**
 * @brief Send new configuration settings for the device's IRIG input
 *
 * @note This is only supported if ::GPS_HAS_IRIG_RX is set in ::RECEIVER_INFO::features
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_irig_rx_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_irig_rx_settings( MBG_MSG_CTL *pmctl,
                                             const XBP_ADDR *p_addr, const IRIG_SETTINGS *p )
{
  GPS_CMD cmd = GPS_IRIG_RX_SETTINGS | OPT_GPS_ACK_CODE;
  IRIG_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.irig_rx_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_irig_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_irig_rx_settings



/*HDR*/
/**
 * @brief Read raw IRIG data
 *
 * The ::MBG_RAW_IRIG_DATA structure can be read for debugging purposes,
 * to decode application defined bits in the control field (CF) segment
 * of the incoming time code, etc. Usually the data set is only updated
 * once per second, even if the code format provides many data frames
 * per second.
 *
 * @note Call ::mbgextio_dev_has_raw_irig_data to check if this is supported.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @see ::mbgextio_dev_has_raw_irig_data
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_raw_irig_data( MBG_MSG_CTL *pmctl,
                                                 const XBP_ADDR *p_addr, MBG_RAW_IRIG_DATA *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_RAW_IRIG_DATA, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_raw_irig_data( p );

  return rc;

}  // mbgextio_get_raw_irig_data



/*HDR*/
/**
 * @brief Read current ref offset to %UTC configuration
 *
 * @note This is only supported if ::GPS_HAS_REF_OFFS is set in ::RECEIVER_INFO::features,
 * usually with IRIG receivers
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_ref_offs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ref_offs( MBG_MSG_CTL *pmctl,
                                         const XBP_ADDR *p_addr, MBG_REF_OFFS *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_REF_OFFS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_ref_offs( p );

  return rc;

}  // mbgextio_get_ref_offs



/*HDR*/
/**
 * @brief Send new ref offset to %UTC settings
 *
 * @note This is only supported if ::GPS_HAS_REF_OFFS is set in ::RECEIVER_INFO::features,
 * usually with IRIG receivers
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ref_offs
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ref_offs( MBG_MSG_CTL *pmctl,
                                    const XBP_ADDR *p_addr, const MBG_REF_OFFS *p )
{
  GPS_CMD cmd = GPS_REF_OFFS | OPT_GPS_ACK_CODE;
  MBG_REF_OFFS *p_data = &pmctl->xmt.pmb->u.msg_data.ref_offs;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_mbg_ref_offs( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ref_offs



/*HDR*/
/**
 * @brief Read current debug status
 *
 * @note This is only supported if ::GPS_HAS_DEBUG_STATUS is set in ::RECEIVER_INFO::features,
 * usually with IRIG receivers
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_debug_status( MBG_MSG_CTL *pmctl,
                                          const XBP_ADDR *p_addr, MBG_DEBUG_STATUS *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_DEBUG_STATUS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_debug_status( p );

  return rc;

}  // mbgextio_get_debug_status



/*HDR*/
/**
 * @brief Read current optional settings and supported options
 *
 * @note This is only supported if ::GPS_HAS_OPT_SETTINGS is set in ::RECEIVER_INFO::features
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_opt_settings
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_opt_info( MBG_MSG_CTL *pmctl,
                                          const XBP_ADDR *p_addr, MBG_OPT_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_OPT_INFO, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_opt_info( p );

  return rc;

}  // mbgextio_get_opt_info



/*HDR*/
/**
 * @brief Send new optional settings flags
 *
 * @note This is only supported if ::GPS_HAS_OPT_SETTINGS is set in ::RECEIVER_INFO::features
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_opt_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_opt_settings( MBG_MSG_CTL *pmctl,
                                   const XBP_ADDR *p_addr, const MBG_OPT_SETTINGS *p )
{
  GPS_CMD cmd = GPS_OPT_SETTINGS | OPT_GPS_ACK_CODE;
  MBG_OPT_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.opt_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_mbg_opt_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_opt_settings



/*HDR*/
/**
 * @brief Read information on a specific supported string format
 *
 * Retrieve a single entry from an array of supported string types.
 * The number of supported string types is specified in ::RECEIVER_INFO::n_str_type.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::RECEIVER_INFO::n_str_type - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_all_str_type_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_str_type_info_idx( MBG_MSG_CTL *pmctl,
                              const XBP_ADDR *p_addr, STR_TYPE_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_STR_TYPE_INFO_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
  {
    _mbg_swab_str_type_info_idx( p );

  #if 0  //##++ TODO: check if received idx matches requested idx
    if ( pii.idx != i )
    {
      printf( "** Info for port %i requested, but for %i received.\n",
              pii.idx, i );
      rc = ...;
    }
  #endif
  }

  return rc;

}  // mbgextio_get_str_type_info_idx



/*HDR*/
/**
 * @brief Read an array of all supported string types
 *
 * The number of supported string types is specified in ::RECEIVER_INFO::n_str_type.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    stii   An array which can hold at least ::RECEIVER_INFO::n_str_type entries
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_str_type_info_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_str_type_info( MBG_MSG_CTL *pmctl,
                                           const XBP_ADDR *p_addr, STR_TYPE_INFO_IDX stii[] )
{
  int rc = MBG_SUCCESS;
  RECEIVER_INFO *p_ri = mbgextio_get_receiver_info_addr( pmctl );
  int n_str_type = p_ri->n_str_type;
  uint16_t i;

  for ( i = 0; i < n_str_type; i++ )
  {
    rc = mbgextio_get_str_type_info_idx( pmctl, p_addr, &stii[i], i );

    if ( mbg_rc_is_error( rc ) )
      break;
  }

  return rc;

}  // mbgextio_get_all_str_type_info



/*HDR*/
/**
 * @brief Read current settings and capabilities of a specific serial port
 *
 * The number of serial ports provided by the device is specified in ::RECEIVER_INFO::n_com_ports.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::RECEIVER_INFO::n_com_ports - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_all_port_info
 * @see ::mbgextio_set_port_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_port_info_idx( MBG_MSG_CTL *pmctl,
                                  const XBP_ADDR *p_addr, PORT_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_PORT_INFO_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
  {
    _mbg_swab_port_info_idx( p );

  #if 0  //##++ TODO: check if received idx matches requested idx
    if ( pii.idx != i )
    {
      printf( "** Info for port %i requested, but for %i received.\n",
              pii.idx, i );
      rc = ...;
    }
  #endif
  }

  return rc;

}  // mbgextio_get_port_info_idx



/*HDR*/
/**
 * @brief Read an array of current settings and capabilities of all serial ports
 *
 * The number of serial ports provided by the device is specified in ::RECEIVER_INFO::n_com_ports.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    pii    An array which can hold at least ::RECEIVER_INFO::n_com_ports entries
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_port_info_idx
 * @see ::mbgextio_set_port_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_port_info( MBG_MSG_CTL *pmctl,
                                          const XBP_ADDR *p_addr, PORT_INFO_IDX pii[] )
{
  int rc = MBG_SUCCESS;
  RECEIVER_INFO *p_ri = mbgextio_get_receiver_info_addr( pmctl );
  int n_com_ports = p_ri->n_com_ports;
  uint16_t i;

  for ( i = 0; i < n_com_ports; i++ )
  {
    rc = mbgextio_get_port_info_idx( pmctl, p_addr, &pii[i], i );

    if ( mbg_rc_is_error( rc ) )
      break;
  }

  return rc;

}  // mbgextio_get_all_port_info



/*HDR*/
/**
 * @brief Send configuration settings for a specific serial port
 *
 * The number of serial ports provided by the device is specified in ::RECEIVER_INFO::n_com_ports.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the serial port to be configured, 0..::RECEIVER_INFO::n_com_ports - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_port_info_idx
 * @see ::mbgextio_get_all_port_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_port_settings_idx( MBG_MSG_CTL *pmctl,
                             const XBP_ADDR *p_addr, const PORT_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_PORT_SETTINGS_IDX | OPT_GPS_ACK_CODE;
  PORT_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.port_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->port_settings = *p;
  p_data->idx = idx;
  _mbg_swab_port_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_port_settings_idx



/*HDR*/
/**
 * @brief Read current settings and capabilities of a specific programmable pulse output
 *
 * The number of supported pulse outputs is specified in ::RECEIVER_INFO::n_prg_out.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::RECEIVER_INFO::n_prg_out - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_all_pout_info
 * @see ::mbgextio_set_pout_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_pout_info_idx( MBG_MSG_CTL *pmctl,
                                const XBP_ADDR *p_addr, POUT_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_POUT_INFO_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
  {
    _mbg_swab_pout_info_idx_on_get( p );

  #if 0  //##++ TODO: check if received idx matches requested idx
    if ( pii.idx != i )
    {
      printf( "** Info for port %i requested, but for %i received.\n",
              pii.idx, i );
      rc = ...;
    }
  #endif
  }

  return rc;

}  // mbgextio_get_pout_info_idx



/*HDR*/
/**
 * @brief Read an array of current settings and capabilities of all programmable pulse outputs
 *
 * The number of supported pulse outputs is specified in ::RECEIVER_INFO::n_prg_out.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    pii    An array which can hold at least ::RECEIVER_INFO::n_prg_out entries
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_pout_info_idx
 * @see ::mbgextio_set_pout_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_pout_info( MBG_MSG_CTL *pmctl,
                                        const XBP_ADDR *p_addr, POUT_INFO_IDX *pii )
{
  int rc = MBG_SUCCESS;
  RECEIVER_INFO *p_ri = mbgextio_get_receiver_info_addr( pmctl );
  int n_prg_out = p_ri->n_prg_out;
  uint16_t i;

  memset( pii, 0, n_prg_out * sizeof( *pii ) );

  for ( i = 0; i < n_prg_out; i++ )
  {
    rc = mbgextio_get_pout_info_idx( pmctl, p_addr, &pii[i], i );

    if ( mbg_rc_is_error( rc ) )
      break;
  }

  return rc;

}  // mbgextio_get_all_pout_info



/*HDR*/
/**
 * @brief Send configuration settings for a specific programmable pulse output
 *
 * The number of supported pulse outputs is specified in ::RECEIVER_INFO::n_prg_out.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the pulse output to be configured, 0..RECEIVER_INFO::n_prg_out - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_pout_info_idx
 * @see ::mbgextio_get_all_pout_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_pout_settings_idx( MBG_MSG_CTL *pmctl,
                             const XBP_ADDR *p_addr, const POUT_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_POUT_SETTINGS_IDX | OPT_GPS_ACK_CODE;
  POUT_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.pout_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->pout_settings = *p;
  p_data->idx = idx;
  _mbg_swab_pout_settings_idx_on_set( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_pout_settings_idx



/*HDR*/
/**
 * @brief Send configuration settings for a specific GPIO
 *
 * @note This is only supported if ::GPS_HAS_GPIO is set in ::RECEIVER_INFO::features
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the pulse output to be configured, 0..MBG_GPIO_CFG_LIMITS::num_io - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_all_gpio_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_gpio_settings_idx( MBG_MSG_CTL *pmctl,
                             const XBP_ADDR *p_addr, const MBG_GPIO_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_GPIO_SETTINGS_IDX | OPT_GPS_ACK_CODE;
  MBG_GPIO_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.gpio_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;

  _mbg_swab_mbg_gpio_settings_idx( p_data, 0 );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_gpio_settings_idx



/*HDR*/
/**
 * @brief Clear the user capture event buffer on-board the device
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ucap
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_clr_ucap_buff( MBG_MSG_CTL *pmctl,
                                                      const XBP_ADDR *p_addr )
{
  return mbgextio_xmt_cmd( pmctl, p_addr, GPS_CLR_UCAP_BUFF );

}  // mbgextio_clr_ucap_buff



/*HDR*/
/**
 * @brief Read time scale configuration parameters
 *
 * @note Some devices may not support a configurable time scale
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_time_scale_settings
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_time_scale_info( MBG_MSG_CTL *pmctl,
                                            const XBP_ADDR *p_addr, MBG_TIME_SCALE_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_TIME_SCALE, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_time_scale_info( p );

  return rc;

}  // mbgextio_get_time_scale_info



/*HDR*/
/**
 * @brief Send new time scale configuration settings
 *
 * @note Some devices may not support a configurable time scale
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_time_scale_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_time_scale_settings( MBG_MSG_CTL *pmctl,
                             const XBP_ADDR *p_addr, const MBG_TIME_SCALE_SETTINGS *p )
{
  GPS_CMD cmd = GPS_TIME_SCALE | OPT_GPS_ACK_CODE;
  MBG_TIME_SCALE_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.time_scale_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_mbg_time_scale_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_time_scale_settings



/*HDR*/
/**
 * @brief Clear the on-board event log
 *
 * @note Some devices don't provide an on-board event log
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_num_evt_log_entries
 * @see ::mbgextio_get_first_evt_log_entry
 * @see ::mbgextio_get_next_evt_log_entry
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_clr_evt_log( MBG_MSG_CTL *pmctl,
                                                    const XBP_ADDR *p_addr )
{
  return mbgextio_xmt_cmd( pmctl, p_addr, GPS_CLR_EVT_LOG );

}  // mbgextio_clr_evt_log



/*HDR*/
/**
 * @brief Read the current number of entries in the on-board event log
 *
 * @note Some devices don't provide an on-board event log
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_clr_evt_log
 * @see ::mbgextio_get_first_evt_log_entry
 * @see ::mbgextio_get_next_evt_log_entry
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_num_evt_log_entries( MBG_MSG_CTL *pmctl,
                                           const XBP_ADDR *p_addr, MBG_NUM_EVT_LOG_ENTRIES *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_NUM_EVT_LOG_ENTRIES, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_num_evt_log_entries( p );

  return rc;

}  // mbgextio_get_num_evt_log_entries



/*HDR*/
/**
 * @brief Return the first entry from the on-board event log
 *
 * This resets an internal counter, so subsequent calls to
 * ::mbgextio_get_next_evt_log_entry will retrieve the following entries.
 *
 * @note Some devices don't provide an on-board event log
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_clr_evt_log
 * @see ::mbgextio_get_num_evt_log_entries
 * @see ::mbgextio_get_next_evt_log_entry
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_first_evt_log_entry( MBG_MSG_CTL *pmctl,
                                               const XBP_ADDR *p_addr, MBG_EVT_LOG_ENTRY *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_FIRST_EVT_LOG_ENTRY, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_evt_log_entry( p );

  return rc;

}  // mbgextio_get_first_evt_log_entry



/*HDR*/
/**
 * @brief Return the next entry from the on-board event log
 *
 * This increments an internal counter, so subsequent calls will
 * return subsequent entries. ::mbgextio_get_first_evt_log_entry
 * should be called first to reset the counter and retrieve the
 * oldest log entry.
 *
 * @note Some devices don't provide an on-board event log
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_clr_evt_log
 * @see ::mbgextio_get_num_evt_log_entries
 * @see ::mbgextio_get_first_evt_log_entry
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_next_evt_log_entry( MBG_MSG_CTL *pmctl,
                                                const XBP_ADDR *p_addr, MBG_EVT_LOG_ENTRY *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_NEXT_EVT_LOG_ENTRY, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_evt_log_entry( p );

  return rc;

}  // mbgextio_get_next_evt_log_entry



/*HDR*/
/**
 * @brief Read the current IMS state and supported IMS features
 *
 * @note This is only supported if ::mbgextio_dev_has_ims returns MBG_SUCCESS
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ims_sensor_state_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ims_state( MBG_MSG_CTL *pmctl,
                                           const XBP_ADDR *p_addr, MBG_IMS_STATE *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_IMS_STATE, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_ims_state( p );

  return rc;

}  // mbgextio_get_ims_state



/*HDR*/
/**
 * @brief Read sensor values from a specified sensor on the device
 *
 * Info on supported sensors can be retrieved using ::mbgextio_get_ims_state.
 * Valid range for the sensor index is [0..::MBG_IMS_STATE::num_sensors - 1].
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    The index of the array element to read
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ims_state
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ims_sensor_state_idx( MBG_MSG_CTL *pmctl,
                              const XBP_ADDR *p_addr, MBG_IMS_SENSOR_STATE_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_IMS_SENSOR_STATE_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_ims_sensor_state_idx( p );

  return rc;

}  // mbgextio_get_ims_sensor_state_idx



/*HDR*/
/**
 * @brief Set the XMR holdover interval
 *
 * @todo In which case is this supported?
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_holdover_interval_counter
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_holdover_interval( MBG_MSG_CTL *pmctl,
                                           const XBP_ADDR *p_addr, const XMR_HOLDOVER_INTV *p )
{
  GPS_CMD cmd = GPS_XMR_HOLDOVER_INTV | OPT_GPS_ACK_CODE;
  XMR_HOLDOVER_INTV *p_data = &pmctl->xmt.pmb->u.msg_data.xmr_holdover_intv;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_xmr_holdover_intv( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_holdover_interval



/*HDR*/
/**
 * @brief Read the XMR holdover interval counter
 *
 * @todo In which case is this supported?
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_holdover_interval
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_holdover_interval_counter( MBG_MSG_CTL *pmctl,
                              const XBP_ADDR *p_addr, XMR_HOLDOVER_INTV *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_XMR_HOLDOVER_INTV, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_xmr_holdover_intv( p );

  return rc;

}  // mbgextio_get_holdover_interval_counter



/*HDR*/
/**
 * @brief Read the multi ref info from a device
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_xmulti_ref_info_idx( MBG_MSG_CTL *pmctl,
                              const XBP_ADDR *p_addr, XMULTI_REF_INFO_IDX *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_MULTI_REF_INFO, p, sizeof( *p ) );

  if ( ( rc == MBG_SUCCESS ) && p )
    *p = pmctl->rcv.pmb->u.msg_data.xmulti_ref_info_idx;

  return rc;

}  // mbgextio_get_xmulti_ref_info_idx



/*HDR*/
/**
 * @brief Read the local time conversion configuration in ::TZCODE format
 *
 * @note Some devices may not support ::TZCODE configuration
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_tzcode
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_tzcode( MBG_MSG_CTL *pmctl,
                                              const XBP_ADDR *p_addr, TZCODE *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, PZF_TZCODE, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_tzcode( p );

  return rc;

}  // mbgextio_get_tzcode



/*HDR*/
/**
 * @brief Set the local time conversion configuration in ::TZCODE format
 *
 * @note Some devices may not support ::TZCODE configuration
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_tzcode
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_tzcode( MBG_MSG_CTL *pmctl,
                                        const XBP_ADDR *p_addr, const TZCODE *p )
{
  GPS_CMD cmd = PZF_TZCODE | OPT_GPS_ACK_CODE;
  TZCODE *p_data = &pmctl->xmt.pmb->u.msg_data.tzcode;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_tzcode( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_tzcode



/*HDR*/
/**
 * @brief Send new configuration settings for the device's HAVEQUICK output
 *
 * @note This is only supported if ::GPS_HAS_HAVEQUICK is set in ::RECEIVER_INFO::features
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see  //##+++++++++++++++++++++
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_hq_tx_settings( MBG_MSG_CTL *pmctl,
                                           const XBP_ADDR *p_addr, const HAVEQUICK_SETTINGS *p )
{
  GPS_CMD cmd = GPS_HAVEQUICK_TX_SETTINGS | OPT_GPS_ACK_CODE;
  HAVEQUICK_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.havequick_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_havequick_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_hq_tx_settings



/*HDR*/
/**
 * @brief Send new configuration settings for the device's HAVEQUICK input
 *
 * @note This is only supported if ::MULTI_REF_HAVEQUICK is > 0 in ::XMULTI_REF_INSTANCES::n_inst  //##+++++++++++++++++++++++ ???
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see  //##+++++++++++++++++++++
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_hq_rx_settings( MBG_MSG_CTL *pmctl,
                                        const XBP_ADDR *p_addr, const HAVEQUICK_SETTINGS *p )
{
  GPS_CMD cmd = GPS_HAVEQUICK_RX_SETTINGS | OPT_GPS_ACK_CODE;
  HAVEQUICK_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.havequick_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_havequick_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_hq_rx_settings



/*HDR*/
/**
 * @brief Read the distance from transmitter ::TR_DISTANCE format
 *
 * @note Some devices may not support ::TR_DISTANCE configuration
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_tr_distance
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_tr_distance( MBG_MSG_CTL *pmctl,
                                            const XBP_ADDR *p_addr, TR_DISTANCE *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, PZF_TR_DISTANCE, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_tr_distance( p );

  return rc;

}  // mbgextio_get_tr_distance



/*HDR*/
/**
 * @brief Set the transmitter distance (km) in ::TR_DISTANCE format
 *
 * @note Some devices may not support ::TR_DISTANCE configuration
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_tr_distance
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_tr_distance( MBG_MSG_CTL *pmctl,
                                      const XBP_ADDR *p_addr, const TR_DISTANCE *p )
{
  GPS_CMD cmd = PZF_TR_DISTANCE | OPT_GPS_ACK_CODE;
  TR_DISTANCE *p_data = &pmctl->xmt.pmb->u.msg_data.tr_distance;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_tr_distance( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_tr_distance



/*HDR*/
/**
 * @brief Read current GNSS mode settings and supported features
 *
 * @note Some devices may not support GNSS configuration
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to a ::MBG_GNSS_MODE_INFO structure to be filled up
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_gnss_mode_settings
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_gnss_mode_info( MBG_MSG_CTL *pmctl,
                                         const XBP_ADDR *p_addr, MBG_GNSS_MODE_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_GNSS_MODE, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_gnss_mode_info( p );

  return rc;

}  // mbgextio_get_gnss_mode_info



/*HDR*/
/**
 * @brief Write GNSS mode settings
 *
 * @note Some devices may not support GNSS configuration
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     p      Pointer to a ::MBG_GNSS_MODE_SETTINGS structure to be sent
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_gnss_mode_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_gnss_mode_settings( MBG_MSG_CTL *pmctl,
                                    const XBP_ADDR *p_addr, const MBG_GNSS_MODE_SETTINGS *p )
{
  GPS_CMD cmd = GPS_GNSS_MODE | OPT_GPS_ACK_CODE;
  MBG_GNSS_MODE_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.gnss_mode_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_mbg_gnss_mode_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_gnss_mode_settings



/*HDR*/
/**
 * @brief ::TODO
 *
 * ### TODO this is obsolete
 * Retrieve a single entry from an array of supported string types.
 * The number of supported string types is specified in ::RECEIVER_INFO::n_str_type.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::TODO
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_gnss_sat_info( MBG_MSG_CTL *pmctl,
                              const XBP_ADDR *p_addr, GNSS_SAT_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_GNSS_SAT_INFO, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_gnss_sat_info( p );

  return rc;

}  // mbgextio_get_gnss_sat_info



/*HDR*/
/**
 * @brief ::TODO
 *
 * ### TODO
 * Retrieve a single entry from an array of supported string types.
 * The number of supported string types is specified in ::RECEIVER_INFO::n_str_type.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::RECEIVER_INFO::n_str_type - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::TODO
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_gnss_sat_info_idx( MBG_MSG_CTL *pmctl,
                              const XBP_ADDR *p_addr, GNSS_SAT_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_GNSS_SAT_INFO_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
  {
    _mbg_swab_gnss_sat_info_idx( p );

  #if 0  //##++ TODO: check if received idx matches requested idx
    if ( pii.idx != i )
    {
      printf( "** Info for port %i requested, but for %i received.\n",
              pii.idx, i );
      rc = ...;
    }
  #endif
  }

  return rc;

}  // mbgextio_get_gnss_sat_info_idx



/*HDR*/
/**
 * @brief ::TODO Read an array of all supported string types
 *
 * The number of supported string types is specified in ::RECEIVER_INFO::n_str_type.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    gsii   ::TODO
 * @param[in]     p_mi   ::TODO
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::TODO
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_gnss_sat_info( MBG_MSG_CTL *pmctl,
                              const XBP_ADDR *p_addr, GNSS_SAT_INFO_IDX gsii[],
                              const MBG_GNSS_MODE_INFO *p_mi )
{
  int rc = MBG_SUCCESS;
  int n_supp = num_bits_set( p_mi->supp_gnss_types );
  uint16_t i;

  for ( i = 0; i < n_supp; i++ )
  {
    rc = mbgextio_get_gnss_sat_info_idx( pmctl, p_addr, &gsii[i], i );

    if ( mbg_rc_is_error( rc ) )
      break;
  }

  return rc;

}  // mbgextio_get_all_gnss_sat_info



/*HDR*/
/**
 * @brief ::TODO
 *
 * ### TODO
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::RECEIVER_INFO::n_str_type - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see @ref group_gnss_sv_status ::TODO
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_gnss_sv_status_idx( MBG_MSG_CTL *pmctl,
                              const XBP_ADDR *p_addr, GNSS_SV_STATUS_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_GNSS_SV_STATUS_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
  {
    _mbg_swab_gnss_sv_status_idx( p );

  #if 0  //##++ TODO: check if received idx matches requested idx
    if ( pii.idx != i )
    {
      printf( "** Info for port %i requested, but for %i received.\n",
              pii.idx, i );
      rc = ...;
    }
  #endif
  }

  return rc;

}  // mbgextio_get_gnss_sv_status_idx



#if 0  ///### TODO

 /*HDR*/
/**
 * @brief ::TODO
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    gsii   ::TODO
 * @param[in]     p_mi   ::TODO
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::TODO
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_gnss_sat_info( MBG_MSG_CTL *pmctl,
                              const XBP_ADDR *p_addr, GNSS_SAT_INFO_IDX gsii[],
                              const MBG_GNSS_MODE_INFO *p_mi )
{
  int rc = MBG_SUCCESS;
  int n_supp = num_bits_set( p_mi->supp_gnss_types );
  uint16_t i;

  for ( i = 0; i < n_supp; i++ )
  {
    rc = mbgextio_get_gnss_sat_info_idx( pmctl, p_addr, &gsii[i], i );

    if ( mbg_rc_is_error( rc ) )
      break;
  }

  return rc;

}  // mbgextio_get_all_gnss_sat_info

#endif



/*HDR*/
/**
 * @brief Read ::XMULTI_REF_INSTANCES
 *
 * Only if ::GPS_HAS_XMULTI_REF is set in ::RECEIVER_INFO::features AND
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_xmr_ext_src_info_idx
 * @see ::mbgextio_get_xmr_info_idx
 * @see ::mbgextio_get_xmr_status_idx
 * @see ::mbgextio_set_xmr_settings_idx
 * @see ::mbgextio_get_xmr_ext_source_stats_idx
 * @see ::mbgextio_get_xmr_ext_source_metrics_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_xmr_instances( MBG_MSG_CTL *pmctl,
                                                          const XBP_ADDR *p_addr,
                                                          XMULTI_REF_INSTANCES *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_XMR_INSTANCES, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) )
    _mbg_swab_xmulti_ref_instances( p );

  return rc;

}  // mbgextio_get_xmr_instances



/*HDR*/
/**
 * @brief Read ::XMR_EXT_SRC_INFO_IDX
 *
 * Only if ::GPS_HAS_XMULTI_REF is set in ::RECEIVER_INFO::features AND
 *         ::XMRIF_MSK_EXT_SRC_INFO_SUPP is set in ::XMULTI_REF_INSTANCES::flags.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::XMULTI_REF_INSTANCES::n_xmr_settings-1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_xmr_instances
 * @see ::mbgextio_get_xmr_holdover_status
 * @see ::mbgextio_get_xmr_info_idx
 * @see ::mbgextio_get_xmr_status_idx
 * @see ::mbgextio_set_xmr_settings_idx
 * @see ::mbgextio_get_xmr_ext_source_stats_idx
 * @see ::mbgextio_get_xmr_ext_source_metrics_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_xmr_ext_src_info_idx( MBG_MSG_CTL *pmctl,
                                                                 const XBP_ADDR *p_addr,
                                                                 XMR_EXT_SRC_INFO_IDX *p,
                                                                 uint16_t idx)
{
  int rc;

  p->idx = idx;

  rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_XMR_EXT_SRC_INFO_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) )
    _mbg_swab_xmr_ext_src_info_idx( p );

  return rc;

}  // mbgextio_get_xmr_ext_src_info_idx



/*HDR*/
/**
 * @brief Read ::XMR_HOLDOVER_STATUS
 *
 * Only if ::GPS_HAS_XMULTI_REF is set in ::RECEIVER_INFO::features AND
 *         ::XMRIF_MSK_HOLDOVER_STATUS_SUPP is set in ::XMULTI_REF_INSTANCES::flags.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_xmr_instances
 * @see ::mbgextio_get_xmr_ext_src_info_idx
 * @see ::mbgextio_get_xmr_info_idx
 * @see ::mbgextio_get_xmr_status_idx
 * @see ::mbgextio_set_xmr_settings_idx
 * @see ::mbgextio_get_xmr_ext_source_stats_idx
 * @see ::mbgextio_get_xmr_ext_source_metrics_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_xmr_holdover_status( MBG_MSG_CTL *pmctl,
                                                                const XBP_ADDR *p_addr,
                                                                XMR_HOLDOVER_STATUS *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_XMR_HOLDOVER_STATUS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) )
    _mbg_swab_xmr_holdover_status( p );

  return rc;

}  // mbgextio_get_xmr_holdover_status



/*HDR*/
/**
 * @brief Read ::XMULTI_REF_INFO_IDX for a specific XMR source
 *
 * Only if ::GPS_HAS_XMULTI_REF is set in ::RECEIVER_INFO::features.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..0..::XMULTI_REF_INSTANCES::n_xmr_settings-1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_xmr_instances
 * @see ::mbgextio_get_xmr_ext_src_info_idx
 * @see ::mbgextio_get_xmr_holdover_status
 * @see ::mbgextio_get_xmr_status_idx
 * @see ::mbgextio_set_xmr_settings_idx
 * @see ::mbgextio_get_xmr_ext_source_stats_idx
 * @see ::mbgextio_get_xmr_ext_source_metrics_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_xmr_info_idx( MBG_MSG_CTL *pmctl,
                                                         const XBP_ADDR *p_addr,
                                                         XMULTI_REF_INFO_IDX *p,
                                                         uint16_t idx)
{
  int rc;

  p->idx = idx;

  rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_XMR_INFO_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) )
    _mbg_swab_xmulti_ref_info_idx( p );

  return rc;

}  // mbgextio_get_xmr_info_idx


/*HDR*/
/**
 * @brief Read ::XMULTI_REF_STATUS_IDX for a specific XMR source
 *
 * Only if ::GPS_HAS_XMULTI_REF is set in ::RECEIVER_INFO::features.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..0..::XMULTI_REF_INSTANCES::n_xmr_settings-1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_xmr_instances
 * @see ::mbgextio_get_xmr_ext_src_info_idx
 * @see ::mbgextio_get_xmr_holdover_status
 * @see ::mbgextio_get_xmr_info_idx
 * @see ::mbgextio_set_xmr_settings_idx
 * @see ::mbgextio_get_xmr_ext_source_stats_idx
 * @see ::mbgextio_get_xmr_ext_source_metrics_idx
 * @see ::mbgextio_get_xmr_ext_source_metrics_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_xmr_status_idx( MBG_MSG_CTL *pmctl,
                                                          const XBP_ADDR *p_addr,
                                                          XMULTI_REF_STATUS_IDX *p,
                                                          uint16_t idx)
{
  int rc = MBG_ERR_UNSPEC;
  GPS_CMD cmd = GPS_XMR_STATUS_IDX;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  //### TODO FIXME  rc = mbgextio_req_data_idx_unlocked( pmctl, p_addr, GPS_XMR_STATUS_IDX, idx, p, sizeof( *p ) );

  rc = xmt_cmd_us( pmctl, p_addr, cmd, idx );                 // request data

  if ( mbg_rc_is_error( rc ) )
    goto out_release;

  for (;;)
  {
    rc = mbgextio_rcv_msg_unlocked( pmctl, p_addr, cmd, p, sizeof( *p ) );   // wait for the reply

    if ( mbg_rc_is_error( rc ) )
      break;

    _mbg_swab_xmulti_ref_status_idx( p );

    if ( p->idx == idx )
      break;

    #if defined( DEBUG )
      fprintf( stderr, "GPS_XMR_STATUS_IDX received reply idx %u, expected %u\n", idx, p->idx );
    #endif
  }

out_release:

  #if _USE_MUTEX
    _mbg_mutex_release( &pmctl->dev_mutex );
  #endif

  return rc;

}  // mbgextio_get_xmr_status_idx


/*HDR*/
/**
 * @brief Read ::XMR_STATS_IDX for a specific XMR source
 *
 * Only if ::mbgextio_dev_xmr_has_ext_source_stats returns ::MBG_SUCCESS for the specific index
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::MULTI_REF_TYPES-1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_xmr_instances
 * @see ::mbgextio_get_xmr_ext_src_info_idx
 * @see ::mbgextio_get_xmr_holdover_status
 * @see ::mbgextio_get_xmr_info_idx
 * @see ::mbgextio_set_xmr_settings_idx
 * @see ::mbgextio_get_xmr_info_idx
 * @see ::mbgextio_get_xmr_status_idx
 * @see ::mbgextio_get_xmr_ext_source_metrics_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_xmr_ext_source_stats_idx( MBG_MSG_CTL *pmctl,
                                                                     const XBP_ADDR *p_addr,
                                                                     XMR_STATS_IDX *p,
                                                                     uint16_t idx)
{
  int rc;

  p->idx = idx;

  rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_XMR_STATS_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) )
    _mbg_swab_xmr_stats_idx( p );

  return rc;

}  // mbgextio_get_xmr_ext_source_stats_idx


/*HDR*/
/**
 * @brief Read ::XMR_METRICS_IDX for a specific XMR source
 *
 * Only if ::mbgextio_dev_xmr_has_ext_source_stats returns ::MBG_SUCCESS for the specific index
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::MULTI_REF_TYPES-1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_xmr_instances
 * @see ::mbgextio_get_xmr_ext_src_info_idx
 * @see ::mbgextio_get_xmr_holdover_status
 * @see ::mbgextio_get_xmr_info_idx
 * @see ::mbgextio_set_xmr_settings_idx
 * @see ::mbgextio_get_xmr_info_idx
 * @see ::mbgextio_get_xmr_status_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_xmr_ext_source_metrics_idx( MBG_MSG_CTL *pmctl,
                                                                       const XBP_ADDR *p_addr,
                                                                       XMR_METRICS_IDX *p,
                                                                       uint16_t idx)
{
  int rc;

  p->idx = idx;

  rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_XMR_METRICS_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) )
    _mbg_swab_xmr_metrics_idx( p );

  return rc;

}  // mbgextio_get_xmr_ext_source_metrics_idx


/*HDR*/
/**
 * @brief Set ::XMULTI_REF_SETTINGS for a specific XMR source
 *
 * Only if ::GPS_HAS_XMULTI_REF is set in ::RECEIVER_INFO::features.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the array element to be retrieved, 0..0..::XMULTI_REF_INSTANCES::n_xmr_settings-1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_xmr_instances
 * @see ::mbgextio_get_xmr_ext_src_info_idx
 * @see ::mbgextio_get_xmr_holdover_status
 * @see ::mbgextio_get_xmr_info_idx
 * @see ::mbgextio_get_xmr_status_idx
 * @see ::mbgextio_get_xmr_ext_source_stats_idx
 * @see ::mbgextio_get_xmr_ext_source_metrics_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_xmr_settings_idx( MBG_MSG_CTL *pmctl,
                                                             const XBP_ADDR *p_addr,
                                                             const XMULTI_REF_SETTINGS *p,
                                                             uint16_t idx)
{
  GPS_CMD cmd = GPS_XMR_SETTINGS_IDX | OPT_GPS_ACK_CODE;
  XMULTI_REF_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.xmulti_ref_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;
  _mbg_swab_xmulti_ref_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_xmr_settings_idx



/*HDR*/
/**
 * @brief Read the lan interface configuration ::LAN_IF_INFO format
 *
 * @note ptp or xmr with ntp feature must set
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see //##+++++++++++++
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_lan_if_info( MBG_MSG_CTL *pmctl,
                                                  const XBP_ADDR *p_addr, LAN_IF_INFO *p )
{
  int rc;
  rc = mbgextio_req_data( pmctl, p_addr, GPS_LAN_IF_INFO, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_lan_if_info( p );

  return rc;

}  // mbgextio_get_lan_if_info



/*HDR*/
/**
 * @brief Read the lan ipv4 configuration state ::IP4_SETTINGS format
 *
 * @note ptp or xmr with ntp feature must set
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see //##+++++++++++++
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ip4_settings( MBG_MSG_CTL *pmctl,
                                                 const XBP_ADDR *p_addr, IP4_SETTINGS *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_IP4_SETTINGS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ip4_settings( p );

  return rc;

}  // mbgextio_get_ip4_settings



/*HDR*/
/**
 * @brief Save the current ipv4 settings ::IP4_SETTINGS format
 *
 * @note ptp or xmr with ntp feature must set
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see //##+++++++++++++
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ip4_settings( MBG_MSG_CTL *pmctl,
                                           const XBP_ADDR *p_addr, const IP4_SETTINGS *p )
{
  GPS_CMD cmd = GPS_IP4_SETTINGS | OPT_GPS_ACK_CODE;
  IP4_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.ip4_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ip4_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ip4_settings



/*HDR*/
/**
 * @brief Read the current lan ipv4 state ::IP4_SETTINGS format
 *
 * @note ptp or xmr with ntp feature must set
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see //##+++++++++++++
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ip4_state( MBG_MSG_CTL *pmctl,
                                              const XBP_ADDR *p_addr, IP4_SETTINGS *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_IP4_STATE, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ip4_settings( p );

  return rc;

}  // mbgextio_get_ip4_state



/*HDR*/
/**
 * @brief Read the current state of PTP device ::PTP_STATE format
 *
 * @note ptp feature must set
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see //##+++++++++++++
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ptp_state( MBG_MSG_CTL *pmctl,
                                                      const XBP_ADDR *p_addr, PTP_STATE *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_PTP_STATE, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ptp_state( p );

  return rc;

}  // mbgextio_get_ptp_state



/*HDR*/
/**
 * @brief Read the ptp configuration ::PTP_CFG_INFO format
 *
 * @note ptp feature must set
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see //##+++++++++++++
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ptp_cfg_info( MBG_MSG_CTL *pmctl,
                                              const XBP_ADDR *p_addr, PTP_CFG_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_PTP_CFG, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ptp_cfg_info( p );

  return rc;

}  // mbgextio_get_ptp_cfg_info



/*HDR*/
/**
 * @brief Read the ptp configuration ::PTP_UC_MASTER_CFG_LIMITS format
 *
 * @note ptp feature must set
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see //##+++++++++++++
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ptp_uc_master_cfg_limits( MBG_MSG_CTL *pmctl,
                                             const XBP_ADDR *p_addr, PTP_UC_MASTER_CFG_LIMITS *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_PTP_UC_MASTER_CFG_LIMITS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ptp_uc_master_cfg_limits( p );

  return rc;

}  // mbgextio_get_ptp_uc_master_cfg_limits



/*HDR*/
/**
 * @brief Read the ptp configuration ::ALL_PTP_UC_MASTER_INFO_IDX format
 *
 * @note ptp feature must set and read number of ptp unicast masters before.
 *
 * @param[in,out] pmctl                     Pointer to a valid message control structure
 * @param[in]     p_addr                    Pointer to an XBP address specifier, or NULL
 * @param[out]    ptp_uc_master_info_idx    Pointer to an array of ::PTP_UC_MASTER_INFO_IDX structures to be filled up
 * @param[in]     ptp_uc_master_cfg_limits  Pointer to a ::PTP_UC_MASTER_CFG_LIMITS structure read before
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see //##+++++++++++++ TODO
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_ptp_uc_master_info( MBG_MSG_CTL *pmctl,
                        const XBP_ADDR *p_addr, PTP_UC_MASTER_INFO_IDX *ptp_uc_master_info_idx,
                        const PTP_UC_MASTER_CFG_LIMITS *ptp_uc_master_cfg_limits )
{
  int rc= MBG_SUCCESS;
  uint16_t i;

  for ( i = 0; i < ptp_uc_master_cfg_limits->n_supp_master; i++ )
  {
    xmt_cmd_us( pmctl, p_addr, GPS_PTP_UC_MASTER_CFG, i );

    rc = mbgextio_rcv_msg_unlocked( pmctl, p_addr, GPS_PTP_UC_MASTER_CFG, NULL, 0 );   //### TODO

    if ( mbg_rc_is_success( rc ) && ptp_uc_master_info_idx )
    {
      PTP_UC_MASTER_INFO_IDX *p = &ptp_uc_master_info_idx[i];
      *p = pmctl->rcv.pmb->u.msg_data.ptp_uc_master_info_idx;
      _mbg_swab_ptp_uc_master_info_idx( p );
    }
    else
      break;
  }

  return rc;

}  // mbgextio_get_all_ptp_uc_master_info



/*HDR*/
/**
 * @brief Send configuration settings for PTP
 *
 * @note This is only supported by PTP devices
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ptp_cfg_settings( MBG_MSG_CTL *pmctl,
                                         const XBP_ADDR *p_addr, const PTP_CFG_SETTINGS *p )
{
  GPS_CMD cmd = GPS_PTP_CFG | OPT_GPS_ACK_CODE;
  PTP_CFG_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.ptp_cfg_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ptp_cfg_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ptp_cfg_settings


/*HDR*/
/**
 * @brief Send PTP unicast master configuration settings for a PTP unicast slave mode device.
 *
 * The number of supported PTP Masters is specified in ::PTP_UC_MASTER_CFG_LIMITS::n_supp_master.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the PTP unicast grandmaster to be configured, 0 ... PTP_UC_MASTER_CFG_LIMITS::n_supp_master - 1 .
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_ptp_cfg_settings
 * @see ::mbgextio_get_all_ptp_uc_master_info
 * @see ::mbgextio_get_ptp_uc_master_cfg_limits
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_get_ptp_state
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ptp_uc_master_settings_idx( MBG_MSG_CTL *pmctl,
                                     const XBP_ADDR *p_addr, const PTP_UC_MASTER_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_PTP_UC_MASTER_CFG | OPT_GPS_ACK_CODE;
  PTP_UC_MASTER_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.ptp_uc_master_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;
  _mbg_swab_ptp_uc_master_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ptp_uc_master_settings_idx



/*HDR*/
/**
 * @brief Read the PTPv1 default dataset in ::MBG_PTP_V1_DEFAULT_DATASET format
 *
 * @note This is only supported if ::PTP_CFG_MSK_HAS_V1_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_set_ptp_v1_default_dataset
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ptp_v1_default_dataset( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                   MBG_PTP_V1_DEFAULT_DATASET *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_PTP_V1_DEFAULT_DS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ptp_v1_default_dataset( p );

  return rc;

}  // mbgextio_get_ptp_v1_default_dataset



/*HDR*/
/**
 * @brief Send PTPv1 default dataset to a device in ::MBG_PTP_V1_DEFAULT_DATASET format
 *
 * @note This is only supported if ::PTP_CFG_MSK_HAS_V1_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_get_ptp_v1_default_dataset
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ptp_v1_default_dataset( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                   const MBG_PTP_V1_DEFAULT_DATASET *p )
{
  GPS_CMD cmd = GPS_PTP_V1_DEFAULT_DS | OPT_GPS_ACK_CODE;
  MBG_PTP_V1_DEFAULT_DATASET *p_data = &pmctl->xmt.pmb->u.msg_data.ptp_v1_default_dataset;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ptp_v1_default_dataset( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ptp_v1_default_dataset



/*HDR*/
/**
 * @brief Read the PTPv1 current dataset in ::MBG_PTP_V1_CURRENT_DATASET format
 *
 * @note only supported if ::PTP_CFG_MSK_HAS_V1_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_set_ptp_v1_current_dataset
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ptp_v1_current_dataset( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                   MBG_PTP_V1_CURRENT_DATASET *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_PTP_V1_CURRENT_DS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ptp_v1_current_dataset( p );

  return rc;

}  // mbgextio_get_ptp_v1_current_dataset



/*HDR*/
/**
 * @brief Send PTPv1 current dataset to a device in ::MBG_PTP_V1_CURRENT_DATASET format
 *
 * @note This is only supported if ::PTP_CFG_MSK_HAS_V1_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_get_ptp_v1_current_dataset
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ptp_v1_current_dataset( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                   const MBG_PTP_V1_CURRENT_DATASET *p )
{
  GPS_CMD cmd = GPS_PTP_V1_CURRENT_DS | OPT_GPS_ACK_CODE;
  MBG_PTP_V1_CURRENT_DATASET *p_data = &pmctl->xmt.pmb->u.msg_data.ptp_v1_current_dataset;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ptp_v1_current_dataset( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ptp_v1_current_dataset



/*HDR*/
/**
 * @brief Read the PTPv1 parent dataset in ::MBG_PTP_V1_PARENT_DATASET format
 *
 * @note only supported if ::PTP_CFG_MSK_HAS_V1_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_set_ptp_v1_parent_dataset
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ptp_v1_parent_dataset( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                  MBG_PTP_V1_PARENT_DATASET *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_PTP_V1_PARENT_DS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ptp_v1_parent_dataset( p );

  return rc;

}  // mbgextio_get_ptp_v1_parent_dataset



/*HDR*/
/**
 * @brief Send PTPv1 parent dataset to a device in ::MBG_PTP_V1_PARENT_DATASET format
 *
 * @note This is only supported if ::PTP_CFG_MSK_HAS_V1_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_get_ptp_v1_parent_dataset
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ptp_v1_parent_dataset( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                   const MBG_PTP_V1_PARENT_DATASET *p )
{
  GPS_CMD cmd = GPS_PTP_V1_PARENT_DS | OPT_GPS_ACK_CODE;
  MBG_PTP_V1_PARENT_DATASET *p_data = &pmctl->xmt.pmb->u.msg_data.ptp_v1_parent_dataset;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ptp_v1_parent_dataset( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ptp_v1_parent_dataset



/*HDR*/
/**
 * @brief Read the PTPv1 time properties dataset in ::MBG_PTP_V1_TIME_PROPERTIES_DATASET format
 *
 * @note only supported if ::PTP_CFG_MSK_HAS_V1_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_set_ptp_v1_time_properties_dataset
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ptp_v1_time_properties_dataset( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                           MBG_PTP_V1_TIME_PROPERTIES_DATASET *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_PTP_V1_TIME_PROP_DS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ptp_v1_time_properties_dataset( p );

  return rc;

}  // mbgextio_get_ptp_v1_time_properties_dataset



/*HDR*/
/**
 * @brief Send PTPv1 time properties dataset to a device in ::MBG_PTP_V1_TIME_PROPERTIES_DATASET format
 *
 * @note This is only supported if ::PTP_CFG_MSK_HAS_V1_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_get_ptp_v1_time_properties_dataset
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ptp_v1_time_properties_dataset( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                           const MBG_PTP_V1_TIME_PROPERTIES_DATASET *p )
{
  GPS_CMD cmd = GPS_PTP_V1_TIME_PROP_DS | OPT_GPS_ACK_CODE;
  MBG_PTP_V1_TIME_PROPERTIES_DATASET *p_data = &pmctl->xmt.pmb->u.msg_data.ptp_v1_time_properties_dataset;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ptp_v1_time_properties_dataset( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ptp_v1_time_properties_dataset



/*HDR*/
/**
 * @brief Read the PTPv1 port dataset with the appropriate index in ::MBG_PTP_V1_PORT_DATASET_IDX format
 *
 * @note only supported if ::PTP_CFG_MSK_HAS_V1_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to return the received data
 * @param[in]     idx       Index of the port dataset be received, 0..::MBG_PTP_V1_DEFAULT_DATASET::number_ports - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_set_ptp_v1_port_dataset_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ptp_v1_port_dataset_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                    MBG_PTP_V1_PORT_DATASET_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_PTP_V1_PORT_DS_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ptp_v1_port_dataset_idx( p );

  return rc;

}  // mbgextio_get_ptp_v1_port_dataset_idx



/*HDR*/
/**
 * @brief Send PTPv1 port dataset with the appropriate index to a device in ::MBG_PTP_V1_PORT_DATASET format
 *
 * @note This is only supported if ::PTP_CFG_MSK_HAS_V1_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to be sent to the device
 * @param[in]     idx       Index of the port dataset to be written, 0..::MBG_PTP_V1_DEFAULT_DATASET::number_ports - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_get_ptp_v1_port_dataset_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ptp_v1_port_dataset_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                    const MBG_PTP_V1_PORT_DATASET *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_PTP_V1_PORT_DS_IDX | OPT_GPS_ACK_CODE;
  MBG_PTP_V1_PORT_DATASET_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.ptp_v1_port_dataset_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->port_dataset = *p;
  p_data->idx = idx;
  _mbg_swab_ptp_v1_port_dataset_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ptp_v1_port_dataset_idx



/*HDR*/
/**
 * @brief Read the PTPv2 default dataset in ::MBG_PTP_V2_DEFAULT_DATASET format
 *
 * @note This is only supported if ::PTP_CFG_MSK_HAS_V2_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_set_ptp_v2_default_dataset
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ptp_v2_default_dataset( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                   MBG_PTP_V2_DEFAULT_DATASET *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_PTP_V2_DEFAULT_DS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ptp_v2_default_dataset( p );

  return rc;

}  // mbgextio_get_ptp_v2_default_dataset



/*HDR*/
/**
 * @brief Send PTPv2 default dataset to a device in ::MBG_PTP_V2_DEFAULT_DATASET format
 *
 * @note This is only supported if ::PTP_CFG_MSK_HAS_V2_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_get_ptp_v2_default_dataset
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ptp_v2_default_dataset( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                   const MBG_PTP_V2_DEFAULT_DATASET *p )
{
  GPS_CMD cmd = GPS_PTP_V2_DEFAULT_DS | OPT_GPS_ACK_CODE;
  MBG_PTP_V2_DEFAULT_DATASET *p_data = &pmctl->xmt.pmb->u.msg_data.ptp_v2_default_dataset;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ptp_v2_default_dataset( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ptp_v2_default_dataset



/*HDR*/
/**
 * @brief Read the PTPv2 current dataset in ::MBG_PTP_V2_CURRENT_DATASET format
 *
 * @note only supported if ::PTP_CFG_MSK_HAS_V2_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_set_ptp_v2_current_dataset
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ptp_v2_current_dataset( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                   MBG_PTP_V2_CURRENT_DATASET *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_PTP_V2_CURRENT_DS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ptp_v2_current_dataset( p );

  return rc;

}  // mbgextio_get_ptp_v2_current_dataset



/*HDR*/
/**
 * @brief Send PTPv2 current dataset to a device in ::MBG_PTP_V2_CURRENT_DATASET format
 *
 * @note This is only supported if ::PTP_CFG_MSK_HAS_V2_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_get_ptp_v2_current_dataset
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ptp_v2_current_dataset( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                   const MBG_PTP_V2_CURRENT_DATASET *p )
{
  GPS_CMD cmd = GPS_PTP_V2_CURRENT_DS | OPT_GPS_ACK_CODE;
  MBG_PTP_V2_CURRENT_DATASET *p_data = &pmctl->xmt.pmb->u.msg_data.ptp_v2_current_dataset;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ptp_v2_current_dataset( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ptp_v2_current_dataset



/*HDR*/
/**
 * @brief Read the PTPv2 parent dataset in ::MBG_PTP_V2_PARENT_DATASET format
 *
 * @note only supported if ::PTP_CFG_MSK_HAS_V2_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_set_ptp_v2_parent_dataset
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ptp_v2_parent_dataset( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                  MBG_PTP_V2_PARENT_DATASET *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_PTP_V2_PARENT_DS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ptp_v2_parent_dataset( p );

  return rc;

}  // mbgextio_get_ptp_v2_parent_dataset



/*HDR*/
/**
 * @brief Send PTPv2 parent dataset to a device in ::MBG_PTP_V2_PARENT_DATASET format
 *
 * @note This is only supported if ::PTP_CFG_MSK_HAS_V2_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_get_ptp_v2_parent_dataset
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ptp_v2_parent_dataset( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                   const MBG_PTP_V2_PARENT_DATASET *p )
{
  GPS_CMD cmd = GPS_PTP_V2_PARENT_DS | OPT_GPS_ACK_CODE;
  MBG_PTP_V2_PARENT_DATASET *p_data = &pmctl->xmt.pmb->u.msg_data.ptp_v2_parent_dataset;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ptp_v2_parent_dataset( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ptp_v2_parent_dataset



/*HDR*/
/**
 * @brief Read the PTPv2 time properties dataset in ::MBG_PTP_V2_TIME_PROPERTIES_DATASET format
 *
 * @note only supported if ::PTP_CFG_MSK_HAS_V2_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_set_ptp_v2_time_properties_dataset
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ptp_v2_time_properties_dataset( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                           MBG_PTP_V2_TIME_PROPERTIES_DATASET *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_PTP_V2_TIME_PROP_DS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ptp_v2_time_properties_dataset( p );

  return rc;

}  // mbgextio_get_ptp_v2_time_properties_dataset



/*HDR*/
/**
 * @brief Send PTPv2 time properties dataset to a device in ::MBG_PTP_V2_TIME_PROPERTIES_DATASET format
 *
 * @note This is only supported if ::PTP_CFG_MSK_HAS_V2_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_get_ptp_v2_time_properties_dataset
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ptp_v2_time_properties_dataset( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                           const MBG_PTP_V2_TIME_PROPERTIES_DATASET *p )
{
  GPS_CMD cmd = GPS_PTP_V2_TIME_PROP_DS | OPT_GPS_ACK_CODE;
  MBG_PTP_V2_TIME_PROPERTIES_DATASET *p_data = &pmctl->xmt.pmb->u.msg_data.ptp_v2_time_properties_dataset;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ptp_v2_time_properties_dataset( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ptp_v2_time_properties_dataset



/*HDR*/
/**
 * @brief Read the PTPv2 port dataset with the appropriate index in ::MBG_PTP_V2_PORT_DATASET_IDX format
 *
 * @note only supported if ::PTP_CFG_MSK_HAS_V2_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to return the received data
 * @param[in]     idx       Index of the port dataset be received, 0..::MBG_PTP_V2_DEFAULT_DATASET::number_ports - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_set_ptp_v2_port_dataset_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ptp_v2_port_dataset_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                    MBG_PTP_V2_PORT_DATASET_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_PTP_V2_PORT_DS_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ptp_v2_port_dataset_idx( p );

  return rc;

}  // mbgextio_get_ptp_v2_port_dataset_idx



/*HDR*/
/**
 * @brief Send PTPv2 port dataset with the appropriate index to a device in ::MBG_PTP_V2_PORT_DATASET format
 *
 * @note This is only supported if ::PTP_CFG_MSK_HAS_V2_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[out]    p         Pointer to the data structure to be sent to the device
 * @param[in]     idx       Index of the port dataset be written, 0..::MBG_PTP_V2_DEFAULT_DATASET::number_ports - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_get_ptp_v2_port_dataset_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ptp_v2_port_dataset_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                    const MBG_PTP_V2_PORT_DATASET *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_PTP_V2_PORT_DS_IDX | OPT_GPS_ACK_CODE;
  MBG_PTP_V2_PORT_DATASET_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.ptp_v2_port_dataset_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->port_dataset = *p;
  p_data->idx = idx;
  _mbg_swab_ptp_v2_port_dataset_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ptp_v2_port_dataset_idx



/*HDR*/
/**
 * @brief Read the monitoring limits in ::MBG_MONITORING_LIMITS format
 *
 * @note extended feature ::MBG_XFEATURE_MONITORING must be set
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_monitoring_limits( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                              MBG_MONITORING_LIMITS *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_MONITORING_LIMITS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_monitoring_limits( p );

  return rc;

}  // mbgextio_get_monitoring_limits



/*HDR*/
/**
 * @brief Read the SNMP global info in ::MBG_SNMP_GLB_INFO format
 *
 * @note ::MBG_MONITORING_TYPE_MSK_SNMP must be set in MBG_MONITORING_LIMITS::supp_types
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 * @see ::mbgextio_set_snmp_glb_settings
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_snmp_glb_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                          MBG_SNMP_GLB_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_SNMP_GLB, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_snmp_glb_info( p );

  return rc;

}  // mbgextio_get_snmp_glb_info



/*HDR*/
/**
 * @brief Send the SNMP global settings in ::MBG_SNMP_GLB_SETTINGS format
 *
 * @note ::MBG_MONITORING_TYPE_MSK_SNMP must be set in MBG_MONITORING_LIMITS::supp_types
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 * @see ::mbgextio_get_snmp_glb_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_snmp_glb_settings( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                              const MBG_SNMP_GLB_SETTINGS *p )
{
  GPS_CMD cmd = GPS_SNMP_GLB | OPT_GPS_ACK_CODE;
  MBG_SNMP_GLB_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.snmp_glb_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_snmp_glb_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_snmp_glb_settings



/*HDR*/
/**
 * @brief Read SNMP v1 or v2 info in ::MBG_SNMP_V12_INFO_IDX format
 *
 * @note ::MBG_MONITORING_TYPE_MSK_SNMP must be set in MBG_MONITORING_LIMITS::supp_types
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the SNMP info to be received, 0 ... MBG_SNMP_GLB_SETTINGS::num_v12_settings - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 * @see ::mbgextio_get_snmp_glb_info
 * @see ::mbgextio_set_snmp_v12_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_snmp_v12_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                              MBG_SNMP_V12_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx(pmctl, p_addr, GPS_SNMP_V12_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_snmp_v12_info_idx( p );

  return rc;

}  // mbgextio_get_snmp_v12_info_idx



/*HDR*/
/**
 * @brief Write SNMP v1 or v2 settings in ::MBG_SNMP_V12_SETTINGS format
 *
 * @note ::MBG_MONITORING_TYPE_MSK_SNMP must be set in MBG_MONITORING_LIMITS::supp_types
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the SNMP settings to be written, 0 ... MBG_SNMP_GLB_SETTINGS::num_v12_settings - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 * @see ::mbgextio_get_snmp_glb_info
 * @see ::mbgextio_get_snmp_v12_info_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_snmp_v12_settings_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                  const MBG_SNMP_V12_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_SNMP_V12_IDX | OPT_GPS_ACK_CODE;
  MBG_SNMP_V12_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.snmp_v12_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;
  _mbg_swab_snmp_v12_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_snmp_v12_settings_idx



/*HDR*/
/**
 * @brief Read SNMP v1 or v2 trap info in ::MBG_SNMP_V12_TRAP_INFO_IDX format
 *
 * @note ::MBG_MONITORING_TYPE_MSK_SNMP must be set in MBG_MONITORING_LIMITS::supp_types
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the SNMP trap info to be received, 0 ... MBG_SNMP_GLB_SETTINGS::num_v12_trap_receivers - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 * @see ::mbgextio_get_snmp_glb_info
 * @see ::mbgextio_set_snmp_v12_trap_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_snmp_v12_trap_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                   MBG_SNMP_V12_TRAP_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx(pmctl, p_addr, GPS_SNMP_V12_TRAP_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_snmp_v12_trap_info_idx( p );

  return rc;

}  // mbgextio_get_snmp_v12_trap_info_idx



/*HDR*/
/**
 * @brief Write SNMP v1 or v2 trap settings in ::MBG_SNMP_V12_TRAP_SETTINGS format
 *
 * @note ::MBG_MONITORING_TYPE_MSK_SNMP must be set in MBG_MONITORING_LIMITS::supp_types
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the SNMP settings to be written, 0 ... MBG_SNMP_GLB_SETTINGS::num_v12_trap_receivers - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 * @see ::mbgextio_get_snmp_glb_info
 * @see ::mbgextio_get_snmp_v12_trap_info_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_snmp_v12_trap_settings_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                       const MBG_SNMP_V12_TRAP_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_SNMP_V12_TRAP_IDX | OPT_GPS_ACK_CODE;
  MBG_SNMP_V12_TRAP_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.snmp_v12_trap_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;
  _mbg_swab_snmp_v12_trap_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_snmp_v12_trap_settings_idx



/*HDR*/
/**
 * @brief Read SNMP v3 info in ::MBG_SNMP_V3_INFO_IDX format
 *
 * @note ::MBG_MONITORING_TYPE_MSK_SNMP must be set in MBG_MONITORING_LIMITS::supp_types
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the SNMP info to be received, 0 ... MBG_SNMP_GLB_SETTINGS::num_v3_settings - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 * @see ::mbgextio_get_snmp_glb_info
 * @see ::mbgextio_set_snmp_v3_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_snmp_v3_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                             MBG_SNMP_V3_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx(pmctl, p_addr, GPS_SNMP_V3_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_snmp_v3_info_idx( p );

  return rc;

}  // mbgextio_get_snmp_v3_info_idx



/*HDR*/
/**
 * @brief Write SNMP v3 settings in ::MBG_SNMP_V3_SETTINGS format
 *
 * @note ::MBG_MONITORING_TYPE_MSK_SNMP must be set in MBG_MONITORING_LIMITS::supp_types
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the SNMP settings to be written, 0 ... MBG_SNMP_GLB_SETTINGS::num_v3_settings - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 * @see ::mbgextio_get_snmp_glb_info
 * @see ::mbgextio_get_snmp_v3_info_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_snmp_v3_settings_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                 const MBG_SNMP_V3_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_SNMP_V3_IDX | OPT_GPS_ACK_CODE;
  MBG_SNMP_V3_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.snmp_v3_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;
  _mbg_swab_snmp_v3_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_snmp_v3_settings_idx



/*HDR*/
/**
 * @brief Read SNMP v3 trap info in ::MBG_SNMP_V3_TRAP_INFO_IDX format
 *
 * @note ::MBG_MONITORING_TYPE_MSK_SNMP must be set in MBG_MONITORING_LIMITS::supp_types
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the SNMP trap info to be received, 0 ... MBG_SNMP_GLB_SETTINGS::num_v3_trap_receivers - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 * @see ::mbgextio_get_snmp_glb_info
 * @see ::mbgextio_set_snmp_v3_trap_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_snmp_v3_trap_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                  MBG_SNMP_V3_TRAP_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx(pmctl, p_addr, GPS_SNMP_V3_TRAP_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_snmp_v3_trap_info_idx( p );

  return rc;

}  // mbgextio_get_snmp_v3_trap_info_idx



/*HDR*/
/**
 * @brief Write SNMP v3 trap settings in ::MBG_SNMP_V3_TRAP_SETTINGS format
 *
 * @note ::MBG_MONITORING_TYPE_MSK_SNMP must be set in MBG_MONITORING_LIMITS::supp_types
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the SNMP trap settings to be written, 0 ... MBG_SNMP_GLB_SETTINGS::num_v3_trap_receivers - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 * @see ::mbgextio_get_snmp_glb_info
 * @see ::mbgextio_get_snmp_v3_trap_info_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_snmp_v3_trap_settings_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                      const MBG_SNMP_V3_TRAP_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_SNMP_V3_TRAP_IDX | OPT_GPS_ACK_CODE;
  MBG_SNMP_V3_TRAP_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.snmp_v3_trap_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;
  _mbg_swab_snmp_v3_trap_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_snmp_v3_trap_settings_idx



/*HDR*/
/**
 * @brief Read monitoring event info in ::MBG_EVENT_INFO_IDX format
 *
 * @note extended feature ::MBG_XFEATURE_MONITORING must be set
 * idx shall be the type of the appropriate event, it should be checked
 * if this event is supported in ::MBG_MONITORING_LIMITS::supp_events
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the event info to be received, type of the appropriate event
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 * @see ::mbgextio_get_monitoring_limits
 * @see ::mbgextio_set_event_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_event_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                           MBG_EVENT_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx(pmctl, p_addr, GPS_EVENT_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_event_info_idx( p );

  return rc;

}  // mbgextio_get_event_info_idx



/*HDR*/
/**
 * @brief Write monitoring event settings in ::MBG_EVENT_SETTINGS format
 *
 * @note extended feature ::MBG_XFEATURE_MONITORING must be set
 * idx shall be the type of the appropriate event, it should be checked
 * if this event is supported in ::MBG_MONITORING_LIMITS::supp_events
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the event settings to be written, type of the appropriate event
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 * @see ::mbgextio_get_monitoring_limits
 * @see ::mbgextio_get_event_info_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_event_settings_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                               const MBG_EVENT_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_EVENT_IDX | OPT_GPS_ACK_CODE;
  MBG_EVENT_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.event_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;
  _mbg_swab_event_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_event_settings_idx



/*HDR*/
/**
 * @brief Read the monitoring status in ::MBG_MONITORING_STATUS format
 *
 * @note extended feature ::MBG_XFEATURE_MONITORING must be set
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_monitoring_status( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                              MBG_MONITORING_STATUS *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_MONITORING_STATUS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_monitoring_status( p );

  return rc;

}  // mbgextio_get_monitoring_status


/*HDR*/
/**
 * @brief Read monitoring event status in ::MBG_EVENT_STATUS_IDX format
 *
 * @note extended feature ::MBG_XFEATURE_MONITORING must be set
 * idx shall be the type of the appropriate event, it should be checked
 * if this event is supported in ::MBG_MONITORING_LIMITS::supp_events
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the event info to be received, type of the appropriate event
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 * @see ::mbgextio_get_monitoring_limits
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_event_status_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                             MBG_EVENT_STATUS_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx(pmctl, p_addr, GPS_EVENT_STAT_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_event_status_idx( p );

  return rc;

}  // mbgextio_get_event_status_idx


/*HDR*/
/**
 * @brief Read the ntp global information ::NTP_GLB_INFO format
 *
 * @note ntp feature must set
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_clnt_mode_info
 * @see ::mbgextio_get_ntp_peer_settings_idx
 * @see ::mbgextio_set_ntp_peer_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ntp_glb_info( MBG_MSG_CTL *pmctl,
                                                  const XBP_ADDR *p_addr, NTP_GLB_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_NTP_GLB_CFG, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ntp_glb_info( p );

  return rc;

}  // mbgextio_get_ntp_glb_info


/*HDR*/
/**
 * @brief Send the NTP global configuration
 *
 * @note This is only supported by NTP devices
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ant_cable_len
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ntp_glb_info( MBG_MSG_CTL *pmctl,
                                   const XBP_ADDR *p_addr, const NTP_GLB_SETTINGS *p )
{
  GPS_CMD cmd = GPS_NTP_GLB_CFG | OPT_GPS_ACK_CODE;
  NTP_GLB_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.ntp_glb_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ntp_glb_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ntp_glb_info



#if defined( _PRELIMINARY_CODE )

/*HDR*/
/**
 * @brief Read the NTP symmetric key limits in ::NTP_SYMM_KEY_LIMITS format
 *
 * @note ::NTP_MSK_SYMM_KEYS must be set in ::NTP_GLB_INFO::supp_flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_glb_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ntp_symm_key_limits( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, NTP_SYMM_KEY_LIMITS *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_NTP_SYMM_KEY_LIMITS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ntp_symm_key_limits( p );

  return rc;

}  // mbgextio_get_ntp_symm_key_limits



/*HDR*/
/**
 * @brief Read the NTP symmetric key info with the given index in ::NTP_SYMM_KEY_INFO_IDX format
 *
 * @note ::NTP_MSK_SYMM_KEYS must be set in ::NTP_GLB_INFO::supp_flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the NTP symmetric key info to be queried, 0 ... ::NTP_GLB_SETTINGS::num_symm_keys - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_glb_info
 * @see ::mbgextio_get_ntp_symm_key_limits
 * @see ::mbgextio_set_ntp_symm_key_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ntp_symm_key_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                  NTP_SYMM_KEY_INFO_IDX *p, uint16_t idx )
{
  int rc;
  rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_NTP_SYMM_KEY_CFG, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ntp_symm_key_info_idx( p );

  return rc;

}  // mbgextio_get_ntp_symm_key_info_idx



/*HDR*/
/**
 * @brief Send NTP symmetric key settings with the given index in ::NTP_SYMM_KEY_SETTINGS format
 *
 * @note ::NTP_MSK_SYMM_KEYS must be set in ::NTP_GLB_INFO::supp_flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the NTP symmetric key to be configured, 0 ... ::NTP_GLB_SETTINGS::num_symm_keys - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_glb_info
 * @see ::mbgextio_get_ntp_symm_key_limits
 * @see ::mbgextio_get_ntp_symm_key_info_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ntp_symm_key_settings_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                      const NTP_SYMM_KEY_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_NTP_SYMM_KEY_CFG | OPT_GPS_ACK_CODE;
  NTP_SYMM_KEY_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.ntp_symm_key_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;
  _mbg_swab_ntp_symm_key_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ntp_symm_key_settings_idx



/*HDR*/
/**
 * @brief Read the NTP trusted key info with the given index in ::NTP_TRUSTED_KEY_INFO_IDX format
 *
 * @note ::NTP_MSK_TRUSTED_KEYS must be set in ::NTP_GLB_INFO::supp_flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the NTP trusted key info to be queried, 0 ... ::NTP_GLB_SETTINGS::num_trusted_keys - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_glb_info
 * @see ::mbgextio_set_ntp_trusted_key_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ntp_trusted_key_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                     NTP_TRUSTED_KEY_INFO_IDX *p, uint16_t idx )
{
  int rc;
  rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_NTP_TRUSTED_KEY_CFG, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ntp_trusted_key_info_idx( p );

  return rc;

}  // mbgextio_get_ntp_trusted_key_info_idx



/*HDR*/
/**
 * @brief Send NTP trusted key settings with the given index in ::NTP_TRUSTED_KEY_SETTINGS format
 *
 * @note ::NTP_MSK_TRUSTED_KEYS must be set in ::NTP_GLB_INFO::supp_flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the NTP trusted key to be configured, 0 ... ::NTP_GLB_SETTINGS::num_trusted_keys - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_glb_info
 * @see ::mbgextio_get_ntp_trusted_key_info_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ntp_trusted_key_settings_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                         const NTP_TRUSTED_KEY_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_NTP_TRUSTED_KEY_CFG | OPT_GPS_ACK_CODE;
  NTP_TRUSTED_KEY_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.ntp_trusted_key_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;
  _mbg_swab_ntp_trusted_key_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ntp_trusted_key_settings_idx



/*HDR*/
/**
 * @brief Read the NTP misc limits in ::NTP_MISC_LIMITS format
 *
 * @note ::NTP_MSK_MISCELLANEOUS must be set in ::NTP_GLB_INFO::supp_flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_glb_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ntp_misc_limits( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, NTP_MISC_LIMITS *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_NTP_MISC_LIMITS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ntp_misc_limits( p );

  return rc;

}  // mbgextio_get_ntp_misc_limits



/*HDR*/
/**
 * @brief Read the NTP orphan mode information in ::NTP_MISC_ORPHAN_MODE_INFO format
 *
 * @note ::NTP_MISC_MSK_ORPHAN_MODE must be set in ::NTP_MISC_LIMITS::supp_flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_glb_info
 * @see ::mbgextio_get_ntp_misc_limits
 * @see ::mbgextio_set_ntp_misc_orphan_mode_settings
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ntp_misc_orphan_mode_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                      NTP_MISC_ORPHAN_MODE_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_NTP_MISC_ORPHAN_MODE, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ntp_misc_orphan_mode_info( p );

  return rc;

}  // mbgextio_get_ntp_misc_orphan_mode_info



/*HDR*/
/**
 * @brief Send the NTP orphan mode configuration in ::NTP_MISC_ORPHAN_MODE_SETTINGS format
 *
 * @note ::NTP_MISC_MSK_ORPHAN_MODE must be set in ::NTP_MISC_LIMITS::supp_flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_glb_info
 * @see ::mbgextio_get_ntp_misc_limits
 * @see ::mbgextio_get_ntp_misc_orphan_mode_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ntp_misc_orphan_mode_settings( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                          const NTP_MISC_ORPHAN_MODE_SETTINGS *p )
{
  GPS_CMD cmd = GPS_NTP_MISC_ORPHAN_MODE | OPT_GPS_ACK_CODE;
  NTP_MISC_ORPHAN_MODE_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.ntp_misc_orphan_mode_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ntp_misc_orphan_mode_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ntp_misc_orphan_mode_settings



/*HDR*/
/**
 * @brief Read the ntp global information ::NTP_CLNT_MODE_INFO format
 *
 * @note ntp feature must set
 * @note ntp client mode must set
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_glb_info
 * @see ::mbgextio_set_ntp_clnt_mode_cfg
 * @see ::mbgextio_get_ntp_peer_settings_idx
 * @see ::mbgextio_set_ntp_peer_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ntp_clnt_mode_info( MBG_MSG_CTL *pmctl,
                                               const XBP_ADDR *p_addr, NTP_CLNT_MODE_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_NTP_CLNT_MODE_CFG, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ntp_clnt_mode_info( p );

  return rc;

}  // mbgextio_get_ntp_clnt_mode_info



/*HDR*/
/**
 * @brief Send the NTP client mode configuration
 *
 * @note This is only supported by NTP devices
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_glb_info
 * @see ::mbgextio_get_ntp_clnt_mode_info
 * @see ::mbgextio_get_ntp_peer_settings_idx
 * @see ::mbgextio_set_ntp_peer_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ntp_clnt_mode_cfg( MBG_MSG_CTL *pmctl,
                                      const XBP_ADDR *p_addr, const NTP_CLNT_MODE_SETTINGS *p )
{
  GPS_CMD cmd = GPS_NTP_CLNT_MODE_CFG | OPT_GPS_ACK_CODE;
  NTP_CLNT_MODE_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.ntp_clnt_mode_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ntp_clnt_mode_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ntp_clnt_mode_cfg



/*HDR*/
/**
 * @brief Read the ntp peer settings of current idx ::NTP_PEER_SETTINGS format
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the NTP peer settings to be configured, 0 ... NTP_CLNT_MODE_INFO::n_supp_peers - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_glb_info
 * @see ::mbgextio_set_ntp_peer_settings_idx
 * @see ::mbgextio_get_ntp_clnt_mode_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ntp_peer_settings_idx( MBG_MSG_CTL *pmctl,
                                        const XBP_ADDR *p_addr, NTP_PEER_SETTINGS_IDX *p, uint16_t idx )
{
  int rc;
  rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_NTP_PEER_SETTINGS_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ntp_peer_settings_idx( p );

  return rc;

}  // mbgextio_get_ntp_peer_settings_idx



/*HDR*/
/**
 * @brief Send configuration settings for a specific NTP peer
 *
 * The number of supported NTP peers is specified in ::NTP_CLNT_MODE_INFO::n_supp_peers -1.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the NTP peer to be configured, 0 ... NTP_CLNT_MODE_INFO::n_supp_peers - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_glb_info
 * @see ::mbgextio_get_ntp_peer_settings_idx
 * @see ::mbgextio_get_ntp_clnt_mode_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ntp_peer_settings_idx( MBG_MSG_CTL *pmctl,
                                     const XBP_ADDR *p_addr, const NTP_PEER_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_NTP_PEER_SETTINGS_IDX | OPT_GPS_ACK_CODE;
  NTP_PEER_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.ntp_peer_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->peer_settings = *p;
  p_data->idx = idx;
  _mbg_swab_ntp_peer_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ntp_peer_settings_idx



/*HDR*/
/**
 * @brief Read the ntp server mode information in ::NTP_SRV_MODE_INFO format
 *
 * @note NTP feature ::GPS_FEAT_NTP must be set
 * @note NTP roles ::NTP_ROLE_SERVER or ::NTP_ROLE_CLIENT_SERVER must be set in ::NTP_GLB_INFO::supp_ntp_roles
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_glb_info
 * @see ::mbgextio_set_ntp_srv_mode_cfg
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ntp_srv_mode_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, NTP_SRV_MODE_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_NTP_SRV_MODE_CFG, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ntp_srv_mode_info( p );

  return rc;

}  // mbgextio_get_ntp_srv_mode_info


/*HDR*/
/**
 * @brief Send the NTP server mode configuration
 *
 * @note NTP feature ::GPS_FEAT_NTP must be set
 * @note NTP roles ::NTP_ROLE_SERVER or ::NTP_ROLE_CLIENT_SERVER must be set in ::NTP_GLB_INFO::supp_ntp_roles
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_glb_info
 * @see ::mbgextio_get_ntp_srv_mode_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ntp_srv_mode_cfg( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, const NTP_SRV_MODE_SETTINGS *p )
{
  GPS_CMD cmd = GPS_NTP_SRV_MODE_CFG | OPT_GPS_ACK_CODE;
  NTP_SRV_MODE_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.ntp_srv_mode_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_ntp_srv_mode_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ntp_srv_mode_cfg


/*HDR*/
/**
 * @brief Read the NTP refclock config info with the given index in ::NTP_REFCLK_CFG_INFO_IDX format
 *
 * @note NTP feature ::GPS_FEAT_NTP must be set
 * @note NTP roles ::NTP_ROLE_SERVER or ::NTP_ROLE_CLIENT_SERVER must be set in ::NTP_GLB_INFO::supp_ntp_roles
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the NTP refclock config to be queried, 0 ... ::NTP_SRV_MODE_SETTINGS::num_refclks - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_glb_info
 * @see ::mbgextio_get_ntp_srv_mode_info
 * @see ::mbgextio_set_ntp_refclk_cfg_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ntp_refclk_cfg_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                    NTP_REFCLK_CFG_INFO_IDX *p, uint16_t idx )
{
  int rc;
  rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_NTP_REFCLK_CFG, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ntp_refclk_cfg_info_idx( p );

  return rc;

}  // mbgextio_get_ntp_refclk_cfg_info_idx



/*HDR*/
/**
 * @brief Send NTP refclk config settings with the given index in ::NTP_REFCLK_CFG_SETTINGS format
 *
 * @note NTP feature ::GPS_FEAT_NTP must be set
 * @note NTP roles ::NTP_ROLE_SERVER or ::NTP_ROLE_CLIENT_SERVER must be set in ::NTP_GLB_INFO::supp_ntp_roles
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the NTP refclock config to be set, 0 ... ::NTP_SRV_MODE_SETTINGS::num_refclks - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_glb_info
 * @see ::mbgextio_get_ntp_srv_mode_info
 * @see ::mbgextio_get_ntp_refclk_cfg_info_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ntp_refclk_cfg_settings_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                        const NTP_REFCLK_CFG_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_NTP_REFCLK_CFG | OPT_GPS_ACK_CODE;
  NTP_REFCLK_CFG_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.ntp_refclk_cfg_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;
  _mbg_swab_ntp_refclk_cfg_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ntp_refclk_cfg_settings_idx



/*HDR*/
/**
 * @brief Read the current system state of ntp device ::NTP_SYS_STATE format
 *
 * @note NTP feature must set
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_ntp_clnt_mode_cfg
 * @see ::mbgextio_get_ntp_glb_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ntp_sys_state( MBG_MSG_CTL *pmctl,
                                             const XBP_ADDR *p_addr, NTP_SYS_STATE *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_NTP_SYS_STATE, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ntp_sys_state( p );

  return rc;

}  // mbgextio_get_ntp_sys_state



/*HDR*/
/**
 * @brief Read the NTP peer state of current idx ::NTP_PEER_STATE format
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the NTP peer state to be configured, 0 ... NTP_CLNT_MODE_INFO::n_supp_peers - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ntp_sys_state
 * @see ::mbgextio_get_ntp_peer_settings_idx
 * @see ::mbgextio_get_ntp_clnt_mode_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ntp_peer_state_idx( MBG_MSG_CTL *pmctl,
                                        const XBP_ADDR *p_addr, NTP_PEER_STATE_IDX *p, uint16_t idx )
{
  int rc;
  rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_NTP_PEER_STATE_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ntp_peer_state_idx( p );

  return rc;

}  // mbgextio_get_ntp_peer_state_idx

#endif  // defined( _PRELIMINARY_CODE )



/*HDR*/
/**
 * @brief Read the network global config information ::MBG_NET_GLB_CFG_INFO format
 *
 * @note GPS_HAS_NET_CFG must set
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_net_glb_cfg_settings
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_net_glb_cfg_info( MBG_MSG_CTL *pmctl,
                                          const XBP_ADDR *p_addr, MBG_NET_GLB_CFG_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_NET_GLB_CFG, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_net_glb_cfg_info( p );

  return rc;

}  // mbgextio_get_net_glb_cfg_info



/*HDR*/
/**
 * @brief Set the device's global network configuration settings
 *
 * @note The function is not supported by all devices.  //##++++++++++++
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_net_glb_cfg_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_net_glb_cfg_settings( MBG_MSG_CTL *pmctl,
                                        const XBP_ADDR *p_addr, const MBG_NET_GLB_CFG_SETTINGS *p )
{
  GPS_CMD cmd = GPS_NET_GLB_CFG | OPT_GPS_ACK_CODE;
  MBG_NET_GLB_CFG_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.net_glb_cfg_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_net_glb_cfg_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_net_glb_cfg_settings



/*HDR*/
/**
 * @brief Read a network DNS server entry in ::MBG_IP_ADDR_IDX format
 *
 * The number of DNS server provided by the device is specified in ::MBG_NET_GLB_CFG_INFO::n_supp_dns_srvr
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::MBG_NET_GLB_CFG_INFO::n_supp_dns_srvr - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_net_dns_srvr_idx
 * @see ::mbgextio_get_net_stat_dns_srvr_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_net_dns_srvr_idx( MBG_MSG_CTL *pmctl,
                                         const XBP_ADDR *p_addr, MBG_IP_ADDR_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_NET_DNS_SRVR, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ip_addr_idx( p );

  return rc;

}  // mbgextio_get_net_dns_srvr_idx



/*HDR*/
/**
 * @brief Send the network DNS server address in ::MBG_IP_ADDR format
 *
 * The number of DNS search domains supported by the device
 * is specified in ::MBG_NET_GLB_CFG_INFO::n_supp_dns_srvr
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the DNS server port to be configured, 0..::MBG_NET_GLB_CFG_INFO::n_supp_dns_srvr - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_net_dns_srvr_idx
 * @see ::mbgextio_get_net_stat_dns_srvr_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_net_dns_srvr_idx( MBG_MSG_CTL *pmctl,
                                          const XBP_ADDR *p_addr, const MBG_IP_ADDR *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_NET_DNS_SRVR | OPT_GPS_ACK_CODE;
  MBG_IP_ADDR_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.ip_addr_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->addr = *p;
  p_data->idx = idx;
  _mbg_swab_ip_addr_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_net_dns_srvr_idx



/*HDR*/
/**
 * @brief Read the network DNS search domain in ::MBG_NET_NAME_IDX format
 *
 * The number of DNS search domains supported by the device
 * is specified in ::MBG_NET_GLB_CFG_INFO::n_supp_dns_srch_dom
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::MBG_NET_GLB_CFG_INFO::n_supp_dns_srch_dom - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_net_dns_srch_dom_idx
 * @see ::mbgextio_get_net_stat_dns_srch_dom_stat_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_net_dns_srch_dom_idx( MBG_MSG_CTL *pmctl,
                                              const XBP_ADDR *p_addr, MBG_NET_NAME_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_NET_DNS_SRCH_DOM, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_net_name_idx( p );

  return rc;

}  // mbgextio_get_net_dns_srch_dom_idx



/*HDR*/
/**
 * @brief Send the network DNS search domain in ::MBG_NET_NAME format
 *
 * The number of DNS search domains supported by the device
 * is specified in ::MBG_NET_GLB_CFG_INFO::n_supp_dns_srch_dom
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the DNS search domain entry to be configured, 0..::MBG_NET_GLB_CFG_INFO::n_supp_dns_srch_dom - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_net_dns_srch_dom_idx
 * @see ::mbgextio_get_net_stat_dns_srch_dom_stat_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_net_dns_srch_dom_idx( MBG_MSG_CTL *pmctl,
                                          const XBP_ADDR *p_addr, const MBG_NET_NAME *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_NET_DNS_SRCH_DOM | OPT_GPS_ACK_CODE;
  MBG_NET_NAME_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.net_name_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->net_name = *p;
  p_data->idx = idx;
  _mbg_swab_net_name( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_net_dns_srch_dom_idx



/*HDR*/
/**
 * @brief Read the current network DNS server in ::MBG_IP_ADDR_IDX format
 *
 * The number of DNS servers supported by the device
 * is specified in ::MBG_NET_GLB_CFG_INFO::n_supp_dns_srvr
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::MBG_NET_GLB_CFG_INFO::n_supp_dns_srvr - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_net_dns_srvr_idx
 * @see ::mbgextio_set_net_dns_srvr_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_net_stat_dns_srvr_idx( MBG_MSG_CTL *pmctl,
                                             const XBP_ADDR *p_addr, MBG_IP_ADDR_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_NET_STAT_DNS_SRVR, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ip_addr_idx( p );

  return rc;

}  // mbgextio_get_net_stat_dns_srvr_idx



/*HDR*/
/**
 * @brief Read the current network DNS search domain in ::MBG_NET_NAME_IDX format
 *
 * The number of DNS search domains supported by the device
 * is specified in ::MBG_NET_GLB_CFG_INFO::n_supp_dns_srch_dom
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::MBG_NET_GLB_CFG_INFO::n_supp_dns_srch_dom - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_net_dns_srch_dom_idx
 * @see ::mbgextio_get_net_dns_srch_dom_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_net_stat_dns_srch_dom_stat_idx( MBG_MSG_CTL *pmctl,
                                              const XBP_ADDR *p_addr, MBG_NET_NAME_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_NET_STAT_DNS_SRCH_DOM, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_net_name_idx( p );

  return rc;

}  // mbgextio_get_net_stat_dns_srch_dom_stat_idx



/*HDR*/
/**
 * @brief Read ::MBG_NET_INTF_LINK_INFO_IDX from the config with the given index
 *
 * @note GPS_HAS_NET_CFG must be set and MBG_NET_GLB_SUPP_STAGE_2_MASK must be set in ::MBG_NET_GLB_CFG_INFO::feat_flags
 *
 * The number of ::MBG_NET_INTF_LINK_INFO_IDX provided by the device is specified in
 * ::MBG_NET_GLB_CFG_INFO::glb_settings::num_intf_link.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::MBG_NET_GLB_CFG_INFO::glb_settings::num_intf_link - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_all_net_cfg_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_net_intf_link_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                   MBG_NET_INTF_LINK_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_NET_INTF_LINK_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_net_intf_link_info_idx( p );

  return rc;

} // mbgextio_get_net_intf_link_info_idx



/*HDR*/
/**
 * @brief Send the network link configuration in ::MBG_NET_INTF_LINK_SETTINGS format
 *
 * @note GPS_HAS_NET_CFG must be set and MBG_NET_GLB_SUPP_STAGE_2_MASK must be set in ::MBG_NET_GLB_CFG_INFO::feat_flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the link to be configured, 0..::MBG_NET_GLB_CFG_INFO::num_intf_link - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_net_intf_link_info_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_net_intf_link_settings_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                       const MBG_NET_INTF_LINK_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_NET_INTF_LINK_IDX | OPT_GPS_ACK_CODE;
  MBG_NET_INTF_LINK_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.net_intf_link_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;
  _mbg_swab_net_intf_link_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_net_intf_link_settings_idx



/*HDR*/
/**
 * @brief Read ::MBG_NET_INTF_ADDR_INFO_IDX from the config with the given index
 *
 * @note GPS_HAS_NET_CFG must be set and MBG_NET_GLB_SUPP_STAGE_2_MASK must be set in ::MBG_NET_GLB_CFG_INFO::feat_flags
 *
 * The number of ::MBG_NET_INTF_ADDR_INFO_IDX provided by the device is specified in
 * ::MBG_NET_GLB_CFG_INFO::glb_settings::num_intf_addr.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::MBG_NET_GLB_CFG_INFO::glb_settings::n_supp_intf_addr - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_all_net_cfg_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_net_intf_addr_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                   MBG_NET_INTF_ADDR_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_NET_INTF_ADDR_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_net_intf_addr_info_idx( p );

  return rc;

} // mbgextio_get_net_intf_addr_info_idx



/*HDR*/
/**
 * @brief Send the network addr configuration in ::MBG_NET_INTF_ADDR_SETTINGS format
 *
 * @note GPS_HAS_NET_CFG must be set and MBG_NET_GLB_SUPP_STAGE_2_MASK must be set in ::MBG_NET_GLB_CFG_INFO::feat_flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the link to be configured, 0..::MBG_NET_GLB_CFG_INFO::n_supp_intf_addr - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_net_intf_addr_info_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_net_intf_addr_settings_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                       const MBG_NET_INTF_ADDR_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_NET_INTF_ADDR_IDX | OPT_GPS_ACK_CODE;
  MBG_NET_INTF_ADDR_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.net_intf_addr_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;
  _mbg_swab_net_intf_addr_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_net_intf_addr_settings_idx



/*HDR*/
/**
 * @brief Read ::MBG_NET_INTF_ROUTE_INFO_IDX from the config with the given index
 *
 * @note GPS_HAS_NET_CFG must be set and MBG_NET_GLB_SUPP_STAGE_2_MASK must be set in ::MBG_NET_GLB_CFG_INFO::feat_flags
 *
 * The number of ::MBG_NET_INTF_ROUTE_INFO_IDX provided by the device is specified in
 * ::MBG_NET_GLB_CFG_INFO::glb_settings::num_intf_route.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::MBG_NET_GLB_CFG_INFO::glb_settings::num_intf_route - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_all_net_cfg_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_net_intf_route_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                    MBG_NET_INTF_ROUTE_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_NET_INTF_ROUTE_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_net_intf_route_info_idx( p );

  return rc;

} // mbgextio_get_net_intf_route_info_idx



/*HDR*/
/**
 * @brief Send the network addr configuration in ::MBG_NET_INTF_ROUTE_SETTINGS format
 *
 * @note GPS_HAS_NET_CFG must be set and MBG_NET_GLB_SUPP_STAGE_2_MASK must be set in ::MBG_NET_GLB_CFG_INFO::feat_flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the link to be configured, 0..::MBG_NET_GLB_CFG_INFO::num_intf_route - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_net_intf_route_info_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_net_intf_route_settings_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                        const MBG_NET_INTF_ROUTE_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_NET_INTF_ROUTE_IDX | OPT_GPS_ACK_CODE;
  MBG_NET_INTF_ROUTE_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.net_intf_route_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;
  _mbg_swab_net_intf_route_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_net_intf_route_settings_idx



/*HDR*/
/**
 * @brief Read the network global status information ::MBG_NET_GLB_CFG_INFO format
 *
 * @note GPS_HAS_NET_CFG must be set and MBG_NET_GLB_SUPP_STAGE_2_MASK must be set in ::MBG_NET_GLB_CFG_INFO::feat_flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_net_glb_cfg_settings
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_net_stat_glb_cfg_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                  MBG_NET_GLB_CFG_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_NET_STAT_GLB_CFG, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_net_glb_cfg_info( p );

  return rc;

}  // mbgextio_get_net_stat_glb_cfg_info



/*HDR*/
/**
 * @brief Read ::MBG_NET_INTF_LINK_INFO_IDX from the status with the given index
 *
 * @note GPS_HAS_NET_CFG must be set and MBG_NET_GLB_SUPP_STAGE_2_MASK must be set in ::MBG_NET_GLB_CFG_INFO::feat_flags
 *
 * The number of ::MBG_NET_INTF_LINK_INFO_IDX provided by the device is specified in
 * ::MBG_NET_GLB_CFG_INFO::glb_settings::num_intf_link.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::MBG_NET_GLB_CFG_INFO::glb_settings::num_intf_link - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_all_net_status_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_net_stat_intf_link_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                        MBG_NET_INTF_LINK_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_NET_STAT_INTF_LINK_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_net_intf_link_info_idx( p );

  return rc;

} // mbgextio_get_net_stat_intf_link_info_idx



/*HDR*/
/**
 * @brief Read ::MBG_NET_INTF_ADDR_INFO_IDX from the status with the given index
 *
 * @note GPS_HAS_NET_CFG must be set and MBG_NET_GLB_SUPP_STAGE_2_MASK must be set in ::MBG_NET_GLB_CFG_INFO::feat_flags
 *
 * The number of ::MBG_NET_INTF_ADDR_INFO_IDX provided by the device is specified in
 * ::MBG_NET_GLB_CFG_INFO::glb_settings::num_intf_addr.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::MBG_NET_GLB_CFG_INFO::glb_settings::num_intf_addr - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_all_net_status_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_net_stat_intf_addr_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                        MBG_NET_INTF_ADDR_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_NET_STAT_INTF_ADDR_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_net_intf_addr_info_idx( p );

  return rc;

} // mbgextio_get_net_stat_intf_addr_info_idx



/*HDR*/
/**
 * @brief Read ::MBG_NET_INTF_ROUTE_INFO_IDX from the status with the given index
 *
 * @note GPS_HAS_NET_CFG must be set and MBG_NET_GLB_SUPP_STAGE_2_MASK must be set in ::MBG_NET_GLB_CFG_INFO::feat_flags
 *
 * The number of ::MBG_NET_INTF_ROUTE_INFO_IDX provided by the device is specified in
 * ::MBG_NET_GLB_CFG_INFO::glb_settings::num_intf_route.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::MBG_NET_GLB_CFG_INFO::glb_settings::num_intf_route - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_all_net_status_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_net_stat_intf_route_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                         MBG_NET_INTF_ROUTE_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_NET_STAT_INTF_ROUTE_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_net_intf_route_info_idx( p );

  return rc;

} // mbgextio_get_net_stat_intf_route_info_idx



/*HDR*/
/**
 * @brief Read the ::XBP_LIMITS to check which XBP features are supported
 *
 * @note Only supported if ::GPS_HAS_XBP is set in ::RECEIVER_INFO::features
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_xbp_node_limits
 * @see ::mbgextio_get_xbp_node_info_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_xbp_limits( MBG_MSG_CTL *pmctl,
                                               const XBP_ADDR *p_addr, XBP_LIMITS *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_XBP_LIMITS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_xbp_limits( p );

  return rc;

}  // mbgextio_get_xbp_limits



/*HDR*/
/**
 * @brief Read the ::XBP_NODE_LIMITS to check how many ports are provided
 *
 * @note Only supported if ::XBP_FEAT_MASK_NODES is set in ::XBP_LIMITS::features.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_xbp_limits
 * @see ::mbgextio_get_xbp_node_info_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_xbp_node_limits( MBG_MSG_CTL *pmctl,
                                             const XBP_ADDR *p_addr, XBP_NODE_LIMITS *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_XBP_NODE_LIMITS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_xbp_node_limits( p );

  return rc;

}  // mbgextio_get_xbp_node_limits



/*HDR*/
/**
 * @brief Read the ::XBP_NODE_INFO for a specific node in ::XBP_NODE_INFO_IDX format
 *
 * The supported number of nodes is provided by ::XBP_NODE_LIMITS::node_count.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::XBP_NODE_LIMITS::node_count-1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_xbp_limits
 * @see ::mbgextio_get_xbp_node_limits
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_xbp_node_info_idx( MBG_MSG_CTL *pmctl,
                                         const XBP_ADDR *p_addr, XBP_NODE_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_XBP_NODE_INFO_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_xbp_node_info_idx( p );

  return rc;

}  // mbgextio_get_xbp_node_info_idx



static /*HDR*/
void handle_xbp_node( XBP_NODE_INFO *p_ni, long idx )
{
  XBP_ADDR * p_addr = &p_ni->addr;
  RECEIVER_INFO *p_ri = &p_ni->ri;
  int i;

  printf( "Node info %li:", idx );

  printf( " addr %02X:", p_addr->hop_count );

  for ( i = 0; i < MAX_XBP_CASC_LVL; i++ )
    printf( "%02X ", p_addr->addr[i] );

  printf( ", state: %i", p_ni->state );

  printf( ", name: \"%s\"", p_ri->model_name );

  printf( "\n" );

}  // handle_xbp_node



/*HDR*/
/**
 * @brief Read the ::XBP_NODE_INFO for a specific node in ::XBP_NODE_INFO_IDX format
 * //##+++++++++++++++++++++++++
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_setup_xbp_node_list( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr )
{
  static int recursion_count;

  XBP_NODE_LIMITS node_limits;
  int rc = MBG_SUCCESS;
  uint16_t i;

  // This function can be called recursively, so first of all check the recursion level.
  if ( recursion_count > MAX_XBP_CASC_LVL )  //##++++ TODO: or even ">=" ??
  {
    // This should never happen, but we check this anyway.
    rc = MBG_ERR_XBP_CASC_LVL;
    goto out;
  }

  recursion_count++;

  rc = mbgextio_get_xbp_node_limits( pmctl, p_addr, &node_limits );

  if ( mbg_rc_is_error( rc ) )
    goto out_dec;

  for ( i = 0; i < node_limits.node_count; i++ )
  {
    XBP_NODE_INFO_IDX node_info_idx;

    rc = mbgextio_get_xbp_node_info_idx( pmctl, p_addr, &node_info_idx, i );

    if ( mbg_rc_is_error( rc ) )
      break;

    #if 1 && DEBUG
      if ( (ulong) node_info_idx.idx != (ulong) i )
        printf( "** Warning: received XBP index %li for idx %li in %s\n",
                (ulong) node_info_idx.idx, (ulong) i, __func__ );
    #endif

    handle_xbp_node( &node_info_idx.node_info, node_info_idx.idx );
  }

out_dec:
  if ( recursion_count > 0 )
    recursion_count--;
  else
  {
    //### recursion count is unexpectedly 0
  }

out:
  return rc;

}  // mbgextio_setup_xbp_node_list



/*HDR*/
/**
 * @brief Read configuration and health data of all satellites from a device
 *
 * @note Only supported by GPS receivers
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_cfgh( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, CFGH *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_CFGH, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_dummy( p );

  return rc;

}  // mbgextio_get_cfgh



/*HDR*/
/**
 * @brief Read the GPS almanach for the specified SV from a device
 *
 * @note Only supported by GPS receivers
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     svno   SV number, for which the ::SV_ALM structure shall be requested
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_sv_alm( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, SV_ALM *p, SVNO svno )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_ALM, svno, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_dummy( p );

  return rc;

}  // mbgextio_get_sv_alm



/*HDR*/
/**
 * @brief Read the ionospheric correction parameters from a device
 *
 * @note Only supported by GPS receivers
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the ::IONO structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_iono( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, IONO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_IONO, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_dummy( p );

  return rc;

}  // mbgextio_get_iono



/*HDR*/
/**
 * @brief Read a ::UTC parameter structure from a device.
 *
 * The ::UTC parameter structure contains the current UTC/GPS time offset
 * as well as information on an eventually upcoming leap second.
 *
 * @note Only supported by GPS/GNSS and some PZF receivers  //##+++++++ TODO Associated feature flag?
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_utc_parm
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_utc_parm( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, UTC *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_UTC, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_utc_parm( p );

  return rc;

}  // mbgextio_get_utc_parm



/*HDR*/
/**
 * @brief Write a ::UTC parameter structure to a device.
 *
 * The ::UTC parameter structure contains the current UTC/GPS time offset
 * as well as information on an eventually upcoming leap second.
 *
 * @note Only supported by GPS/GNSS and some PZF receivers  //##+++++++ TODO Associated feature flag?
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_utc_parm
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_utc_parm( MBG_MSG_CTL *pmctl, XBP_ADDR *p_addr, const UTC *p )
{
  GPS_CMD cmd = GPS_UTC | OPT_GPS_ACK_CODE;
  UTC *p_data = &pmctl->xmt.pmb->u.msg_data.utc;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_utc_parm( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_utc_parm



/*HDR*/
/**
 * @brief Read SCU_STAT_INFO from the device
 *
 * @note This is only supported if ::GPS_MODEL_HAS_SCU_STAT is set in
 * the builtin features of the device
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_scu_stat
 * @see ::mbgextio_set_scu_stat_settings
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_scu_stat_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, SCU_STAT_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_SCU_STAT, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_scu_stat_info( p );

  return rc;

}  // mbgextio_get_scu_stat_info



/*HDR*/
/**
 * @brief Send SCU_STAT_SETTINGS to the device
 *
 * @note This is only supported if ::GPS_MODEL_HAS_SCU_STAT is set in
 * the builtin features of the device
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_scu_stat
 * @see ::mbgextio_get_scu_stat_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_scu_stat_settings( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, const SCU_STAT_SETTINGS *p )
{
  // TODO: request ACK for SCU_STAT_SETTINGS (at the moment, this is not implemented in RSC/MDU)
  GPS_CMD cmd = GPS_SCU_STAT;
  SCU_STAT_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.scu_stat_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_scu_stat_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_scu_stat_settings



#if _USE_TIME_MON && defined( _PRELIMINARY_CODE )

/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_time_mon_get_inst_info( MBG_MSG_CTL *pmctl,
                                          const XBP_ADDR *p_addr, MBG_TIME_MON_INST_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_TIME_MON_INST_INFO, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_time_mon_inst_info( p );

  return rc;

}  // mbgextio_time_mon_get_inst_info



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_time_mon_send_inst_info( MBG_MSG_CTL *pmctl,
                                        const XBP_ADDR *p_addr, const MBG_TIME_MON_INST_INFO *p )
{
  GPS_CMD cmd = GPS_TIME_MON_INST_INFO | OPT_GPS_ACK_CODE;
  MBG_TIME_MON_INST_INFO *p_data = &pmctl->xmt.pmb->u.msg_data.time_mon_inst_info;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_time_mon_inst_info( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_time_mon_send_inst_info



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_time_mon_send_target_settings( MBG_MSG_CTL *pmctl,
                                        const XBP_ADDR *p_addr, const MBG_TIME_MON_TARGET_SETTINGS *p )
{
  GPS_CMD cmd = GPS_TIME_MON_TARGET_SETTINGS | OPT_GPS_ACK_CODE;
  MBG_TIME_MON_TARGET_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.time_mon_target_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_time_mon_target_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_time_mon_send_target_settings



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_time_mon_get_target_status_idx( MBG_MSG_CTL *pmctl,
                                          const XBP_ADDR *p_addr, MBG_TIME_MON_TARGET_STATUS_IDX *p, uint16_t idx  )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_TIME_MON_TARGET_STATUS_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_time_mon_target_status_idx( p );

  return rc;

}  // mbgextio_time_mon_get_target_status_idx



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_time_mon_get_target_ext_data_set_idx( MBG_MSG_CTL *pmctl,
                                          const XBP_ADDR *p_addr, MBG_TIME_MON_TARGET_EXT_DATA_SET_IDX *p, uint16_t idx  )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_TIME_MON_TARGET_EXT_DATA_SET_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_time_mon_target_ext_data_set_idx( p );

  return rc;

}  // mbgextio_time_mon_get_target_ext_data_set_idx()

#endif   // _USE_TIME_MON && defined( _PRELIMINARY_CODE )



#if defined( _PRELIMINARY_CODE )

/*HDR*/
/**
* @brief Read a ::MBG_GPIO_FREQ parameter structure from a device.
*
* The ::MBG_GPIO_FREQ parameter structure contains the current frequency
*
* @note Only supported if ::MBG_IMS_STATE_FLAG_BIT_HAS_FDM is set in ::MBG_IMS_STATE::flags
*
* @param[in,out] pmctl  Pointer to a valid message control structure
* @param[in]     p_addr Pointer to an XBP address specifier, or NULL
* @param[out]    p      Pointer to the data structure to return the received data
*
* @return One of the @ref MBG_RETURN_CODES
*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_fdm_freq( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, MBG_GPIO_FREQ *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_FDM_FREQ, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_gpio_freq( p );

  return rc;

}  // mbgextio_get_fdm_freq



/*HDR*/
/**
* @brief Read a ::MBG_IMS_FDM_INFO parameter structure from a device.
*
* The ::MBG_IMS_FDM_INFO parameter structure contains the ::MBG_IMS_FDM_SETTINGS, the supported nominal frequencies and flags
*
* @note Only supported if ::MBG_IMS_STATE_FLAG_BIT_HAS_FDM is set in ::MBG_IMS_STATE::flags
*
* @param[in,out] pmctl  Pointer to a valid message control structure
* @param[in]     p_addr Pointer to an XBP address specifier, or NULL
* @param[out]    p      Pointer to the data structure to return the received data
*
* @return One of the @ref MBG_RETURN_CODES
*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ims_fdm_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, MBG_IMS_FDM_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_FDM_INFO, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_ims_fdm_info( p );

  return rc;

}  // mbgextio_get_ims_fdm_info



/*HDR*/
/**
* @brief Read a ::MBG_IMS_FDM_LIMITS parameter structure from a device.
*
* The ::MBG_IMS_FDM_LIMITS parameter structure contains the limits of the configurable parameters and number of outpus
*
* @note Only supported if ::MBG_IMS_STATE_FLAG_BIT_HAS_FDM is set in ::MBG_IMS_STATE::flags
*
* @param[in,out] pmctl  Pointer to a valid message control structure
* @param[in]     p_addr Pointer to an XBP address specifier, or NULL
* @param[out]    p      Pointer to the data structure to return the received data
*
* @return One of the @ref MBG_RETURN_CODES
*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ims_fdm_limits( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, MBG_IMS_FDM_LIMITS *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_FDM_LIMITS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_ims_fdm_limits( p );

  return rc;

}  // mbgextio_get_ims_fdm_limits



/*HDR*/
/**
* @brief Read a ::MBG_IMS_FDM_OUTPUT_INFO_IDX parameter structure from a device.
*
* The ::MBG_IMS_FDM_OUTPUT_INFO parameter structure contains the ::MBG_IMS_FDM_OUTPUT_SETTINGS, the supported modes and specific DAC limits
*
* @note Only supported if ::MBG_IMS_STATE_FLAG_BIT_HAS_FDM is set in ::MBG_IMS_STATE::flags
*
* @param[in,out] pmctl  Pointer to a valid message control structure
* @param[in]     p_addr Pointer to an XBP address specifier, or NULL
* @param[out]    p      Pointer to the data structure to return the received data
* @param[in]     idx    Index of the array element to be retrieved, 0..::MBG_IMS_FDM_LIMITS::n_outputs-1
*
* @return One of the @ref MBG_RETURN_CODES
*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ims_fdm_output_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, MBG_IMS_FDM_OUTPUT_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_FDM_OUTPUT_INFO_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_ims_fdm_output_info_idx( p );

  return rc;

}  // mbgextio_get_ims_fdm_output_info_idx



/*HDR*/
/**
* @brief Read a ::MBG_IMS_FDM_OUTPUT_STATE_IDX parameter structure from a device.
*
* The ::MBG_IMS_FDM_OUTPUT_STATE parameter structure contains the current DAC val and specs and the configured mode
*
* @note Only supported if ::MBG_IMS_STATE_FLAG_BIT_HAS_FDM is set in ::MBG_IMS_STATE::flags
*
* @param[in,out] pmctl  Pointer to a valid message control structure
* @param[in]     p_addr Pointer to an XBP address specifier, or NULL
* @param[out]    p      Pointer to the data structure to return the received data
* @param[in]     idx    Index of the array element to be retrieved, 0..::MBG_IMS_FDM_LIMITS::n_outputs-1
*
* @return One of the @ref MBG_RETURN_CODES
*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ims_fdm_output_state_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, MBG_IMS_FDM_OUTPUT_STATE_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_FDM_OUTPUT_STATE_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_ims_fdm_output_state_idx( p );

  return rc;

}  // mbgextio_get_ims_fdm_output_state_idx



/*HDR*/
/**
* @brief Read a ::MBG_IMS_FDM_STATE parameter structure from a device.
*
* The ::MBG_IMS_FDM_STATE parameter structure contains the current frequency, the reference, powerline and sync time, the configured nominal frequency and flags
*
* @note Only supported if ::MBG_IMS_STATE_FLAG_BIT_HAS_FDM is set in ::MBG_IMS_STATE::flags
*
* @param[in,out] pmctl  Pointer to a valid message control structure
* @param[in]     p_addr Pointer to an XBP address specifier, or NULL
* @param[out]    p      Pointer to the data structure to return the received data
*
* @return One of the @ref MBG_RETURN_CODES
*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ims_fdm_state( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, MBG_IMS_FDM_STATE *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_FDM_STATE, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_ims_fdm_state( p );

  return rc;

}  // mbgextio_get_ims_fdm_state



/*HDR*/
/**
* @brief Write a ::MBG_IMS_FDM_SETTINGS parameter structure to a device.
*
* The ::MBG_IMS_FDM_SETTINGS parameter structure contains the limits for time and frequency deviation and the nominal frequency
*
* @note Only supported if ::MBG_IMS_STATE_FLAG_BIT_HAS_FDM is set in ::MBG_IMS_STATE::flags
*
* @param[in,out] pmctl  Pointer to a valid message control structure
* @param[in]     p_addr Pointer to an XBP address specifier, or NULL
* @param[in]     p      Pointer to the data structure to be sent to the device
*
* @return One of the @ref MBG_RETURN_CODES
*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ims_fdm_settings( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, const MBG_IMS_FDM_SETTINGS *p )
{
  GPS_CMD cmd = GPS_FDM_SETTINGS | OPT_GPS_ACK_CODE;
  MBG_IMS_FDM_SETTINGS *p_data = &pmctl->xmt.pmb->u.msg_data.fdm_settings;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_mbg_ims_fdm_settings( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ims_fdm_settings



/*HDR*/
/**
 * @brief Write a ::MBG_IMS_FDM_OUTPUT_SETTINGS parameter structure to a device.
 *
 * The ::MBG_IMS_FDM_SETTINGS parameter structure contains the analog output mode
 *
 * @note Only supported if ::MBG_IMS_STATE_FLAG_BIT_HAS_FDM is set in ::MBG_IMS_STATE::flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the array element to be set, 0..::MBG_IMS_FDM_LIMITS::n_outputs-1
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_ims_fdm_output_settings_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                       const MBG_IMS_FDM_OUTPUT_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_FDM_OUTPUT_SETTINGS_IDX | OPT_GPS_ACK_CODE;
  MBG_IMS_FDM_OUTPUT_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.fdm_output_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;
  _mbg_swab_mbg_ims_fdm_output_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_ims_fdm_output_settings_idx



/*HDR*/
/**
 * @brief Set time deviation of powerline time in relation to the current reference time
 *
 * @note Only supported if ::MBG_IMS_STATE_FLAG_BIT_HAS_FDM is set in ::MBG_IMS_STATE::flags and
 *                         ::MBG_IMS_FDM_FLAG_CAN_SET_TDEV is set in ::MBG_IMS_FDM_INFO::flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_fdm_tdev( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, const NANO_TIME_64 *p )
{
  GPS_CMD cmd = GPS_FDM_SET_TD | OPT_GPS_ACK_CODE;
  NANO_TIME_64 *p_data = &pmctl->xmt.pmb->u.msg_data.nano_time_64;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_nano_time_64( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_fdm_tdev

#endif  // defined( _PRELIMINARY_CODE )



/*HDR*/
/**
 * @brief Send a command to let the device save it's current configuration as default
 *
 * Call ::mbgextio_dev_has_cmd_save_cfg to check if this function is supported.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_cmd_save_cfg
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_cmd_save_cfg( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr )
{
  return mbgextio_xmt_cmd( pmctl, p_addr, GPS_SAVE_CFG | OPT_GPS_ACK_CODE );

  //### TODO: check if ACK works

}  // mbgextio_cmd_save_cfg



/*HDR*/
/**
 * @brief Read a ::MBG_IMS_FDM_STATE parameter structure from a device.
 *  ::TODO ###
 *
 * @note Only supported if ::MBG_IMS_STATE_FLAG_BIT_HAS_FDM is set in ::TODO MBG_IMS_STATE::flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_lne_limits( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, MBG_LNE_LIMITS *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_LNE_LIMITS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_lne_limits( p );

  return rc;

}  // mbgextio_get_lne_limits



/*HDR*/
/**
 * @brief Read a ::MBG_LNE_PORT_INFO_IDX parameter structure from a device.
 *
 * @note ### ::TODO Only supported if ::MBG_IMS_STATE_FLAG_BIT_HAS_FDM is set in ::MBG_IMS_STATE::flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::MBG_LNE_LIMITS::num_ports-1
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_lne_port_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, MBG_LNE_PORT_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_LNE_PORT_INFO_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_lne_port_info_idx( p );

  return rc;

}  // mbgextio_get_lne_port_info_idx



/*HDR*/
/**
 * @brief Read a ::MBG_LNE_PORT_SETTINGS_IDX parameter structure from a device.
 *
 * @note ### ::TODO Only supported if ::MBG_IMS_STATE_FLAG_BIT_HAS_FDM is set in ::MBG_IMS_STATE::flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::MBG_LNE_LIMITS::num_ports
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_lne_port_settings_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, MBG_LNE_PORT_SETTINGS_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_LNE_PORT_SETTINGS_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_lne_port_settings_idx( p );

  return rc;

}  // mbgextio_get_lne_port_settings_idx



/*HDR*/
/**
 * @brief Write a ::MBG_LNE_PORT_SETTINGS parameter structure to a device.
 *
 * @note ### ::TODO Only supported if ::MBG_IMS_STATE_FLAG_BIT_HAS_FDM is set in ::MBG_IMS_STATE::flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the array element to be set, 0..::MBG_LNE_LIMITS::num_ports - 1
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_lne_port_settings_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, const MBG_LNE_PORT_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_LNE_PORT_SETTINGS_IDX | OPT_GPS_ACK_CODE;
  MBG_LNE_PORT_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.lne_port_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;
  _mbg_swab_mbg_lne_port_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_lne_port_settings_idx



/*HDR*/
/**
 * @brief Write a ::MBG_PWR_CTL parameter structure to a device.
 *
 * @note Only supported if ::MBG_XFEATURE_PWR_CTL_API is set
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_pwr_ctl( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, const MBG_PWR_CTL *p )
{
  GPS_CMD cmd = GPS_PWR_CTL | OPT_GPS_ACK_CODE;
  MBG_PWR_CTL *p_data = &pmctl->xmt.pmb->u.msg_data.pwr_ctl;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *p_data = *p;
  _mbg_swab_mbg_pwr_ctl( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_pwr_ctl



/*HDR*/
/**
 * @brief Read a ::MBG_PWR_CTL parameter structure from a device.
 *
 * @note ### ::TODO Only supported if ::MBG_XFEATURE_PWR_CTL_API is set
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_pwr_ctl( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, MBG_PWR_CTL *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_PWR_CTL, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_pwr_ctl( p );

  return rc;

}  // mbgextio_get_pwr_ctl



/*HDR*/
/**
 * @brief Read the ::MBG_LED_LIMITS to check how many LEDs are provided
 *
 * ::mbgextio_dev_has_led_api should be used to check if this is supported
 *
 * @note Only supported if ::MBG_IMS_STATE_FLAG_BIT_HAS_FDM is set in ::MBG_IMS_STATE::flags
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @ingroup group_led_api
 * @see ::mbgextio_dev_has_led_api
 * @see ::mbgextio_get_led_limits
 * @see ::mbgextio_get_led_info_idx
 * @see ::mbgextio_get_led_settings_idx
 * @see ::mbgextio_set_led_settings_idx
 * @see @ref group_led_api
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_led_limits( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                       MBG_LED_LIMITS *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_LED_LIMITS, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_led_limits( p );

  return rc;

}  // mbgextio_get_led_limits



/*HDR*/
/**
 * @brief Read the current settings and features of a particular LED
 *
 * ::mbgextio_dev_has_led_api should be used to check if this is supported
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::MBG_LED_LIMITS::num_leds-1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @ingroup group_led_api
 * @see ::mbgextio_dev_has_led_api
 * @see ::mbgextio_get_led_limits
 * @see ::mbgextio_get_led_settings_idx
 * @see ::mbgextio_set_led_settings_idx
 * @see @ref group_led_api
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_led_info_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                         MBG_LED_INFO_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_LED_INFO_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_led_info_idx( p );

  return rc;

}  // mbgextio_get_led_info_idx



/*HDR*/
/**
 * @brief Read a ::MBG_LED_SETTINGS_IDX parameter structure from a device.
 *
 * ::mbgextio_dev_has_led_api should be used to check if this is supported
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p      Pointer to the data structure to return the received data
 * @param[in]     idx    Index of the array element to be retrieved, 0..::num_leds-1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @ingroup group_led_api
 * @see ::mbgextio_dev_has_led_api
 * @see ::mbgextio_get_led_limits
 * @see ::mbgextio_get_led_info_idx
 * @see ::mbgextio_set_led_settings_idx
 * @see @ref group_led_api
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_led_settings_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, MBG_LED_SETTINGS_IDX *p, uint16_t idx )
{
  int rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_LED_SETTINGS_IDX, idx, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_mbg_led_settings_idx( p );

  return rc;

}  // mbgextio_get_led_settings_idx



/*HDR*/
/**
 * @brief Write a ::MBG_LED_SETTINGS parameter structure to a device.
 *
 * ::mbgextio_dev_has_led_api should be used to check if this is supported
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     p      Pointer to the data structure to be sent to the device
 * @param[in]     idx    Index of the array element to be set, 0..::MBG_LNE_LIMITS::num_ports-1
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @ingroup group_led_api
 * @see ::mbgextio_dev_has_led_api
 * @see ::mbgextio_get_led_limits
 * @see ::mbgextio_get_led_info_idx
 * @see ::mbgextio_get_led_settings_idx
 * @see @ref group_led_api
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_set_led_settings_idx( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, const MBG_LED_SETTINGS *p, uint16_t idx )
{
  GPS_CMD cmd = GPS_LED_SETTINGS_IDX | OPT_GPS_ACK_CODE;
  MBG_LED_SETTINGS_IDX *p_data = &pmctl->xmt.pmb->u.msg_data.led_settings_idx;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  p_data->settings = *p;
  p_data->idx = idx;
  _mbg_swab_mbg_led_settings_idx( p_data );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *p_data ) );

}  // mbgextio_set_led_settings_idx



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_ext_sys_info( MBG_MSG_CTL *pmctl,
                                                         const XBP_ADDR *p_addr,
                                                         MBG_EXT_SYS_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, GPS_EXT_SYS_INFO, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_ext_sys_info( p );

  return rc;

}  // mbgextio_get_ext_sys_info


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_corr_info( MBG_MSG_CTL *pmctl,
                                                         const XBP_ADDR *p_addr,
                                                         CORR_INFO *p )
{
  int rc = mbgextio_req_data( pmctl, p_addr, PZF_CORR_INFO, p, sizeof( *p ) );

  if ( mbg_rc_is_success( rc ) && p )
    _mbg_swab_corr_info( p );

  return rc;

}  // mbgextio_get_core_info





