
/**************************************************************************
 *
 *  $Id: cfg_hlp.h 1.3.1.100 2017/03/29 12:33:26 philipp TEST $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    Definitions and prototypes for configuration programs.
 *
 *    WARNING: Changing the constants defined here affects the size of
 *    the related structures and arrays, and thus would break compatibility
 *    if used in DLLs / shared object libraries.
 *
 *    Care must be taken that the number of objects supported by
 *    any particular device (which can be only determined at runtime)
 *    does not exceed the max. number of objects specified here
 *    for the configuration programs.
 *
 * -----------------------------------------------------------------------
 *  $Log: cfg_hlp.h $
 *  Revision 1.3.1.100  2017/03/29 12:33:26  philipp
 *  Extended event info and event status index structures by data pointer
 *  Revision 1.3.1.99  2017/03/27 10:37:18  thomas-b
 *  Added ALL_PTP_V1_COMMON_DATASETS and the appropriate free function
 *  Revision 1.3.1.98  2017/03/22 09:07:53  thomas-b
 *  Added function get_io_port_type_info_idx
 *  Revision 1.3.1.97  2017/03/17 12:01:02  martin
 *  Moved definitions of PCPS_HRT_FRAC_CONVERSION_TYPE,
 *  PCPS_HRT_BIN_FRAC_SCALE, and PCPS_HRT_FRAC_SCALE_FMT
 *  here.
 *  Revision 1.3.1.96  2017/03/17 11:45:17  martin
 *  Moved binary fraction conversion functions here.
 *  Revision 1.3.1.95  2017/03/02 14:03:00  gregoire
 *  ALL_NTP_CFG_INFO reworked
 *  Revision 1.3.1.94  2017/03/02 08:34:08  gregoire
 *  ALL_NTP_CFG_INFO: NTP_REFCLK_CFG_INFO_IDX added
 *  Revision 1.3.1.93  2017/02/28 15:25:17  gregoire
 *  ALL_NTP_CFG_INFO extended
 *  Revision 1.3.1.92  2017/02/24 09:38:25  philipp
 *  Do not use fixed buffers for monitorung event info and status
 *  Revision 1.3.1.91  2017/02/22 11:40:17  martin
 *  Account for preliminary code.
 *  Revision 1.3.1.90  2017/02/21 15:51:40  thomas-b
 *  Added ALL_MONITORING_STATUS and free function and extended ALL_MONITORING_INFO by events
 *  Revision 1.3.1.89  2017/02/16 12:58:01  thomas-b
 *  Added ALL_PTP_V2_COMMON_DATASETS structure and the appropriate free function
 *  Revision 1.3.1.88  2017/02/08 13:16:08  thomas-b
 *  Added ALL_MONITORING_INFO and the appropriate free function
 *  Revision 1.3.1.87  2017/02/07 09:46:51  thomas-b
 *  Added ALL_SNMP_INFO and the appropriate free function
 *  Revision 1.3.1.86  2016/12/19 12:13:55  philipp
 *  Added GPIO associated index helper
 *  Revision 1.3.1.85  2016/12/13 13:46:22  martin
 *  Increased MAX_PARM_PORT and MAX_PARM_POUT to 10.
 *  Revision 1.3.1.84  2016/11/22 12:40:46Z  philipp
 *  Added I/O port helper functions
 *  Revision 1.3.1.83  2016/11/08 17:21:35  martin
 *  Updated function prototypes.
 *  Revision 1.3.1.82  2016/11/01 09:24:02  martin
 *  *** empty log message ***
 *  Revision 1.3.1.81  2016/10/25 07:48:27  martin
 *  Doxygen fixes.
 *  Revision 1.3.1.80  2016/10/21 09:39:26  thomas-b
 *  Added struct ALL_XBP_INFO and appropriate free function
 *  Revision 1.3.1.79  2016/10/14 11:10:03  thomas-b
 *  Added ALL_UCAP_NET_INFO and the appropriate free function
 *  Revision 1.3.1.78  2016/09/28 15:04:03  thomas-b
 *  Added function to check whether NET_CFG_API stage 2 is supported
 *  Revision 1.3.1.77  2016/08/11 13:25:06  martin
 *  Use mbgklist.h instead of mbg_klist.h.
 *  Revision 1.3.1.76  2016/08/11 08:36:56Z  martin
 *  Exclude function prototypes from build for kernel space.
 *  Revision 1.3.1.75  2016/08/10 12:25:02Z  martin
 *  Support FreeBSD.
 *  Revision 1.3.1.74  2016/08/09 07:09:28  martin
 *  Support QNX Neutrino.
 *  Revision 1.3.1.73  2016/06/21 12:06:59  thomas-b
 *  Added function calloc_ucap_entry
 *  Revision 1.3.1.72  2016/06/21 07:39:21  philipp
 *  Use mbg_klist for ucap events and not an array -> Easier to handle MAX_UCAP_ENTRIES
 *  Revision 1.3.1.71  2016/06/16 07:46:45  philipp
 *  Fixed checking flags for IMS sensor specific features
 *  Revision 1.3.1.70  2016/06/15 14:04:27  thomas-b
 *  Added ALL_UCAP_INFO structure and function free_all_ucap_info
 *  Revision 1.3.1.69  2016/06/02 10:24:22  philipp
 *  Renaming all revision macros and helper functions
 *  Revision 1.3.1.68  2016/05/31 11:29:15  philipp
 *  Extended ALL_XMULTI_REF_STATUS by flags if at least one ref type supports stats and/or metrics
 *  Revision 1.3.1.67  2016/05/27 05:48:08  philipp
 *  Refactoring to support XMR_METRICS properly
 *  Revision 1.3.1.66  2016/05/26 11:01:03  thomas-b
 *  Removed info structures from ALL_NTP_STATUS
 *  Moved check functions of specific features to cfg_hlp
 *  Revision 1.3.1.65  2016/05/25 08:43:36  philipp
 *  Redesign of ALL_[xxx]_[XXX] structures and (helper) functions
 *  Revision 1.3.1.64  2016/05/23 09:37:40  philipp
 *  Extended ALL_XMULTI_REF_STATUS by holdover_status
 *  Revision 1.3.1.63  2016/05/23 08:59:03  philipp
 *  New function free_all_gpio_state
 *  Revision 1.3.1.62  2016/05/23 08:24:38  philipp
 *  New function free_all_ims_state
 *  Revision 1.3.1.61  2016/05/11 13:20:59  thomas-b
 *  Added ALL_NET_STATUS_INFO and the appropriate free function
 *  Revision 1.3.1.60  2016/04/26 14:21:58  thomas-b
 *  Renamed ALL_NET_CFG_INFO structure members
 *  Revision 1.3.1.59  2016/04/26 08:23:45  thomas-b
 *  Extended ALL_NET_CFG_INFO by DNS configurations
 *  Revision 1.3.1.58  2016/04/26 06:29:30  thomas-b
 *  Added ALL_NET_CFG_INFO for network configuration
 *  Revision 1.3.1.57  2016/04/25 11:22:31  martin
 *  Revision 1.3.1.56  2016/04/25 10:43:38Z  martin
 *  *** empty log message ***
 *  Revision 1.3.1.55  2016/04/22 07:11:50  philipp
 *  Use pointer to pointer in get_all_* functions
 *  Revision 1.3.1.54  2016/04/20 14:45:46  thomas-b
 *  Renamed ALL_NTP_STATE_INFO to ALL_NTP_STATUS
 *  Revision 1.3.1.53  2016/04/20 13:49:13  thomas-b
 *  Added structure definitions and free functions for NTP
 *  Revision 1.3.1.52  2016/04/20 12:37:53  thomas-b
 *  Moved free functions for ALL_XMULTI_REF_INFO and ALL_XMULTI_REF_STATUS to cfg_hlp
 *  Revision 1.3.1.51  2016/04/12 13:28:39  philipp
 *  New helper functions to get all feature related structures at once
 *  Revision 1.3.1.50  2016/04/12 08:25:10  thomas-b
 *  Added ALL_XMULTI_REF_STATUS structure
 *  Revision 1.3.1.49  2016/04/11 13:56:24  thomas-b
 *  Added ALL_XMULTI_REF_INFO structure
 *  Revision 1.3.1.48  2016/04/08 07:54:26  philipp
 *  Added function prototype mbg_print_ext_rev_info
 *  Revision 1.3.1.47  2016/04/04 15:08:45  martin
 *  Replaced chk_model_is_vsg() by xdevfeat::xdevfeat_is_vsg().
 *  Revision 1.3.1.46  2016/03/03 11:20:30  martin
 *  Updated function prototypes.
 *  Revision 1.3.1.45  2016/02/17 12:00:07Z  gregoire
 *  new function chk_model_is_vsg
 *  Revision 1.3.1.44  2016/02/10 15:45:32Z  martin
 *  *** empty log message ***
 *  Revision 1.3.1.43  2015/11/24 13:12:04  philipp
 *  Extended / modified TLV functions
 *  Revision 1.3.1.42  2015/11/23 14:15:40  philipp
 *  Moved TLV related initializing functions to here
 *  Revision 1.3.1.41  2015/11/20 14:51:10  martin
 *  Revision 1.3.1.40  2015/11/02 11:00:48Z  martin
 *  *** empty log message ***
 *  Revision 1.3.1.39  2015/11/02 09:20:02  martin
 *  *** empty log message ***
 *  Revision 1.3.1.38  2015/10/27 16:21:31  martin
 *  Older defines N_SUPP_DEV, PCPS_MAX_DDEVS, and MBG_MAX_DEVICES
 *  have been obsoleted by new defines N_SUPP_DEV_BUS, N_SUPP_DEV_EXT,
 *  and N_SUPP_DEV_TOTAL.
 *  Revision 1.3.1.37  2015/10/26 16:31:53  martin
 *  *** empty log message ***
 *  Revision 1.3.1.36  2015/10/26 14:18:34  martin
 *  Revision 1.3.1.35  2015/10/15 12:49:10Z  marvin
 *  Revision 1.3.1.34  2015/10/12 10:01:59Z  martin
 *  *** empty log message ***
 *  Revision 1.3.1.33  2015/10/08 13:27:37  martin
 *  *** empty log message ***
 *  Revision 1.3.1.32  2015/10/08 10:32:16  martin
 *  *** empty log message ***
 *  Revision 1.3.1.31  2015/10/08 09:30:34  martin
 *  *** empty log message ***
 *  Revision 1.3.1.30  2015/10/07 10:12:08  martin
 *  *** empty log message ***
 *  Revision 1.3.1.29  2015/10/07 10:08:34  martin
 *  *** empty log message ***
 *  Revision 1.3.1.28  2015/10/07 09:59:00  martin
 *  More common GNSS support.
 *  Revision 1.3.1.27  2015/09/15 09:10:40  martin
 *  Updated function prototypes.
 *  Revision 1.3.1.26  2015/09/11 12:09:09  thomas-b
 *  Added nano_time_64_to_tm_gps and tm_gps_to_nano_time_64 functions
 *  Revision 1.3.1.25  2015/08/31 14:55:11  martin
 *  Moved string trim functions to str_util module.
 *  Revision 1.3.1.24  2015/08/31 14:49:25  martin
 *  Revision 1.3.1.23  2015/08/31 09:58:08  martin
 *  Revision 1.3.1.22  2015/08/27 16:30:10Z  martin
 *  Revision 1.3.1.21  2015/08/26 07:31:51  martin
 *  Revision 1.3.1.20  2015/08/21 14:22:59  martin
 *  Revision 1.3.1.19  2014/10/29 16:25:31  martin
 *  Moved some functions and structures to more convenient files.
 *  Revision 1.3.1.18  2014/10/29 16:00:37  martin
 *  Revision 1.3.1.17  2014/10/29 14:21:55  martin
 *  Revision 1.3.1.16  2014/09/26 11:43:24  martin
 *  Revision 1.3.1.15  2014/07/22 13:05:35  martin
 *  Revision 1.3.1.14  2014/07/14 15:42:45Z  martin
 *  Revision 1.3.1.13  2014/06/25 15:15:20  martin
 *  Revision 1.3.1.12  2014/06/25 08:51:36Z  martin
 *  Support GPIO status.
 *  Revision 1.3.1.11  2014/05/22 16:15:16Z  martin
 *  Revision 1.3.1.10  2014/05/14 12:43:53  martin
 *  Revision 1.3.1.9  2014/05/13 08:23:24  martin
 *  Revision 1.3.1.8  2014/05/13 08:19:34Z  martin
 *  Revision 1.3.1.7  2014/04/28 12:33:09  martin
 *  Revision 1.3.1.6  2014/04/28 12:04:32  martin
 *  Revision 1.3.1.5  2014/04/25 09:16:38  martin
 *  Updated function prototypes.
 *  Revision 1.3.1.4  2013/12/18 14:51:37  martin
 *  Moved inline function num_bits_set() here.
 *  Revision 1.3.1.3  2013/11/13 10:00:09  martin
 *  Revision 1.3.1.2  2013/11/12 12:12:40  marvin
 *  Changed calls for NTP info and settings.
 *  Revision 1.3.1.1  2013/09/25 10:14:38Z  martin
 *  Started to support NTP configuration.
 *  Revision 1.3  2013/09/25 10:02:15  martin
 *  Added ALL_PTP_CFG_INFO, ALL_GNSS_SAT_INFO_IDX and
 *  related definitions.
 *  Added doxygen comments.
 *  Revision 1.2  2012/10/02 18:16:26  martin
 *  Modified some typedefs to be more compliant with the underlying types.
 *  Revision 1.1  2011/09/21 15:59:59  martin
 *  Initial revision.
 *
 **************************************************************************/

#ifndef _CFG_HLP_H
#define _CFG_HLP_H


/* Other headers to be included */

#include <gpsdefs.h>
#include <mbgklist.h>

#if !defined( MBG_TGT_KERNEL )
  #include <stdlib.h>
  #include <string.h>
#endif

#if defined( _PRELIMINARY_CODE )
  #if defined( MBG_TGT_POSIX )
    #include <sys/stat.h>
    #include <sys/sysinfo.h>
    #include <time.h>
  #endif // MBG_TGT_POSIX
#endif // _PRELIMINARY_CODE

#ifdef _CFG_HLP
 #define _ext
 #define _DO_INIT
#else
 #define _ext extern
#endif


/* Start of header body */

#ifdef __cplusplus
extern "C" {
#endif


#if 1  // ### TODO cleanup

#define N_SUPP_DEV_BUS   16
#define N_SUPP_DEV_EXT    1

#define N_SUPP_DEV_TOTAL   ( N_SUPP_DEV_BUS + N_SUPP_DEV_EXT )

typedef struct _DEVICE_INFO
{
  char *hw_id;
  char fw_id[100];

} DEVICE_INFO;

_ext DEVICE_INFO device_list[N_SUPP_DEV_TOTAL];

#endif



/// @brief The max number of serial ports supported by configuration programs
#define MAX_PARM_PORT        10

/// @brief The max number of serial string types supported by configuration programs
#define MAX_PARM_STR_TYPE    20

/// @brief The max number of programmable pulse outputs supported by configuration programs
#define MAX_PARM_POUT        10

/// @brief The max number of GNSS settings supported by configuration programs
#define MAX_PARM_GNSS_SAT  N_GNSS_TYPES

/// @brief The max number of PTP unicast masters supported by configuration programs
#define MAX_PARM_PTP_UC_MASTER    3

/// @brief The max number of external NTP server associations to be handled by configuration programs
#define MAX_PARM_EXT_NTP_SRVR  20

/// @brief The max number of GPIO ports supported by configuration programs
#define MAX_PARM_GPIO    10

/// @brief The max number of XMR sources supported by configuration programs
#define MAX_PARM_XMR     10

/// @brief The max number of external NTP servers supported by configuration programs
#define MAX_EXT_NTP_SERVERS  20

/// @brief The max. number of time monitoring modules supported by configuration programs
/// Each module may support a different number of targets to be monitored.
/// @see ### TODO
#define MAX_MBG_TIME_MON_MODULES 10

/// @brief The max. number of time monitoring targets supported by configuration programs
/// This is the sum of all targets from all monitoring modules.
/// @see ### TODO
#define MAX_MBG_TIME_MON_TARGETS 100



/// @brief An array of configuration settings for all serial ports
typedef PORT_INFO_IDX ALL_PORT_INFO_IDX[MAX_PARM_PORT];

/// @brief An array of configuration settings for all serial string types
typedef STR_TYPE_INFO_IDX ALL_STR_TYPE_INFO_IDX[MAX_PARM_STR_TYPE];

/**
 * @brief All configuration parameters for all serial ports
 *
 * Used to collect all configuration parameters of a clock's serial ports
 * that can be handled by a configuration program.
 *
 * @see ::RECEIVER_INFO::n_com_ports
 * @see ::RECEIVER_INFO::n_str_type
 */
typedef struct
{
  ALL_PORT_INFO_IDX pii;       ///< all serial port configuration settings
  ALL_STR_TYPE_INFO_IDX stii;  ///< all supported serial string types
  PORT_PARM tmp_pp;            ///< used internally only, for compatibility

} RECEIVER_PORT_CFG;



/**
 * @brief All XBP information of a XBP supporting device
 *
 * This structure represents a list of connected devices
 *
 * @see ::GPS_HAS_XBP
 */
typedef struct
{
  XBP_LIMITS limits;
  XBP_NODE_LIMITS* node_limits;
  XBP_NODE_INFO_IDX* node_infos;
} ALL_XBP_INFO;



/**
 * @brief An array of configuration settings for all programmable pulse outputs
 *
 * Used to collect all configuration parameters of a clock's programmable pulse outputs
 * that can be handled by a configuration program.
 *
 * @see ::RECEIVER_INFO::n_prg_out
 */
typedef POUT_INFO_IDX ALL_POUT_INFO_IDX[MAX_PARM_POUT];



/**
 * @brief All network configuration parameters
 *
 * Used to collect all configuration parameters for networking
 *
 * @see ::GPS_HAS_NET_CFG
 * @see ::GPS_HAS_LAN_IP4
 */
typedef struct
{
  MBG_NET_GLB_CFG_INFO glb_cfg_info;
  MBG_NET_INTF_LINK_INFO_IDX *link_infos;
  MBG_NET_INTF_ADDR_INFO_IDX *addr_infos;
  MBG_IP_ADDR_IDX *dns_srvrs;
  MBG_NET_NAME_IDX *dns_srch_doms;
  MBG_NET_INTF_ROUTE_INFO_IDX *route_infos;
} ALL_NET_CFG_INFO;

typedef ALL_NET_CFG_INFO ALL_NET_STATUS_INFO;



/**
 * @brief All SNMP configuration information
 *
 * Used to collect all configuration parameters for monitoring via SNMP
 * Can be used, if ::MBG_MONITORING_TYPE_MSK_SNMP is set in ::MBG_MONITORING_LIMITS::supp_types
 *
 * @see ::MBG_XFEATURE_MONITORING
 */
typedef struct
{
  MBG_SNMP_GLB_INFO glb_info;
  MBG_SNMP_V12_INFO_IDX *v12_infos;
  MBG_SNMP_V12_TRAP_INFO_IDX *v12_trap_infos;
  MBG_SNMP_V3_INFO_IDX *v3_infos;
  MBG_SNMP_V3_TRAP_INFO_IDX *v3_trap_infos;

} ALL_SNMP_INFO;


/**
 * @brief All monitoring information
 *
 * Used to collect all configuration parameters for monitoring of a device
 * Depending on the ::MBG_MONITORING_LIMITS::supp_types,
 * the approriate configurations can be found in the sub structures
 *
 * @see ::MBG_XFEATURE_MONITORING
 */
typedef struct
{
  MBG_EVENT_INFO_IDX info;
  void *priv_data;

} MBG_EVENT_INFO_IDX_DATA;

typedef struct
{
  MBG_MONITORING_LIMITS limits;
  ALL_SNMP_INFO *all_snmp_info;
  MBG_EVENT_INFO_IDX_DATA *event_infos;

} ALL_MONITORING_INFO;



/**
 * @brief All monitoring status
 *
 * Used to collect all status information for monitoring of a device
 * Depending on the ::MBG_MONITORING_LIMITS::supp_events, the
 * appropriate event status for each event can be found in ::event_stati.
 */
typedef struct
{
  MBG_EVENT_STATUS_IDX status;
  void *priv_data;

} MBG_EVENT_STATUS_IDX_DATA;

typedef struct
{
  MBG_MONITORING_STATUS status;
  MBG_EVENT_STATUS_IDX_DATA *event_stati;

} ALL_MONITORING_STATUS;


/// @brief Configuration settings for all unicast master specifications
typedef PTP_UC_MASTER_INFO_IDX ALL_PTP_UC_MASTER_INFO_IDX[MAX_PARM_PTP_UC_MASTER];

/**
 * @brief All PTP configuration parameters
 *
 * Used to collect all configuration parameters for a PTP daemon
 * that can be handled by a configuration program.
 *
 * @see ::GPS_HAS_PTP
 * @see ::PTP_UC_MASTER_CFG_LIMITS::n_supp_master
 */
typedef struct
{
  PTP_CFG_INFO ptp_cfg_info;
  PTP_UC_MASTER_CFG_LIMITS ptp_uc_master_cfg_limits;
  ALL_PTP_UC_MASTER_INFO_IDX all_ptp_uc_master_info_idx;

} ALL_PTP_CFG_INFO;



/**
 * @brief All PTPv1 common datasets for a PTP device
 *
 * Contains one of each common datasets plus one port dataset
 * for each port of a device. The number of port datasets is
 * defined in ::default_dataset::number_ports.
 *
 * @see ::MBG_PTP_V1_DEFAULT_DATASET
 * @see ::MBG_PTP_V1_CURRENT_DATASET
 * @see ::MBG_PTP_V1_PARENT_DATASET
 * @see ::MBG_PTP_V1_TIME_PROPERTIES_DATASET
 * @see ::MBG_PTP_V1_PORT_DATASET_IDX
 */
typedef struct
{
  MBG_PTP_V1_DEFAULT_DATASET default_dataset;
  MBG_PTP_V1_CURRENT_DATASET current_dataset;
  MBG_PTP_V1_PARENT_DATASET parent_dataset;
  MBG_PTP_V1_TIME_PROPERTIES_DATASET time_properties_dataset;
  MBG_PTP_V1_PORT_DATASET_IDX *port_datasets;

} ALL_PTP_V1_COMMON_DATASETS;



/**
 * @brief All PTPv2 common datasets for a PTP device
 *
 * Contains one of each common datasets plus one port dataset
 * for each port of a device. The number of port datasets is
 * defined in ::default_dataset::number_ports.
 *
 * @see ::MBG_PTP_V2_DEFAULT_DATASET
 * @see ::MBG_PTP_V2_CURRENT_DATASET
 * @see ::MBG_PTP_V2_PARENT_DATASET
 * @see ::MBG_PTP_V2_TIME_PROPERTIES_DATASET
 * @see ::MBG_PTP_V2_PORT_DATASET_IDX
 */
typedef struct
{
  MBG_PTP_V2_DEFAULT_DATASET default_dataset;
  MBG_PTP_V2_CURRENT_DATASET current_dataset;
  MBG_PTP_V2_PARENT_DATASET parent_dataset;
  MBG_PTP_V2_TIME_PROPERTIES_DATASET time_properties_dataset;
  MBG_PTP_V2_PORT_DATASET_IDX *port_datasets;

} ALL_PTP_V2_COMMON_DATASETS;



/**
 * @brief An array of configuration settings for all programmable pulse outputs
 *
 * Used to collect all configuration parameters of a clock's programmable pulse outputs
 * that can be handled by a configuration program.
 */
typedef GNSS_SAT_INFO_IDX  ALL_GNSS_SAT_INFO_IDX[MAX_PARM_GNSS_SAT];



/**
 * @brief An array of configuration settings for all programmable pulse outputs ####
 *
 * Used to collect all configuration parameters of a clock's programmable pulse outputs
 * that can be handled by a configuration program. #####
 */
typedef struct
{
  STAT_INFO stat_info;
  int n_gnss_supp;
  MBG_GNSS_MODE_INFO gnss_mode_info;
  ALL_GNSS_SAT_INFO_IDX gnss_sat_info_idx;

} ALL_GNSS_INFO;



/// @brief Configuration settings for all NTP server associatioions
typedef NTP_PEER_SETTINGS ALL_NTP_PEER_SETTINGS[MAX_EXT_NTP_SERVERS];

/**
 * @brief All NTP configuration parameters
 *
 * Used to collect all configuration parameters for an NTP daemon
 * that can be handled by a configuration program.
 *
 * @see ::GPS_HAS_NTP
 */
typedef struct
{
  ALL_NTP_PEER_SETTINGS all_ntp_peer_settings;

} NTP_CLIENT_CFG_PEER_SETTINGS;


typedef struct
{
  NTP_GLB_INFO glb_info;
  NTP_SYMM_KEY_LIMITS *symm_key_limits;
  NTP_SYMM_KEY_INFO_IDX *symm_key_info_idx;
  NTP_TRUSTED_KEY_INFO_IDX *trusted_key_info_idx;

  NTP_CLNT_MODE_INFO *clnt_info;
  NTP_PEER_SETTINGS_IDX *peer_settings_idx;

  NTP_SRV_MODE_INFO *srv_info;
  NTP_REFCLK_CFG_INFO_IDX *refclk_info_idx;
  NTP_MISC_LIMITS *misc_limits;
  NTP_MISC_ORPHAN_MODE_INFO *orphan_mode_info;

} ALL_NTP_CFG_INFO;


typedef struct
{
  NTP_SYS_STATE sys_state;
  NTP_PEER_STATE_IDX *peer_states;
} ALL_NTP_STATUS;



/// @brief Configuration settings for all GPIO ports
typedef MBG_GPIO_INFO_IDX ALL_GPIO_INFO_IDX[MAX_PARM_GPIO];

/// @brief Status information on all GPIO ports
typedef MBG_GPIO_STATUS_IDX ALL_GPIO_STATUS_IDX[MAX_PARM_GPIO];




/// @brief Status of all XMR inputs
typedef XMULTI_REF_STATUS_IDX ALL_XMULTI_REF_STATUS_IDX[MAX_PARM_XMR];

/// @brief Configuration settings for all XMR inputs
typedef XMULTI_REF_INFO_IDX ALL_XMULTI_REF_INFO_IDX[MAX_PARM_XMR];



typedef struct
{
  XMULTI_REF_INSTANCES instances;
  XMULTI_REF_INFO_IDX *infos;
  XMR_EXT_SRC_INFO_IDX *ext_src_infos;
} ALL_XMULTI_REF_INFO;


typedef struct
{
  XMULTI_REF_STATUS_IDX *status;
  XMR_HOLDOVER_STATUS *holdover_status;
  XMR_STATS_IDX *stats_idx;
  XMR_METRICS_IDX *metrics_idx;
  /* ALL_XMULTI_REF_STATUS related flag if at least one ref type supports stats */
  unsigned char has_stats;
  /* ALL_XMULTI_REF_STATUS related flag if at least one ref type supports metrics */
  unsigned char has_metrics;
} ALL_XMULTI_REF_STATUS;



typedef struct
{
  MBG_IMS_STATE state;
  MBG_IMS_FDM_INFO *fdm_info;
  MBG_IMS_FDM_LIMITS *fdm_limits;
  MBG_IMS_FDM_OUTPUT_INFO_IDX *fdm_outinfo_idx;
} ALL_IMS_INFO;


typedef struct
{
  MBG_IMS_SENSOR_STATE_IDX *sensor_state_idx;
  MBG_IMS_FDM_STATE *fdm_state;
  MBG_IMS_FDM_OUTPUT_STATE_IDX *fdm_output_state_idx;
} ALL_IMS_STATE;



typedef struct
{
  MBG_GPIO_CFG_LIMITS cfg_limits;
  MBG_GPIO_INFO_IDX *infos;
} ALL_GPIO_INFO;


typedef struct
{
  MBG_GPIO_STATUS_IDX *states;
} ALL_GPIO_STATE;


typedef struct
{
  MBG_IO_PORT_LIMITS limits;
  MBG_IO_PORT_INFO_IDX *p_infos;
  MBG_IO_PORT_TYPE_INFO_IDX **pt_infos;
} ALL_IO_PORT_INFO;


typedef struct
{
  MBG_IO_PORT_STATUS_IDX *status;
} ALL_IO_PORT_STATUS;


#ifndef MAX_UCAP_ENTRIES
/*
 * According to Andre's GPS firmware this is the maximum
 * number of user captures that are preserved.
 */
#define MAX_UCAP_ENTRIES 585
#endif

typedef struct
{
  struct mbg_klist_head head;
  TTM ttm;
} UCAP_ENTRY;

typedef struct
{
  uint32_t num_ucaps; /// User capture counter, see ::MAX_UCAP_ENTRIES
  struct mbg_klist_head list;
} ALL_UCAP_INFO;

// User Captures via Network configuration, see ::MBG_XFEATURE_UCAP_NET
typedef struct
{
  MBG_UCAP_NET_GLB_INFO glb_info;
  MBG_UCAP_NET_RECV_INFO_IDX *recv_infos;
} ALL_UCAP_NET_INFO;



/**
 * @brief A mode specifying how to interpret a ::PCPS_SIG_VAL
 *
 * Used with ::PCPS_TIME_EXT::comp_sig_mode. Depending on this mode
 * a signal value can be interpreted e.g. as signal strength (with
 * long wave or IRIG time code receivers), or as indicator whether an
 * antenna is connected (satellite receivers), or a network link is
 * available (PTP slaves) or not, and an appropriate status message
 * can be displayed.
 *
 * @see @ref PCPS_SIG_VAL_DEFS
 */
enum COMP_SIG_MODES
{
  COMP_SIG_MODE_NONE,       ///< signal value not used
  COMP_SIG_MODE_SIGNAL,     ///< input signal strength
  COMP_SIG_MODE_ANT_CONN,   ///< antenna connection state
  COMP_SIG_MODE_PORT_LINK,  ///< port link state
  N_CONN_SIG_MODES
};


/**
 * @brief Flag bits indicating if some extended status is available
 *
 * @see ::PCPS_TIME_EXT_FLAGS
 */
enum PCPS_TIME_EXT_FLAG_BITS
{
  PCPS_TIME_EXT_FLAG_BIT_UTC_VALID,   ///< ::PCPS_TIME_EXT::utc_offs field is valid
  N_PCPS_TIME_EXT_FLAG_BITS
};


/**
 * @brief Flag masks used with ::PCPS_TIME_EXT::flags
 *
 * @see ::PCPS_TIME_EXT_FLAG_BITS
 */
enum PCPS_TIME_EXT_FLAGS
{
  PCPS_TIME_EXT_FLAG_UTC_VALID = ( 1UL << PCPS_TIME_EXT_FLAG_BIT_UTC_VALID )  ///< see ::PCPS_TIME_EXT_FLAG_BIT_UTC_VALID
};



_ext BAUD_RATE mbg_baud_rate[N_MBG_BAUD_RATES]
#ifdef _DO_INIT
 = MBG_BAUD_RATES
#endif
;

_ext const char *mbg_baud_str[N_MBG_BAUD_RATES]
#ifdef _DO_INIT
 = MBG_BAUD_STRS
#endif
;

_ext const char *mbg_framing_str[N_MBG_FRAMINGS]
#ifdef _DO_INIT
 = MBG_FRAMING_STRS
#endif
;

_ext const char *str_unknown
#ifdef _DO_INIT
 = "unknown"
#endif
;


_ext const char *str_undefined
#ifdef _DO_INIT
 = "(undefined)"
#endif
;


_ext const char *str_not_spc
#ifdef _DO_INIT
 = "not "
#endif
;



//### TODO
#define DEFAULT_MAX_STR_TYPE    2  // DEFAULT_N_STR_TYPE_GPS ?

_ext STR_TYPE_INFO default_str_type_info[DEFAULT_MAX_STR_TYPE]
#ifdef _DO_INIT
  = {
      { DEFAULT_STR_MODES,      "Default Time String", "Time", 0 },
      { DEFAULT_STR_MODES_UCAP, "Capture String",      "Cap",  0 }
    }
#endif
;



_ext const char *mbg_gpio_type_names[N_MBG_GPIO_TYPES]
#ifdef _DO_INIT
 = DEFAULT_GPIO_TYPES_SHORT_STRS
#endif
;

#define _get_gpio_type_name( _i ) \
  ( ( (_i) < N_MBG_GPIO_TYPES ) ? mbg_gpio_type_names[_i] : str_unknown )



_ext const char *mbg_gpio_port_state_names[N_MBG_GPIO_PORT_STATES]
#ifdef _DO_INIT
 = DEFAULT_GPIO_PORT_STATE_NAMES
#endif
;

#define _get_gpio_port_state_name( _i ) \
  ( ( (_i) < N_MBG_GPIO_PORT_STATES ) ? mbg_gpio_port_state_names[_i] : str_unknown )



_ext const char *mbg_gpio_signal_shape_names[N_MBG_GPIO_SIGNAL_SHAPES]
#ifdef _DO_INIT
 = DEFAULT_GPIO_SIGNAL_SHAPE_NAMES
#endif
;

#define _get_gpio_signal_shape_name( _i ) \
  ( ( (_i) < N_MBG_GPIO_SIGNAL_SHAPES ) ? mbg_gpio_signal_shape_names[_i] : str_unknown )



_ext const char *mbg_gpio_fixed_freq_strs[N_MBG_GPIO_FIXED_FREQ]
#ifdef _DO_INIT
 = MBG_GPIO_FIXED_FREQ_STRS
#endif
;

#define _get_gpio_fixed_freq_str( _i ) \
  ( ( (_i) < N_MBG_GPIO_FIXED_FREQ ) ? mbg_gpio_fixed_freq_strs[_i] : str_unknown )



_ext const char *xmr_holdover_status_mode_names[N_XMR_HOLDOVER_STATUS_MODES]
#ifdef _DO_INIT
 = XMR_HOLDOVER_STATUS_MODE_NAMES
#endif
;

#define _get_xmr_holdover_status_mode_name( _i ) \
  ( ( (_i) < N_XMR_HOLDOVER_STATUS_MODES ) ? xmr_holdover_status_mode_names[_i] : str_unknown )



/**
 * @brief Count the number of bits which are not 0
 *
 * @param[in] val  Value to be tested
 *
 * @return The number of non-zero bits in val
 */
static __mbg_inline
int num_bits_set( long val )
{
  int bits_set = 0;
  size_t i;

  for ( i = 0; i < ( 8 * sizeof( val ) ); i++ )
  {
    if ( val & 1 )
      bits_set++;

    val >>= 1;
  }

  return bits_set;

}  // num_bits_set



/**
 * @brief Check if a device ID refers to a serial port
 *
 * @param[in] dev_id  A string with the device name or port name
 *
 * @see ::DEFAULT_SERIAL_DEVICE_NAME
 *
 * @return true if the device id contains the name of a serial port, else false
 */
static __mbg_inline
bool device_id_is_serial( const char *dev_id )
{
  #if defined( MBG_TGT_WIN32 )
    //##++++ There may be also serial ports under Windows
    // which don't have "COM" in their name.
    return strstr( dev_id, "COM" ) != NULL;
  #elif defined( MBG_TGT_LINUX )
    return strstr( dev_id, "/dev/ttyS" ) != NULL     // standard serial device
        || strstr( dev_id, "/dev/ttyUSB" ) != NULL;  // serial-to-USB adapter
  #elif defined( MBG_TGT_FREEBSD )
    return strstr( dev_id, "/dev/ttyu" ) != NULL     // dial-in device (standard), FreeBSD 10 and newer
        || strstr( dev_id, "/dev/cuau" ) != NULL     // dial out device, FreeBSD 10 and newer
        || strstr( dev_id, "/dev/ttyd" ) != NULL     // dial-in device (standard), before FreeBSD 10
        || strstr( dev_id, "/dev/cuad" ) != NULL;    // dial-out device, before FreeBSD 10
  #elif defined( MBG_TGT_QNX_NTO )
    return strstr( dev_id, "/dev/ser" ) != NULL;
  #elif defined( MBG_TGT_DOS )
    return strstr( dev_id, "COM" ) != NULL;
  #else
    #error device_id_is_serial() needs to be implemented for this platform
  #endif

}  // device_id_is_serial



/**
 * @brief Check if a device ID refers to a LAN connection
 *
 * @param[in] dev_id  A string with the device ID
 *
 * @return true if the device id specifies a LAN connection, else false
 */
static __mbg_inline
bool device_id_is_lan( const char *dev_id )
{
  return strstr( dev_id, "LAN" ) != NULL;

}  // device_id_is_lan



#if defined( _PRELIMINARY_CODE )

static __mbg_inline
MBG_TLV_UID mbg_tlv_create_id( void )
{
#if defined( MBG_TGT_POSIX )
  struct sysinfo info;

  // Linux specific, implement Windows equivalent
  sysinfo( &info );
  return ( (MBG_TLV_UID) ( ( time( NULL ) >> 16 ) | ( info.uptime << 16 ) ) );
#else
  return 0;
#endif
} // mbg_tlv_create_id

#endif  // defined( _PRELIMINARY_CODE )



// Depending on the target environment define a data type
// which can be used to convert binary fractions without
// range overflow.
#if defined( MBG_TGT_MISSING_64_BIT_TYPES )
  #define PCPS_HRT_FRAC_CONVERSION_TYPE double
#else
  #define PCPS_HRT_FRAC_CONVERSION_TYPE int64_t
#endif

/**
 * @brief Constant used to convert ::PCPS_TIME_STAMP::frac values
 *
 * Max value of ::PCPS_TIME_STAMP::frac + 1, used for scaling
 */
#define PCPS_HRT_BIN_FRAC_SCALE  ( (PCPS_HRT_FRAC_CONVERSION_TYPE) 4294967296.0  )  // == 0x100000000


#ifndef PCPS_HRT_FRAC_SCALE
  /**
   * @brief Scale to be used to print ::PCPS_TIME_STAMP::frac values
   *
   * The function ::frac_sec_from_bin can be used for the conversion.
   *
   * @see ::PCPS_HRT_FRAC_SCALE_FMT
   */
  #define PCPS_HRT_FRAC_SCALE       10000000UL
#endif

#ifndef PCPS_HRT_FRAC_SCALE_FMT
  /**
   * @brief Format specifier used to print ::PCPS_TIME_STAMP::frac values
   *
   * Used to print values scaled with ::frac_sec_from_bin called
   * with ::PCPS_HRT_FRAC_SCALE.
   *
   * @see ::PCPS_HRT_FRAC_SCALE
   */
  #define PCPS_HRT_FRAC_SCALE_FMT   "%07lu"
#endif


static __mbg_inline
uint32_t bin_frac_16_to_dec_frac( uint16_t bin, uint32_t scale )
{
  return (uint32_t) ( (PCPS_HRT_FRAC_CONVERSION_TYPE) bin * scale
                      / 0x10000UL );

} // bin_frac_16_to_dec_frac


static __mbg_inline
uint32_t bin_frac_32_to_dec_frac( uint32_t bin, uint32_t scale )
{
  return (uint32_t) ( (PCPS_HRT_FRAC_CONVERSION_TYPE) bin * scale
                      / PCPS_HRT_BIN_FRAC_SCALE );

} // bin_frac_32_to_dec_frac


#if !defined( MBG_TGT_MISSING_64_BIT_TYPES )

// On targets which don't provide 64 bit data types
// PCPS_HRT_FRAC_CONVERSION_TYPE is defined as double,
// in which case the ">> 1" operation in the 2 functions
// below yields an "invalid use of floating point" error.
// This could probably be fixed by a different way of
// casting, at least for a partial expression.

static __mbg_inline
uint16_t dec_frac_to_bin_frac_16( uint32_t dec, uint32_t scale )
{
  return (uint16_t) ( ( ( (PCPS_HRT_FRAC_CONVERSION_TYPE) dec * 0x20000 / scale ) + 1 ) >> 1 );

} // dec_frac_to_bin_frac_16


static __mbg_inline
uint32_t dec_frac_to_bin_frac_32( uint32_t dec, uint32_t scale )
{
  return (uint32_t) ( ( ( (PCPS_HRT_FRAC_CONVERSION_TYPE) dec * PCPS_HRT_BIN_FRAC_SCALE * 2 / scale ) + 1 ) >> 1 );

} // dec_frac_to_bin_frac_32

#endif


#define bin_frac_32_to_msec( _bin ) bin_frac_32_to_dec_frac( (_bin), 1000 )
#define bin_frac_32_to_usec( _bin ) bin_frac_32_to_dec_frac( (_bin), 1000000 )
#define bin_frac_32_to_nsec( _bin ) bin_frac_32_to_dec_frac( (_bin), 1000000000 )
#define bin_frac_16_to_msec( _bin ) bin_frac_16_to_dec_frac( (_bin), 1000 )
#define bin_frac_16_to_usec( _bin ) bin_frac_16_to_dec_frac( (_bin), 1000000 )
#define bin_frac_16_to_nsec( _bin ) bin_frac_16_to_dec_frac( (_bin), 1000000000 )


#define msec_to_bin_frac_32( _msec ) dec_frac_to_bin_frac_32( (_msec), 1000 )
#define usec_to_bin_frac_32( _usec ) dec_frac_to_bin_frac_32( (_usec), 1000000 )
#define nsec_to_bin_frac_32( _nsec ) dec_frac_to_bin_frac_32( (_nsec), 1000000000 )
#define msec_to_bin_frac_16( _msec ) dec_frac_to_bin_frac_16( (_msec), 1000 )
#define usec_to_bin_frac_16( _usec ) dec_frac_to_bin_frac_16( (_usec), 1000000 )
#define nsec_to_bin_frac_16( _nsec ) dec_frac_to_bin_frac_16( (_nsec), 1000000000 )



/**
 * @brief Convert a fraction of a second from binary
 *
 * Convert a fraction of a second from binary format (as returned
 * as part of the ::PCPS_HR_TIME structure) to a decimal fraction,
 * using a specified scale factor.
 *
 * @param[in] b      The binary fraction
 * @param[in] scale  The scale factor
 *
 * @return The calculated number
 *
 * @see ::PCPS_HRT_FRAC_SCALE
 * @see ::PCPS_HRT_FRAC_SCALE_FMT
 */
static __mbg_inline
uint32_t frac_sec_from_bin( uint32_t b, uint32_t scale )
{
  return bin_frac_32_to_dec_frac( b, scale );

}  // frac_sec_from_bin



/**
 * @brief Convert a fraction of a second to double
 *
 * Convert a fraction of a second from binary format (as returned
 * as part of the ::PCPS_HR_TIME structure) to a double with the
 * units of seconds, e.g. 0xFFFFFFFF yields 0.9999999999....
 *
 * @param[in] b      The binary fraction
 *
 * @return The calculated fraction
 *
 * @see ::PCPS_HRT_FRAC_SCALE
 */
static __mbg_inline
double dfrac_sec_from_bin( uint32_t b )
{
  return (double) b / (double) PCPS_HRT_BIN_FRAC_SCALE;

}  // dfrac_sec_from_bin



#if !defined( MBG_TGT_KERNEL )

/* function prototypes: */

/* ----- function prototypes begin ----- */

/* This section was generated automatically */
/* by MAKEHDR, do not remove the comments. */

 /**
 * @brief Normalize a ::NANO_TIME_64 struct
 *
 * After normalization, the following can be assumed:<br>
 * - nano_secs is in the range [-10^9 + 1, 10^9 - 1]<br>
 * - if secs is not 0, secs and nano_secs have the same sign
 *
 * @param[in,out] nt  The NANO_TIME_64 to be normalized
 */
 void normalize_nano_time_64( NANO_TIME_64 *nt ) ;

 /**
 * @brief Print a normalized ::NANO_TIME_64 into a string buffer
 *
 * @param[out] s        The string buffer to be filled
 * @param[in]  max_len  Size of the output buffer for 0-terminated string
 * @param[in]  nt       The ::NANO_TIME_64 to be printed
 */
 size_t snprint_nano_time_64( char *s, size_t max_len, const NANO_TIME_64 *nt ) ;

 /**
 * @brief Print nano time into string buffer
 *
 * @param[out] s        The string buffer to be filled
 * @param[in]  max_len  Size of the output buffer for 0-terminated string
 * @param[in]  nt       The ::NANO_TIME to be printed
 */
 size_t snprint_nano_time( char *s, size_t max_len, const NANO_TIME *nt ) ;

 /**
 * @brief Convert ::NANO_TIME_64 to ::TM_GPS
 *
 * @param[out] tm_gps  The ::TM_GPS to be filled
 * @param[in]  nt      The ::NANO_TIME_64 to be converted
 *
 * @return 1 on success, 0 on error
 *
 * @see ::tm_gps_to_nano_time_64
 */
 int nano_time_64_to_tm_gps( TM_GPS *tm_gps, const NANO_TIME_64 *nt ) ;

 /**
 * @brief Convert ::TM_GPS to ::NANO_TIME_64
 *
 * @param[out] nt      The ::NANO_TIME_64 to be filled
 * @param[in]  tm      The ::TM_GPS to be converted
 *
 * @return 1 on success, 0 on error
 *
 * @see ::nano_time_64_to_tm_gps
 */
 int tm_gps_to_nano_time_64( NANO_TIME_64 *nt, const TM_GPS *tm ) ;

 /**
 * @brief Check if a software revision name should be displayed
 *
 * The software revision name is usually empty, except if the
 * firmware is a customized version, in which case the field
 * contains an identifier string.
 *
 * There are some standard firmware versions where this string
 * is not empty but padded with spaces, etc., so we try to
 * clean this up and display the string properly, if appropriate.
 *
 * @param[in,out] p        The ::SW_REV name to check
 * @param[in]     verbose  The app's verbosity level
 *
 * @return != 0 if SW name should be displayed
 */
 int chk_sw_rev_name( SW_REV *p, int verbose ) ;

 int get_str_idx( const char *search, const char *str_table[], int n_entries ) ;
 int get_baud_rate_idx( BAUD_RATE baud_rate ) ;
 int get_framing_idx( const char *framing ) ;
 void port_settings_from_port_parm_mode( PORT_SETTINGS *p_ps, uint8_t pp_mode, int str_type_cap ) ;
 void port_parm_mode_from_port_settings( uint8_t *pp_mode, const PORT_SETTINGS *p_ps, int str_type_cap ) ;
 void port_settings_from_port_parm( PORT_SETTINGS *p_ps, int port_num, const PORT_PARM *p_pp, int cap_str_idx ) ;
 void port_parm_from_port_settings( PORT_PARM *p_pp, int port_num, const PORT_SETTINGS *p_ps, int cap_str_idx ) ;
 uint32_t check_valid_port_info( const PORT_INFO *p_pi, const STR_TYPE_INFO_IDX str_type_info_idx[], int n_str_type ) ;
 int valid_port_info( const PORT_INFO *p_pi, const STR_TYPE_INFO_IDX str_type_info_idx[], int n_str_type ) ;
 int setup_port_info_from_port_settings( PORT_INFO_IDX pii[], const PORT_PARM *p_pp, const RECEIVER_INFO *p_ri ) ;
 int setup_default_str_type_info_idx( STR_TYPE_INFO_IDX stii[], const RECEIVER_INFO *p_ri ) ;
 int chk_set_n_gnss_supp( ALL_GNSS_INFO *p_agi ) ;
 /**
 * @brief
 *
 * ### Setup GNSS info from stat_info so we can use the same printing routine
 */
 void setup_gps_only_sat_info_idx_from_statinfo( ALL_GNSS_INFO *p_agi ) ;

 /**
 * @brief
 *
 * Setup GNSS info from stat_info so we can use the same printing routine
 */
 int setup_gps_only_gnss_info_from_statinfo( ALL_GNSS_INFO *p_agi ) ;

 void chk_free_dev_hw_id( DEVICE_INFO *p ) ;
 int alloc_dev_hw_id( DEVICE_INFO *p, size_t len ) ;
 const char *get_fw_id_from_hw_id( const char *hw_id ) ;
 const char *get_hw_id_from_fw_id( const char *fw_id ) ;
 /**
 * @brief Returns the currently used ::MBG_IO_PORT_TYPE_INFO_IDX for the appropriate ::MBG_IO_PORT_INFO_IDX
 *
 * @param[in]   all_io_port_info    Pointer to the ::ALL_IO_PORT_INFO, containing the current configuration
 * @param[in]   io_port_info_idx    Pointer to the ::MBG_IO_PORT_INFO_IDX, for which the ::MBG_IO_PORT_TYPE_INFO_IDX shall be found
 *
 * @return Pointer to the found ::MBG_IO_PORT_TYPE_INFO_IDX or NULL
 *
 */
 MBG_IO_PORT_TYPE_INFO_IDX *get_io_port_type_info_idx(ALL_IO_PORT_INFO *all_io_port_info, MBG_IO_PORT_INFO_IDX *io_port_info_idx) ;

 /**
 * @brief Initializes a ::MBG_TLV_ANNOUNCE structure
 *
 * @param[out] tlv            Pointer to a ::MBG_TLV_ANNOUNCE structure
 * @param[in]  uid            Unique sender ID used as identifier with all
 *                            subsequent messages related to this transaction.
 * @param[in]  tlv_feat_type  One of the ::MBG_TLV_FEAT_TYPES
 * @param[in]  total_bytes    Total number of bytes of all upcoming TLVs
 */
 void mbg_tlv_announce_init( MBG_TLV_ANNOUNCE *tlv, MBG_TLV_UID uid, MBG_TLV_TYPE tlv_feat_type, uint32_t total_bytes ) ;

 /**
 * @brief Initializes a ::MBG_TLV
 *
 * @param[out] tlv           Pointer to a valid ::MBG_TLV structure
 * @param[in]  uid           Unique sender ID used as identifier for each further
 *                           TLV message related to this type.
 * @param[in]  tlv_type      Type identifier, see ::MBG_TLV_TYPES
 * @param[in]  total_bytes   Total number of bytes belonging to this
 *                           TLV transaction (which is very likely split into several TLVs)
 */
 void mbg_tlv_init( MBG_TLV *tlv, MBG_TLV_UID uid, MBG_TLV_TYPE tlv_type, uint32_t total_bytes ) ;

 /**
 * @brief Initializes ::MBG_TLV_RCV_STATE structure
 *
 * @param[in,out] state         Pointer to ::MBG_TLV_RCV_STATE structure
 * @param[in]     uid           Unique sender ID used as identifier for each further
 *                              TLV message related to this type.
 * @param[in]     total_bytes   Total number of bytes belonging to this
 *                              TLV transaction (which is very likely split into several TLVS)
 */
 void mbg_tlv_rcv_state_init( MBG_TLV_RCV_STATE *state, MBG_TLV_UID uid, uint32_t total_bytes ) ;

 size_t mbg_snprint_revision( char *buf, size_t buflen, const char *prefix, const char *suffix, uint32_t rev) ;
 _NO_MBG_API_ATTR int _MBG_API chk_dev_xbp_supp_nodes( const ALL_XBP_INFO *info ) ;
 _NO_MBG_API_ATTR int _MBG_API chk_dev_net_cfg_supp_stage_2( const ALL_NET_CFG_INFO *info ) ;
 _NO_MBG_API_ATTR int _MBG_API chk_dev_ntp_supp_client( const ALL_NTP_CFG_INFO *info ) ;
 _NO_MBG_API_ATTR int _MBG_API chk_dev_ntp_supp_server( const ALL_NTP_CFG_INFO *info ) ;
 _NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_ref_supp_mrf_none( const ALL_XMULTI_REF_INFO *info ) ;
 _NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_ref_supp_ext_src_info( const ALL_XMULTI_REF_INFO *info ) ;
 _NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_ref_supp_holdover_status( const ALL_XMULTI_REF_INFO *info ) ;
 /*
 * Type is NOT an index of ::XMULTI_REF_INSTANCES::n_xmr_settings,
 * but of ::MULTI_REF_TYPES.
 * Depends on chk_dev_supp_xmulti_ref_ext_src_info.
 * @see chk_dev_supp_xmulti_ref_ext_src_info
 */
 _NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_ref_supp_ext_source_stats( const ALL_XMULTI_REF_INFO *info, int type ) ;

 /*
 * Type is NOT an index of ::XMULTI_REF_INSTANCES::n_xmr_settings,
 * but of ::MULTI_REF_TYPES.
 * Depends on chk_dev_supp_xmulti_ref_ext_src_info.
 * @see chk_dev_supp_xmulti_ref_ext_src_info
 */
 _NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_ref_supp_ext_source_metrics( const ALL_XMULTI_REF_INFO *info, int type ) ;

 _NO_MBG_API_ATTR int _MBG_API chk_dev_ims_has_fdm( const ALL_IMS_INFO *info ) ;
 _NO_MBG_API_ATTR int _MBG_API chk_dev_ims_is_volt_out_enabled( const ALL_IMS_STATE *ims_state, unsigned idx ) ;
 _NO_MBG_API_ATTR int _MBG_API chk_dev_ims_is_volt_out_overload( const ALL_IMS_STATE *ims_state, unsigned idx ) ;
 _NO_MBG_API_ATTR int _MBG_API chk_dev_ims_is_pll_locked( const ALL_IMS_STATE *ims_state, unsigned idx ) ;
 _NO_MBG_API_ATTR int _MBG_API chk_dev_gpio_supp_ass_idx( const ALL_GPIO_INFO *gpio_info, unsigned idx ) ;
 _NO_MBG_API_ATTR int _MBG_API chk_dev_gpio_dep_on_ass_idx( const ALL_GPIO_INFO *gpio_info, unsigned idx ) ;
 /**
 * @brief Checks whether GPIO supports status function
 *
 * @param[out]  info  Pointer to a ::ALL_GPIO_INFO structure to be filled up
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_gpio
 * @see ::mbg_chk_dev_supp_gpio
 * @see ::free_all_gpio_info
 */
 _NO_MBG_API_ATTR int _MBG_API chk_dev_gpio_has_status( const ALL_GPIO_INFO *info ) ;

 /**
 * @brief Frees ::ALL_XBP_INFO structure
 *
 * @param[in]     p       Pointer to the ::ALL_XBP_INFO structure, which will be freed
 *
 * @see ::mbgextio_get_all_xbp_info
 */
 void free_all_xbp_info ( ALL_XBP_INFO *p ) ;

 /**
 * @brief Frees ::ALL_NET_CFG_INFO structure
 *
 * @param[in]     p       Pointer to the ::ALL_NET_CFG_INFO structure, which will be freed
 *
 * @see ::mbgextio_get_all_net_cfg_info
 */
 void free_all_net_cfg_info ( ALL_NET_CFG_INFO *p ) ;

 /**
 * @brief Frees ::ALL_NET_STATUS_INFO structure
 *
 * @param[in]     p       Pointer to the ::ALL_NET_STATUS_INFO structure, which will be freed
 *
 * @see ::mbgextio_get_all_net_status_info
 */
 void free_all_net_status_info ( ALL_NET_STATUS_INFO *p ) ;

 /**
 * @brief Frees ::ALL_SNMP_INFO structure
 *
 * @param[in]     p       Pointer to the ::ALL_SNMP_INFO structure, which will be freed
 *
 */
 void free_all_snmp_info ( ALL_SNMP_INFO *p ) ;

 /**
 * @brief Frees ::ALL_MONITORING_INFO structure
 *
 * @param[in]     p       Pointer to the ::ALL_MONITORING_INFO structure, which will be freed
 *
 */
 void free_all_monitoring_info ( ALL_MONITORING_INFO *p ) ;

 /**
 * @brief Frees ::ALL_MONITORING_STATUS structure
 *
 * @param[in]     p       Pointer to the ::ALL_MONITORING_STATUS structure, which will be freed
 *
 */
 void free_all_monitoring_status ( ALL_MONITORING_STATUS *p ) ;

 /**
 * @brief Frees ::ALL_XMULTI_REF_INFO structure
 *
 * @param[in]     p       Pointer to the ::ALL_XMULTI_REF_INFO structure, which will be freed
 *
 * @see ::mbgextio_get_all_xmulti_ref_info
 * @see ::mbg_get_all_xmulti_ref_info
 */
 void free_all_xmulti_ref_info( ALL_XMULTI_REF_INFO *p ) ;

 /**
 * @brief Frees ::ALL_XMULTI_REF_STATUS structure
 *
 * @param[in]     p       Pointer to the ::ALL_XMULTI_REF_STATUS structure, which will be freed
 *
 * @see ::mbgextio_get_all_xmulti_ref_status
 * @see ::mbg_get_all_xmulti_ref_status
 */
 void free_all_xmulti_ref_status( ALL_XMULTI_REF_STATUS *p ) ;

 /**
 * @brief Frees ::ALL_PTP_V1_COMMON_DATASETS structure allocated by ::mbgextio_get_all_ptp_v1_common_datasets
 *
 * @param[in]     p       Pointer to the ::ALL_PTP_V1_COMMON_DATASETS structure, which will be freed
 *
 * @see ::mbgextio_get_all_ptp_v1_common_datasets
 */
 void free_all_ptp_v1_common_datasets( ALL_PTP_V1_COMMON_DATASETS *p ) ;

 /**
 * @brief Frees ::ALL_PTP_V2_COMMON_DATASETS structure allocated by ::mbgextio_get_all_ptp_v2_common_datasets
 *
 * @param[in]     p       Pointer to the ::ALL_PTP_V2_COMMON_DATASETS structure, which will be freed
 *
 * @see ::mbgextio_get_all_ptp_v2_common_datasets
 */
 void free_all_ptp_v2_common_datasets( ALL_PTP_V2_COMMON_DATASETS *p ) ;

 /**
 * @brief Frees ::ALL_NTP_CFG_INFO structure
 *
 * @param[in]     p       Pointer to the ::ALL_NTP_CFG_INFO structure, which will be freed
 *
 * @see ::mbgextio_get_all_ntp_cfg_info
 */
 void free_all_ntp_cfg_info( ALL_NTP_CFG_INFO *p ) ;

 /**
 * @brief Frees ::ALL_NTP_STATUS structure
 *
 * @param[in]     p       Pointer to the ::ALL_NTP_STATUS structure, which will be freed
 *
 * @see ::mbgextio_get_all_ntp_status
 */
 void free_all_ntp_status( ALL_NTP_STATUS *p ) ;

 /**
 * @brief Frees memory allocated by ::mbgextio_get_all_ims_info
 *
 * @param[in]     p       Pointer to the ::ALL_IMS_INFO structure, which will be freed
 *
 * @see ::mbgextio_dev_has_ims
 * @see ::mbgextio_dev_ims_has_fdm
 * @see ::mbgextio_get_all_ims_info
 * @see ::mbgextio_get_all_ims_state
 */
 void free_all_ims_info( ALL_IMS_INFO *p ) ;

 /**
 * @brief Frees memory allocated by ::mbgextio_get_all_ims_state
 *
 * @param[in]     p       Pointer to the ::ALL_IMS_STATE structure, which will be freed
 *
 * @see ::mbgextio_dev_has_ims
 * @see ::mbgextio_dev_ims_has_fdm
 * @see ::mbgextio_get_all_ims_info
 * @see ::mbgextio_get_all_ims_state
 */
 void free_all_ims_state( ALL_IMS_STATE *p ) ;

 /**
 * @brief Frees memory allocated by ::mbgextio_get_all_gpio_info
 *
 * @param[in]     p       Pointer to the ::ALL_GPIO_INFO structure, which will be freed
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_gpio
 * @see ::mbgextio_get_all_gpio_info
 */
 void free_all_gpio_info( ALL_GPIO_INFO *p ) ;

 /**
 * @brief Frees memory allocated by ::mbgextio_get_all_io_port_info
 *
 * @param[in]     p       Pointer to the ::ALL_IO_PORT_INFO structure, which will be freed
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_io_ports
 * @see ::mbgextio_get_all_io_port_info
 * @see ::mbgextio_get_all_io_port_status
 * @see ::free_all_io_port_status
 */
 void free_all_io_port_info( ALL_IO_PORT_INFO *p ) ;

 /**
 * @brief Frees memory allocated by ::mbgextio_get_all_io_port_status
 *
 * @param[in]     p       Pointer to the ::ALL_IO_PORT_STATUS structure, which will be freed
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_io_ports
 * @see ::mbgextio_get_all_io_port_info
 * @see ::mbgextio_get_all_io_port_status
 * @see ::free_all_io_port_info
 */
 void free_all_io_port_status( ALL_IO_PORT_STATUS *p ) ;

 /**
 * @brief Frees memory allocated by ::mbgextio_get_all_gpio_state
 *
 * @param[in]     p       Pointer to the ::ALL_GPIO_STATE structure, which will be freed
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_gpio
 * @see ::mbgextio_get_all_gpio_state
 */
 void free_all_gpio_state( ALL_GPIO_STATE *p ) ;

 /**
 * Allocates memory for a new ::UCAP_ENTRY structure
 *
 * @return The new allocated ::UCAP_ENTRY or null if the calloc call was not successful
 */
 UCAP_ENTRY* calloc_ucap_entry(void) ;

 /**
 * @brief Frees memory allocated by ::mbgextio_get_all_ucap_info
 *
 * @param[in]     p       Pointer to the ::ALL_UCAP_INFO structure, which will be freed
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_ucap
 * @see ::mbg_chk_dev_has_ucap
 * @see ::mbgextio_get_all_ucap_info
 * @see ::mbg_get_all_ucap_info
 */
 void free_all_ucap_info( ALL_UCAP_INFO *p ) ;

 /**
 * @brief Frees ::ALL_UCAP_NET_INFO structure
 *
 * @param[in]     p       Pointer to the ::ALL_UCAP_NET_INFO structure, which will be freed
 *
 * @see ::mbgextio_get_all_ucap_net_info
 */
 void free_all_ucap_net_info( ALL_UCAP_NET_INFO *p ) ;


/* ----- function prototypes end ----- */

#endif  // !defined( MBG_TGT_KERNEL )

#ifdef __cplusplus
}
#endif


/* End of header body */

#undef _ext
#undef _DO_INIT

#endif  /* _CFG_HLP_H */
