
/**************************************************************************
 *
 *  $Id: cfg_hlp.c 1.1.1.85 2017/04/11 14:47:08 martin TEST $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    Meinberg device configuration helper functions.
 *
 * -----------------------------------------------------------------------
 *  $Log: cfg_hlp.c $
 *  Revision 1.1.1.85  2017/04/11 14:47:08  martin
 *  Revision 1.1.1.84  2017/04/06 09:16:16Z  thomas-b
 *  mbg_snprint_revision is not preliminary any longer
 *  Revision 1.1.1.83  2017/03/27 10:37:18  thomas-b
 *  Added ALL_PTP_V1_COMMON_DATASETS and the appropriate free function
 *  Revision 1.1.1.82  2017/03/22 09:07:52  thomas-b
 *  Added function get_io_port_type_info_idx
 *  Revision 1.1.1.81  2017/03/02 15:29:27  thomas-b
 *  Adapted free_all_ntp_cfg_info for new ALL_NTP_CFG_INFO structure
 *  Revision 1.1.1.80  2017/02/28 15:24:27  gregoire
 *  peer strucutre in ALL_NTP_CFG_INFO renamed
 *  Revision 1.1.1.79  2017/02/24 09:38:25  philipp
 *  Do not use fixed buffers for monitorung event info and status
 *  Revision 1.1.1.78  2017/02/22 14:28:57  martin
 *  Fixed some warnings from clang compiler.
 *  Revision 1.1.1.77  2017/02/22 11:40:17  martin
 *  Account for preliminary code.
 *  Revision 1.1.1.76  2017/02/21 15:51:40  thomas-b
 *  Added ALL_MONITORING_STATUS and free function and extended ALL_MONITORING_INFO by events
 *  Revision 1.1.1.75  2017/02/16 12:58:01  thomas-b
 *  Added ALL_PTP_V2_COMMON_DATASETS structure and the appropriate free function
 *  Revision 1.1.1.74  2017/02/08 13:16:07  thomas-b
 *  Added ALL_MONITORING_INFO and the appropriate free function
 *  Revision 1.1.1.73  2017/02/07 09:46:51  thomas-b
 *  Added ALL_SNMP_INFO and the appropriate free function
 *  Revision 1.1.1.72  2016/12/19 12:13:55  philipp
 *  Added GPIO associated index helper
 *  Revision 1.1.1.71  2016/11/22 12:40:46  philipp
 *  Added I/O port helper functions
 *  Revision 1.1.1.70  2016/11/08 17:21:19  martin
 *  Doxygen fixes.
 *  Revision 1.1.1.69  2016/11/01 14:51:54  udo
 *  *** empty log message ***
 *  Revision 1.1.1.68  2016/11/01 09:24:01  martin
 *  *** empty log message ***
 *  Revision 1.1.1.67  2016/10/25 07:48:27  martin
 *  Doxygen fixes.
 *  Revision 1.1.1.66  2016/10/21 09:39:25  thomas-b
 *  Added struct ALL_XBP_INFO and appropriate free function
 *  Revision 1.1.1.65  2016/10/14 11:10:02  thomas-b
 *  Added ALL_UCAP_NET_INFO and the appropriate free function
 *  Revision 1.1.1.64  2016/09/28 15:04:02  thomas-b
 *  Added function to check whether NET_CFG_API stage 2 is supported
 *  Revision 1.1.1.63  2016/07/26 08:07:10  philipp
 *  Fixed compiler errors (name clash, syntax error)
 *  Revision 1.1.1.62  2016/07/15 14:09:48  martin
 *  Use functions from new module timeutil.
 *  Revision 1.1.1.61  2016/06/21 13:51:23  philipp
 *  Fixed nasty segfault due to pointer corruption
 *  Revision 1.1.1.60  2016/06/21 12:06:59  thomas-b
 *  Added function calloc_ucap_entry
 *  Revision 1.1.1.59  2016/06/21 10:25:30  philipp
 *  Freeing ucap entry list is now portable
 *  Revision 1.1.1.58  2016/06/21 07:39:21  philipp
 *  Use mbg_klist for ucap events and not an array -> Easier to handle MAX_UCAP_ENTRIES
 *  Revision 1.1.1.57  2016/06/16 07:46:45  philipp
 *  Fixed checking flags for IMS sensor specific features
 *  Revision 1.1.1.56  2016/06/15 14:04:27  thomas-b
 *  Added ALL_UCAP_INFO structure and function free_all_ucap_info
 *  Revision 1.1.1.55  2016/06/03 09:00:13  thomas-b
 *  tm structure does not have gmtoff field under Windows
 *  Revision 1.1.1.54  2016/06/02 10:24:22  philipp
 *  Renaming all revision macros and helper functions
 *  Revision 1.1.1.53  2016/05/27 07:27:43  philipp
 *  Fixed (potential) memleak
 *  Revision 1.1.1.52  2016/05/27 05:48:08  philipp
 *  Refactoring to support XMR_METRICS properly
 *  Revision 1.1.1.51  2016/05/26 11:01:09  thomas-b
 *  Removed info structures from ALL_NTP_STATUS
 *  Moved check functions of specific features to cfg_hlp
 *  Revision 1.1.1.50  2016/05/25 08:43:35  philipp
 *  Redesign of ALL_[xxx]_[XXX] structures and (helper) functions
 *  Revision 1.1.1.49  2016/05/23 09:37:40  philipp
 *  Extended ALL_XMULTI_REF_STATUS by holdover_status
 *  Revision 1.1.1.48  2016/05/23 08:59:03  philipp
 *  New function free_all_gpio_state
 *  Revision 1.1.1.47  2016/05/23 08:24:37  philipp
 *  New function free_all_ims_state
 *  Revision 1.1.1.46  2016/05/11 13:20:55  thomas-b
 *  Added ALL_NET_STATUS_INFO and the appropriate free function
 *  Revision 1.1.1.45  2016/04/26 14:21:58  thomas-b
 *  Renamed ALL_NET_CFG_INFO structure members
 *  Revision 1.1.1.44  2016/04/26 13:45:36  martin
 *  Tried portable printing of int64_t types.
 *  Revision 1.1.1.43  2016/04/26 08:40:17Z  philipp
 *  Fixed compiler warning due to invalid format specifier
 *  Revision 1.1.1.42  2016/04/26 08:23:45  thomas-b
 *  Extended ALL_NET_CFG_INFO by DNS configurations
 *  Revision 1.1.1.41  2016/04/26 06:29:27  thomas-b
 *  Added ALL_NET_CFG_INFO for network configuration
 *  Revision 1.1.1.40  2016/04/25 14:55:32  martin
 *  *** empty log message ***
 *  Revision 1.1.1.39  2016/04/25 14:42:18  martin
 *  *** empty log message ***
 *  Revision 1.1.1.38  2016/04/25 11:22:41  martin
 *  Revision 1.1.1.37  2016/04/25 09:01:29Z  martin
 *  *** empty log message ***
 *  Revision 1.1.1.36  2016/04/22 07:11:50  philipp
 *  Use pointer to pointer in get_all_* functions
 *  Revision 1.1.1.35  2016/04/20 14:45:46  thomas-b
 *  Renamed ALL_NTP_STATE_INFO to ALL_NTP_STATUS
 *  Revision 1.1.1.34  2016/04/20 13:49:12  thomas-b
 *  Added structure definitions and free functions for NTP
 *  Revision 1.1.1.33  2016/04/20 12:37:53  thomas-b
 *  Moved free functions for ALL_XMULTI_REF_INFO and ALL_XMULTI_REF_STATUS to cfg_hlp
 *  Revision 1.1.1.32  2016/04/08 07:54:17  philipp
 *  Added function mbg_print_ext_rev_info
 *  Revision 1.1.1.31  2016/04/07 08:07:38  philipp
 *  Added function normalize_nano_time_64 (Clemens)
 *  Revision 1.1.1.30  2016/04/04 15:08:45  martin
 *  Replaced chk_model_is_vsg() by xdevfeat::xdevfeat_is_vsg().
 *  Revision 1.1.1.29  2016/03/03 11:16:50  martin
 *  Utility functions to alloc and free memory for hardware ID.
 *  Revision 1.1.1.28  2016/02/17 12:00:06Z  gregoire
 *  new function chk_model_is_vsg
 *  Revision 1.1.1.27  2015/11/24 13:12:04Z  philipp
 *  Extended / modified TLV functions
 *  Revision 1.1.1.26  2015/11/23 14:15:40  philipp
 *  Moved TLV related initializing functions to here
 *  Revision 1.1.1.25  2015/10/30 15:33:48  martin
 *  Revision 1.1.1.24  2015/10/27 16:17:32Z  martin
 *  *** empty log message ***
 *  Revision 1.1.1.23  2015/10/26 16:31:52  martin
 *  *** empty log message ***
 *  Revision 1.1.1.22  2015/10/26 13:36:44  martin
 *  Revision 1.1.1.21  2015/10/15 12:49:09Z  marvin
 *  Revision 1.1.1.20  2015/10/12 10:01:59Z  martin
 *  *** empty log message ***
 *  Revision 1.1.1.19  2015/10/09 11:09:13  martin
 *  *** empty log message ***
 *  Revision 1.1.1.18  2015/10/08 10:32:16  martin
 *  *** empty log message ***
 *  Revision 1.1.1.17  2015/10/08 09:30:34  martin
 *  *** empty log message ***
 *  Revision 1.1.1.16  2015/10/07 16:03:19  martin
 *  *** empty log message ***
 *  Revision 1.1.1.15  2015/10/07 09:59:00  martin
 *  More common GNSS support.
 *  Revision 1.1.1.14  2015/10/05 13:29:45  martin
 *  Revision 1.1.1.13  2015/10/01 10:58:35Z  thomas-b
 *  fixed call of mbgmktm with correct month and day values in nano_time_64_to_tm_gps
 *  Revision 1.1.1.12  2015/09/15 09:11:13  martin
 *  Moved some functions into a _PRELIMINARY_CODE section.
 *  Revision 1.1.1.11  2015/09/14 08:56:05  thomas-b
 *  Sustract 1900 from year when calling mbg_mktime
 *  Revision 1.1.1.10  2015/09/14 07:34:08  thomas-b
 *  Added missing includes of time.h and mbgmktm.h
 *  Revision 1.1.1.9  2015/09/11 12:09:04  thomas-b
 *  Added nano_time_64_to_tm_gps and tm_gps_to_nano_time_64 functions
 *  Revision 1.1.1.8  2015/08/31 14:55:11  martin
 *  Moved string trim functions to str_util module.
 *  Revision 1.1.1.7  2015/08/31 13:41:56  martin
 *  Revision 1.1.1.6  2014/10/29 16:25:31  martin
 *  Moved some functions and structures to more convenient files.
 *  Revision 1.1.1.5  2014/10/14 10:20:18  martin
 *  Revision 1.1.1.4  2014/10/14 10:05:10  martin
 *  Revision 1.1.1.3  2014/09/26 11:43:24Z  martin
 *  Revision 1.1.1.2  2014/07/22 13:05:34  martin
 *  Revision 1.1.1.1  2014/04/28 12:29:45Z  martin
 *  Revision 1.1  2014/04/25 09:14:49  martin
 *  Initial revision.
 *
 **************************************************************************/

#define _CFG_HLP
 #include <cfg_hlp.h>
#undef _CFG_HLP

#include <mbgerror.h>
#include <timeutil.h>
#include <str_util.h>
#include <myutil.h>
#include <mbgtime.h>

#if defined( _PRELIMINARY_CODE )
  #include <mbgmktm.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>


#if !defined( MBG_TGT_MISSING_64_BIT_TYPES )

/*HDR*/
/**
 * @brief Normalize a ::NANO_TIME_64 struct
 *
 * After normalization, the following can be assumed:<br>
 * - nano_secs is in the range [-10^9 + 1, 10^9 - 1]<br>
 * - if secs is not 0, secs and nano_secs have the same sign
 *
 * @param[in,out] nt  The NANO_TIME_64 to be normalized
 */
void normalize_nano_time_64( NANO_TIME_64 *nt )
{
  int64_t additional_secs;

  // Step 1: Make sure nano seconds are in the interval [-10^9 + 1, 10^9 - 1]
  additional_secs = nt->nano_secs / NSECS_PER_SEC;
  nt->nano_secs -= additional_secs * NSECS_PER_SEC;
  nt->secs += additional_secs;

  // Step 2: Make sure seconds and nanoseconds have same sign if seconds is not 0
  if ( nt->secs > 0 && nt->nano_secs < 0 )
  {
    nt->secs -= 1;
    nt->nano_secs += NSECS_PER_SEC;
  }
  else if ( nt->secs < 0 && nt->nano_secs > 0 )
  {
    nt->secs += 1;
    nt->nano_secs -= NSECS_PER_SEC;
  }

}  // normalize_nano_time_64

#endif  // !defined( MBG_TGT_MISSING_64_BIT_TYPES )



#if !defined( MBG_TGT_MISSING_64_BIT_TYPES )

/*HDR*/
/**
 * @brief Print a normalized ::NANO_TIME_64 into a string buffer
 *
 * @param[out] s        The string buffer to be filled
 * @param[in]  max_len  Size of the output buffer for 0-terminated string
 * @param[in]  nt       The ::NANO_TIME_64 to be printed
 */
size_t snprint_nano_time_64( char *s, size_t max_len, const NANO_TIME_64 *nt )
{
  size_t n = snprintf_safe( s, max_len, "%c%" PRId64 ".%09" PRId64,
                            _nano_time_negative( nt ) ? '-' : '+',
                            _abs64( nt->secs ),
                            _abs64( nt->nano_secs ) );
  return n;

}  // snprint_nano_time_64

#endif  // !defined( MBG_TGT_MISSING_64_BIT_TYPES )



/*HDR*/
/**
 * @brief Print nano time into string buffer
 *
 * @param[out] s        The string buffer to be filled
 * @param[in]  max_len  Size of the output buffer for 0-terminated string
 * @param[in]  nt       The ::NANO_TIME to be printed
 */
size_t snprint_nano_time( char *s, size_t max_len, const NANO_TIME *nt )
{
  size_t n = snprintf_safe( s, max_len, "%c%li.%09li",
                            _nano_time_negative( nt ) ? '-' : '+',
                            labs( (long) nt->secs ),
                            labs( (long) nt->nano_secs ) );
  return n;

}  // snprint_nano_time



#if defined( _PRELIMINARY_CODE )

//### TODO: eventually move these functions to a different module

#if !defined( MBG_TGT_MISSING_64_BIT_TYPES )

/*HDR*/
/**
 * @brief Convert ::NANO_TIME_64 to ::TM_GPS
 *
 * @param[out] tm_gps  The ::TM_GPS to be filled
 * @param[in]  nt      The ::NANO_TIME_64 to be converted
 *
 * @return 1 on success, 0 on error
 *
 * @see ::tm_gps_to_nano_time_64
 */
int nano_time_64_to_tm_gps( TM_GPS *tm_gps, const NANO_TIME_64 *nt )
{
  struct tm tm = { 0 };
  time_t t = cvt_to_time_t( nt->secs );
  int rc = mbg_gmtime( &tm, &t );

  if ( mbg_rc_is_success( rc ) )
  {
    tm_gps->year = tm.tm_year + 1900;
    tm_gps->month = tm.tm_mon + 1;
    tm_gps->mday = tm.tm_mday;
    tm_gps->yday = tm.tm_yday;
    tm_gps->wday = tm.tm_wday;
    tm_gps->hour = tm.tm_hour;
    tm_gps->min = tm.tm_min;
    tm_gps->sec = tm.tm_sec;
    tm_gps->frac = 0;         //### TODO convert fractions
#if defined( MBG_TGT_POSIX )
    tm_gps->offs_from_utc = tm.tm_gmtoff;  //### TODO Is tm.tm_gmtoff even valid?
#endif
    return 1;  //### TODO return MBG_SUCCESS
  }

  return 0;  //### TODO return rc

}  // nano_time_64_to_tm_gps



/*HDR*/
/**
 * @brief Convert ::TM_GPS to ::NANO_TIME_64
 *
 * @param[out] nt      The ::NANO_TIME_64 to be filled
 * @param[in]  tm      The ::TM_GPS to be converted
 *
 * @return 1 on success, 0 on error
 *
 * @see ::nano_time_64_to_tm_gps
 */
int tm_gps_to_nano_time_64( NANO_TIME_64 *nt, const TM_GPS *tm )
{
  time_t t = mbg_mktime( tm->year - 1900, tm->month - 1, tm->mday - 1, tm->hour, tm->min, tm->sec );

  if ( t != (time_t) -1 )
  {
    nt->secs = (uint64_t) t;
    nt->nano_secs = 0;    //### TODO: convert fractions
    return 1;
  }

  return 0;

}  // tm_gps_to_nano_time_64

#endif  // !defined( MBG_TGT_MISSING_64_BIT_TYPES )

#endif  // defined( _PRELIMINARY_CODE )



/*HDR*/
/**
 * @brief Check if a software revision name should be displayed
 *
 * The software revision name is usually empty, except if the
 * firmware is a customized version, in which case the field
 * contains an identifier string.
 *
 * There are some standard firmware versions where this string
 * is not empty but padded with spaces, etc., so we try to
 * clean this up and display the string properly, if appropriate.
 *
 * @param[in,out] p        The ::SW_REV name to check
 * @param[in]     verbose  The app's verbosity level
 *
 * @return != 0 if SW name should be displayed
 */
int chk_sw_rev_name( SW_REV *p, int verbose )
{
  if ( verbose > 1 )  // more than just verbose
    return 1;         // just return raw string

  trim_whitespace( p->name );

  // Some firmware versions have "CC_STANDARD" in their standard version,
  // which doesn't provide any valuable information.
  // We discard this by default.
  if ( strstr( p->name, "CC_STANDARD" ) )
    p->name[0] = 0;

  if ( verbose )
    return 1;  // calling app should display string, even if empty

  // calling app should display string only if not empty
  return strlen( p->name ) != 0;

}  // chk_sw_rev_name



/*HDR*/
int get_str_idx( const char *search,
                 const char *str_table[],
                 int n_entries )
{
  int i;

  for ( i = 0; i < n_entries; i++ )
    if ( strcmp( search, str_table[i] ) == 0 )
      return i;

  return -1;

}  // get_str_idx



/*HDR*/
int get_baud_rate_idx( BAUD_RATE baud_rate )
{
  int i;

  for ( i = 0; i < N_MBG_BAUD_RATES; i++ )
    if ( baud_rate == mbg_baud_rate[i] )
      return i;

  return -1;

}  // get_baud_rate_idx



/*HDR*/
int get_framing_idx( const char *framing )
{
  return get_str_idx( framing, mbg_framing_str, N_MBG_FRAMINGS );

}  // get_framing_idx



/*HDR*/
void port_settings_from_port_parm_mode(
       PORT_SETTINGS *p_ps,
       uint8_t pp_mode,
       int str_type_cap
     )
{
  if ( pp_mode >= STR_UCAP )
  {
    p_ps->str_type = str_type_cap;
    p_ps->mode = ( pp_mode == STR_UCAP ) ? STR_AUTO : STR_ON_REQ;
  }
  else
  {
    p_ps->str_type = 0;
    p_ps->mode = pp_mode;
  }

}  // port_settings_from_port_parm_mode



/*HDR*/
void port_parm_mode_from_port_settings(
       uint8_t *pp_mode,
       const PORT_SETTINGS *p_ps,
       int str_type_cap
     )
{
  if ( p_ps->str_type == str_type_cap )
    *pp_mode = ( p_ps->mode == STR_ON_REQ ) ? STR_UCAP_REQ : STR_UCAP;
  else
    *pp_mode = p_ps->mode;

}  // port_parm_mode_from_port_settings



/*HDR*/
void port_settings_from_port_parm(
  PORT_SETTINGS *p_ps,
  int port_num,
  const PORT_PARM *p_pp,
  int cap_str_idx
)
{
  p_ps->parm = p_pp->com[port_num];

  port_settings_from_port_parm_mode( p_ps, p_pp->mode[port_num],
                                     cap_str_idx );

}  // port_info_from_port_parm



/*HDR*/
void port_parm_from_port_settings(
  PORT_PARM *p_pp,
  int port_num,
  const PORT_SETTINGS *p_ps,
  int cap_str_idx
)
{
  p_pp->com[port_num] = p_ps->parm;

  port_parm_mode_from_port_settings( &p_pp->mode[port_num],
                                     p_ps, cap_str_idx );

}  // port_parm_from_port_settings



/*HDR*/
uint32_t check_valid_port_info( const PORT_INFO *p_pi,
                                const STR_TYPE_INFO_IDX str_type_info_idx[],
                                int n_str_type )
{
  const PORT_SETTINGS *p_ps = &p_pi->port_settings;
  int idx;
  uint32_t flags = 0;


  if ( p_pi->supp_baud_rates & ~_mask( N_MBG_BAUD_RATES ) )
    flags |= MBG_PS_MSK_BAUD_RATE_OVR_SW;  // dev. supports more baud rates than driver

  idx = get_baud_rate_idx( p_ps->parm.baud_rate );

  if ( !_inrange( idx, 0, N_MBG_BAUD_RATES ) ||
       !_is_supported( idx, p_pi->supp_baud_rates ) )
    flags |= MBG_PS_MSK_BAUD_RATE;


  if ( p_pi->supp_framings & ~_mask( N_MBG_FRAMINGS ) )
    flags |= MBG_PS_MSK_FRAMING_OVR_SW;    // dev. supports more framings than driver

  idx = get_framing_idx( p_ps->parm.framing );

  if ( !_inrange( idx, 0, N_MBG_FRAMINGS ) ||
       !_is_supported( idx, p_pi->supp_framings ) )
    flags |= MBG_PS_MSK_FRAMING;


  if ( p_ps->parm.handshake >= N_COM_HS )
    flags |= MBG_PS_MSK_HS_OVR_SW;         // handshake index exceeds max.

  if ( p_ps->parm.handshake != HS_NONE )   // currently no device supports any handshake
    flags |= MBG_PS_MSK_HS;                // handshake mode not supp. by dev.


  if ( p_pi->supp_str_types & ~_mask( n_str_type ) )
    flags |= MBG_PS_MSK_STR_TYPE_OVR_SW;   // firmware error: more string types supported than reported

  idx = p_ps->str_type;

  if ( idx >= n_str_type )
    flags |= MBG_PS_MSK_STR_TYPE_OVR_DEV;  // string type index exceeds max.
  else
  {
    if ( !_is_supported( idx, p_pi->supp_str_types ) )
      flags |= MBG_PS_MSK_STR_TYPE;        // string type not supported by this port
    else
    {
      // Use the str_type index to get the supported output mode mask
      // from the string type info table. This is required to check
      // whether the selected mode is supported by the selected
      // string type.
      ulong supp_modes = str_type_info_idx[idx].str_type_info.supp_modes;

      if ( supp_modes & ~_mask( N_STR_MODE ) )
        flags |= MBG_PS_MSK_STR_MODE_OVR_SW;  // dev. supports more string modes than driver

      idx = p_ps->mode;

      if ( idx >= N_STR_MODE )                // mode is always >= 0
        flags |= MBG_PS_MSK_STR_MODE_OVR_SW;  // string mode index exceeds max.
      else
        if ( !_is_supported( idx, supp_modes ) )
          flags |= MBG_PS_MSK_STR_MODE;       // string mode not supp. by this string type and port
    }
  }


  if ( p_ps->flags != 0 )            /* currently always 0 */
    flags |= MBG_PS_MSK_FLAGS_OVR_SW | MBG_PS_MSK_FLAGS;


  return flags;

}  // check_valid_port_info



/*HDR*/
int valid_port_info( const PORT_INFO *p_pi,
                     const STR_TYPE_INFO_IDX str_type_info_idx[],
                     int n_str_type )
{
  return check_valid_port_info( p_pi, str_type_info_idx, n_str_type ) == 0;

}  // valid_port_info



/*HDR*/
int setup_port_info_from_port_settings( PORT_INFO_IDX pii[], const PORT_PARM *p_pp,
                                        const RECEIVER_INFO *p_ri )
{
  int i;

  for ( i = 0; i < p_ri->n_com_ports; i++ )
  {
    PORT_INFO_IDX *p_pii = &pii[i];
    PORT_INFO *p_pi = &p_pii->port_info;

    p_pii->idx = i;
    port_settings_from_port_parm( &p_pi->port_settings, i, p_pp, 1 );

    p_pi->supp_baud_rates = DEFAULT_GPS_BAUD_RATES_C166;
    p_pi->supp_framings = DEFAULT_GPS_FRAMINGS_C166;
    p_pi->supp_str_types = DEFAULT_SUPP_STR_TYPES_GPS;
  }

  return MBG_SUCCESS;

}  // setup_port_info_from_port_settings



/*HDR*/
int setup_default_str_type_info_idx( STR_TYPE_INFO_IDX stii[], const RECEIVER_INFO *p_ri )
{
  int i;

  for ( i = 0; i < p_ri->n_str_type; i++ )
  {
    STR_TYPE_INFO_IDX *stip = &stii[i];
    stip->idx = i;
    stip->str_type_info = default_str_type_info[i];
  }

  return MBG_SUCCESS;

}  // setup_default_str_type_info_idx



/*HDR*/
int chk_set_n_gnss_supp( ALL_GNSS_INFO *p_agi )
{
  p_agi->n_gnss_supp = num_bits_set( p_agi->gnss_mode_info.supp_gnss_types );

  if ( p_agi->n_gnss_supp > N_GNSS_TYPES )
    return MBG_ERR_N_GNSS_EXCEEDS_SUPP;

  return MBG_SUCCESS;

}  // chk_set_n_gnss_supp



/*HDR*/
/**
 * @brief
 *
 * ### Setup GNSS info from stat_info so we can use the same printing routine
 */
void setup_gps_only_sat_info_idx_from_statinfo( ALL_GNSS_INFO *p_agi )
{
  STAT_INFO *p_si = &p_agi->stat_info;
  GNSS_SAT_INFO_IDX *p_gsii = &p_agi->gnss_sat_info_idx[GNSS_TYPE_GPS];
  GNSS_SAT_INFO *p_gsi = &p_gsii->gnss_sat_info;

  memset( p_gsii, 0, sizeof( *p_gsii ) );
  p_gsii->idx = GNSS_TYPE_GPS;

  p_gsi->gnss_type = GNSS_TYPE_GPS;
  p_gsi->svs_in_view = p_si->svs_in_view;
  p_gsi->good_svs = p_si->good_svs;

}  // setup_gps_only_sat_info_idx_from_statinfo



/*HDR*/
/**
 * @brief
 *
 * Setup GNSS info from stat_info so we can use the same printing routine
 */
int setup_gps_only_gnss_info_from_statinfo( ALL_GNSS_INFO *p_agi )
{
  MBG_GNSS_MODE_INFO *p_gmi = &p_agi->gnss_mode_info;

  memset( p_gmi, 0, sizeof( *p_gmi ) );

  p_gmi->supp_gnss_types = MBG_GNSS_TYPE_MSK_GPS;
  p_gmi->settings.gnss_set = p_gmi->supp_gnss_types;

  memset( p_agi->gnss_sat_info_idx, 0, sizeof( p_agi->gnss_sat_info_idx ) );

  setup_gps_only_sat_info_idx_from_statinfo( p_agi );

  return chk_set_n_gnss_supp( p_agi );

}  // setup_gps_only_gnss_info_from_statinfo



/*HDR*/
void chk_free_dev_hw_id( DEVICE_INFO *p )
{
  if ( p->hw_id )
  {
    free( p->hw_id );
    p->hw_id = NULL;
  }

}  // chk_free_dev_hw_id



/*HDR*/
int alloc_dev_hw_id( DEVICE_INFO *p, size_t len )
{
  if ( p->hw_id )
    return MBG_ERR_ALREADY_ALLOC;


  p->hw_id = (char *) malloc( len );

  if ( p->hw_id == NULL )
    return MBG_ERR_NO_MEM;

  return MBG_SUCCESS;

}  // alloc_dev_hw_id



/*HDR*/
const char *get_fw_id_from_hw_id( const char *hw_id )
{
  int i;

  for ( i = 0; i < N_SUPP_DEV_TOTAL; i++ )
  {
    DEVICE_INFO *p = &device_list[i];

    if ( strlen( p->fw_id ) && p->hw_id )  //### TODO check if this still works as expected
    {
      if ( strcmp( hw_id, p->hw_id ) == 0 )
      {
        if ( strlen( p->fw_id ) > 0 )
          return p->fw_id;

        #if defined( DEBUG )
          fprintf( stderr, "Length of fw_id is 0 in %s for device %i (%s)\n",
                   __func__, i, p->hw_id );
        #endif
      }
    }
  }

  return NULL;

}  // get_fw_id_from_hw_id



/*HDR*/
const char *get_hw_id_from_fw_id( const char *fw_id )
{
  int i;

  for ( i = 0; i < N_SUPP_DEV_TOTAL; i++ )
  {
    DEVICE_INFO *p = &device_list[i];

    if ( strlen( p->fw_id ) && p->hw_id )  //### TODO check if this still works as expected
    {
      if ( strcmp( fw_id, p->fw_id ) == 0 )
      {
        if ( strlen( p->hw_id ) > 0 )
          return p->hw_id;

        #if defined( DEBUG )
          fprintf( stderr, "Length of hw_id is 0 in %s for device %i (%s)\n",
                   __func__, i, p->fw_id );
        #endif
      }
    }
  }

  return NULL;

}  // get_hw_id_from_fw_id



/*HDR*/
/**
 * @brief Returns the currently used ::MBG_IO_PORT_TYPE_INFO_IDX for the appropriate ::MBG_IO_PORT_INFO_IDX
 *
 * @param[in]   all_io_port_info    Pointer to the ::ALL_IO_PORT_INFO, containing the current configuration
 * @param[in]   io_port_info_idx    Pointer to the ::MBG_IO_PORT_INFO_IDX, for which the ::MBG_IO_PORT_TYPE_INFO_IDX shall be found
 *
 * @return Pointer to the found ::MBG_IO_PORT_TYPE_INFO_IDX or NULL
 *
 */
MBG_IO_PORT_TYPE_INFO_IDX *get_io_port_type_info_idx(ALL_IO_PORT_INFO *all_io_port_info, MBG_IO_PORT_INFO_IDX *io_port_info_idx)
{
  uint8_t i;

  if(!all_io_port_info || !io_port_info_idx)
    return NULL;

  for(i = 0; i < io_port_info_idx->info.num_types; ++i)
  {
    if(all_io_port_info->pt_infos[io_port_info_idx->idx][i].info.port_type == io_port_info_idx->info.settings.type)
    {
      if(all_io_port_info->pt_infos[io_port_info_idx->idx][i].info.port_type == MBG_IO_PORT_TYPE_GPIO)
      {
        if(all_io_port_info->pt_infos[io_port_info_idx->idx][i].info.data.gpio_limits.type == io_port_info_idx->info.settings.data.gpio_settings.type)
          return &all_io_port_info->pt_infos[io_port_info_idx->idx][i];
      }
      else return &all_io_port_info->pt_infos[io_port_info_idx->idx][i];
    }
  }

  return NULL;

}



/*HDR*/
/**
 * @brief Initializes a ::MBG_TLV_ANNOUNCE structure
 *
 * @param[out] tlv            Pointer to a ::MBG_TLV_ANNOUNCE structure
 * @param[in]  uid            Unique sender ID used as identifier with all
 *                            subsequent messages related to this transaction.
 * @param[in]  tlv_feat_type  One of the ::MBG_TLV_FEAT_TYPES
 * @param[in]  total_bytes    Total number of bytes of all upcoming TLVs
 */
void mbg_tlv_announce_init( MBG_TLV_ANNOUNCE *tlv, MBG_TLV_UID uid,
                            MBG_TLV_TYPE tlv_feat_type, uint32_t total_bytes )
{
  memset( tlv, 0, sizeof( *tlv ) );
  tlv->data.uid = uid;
  tlv->data.type = tlv_feat_type;
  tlv->data.total_bytes = total_bytes;
  tlv->data.reserved_1 = 0;
  tlv->reserved_1 = 0;
  tlv->reserved_2 = 0;

}  // mbg_tlv_announce_init



/*HDR*/
/**
 * @brief Initializes a ::MBG_TLV
 *
 * @param[out] tlv           Pointer to a valid ::MBG_TLV structure
 * @param[in]  uid           Unique sender ID used as identifier for each further
 *                           TLV message related to this type.
 * @param[in]  tlv_type      Type identifier, see ::MBG_TLV_TYPES
 * @param[in]  total_bytes   Total number of bytes belonging to this
 *                           TLV transaction (which is very likely split into several TLVs)
 */
void mbg_tlv_init( MBG_TLV *tlv, MBG_TLV_UID uid,
                   MBG_TLV_TYPE tlv_type, uint32_t total_bytes )
{
  memset( tlv, 0, sizeof( *tlv ) );
  tlv->hdr.uid = uid;
  tlv->hdr.tlv_type = tlv_type;
  tlv->hdr.cur_bytes = 0;
  tlv->hdr.trans_bytes = 0;
  tlv->hdr.total_bytes = total_bytes;
  tlv->hdr.reserved_1 = 0;
  tlv->hdr.reserved_2 = 0;
  tlv->hdr.reserved_3 = 0;

}  // mbg_tlv_init



/*HDR*/
/**
 * @brief Initializes ::MBG_TLV_RCV_STATE structure
 *
 * @param[in,out] state         Pointer to ::MBG_TLV_RCV_STATE structure
 * @param[in]     uid           Unique sender ID used as identifier for each further
 *                              TLV message related to this type.
 * @param[in]     total_bytes   Total number of bytes belonging to this
 *                              TLV transaction (which is very likely split into several TLVS)
 */
void mbg_tlv_rcv_state_init( MBG_TLV_RCV_STATE *state, MBG_TLV_UID uid, uint32_t total_bytes )
{
  state->data.uid = uid;
  state->data.type = 0;
  state->data.total_bytes = total_bytes;
  state->data.reserved_1 = 0;
  state->read_bytes = 0;
  state->reserved_1 = 0;

}  // mbg_tlv_state_init



/*HDR*/
size_t mbg_snprint_revision( char *buf, size_t buflen,
                             const char *prefix, const char *suffix,
                             uint32_t rev)
{
    size_t bytes = 0;
    uint32_t major, minor, patch;

    if ( prefix )
      bytes += snprintf_safe( &buf[bytes], buflen, "%s", prefix );

    _mbg_decode_revision( rev, major, minor, patch );
    bytes += snprintf_safe( &buf[bytes], buflen - bytes, "%u.%u.%u",
                            major, minor, patch);

    if ( suffix )
        bytes += snprintf_safe( &buf[bytes], buflen - bytes, "%s", suffix );

    return bytes;

}  // mbg_snprint_revision



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_xbp_supp_nodes( const ALL_XBP_INFO *info )
{
  if ( info )
  {
    if ( ( info->limits.features & XBP_FEAT_MASK_NODES ) == XBP_FEAT_MASK_NODES )
      return MBG_SUCCESS;

    return MBG_ERR_NOT_SUPP_BY_DEV;
  }

  return MBG_ERR_INV_PARM;

} // chk_dev_xbp_supp_nodes



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_net_cfg_supp_stage_2( const ALL_NET_CFG_INFO *info )
{
  if ( info->glb_cfg_info.feat_flags & MBG_NET_GLB_SUPP_STAGE_2_MASK )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

} // chk_dev_net_cfg_supp_stage_2



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_ntp_supp_client( const ALL_NTP_CFG_INFO *info )
{
  if ( ( ( info->glb_info.supp_ntp_roles & NTP_MSK_ROLE_CLIENT ) == NTP_MSK_ROLE_CLIENT ) ||
       ( ( info->glb_info.supp_ntp_roles & NTP_MSK_ROLE_CLIENT_SERVER ) == NTP_MSK_ROLE_CLIENT_SERVER ) )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

} // chk_dev_ntp_supp_client


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_ntp_supp_server( const ALL_NTP_CFG_INFO *info )
{
  if ( ( ( info->glb_info.supp_ntp_roles & NTP_MSK_ROLE_SERVER ) == NTP_MSK_ROLE_SERVER ) ||
       ( ( info->glb_info.supp_ntp_roles & NTP_MSK_ROLE_CLIENT_SERVER ) == NTP_MSK_ROLE_CLIENT_SERVER ) )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

} // chk_dev_ntp_supp_server


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_ref_supp_mrf_none( const ALL_XMULTI_REF_INFO *info )
{
  if ( ( info->instances.flags & XMRIF_MSK_MRF_NONE_SUPP ) == XMRIF_MSK_MRF_NONE_SUPP )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

} // chk_dev_xmulti_ref_supp_mrf_none


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_ref_supp_ext_src_info( const ALL_XMULTI_REF_INFO *info )
{
  if ( ( info->instances.flags & XMRIF_MSK_EXT_SRC_INFO_SUPP ) == XMRIF_MSK_EXT_SRC_INFO_SUPP )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

} // chk_dev_xmulti_ref_supp_ext_src_info


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_ref_supp_holdover_status( const ALL_XMULTI_REF_INFO *info )
{
  if ( ( info->instances.flags & XMRIF_MSK_HOLDOVER_STATUS_SUPP ) == XMRIF_MSK_HOLDOVER_STATUS_SUPP )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

} // chk_dev_xmulti_ref_supp_holdover_status


/*HDR*/
/*
 * Type is NOT an index of ::XMULTI_REF_INSTANCES::n_xmr_settings,
 * but of ::MULTI_REF_TYPES.
 * Depends on chk_dev_supp_xmulti_ref_ext_src_info.
 * @see chk_dev_supp_xmulti_ref_ext_src_info
 */
_NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_ref_supp_ext_source_stats( const ALL_XMULTI_REF_INFO *info, int type )
{
  if (
       ( type < N_MULTI_REF ) &&
       (( info->ext_src_infos[type].info.feat_flags & XMR_EXT_SRC_FEAT_FLAG_MSK_STATS ) == XMR_EXT_SRC_FEAT_FLAG_MSK_STATS )
     ) return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

} // chk_dev_xmulti_ref_supp_ext_source_stats


/*HDR*/
/*
 * Type is NOT an index of ::XMULTI_REF_INSTANCES::n_xmr_settings,
 * but of ::MULTI_REF_TYPES.
 * Depends on chk_dev_supp_xmulti_ref_ext_src_info.
 * @see chk_dev_supp_xmulti_ref_ext_src_info
 */
_NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_ref_supp_ext_source_metrics( const ALL_XMULTI_REF_INFO *info, int type )
{
  if (
       ( type < N_MULTI_REF ) &&
       (( info->ext_src_infos[type].info.feat_flags & XMR_EXT_SRC_FEAT_FLAG_MSK_METRICS ) == XMR_EXT_SRC_FEAT_FLAG_MSK_METRICS )
     ) return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

} // chk_dev_xmulti_ref_supp_ext_source_metrics


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_ims_has_fdm( const ALL_IMS_INFO *info )
{
  if ( ( info->state.flags & MBG_IMS_STATE_FLAG_MSK_HAS_FDM ) == MBG_IMS_STATE_FLAG_MSK_HAS_FDM )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

} // chk_dev_ims_has_fdm


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_ims_is_volt_out_enabled( const ALL_IMS_STATE *ims_state, unsigned idx )
{
  const MBG_IMS_SENSOR_STATE *sstate = &ims_state->sensor_state_idx[idx].state;

  if (
        ( sstate->type == MBG_IMS_SENSOR_VOLTAGE ) &&
        ( ( sstate->flags & MBG_IMS_SENSOR_VOLTAGE_OUT_ENB ) == MBG_IMS_SENSOR_VOLTAGE_OUT_ENB )
     ) return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

} // chk_dev_ims_is_volt_out_enabled


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_ims_is_volt_out_overload( const ALL_IMS_STATE *ims_state, unsigned idx )
{
  const MBG_IMS_SENSOR_STATE *sstate = &ims_state->sensor_state_idx[idx].state;

  if (
        ( sstate->type == MBG_IMS_SENSOR_VOLTAGE ) &&
        ( ( sstate->flags & MBG_IMS_SENSOR_VOLTAGE_OUT_OVR ) == MBG_IMS_SENSOR_VOLTAGE_OUT_OVR )
     ) return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

} // chk_dev_ims_is_volt_out_overload


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_ims_is_pll_locked( const ALL_IMS_STATE *ims_state, unsigned idx )
{
  const MBG_IMS_SENSOR_STATE *sstate = &ims_state->sensor_state_idx[idx].state;

  if (
        ( sstate->type == MBG_IMS_SENSOR_PLL ) &&
        ( ( sstate->flags & MBG_IMS_SENSOR_PLL_LOCKED ) == MBG_IMS_SENSOR_PLL_LOCKED )
     ) return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

} // chk_dev_ims_is_pll_locked


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_gpio_supp_ass_idx( const ALL_GPIO_INFO *gpio_info, unsigned idx )
{
  const MBG_GPIO_LIMITS *limits = &gpio_info->infos[idx].info.limits;

  if ( ( limits->supp_flags & MSK_MBG_GPIO_DEPENDS_ON_ASS_IO_IDX ) == MSK_MBG_GPIO_DEPENDS_ON_ASS_IO_IDX )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

} // chk_dev_gpio_supp_ass_idx


/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_gpio_dep_on_ass_idx( const ALL_GPIO_INFO *gpio_info, unsigned idx )
{
  const MBG_GPIO_SETTINGS *settings = &gpio_info->infos[idx].info.settings;

  if ( ( settings->flags & MSK_MBG_GPIO_DEPENDS_ON_ASS_IO_IDX ) == MSK_MBG_GPIO_DEPENDS_ON_ASS_IO_IDX )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

} // chk_dev_gpio_dep_on_ass_idx


/*HDR*/
/**
 * @brief Checks whether GPIO supports status function
 *
 * @param[out]  info  Pointer to a ::ALL_GPIO_INFO structure to be filled up
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_gpio
 * @see ::mbg_chk_dev_supp_gpio
 * @see ::free_all_gpio_info
 */
_NO_MBG_API_ATTR int _MBG_API chk_dev_gpio_has_status( const ALL_GPIO_INFO *info )
{
  if ( ( info->cfg_limits.flags & MBG_GPIO_CFG_LIMIT_FLAG_MASK_STATUS_SUPP ) == MBG_GPIO_CFG_LIMIT_FLAG_MASK_STATUS_SUPP )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

} // chk_dev_gpio_has_status



/*HDR*/
/**
 * @brief Frees ::ALL_XBP_INFO structure
 *
 * @param[in]     p       Pointer to the ::ALL_XBP_INFO structure, which will be freed
 *
 * @see ::mbgextio_get_all_xbp_info
 */
void free_all_xbp_info ( ALL_XBP_INFO *p )
{
  if ( p )
  {
    if ( p->node_limits )
      free( p->node_limits );

    if ( p->node_infos )
      free( p->node_infos );

    free( p );
  }

} // free_all_xbp_info



/*HDR*/
/**
 * @brief Frees ::ALL_NET_CFG_INFO structure
 *
 * @param[in]     p       Pointer to the ::ALL_NET_CFG_INFO structure, which will be freed
 *
 * @see ::mbgextio_get_all_net_cfg_info
 */
void free_all_net_cfg_info ( ALL_NET_CFG_INFO *p )
{
  if ( p )
  {
    if ( p->link_infos )
      free( p->link_infos );

    if ( p->addr_infos )
      free( p->addr_infos );

    if ( p->dns_srvrs )
      free( p->dns_srvrs );

    if ( p->dns_srch_doms )
      free( p->dns_srch_doms );

    if ( p->route_infos )
      free( p->route_infos );

    free( p );
  }

} // free_all_net_cfg_info



/*HDR*/
/**
 * @brief Frees ::ALL_NET_STATUS_INFO structure
 *
 * @param[in]     p       Pointer to the ::ALL_NET_STATUS_INFO structure, which will be freed
 *
 * @see ::mbgextio_get_all_net_status_info
 */
void free_all_net_status_info ( ALL_NET_STATUS_INFO *p )
{
  if ( p )
  {
    if ( p->link_infos )
      free( p->link_infos );

    if ( p->addr_infos )
      free( p->addr_infos );

    if ( p->dns_srvrs )
      free( p->dns_srvrs );

    if ( p->dns_srch_doms )
      free( p->dns_srch_doms );

    if ( p->route_infos )
      free( p->route_infos );

    free( p );
  }

} // free_all_net_status_info



/*HDR*/
/**
 * @brief Frees ::ALL_SNMP_INFO structure
 *
 * @param[in]     p       Pointer to the ::ALL_SNMP_INFO structure, which will be freed
 *
 */
void free_all_snmp_info ( ALL_SNMP_INFO *p )
{
  if ( p )
  {
    if ( p->v12_infos )
      free( p->v12_infos );

    if ( p->v12_trap_infos )
      free( p->v12_trap_infos );

    if ( p->v3_infos )
      free( p->v3_infos );

    if ( p->v3_trap_infos )
      free( p->v3_trap_infos );

    free( p );
  }

} // free_all_snmp_info



/*HDR*/
/**
 * @brief Frees ::ALL_MONITORING_INFO structure
 *
 * @param[in]     p       Pointer to the ::ALL_MONITORING_INFO structure, which will be freed
 *
 */
void free_all_monitoring_info ( ALL_MONITORING_INFO *p )
{
  if ( p )
  {
    if ( p->all_snmp_info )
      free_all_snmp_info( p->all_snmp_info );

    if ( p->event_infos )
        free( p->event_infos );

    free ( p );
  }
}


/*HDR*/
/**
 * @brief Frees ::ALL_MONITORING_STATUS structure
 *
 * @param[in]     p       Pointer to the ::ALL_MONITORING_STATUS structure, which will be freed
 *
 */
void free_all_monitoring_status ( ALL_MONITORING_STATUS *p )
{
  if ( p )
  {
    if ( p->event_stati )
      free( p->event_stati );

    free ( p );
  }
}



/*HDR*/
/**
 * @brief Frees ::ALL_XMULTI_REF_INFO structure
 *
 * @param[in]     p       Pointer to the ::ALL_XMULTI_REF_INFO structure, which will be freed
 *
 * @see ::mbgextio_get_all_xmulti_ref_info
 * @see ::mbg_get_all_xmulti_ref_info
 */
void free_all_xmulti_ref_info( ALL_XMULTI_REF_INFO *p )
{
  if ( p )
  {
    if ( p->infos )
      free( p->infos );

    if ( p->ext_src_infos )
      free( p->ext_src_infos );

    free ( p );
  }

}  // free_all_xmulti_ref_info



/*HDR*/
/**
 * @brief Frees ::ALL_XMULTI_REF_STATUS structure
 *
 * @param[in]     p       Pointer to the ::ALL_XMULTI_REF_STATUS structure, which will be freed
 *
 * @see ::mbgextio_get_all_xmulti_ref_status
 * @see ::mbg_get_all_xmulti_ref_status
 */
void free_all_xmulti_ref_status( ALL_XMULTI_REF_STATUS *p )
{
  if ( p )
  {
    if ( p->status )
      free( p->status );

    if ( p->holdover_status )
      free( p->holdover_status );

    if ( p->stats_idx )
      free( p->stats_idx );

    if ( p->metrics_idx )
      free( p->metrics_idx );

    free( p );
  }

}  // free_all_xmulti_ref_status



/*HDR*/
/**
 * @brief Frees ::ALL_PTP_V1_COMMON_DATASETS structure allocated by ::mbgextio_get_all_ptp_v1_common_datasets
 *
 * @param[in]     p       Pointer to the ::ALL_PTP_V1_COMMON_DATASETS structure, which will be freed
 *
 * @see ::mbgextio_get_all_ptp_v1_common_datasets
 */
void free_all_ptp_v1_common_datasets( ALL_PTP_V1_COMMON_DATASETS *p )
{
  if ( p )
  {
    if ( p->port_datasets )
      free( p->port_datasets );

    free( p );
  }

}  // free_all_ptp_v1_common_datasets



/*HDR*/
/**
 * @brief Frees ::ALL_PTP_V2_COMMON_DATASETS structure allocated by ::mbgextio_get_all_ptp_v2_common_datasets
 *
 * @param[in]     p       Pointer to the ::ALL_PTP_V2_COMMON_DATASETS structure, which will be freed
 *
 * @see ::mbgextio_get_all_ptp_v2_common_datasets
 */
void free_all_ptp_v2_common_datasets( ALL_PTP_V2_COMMON_DATASETS *p )
{
  if ( p )
  {
    if ( p->port_datasets )
      free( p->port_datasets );

    free( p );
  }

}  // free_all_ptp_v2_common_datasets



/*HDR*/
/**
 * @brief Frees ::ALL_NTP_CFG_INFO structure
 *
 * @param[in]     p       Pointer to the ::ALL_NTP_CFG_INFO structure, which will be freed
 *
 * @see ::mbgextio_get_all_ntp_cfg_info
 */
void free_all_ntp_cfg_info( ALL_NTP_CFG_INFO *p )
{
  if ( p )
  {
    if ( p->symm_key_limits )
      free( p->symm_key_limits );

    if ( p->symm_key_info_idx )
      free( p->symm_key_info_idx );

    if ( p->trusted_key_info_idx )
      free( p->trusted_key_info_idx );

    if ( p->clnt_info )
      free( p->clnt_info );

    if ( p->peer_settings_idx )
      free( p->peer_settings_idx );

    if ( p->srv_info )
      free( p->srv_info );

    if ( p->refclk_info_idx )
      free( p->refclk_info_idx );

    if ( p->misc_limits )
      free( p->misc_limits );

    if ( p->orphan_mode_info )
      free( p->orphan_mode_info );

    free( p );
  }

}  // free_all_ntp_cfg_info



/*HDR*/
/**
 * @brief Frees ::ALL_NTP_STATUS structure
 *
 * @param[in]     p       Pointer to the ::ALL_NTP_STATUS structure, which will be freed
 *
 * @see ::mbgextio_get_all_ntp_status
 */
void free_all_ntp_status( ALL_NTP_STATUS *p )
{
  if ( p )
  {
    if ( p->peer_states )
      free( p->peer_states );

    free( p );
  }

}  // free_all_ntp_status



/*HDR*/
/**
 * @brief Frees memory allocated by ::mbgextio_get_all_ims_info
 *
 * @param[in]     p       Pointer to the ::ALL_IMS_INFO structure, which will be freed
 *
 * @see ::mbgextio_dev_has_ims
 * @see ::mbgextio_dev_ims_has_fdm
 * @see ::mbgextio_get_all_ims_info
 * @see ::mbgextio_get_all_ims_state
 */
void free_all_ims_info( ALL_IMS_INFO *p )
{
  if ( p )
  {
    if ( p->fdm_info )
      free( p->fdm_info );

    if ( p->fdm_limits )
      free( p->fdm_limits );

    if ( p->fdm_outinfo_idx )
      free( p->fdm_outinfo_idx );

    free( p );
  }

}  // free_all_ims_info



/*HDR*/
/**
 * @brief Frees memory allocated by ::mbgextio_get_all_ims_state
 *
 * @param[in]     p       Pointer to the ::ALL_IMS_STATE structure, which will be freed
 *
 * @see ::mbgextio_dev_has_ims
 * @see ::mbgextio_dev_ims_has_fdm
 * @see ::mbgextio_get_all_ims_info
 * @see ::mbgextio_get_all_ims_state
 */
void free_all_ims_state( ALL_IMS_STATE *p )
{
  if ( p )
  {
    if ( p->sensor_state_idx )
      free( p->sensor_state_idx );

    if ( p->fdm_state )
      free( p->fdm_state );

    if ( p->fdm_output_state_idx )
      free( p->fdm_output_state_idx );

    free( p );
  }

}  // free_all_ims_state



/*HDR*/
/**
 * @brief Frees memory allocated by ::mbgextio_get_all_gpio_info
 *
 * @param[in]     p       Pointer to the ::ALL_GPIO_INFO structure, which will be freed
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_gpio
 * @see ::mbgextio_get_all_gpio_info
 */
void free_all_gpio_info( ALL_GPIO_INFO *p )
{
  if ( p )
  {
    if ( p->infos )
      free( p->infos );

    free( p );
  }

} // free_all_gpio_info



/*HDR*/
/**
 * @brief Frees memory allocated by ::mbgextio_get_all_io_port_info
 *
 * @param[in]     p       Pointer to the ::ALL_IO_PORT_INFO structure, which will be freed
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_io_ports
 * @see ::mbgextio_get_all_io_port_info
 * @see ::mbgextio_get_all_io_port_status
 * @see ::free_all_io_port_status
 */
void free_all_io_port_info( ALL_IO_PORT_INFO *p )
{
  uint8_t i;

  if ( p )
  {
    if ( p->pt_infos )
    {
      for ( i = 0; i < p->limits.num_ports; ++i )
      {
        if ( p->pt_infos[i] )
          free( p->pt_infos[i] );
      }

      free ( p->pt_infos );
    }

    if ( p->p_infos )
      free( p->p_infos );

    free( p );
  }

} // free_all_io_port_info



/*HDR*/
/**
 * @brief Frees memory allocated by ::mbgextio_get_all_io_port_status
 *
 * @param[in]     p       Pointer to the ::ALL_IO_PORT_STATUS structure, which will be freed
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_io_ports
 * @see ::mbgextio_get_all_io_port_info
 * @see ::mbgextio_get_all_io_port_status
 * @see ::free_all_io_port_info
 */
void free_all_io_port_status( ALL_IO_PORT_STATUS *p )
{
  if ( p )
  {
    if ( p->status )
      free ( p->status );

    free( p );
  }

} // free_all_io_port_status



/*HDR*/
/**
 * @brief Frees memory allocated by ::mbgextio_get_all_gpio_state
 *
 * @param[in]     p       Pointer to the ::ALL_GPIO_STATE structure, which will be freed
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_gpio
 * @see ::mbgextio_get_all_gpio_state
 */
void free_all_gpio_state( ALL_GPIO_STATE *p )
{
  if ( p )
  {
    if ( p->states )
      free( p->states );

    free( p );
  }

} // free_all_gpio_state



/*HDR*/
/**
 * Allocates memory for a new ::UCAP_ENTRY structure
 *
 * @return The new allocated ::UCAP_ENTRY or NULL if the calloc call was not successful
 */
UCAP_ENTRY* calloc_ucap_entry( void )
{
  UCAP_ENTRY *entry = (UCAP_ENTRY *) calloc( 1, sizeof( *entry ) );
  if ( entry )
  {
    mbg_klist_init(&entry->head);
  }
  return entry;

} // calloc_ucap_entry


/*HDR*/
/**
 * @brief Frees memory allocated by ::mbgextio_get_all_ucap_info
 *
 * @param[in]     p       Pointer to the ::ALL_UCAP_INFO structure, which will be freed
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_ucap
 * @see ::mbg_chk_dev_has_ucap
 * @see ::mbgextio_get_all_ucap_info
 * @see ::mbg_get_all_ucap_info
 */
void free_all_ucap_info( ALL_UCAP_INFO *p )
{
  if ( p )
  {
    UCAP_ENTRY *entry;

    while ( !mbg_klist_is_empty( &p->list ) )
    {
      entry = mbg_klist_first_entry( &p->list, UCAP_ENTRY, head );
      mbg_klist_delete_item( &entry->head );
      free( entry );
    }

    free( p );
  }

} // free_all_ucap_info


/*HDR*/
/**
 * @brief Frees ::ALL_UCAP_NET_INFO structure
 *
 * @param[in]     p       Pointer to the ::ALL_UCAP_NET_INFO structure, which will be freed
 *
 * @see ::mbgextio_get_all_ucap_net_info
 */
void free_all_ucap_net_info( ALL_UCAP_NET_INFO *p )
{
  if ( p )
  {
    if ( p->recv_infos )
      free( p->recv_infos );

    free( p );
  }

} // free_all_ucap_net_info


