
/**************************************************************************
 *
 *  $Id: aes128.c 1.2.1.1 2009/12/22 12:23:00 martin TEST $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    FIPS-197 compliant AES implementation
 *
 * -----------------------------------------------------------------------
 *  $Log: aes128.c $
 *  Revision 1.2.1.1  2009/12/22 12:23:00  martin
 *  Started to fix possible 32/64 bit issues.
 *  Revision 1.2  2009/10/01 14:03:09  martin
 *  Added standard file header.
 *  Fixed compiler warnings.
 *  Support prototype generation by makehdr tool.
 *  Use standard fixed-size data types.
 *  Code cleanup.
 *  Revision 1.1  2004/05/25 12:36:21  andre
 *  Initial revision.
 *
 **************************************************************************/

#define _AES128
  #include <aes128.h>
#undef _AES128


#include <string.h>
#include <stdio.h>




/* forward S-box & tables */

#define FTABLE_ENTRIES 256

static uint32_t FSb[FTABLE_ENTRIES];
static uint32_t FT0[FTABLE_ENTRIES];
static uint32_t FT1[FTABLE_ENTRIES];
static uint32_t FT2[FTABLE_ENTRIES];
static uint32_t FT3[FTABLE_ENTRIES];


/* rounding constants */
#define RCON_TABLE_ENTRIES 10
uint32_t RCON[RCON_TABLE_ENTRIES];

/* tables generation flag */
static int initialized;



#define ROTR8( _x ) ( ( ( (_x) << 24 ) & 0xFFFFFFFFUL ) | \
                      ( ( (_x) & 0xFFFFFFFFUL ) >>  8 ) )

#define XTIME( _x ) ( ( (_x) <<  1 ) ^ ( ( (_x) & 0x80 ) ? 0x1B : 0x00 ) )

#define MUL( _x, _y ) ( ( (_x) &&  (_y) ) ? pow[(log[_x] + log[_y]) % 255] : 0 )



/*HDR*/
void aes_gen_tables( void )
{
  int i;
  uint8_t x;
  uint8_t y;
  uint8_t pow[FTABLE_ENTRIES];
  uint8_t log[FTABLE_ENTRIES];

  /* compute pow and log tables over GF(2^8) */

  for ( i = 0, x = 1; i < FTABLE_ENTRIES; i++, x ^= XTIME( x ) )
  {
    pow[i] = x;
    log[x] = i;
  }

  /* calculate the round constants */

  for ( i = 0, x = 1; i < RCON_TABLE_ENTRIES; i++, x = XTIME( x ) )
    RCON[i] = (uint32_t) x << 24;

  /* generate the forward and reverse S-boxes */

  FSb[0x00] = 0x63;

  for ( i = 1; i < FTABLE_ENTRIES; i++ )
  {
    x = pow[255 - log[i]];

    y = x;  y = ( y << 1 ) | ( y >> 7 );
    x ^= y; y = ( y << 1 ) | ( y >> 7 );
    x ^= y; y = ( y << 1 ) | ( y >> 7 );
    x ^= y; y = ( y << 1 ) | ( y >> 7 );
    x ^= y ^ 0x63;

    FSb[i] = x;
  }

  /* generate the forward and reverse tables */

  for ( i = 0; i < FTABLE_ENTRIES; i++ )
  {
    x = (uint8_t) FSb[i];
    y = XTIME( x );

    FT0[i] = (uint32_t) ( x ^ y ) ^
             ( (uint32_t) x <<  8 ) ^
             ( (uint32_t) x << 16 ) ^
             ( (uint32_t) y << 24 );

    FT0[i] &= 0xFFFFFFFFUL;

    FT1[i] = ROTR8( FT0[i] );
    FT2[i] = ROTR8( FT1[i] );
    FT3[i] = ROTR8( FT2[i] );
  }

}  /* aes_gen_tables */



/* platform-independant 32-bit integer manipulation macros */

#define GET_ulong( _n, _b, _i )                \
{                                              \
  (_n) = ( (uint32_t) (_b)[(_i)    ] << 24 )   \
       | ( (uint32_t) (_b)[(_i) + 1] << 16 )   \
       | ( (uint32_t) (_b)[(_i) + 2] <<  8 )   \
       | ( (uint32_t) (_b)[(_i) + 3]       );  \
}

#define PUT_ulong( _n, _b, _i )                \
{                                              \
  (_b)[(_i)    ] = (uint8_t) ( (_n) >> 24 );   \
  (_b)[(_i) + 1] = (uint8_t) ( (_n) >> 16 );   \
  (_b)[(_i) + 2] = (uint8_t) ( (_n) >>  8 );   \
  (_b)[(_i) + 3] = (uint8_t) ( (_n)       );   \
}




/* decryption key schedule tables */


/* AES key scheduling routine */

/*HDR*/
int aes128_set_key( aes128_context *ctx, uint8_t *key )
{
  int i;
  uint32_t *RK;

  if ( !initialized )
  {
    aes_gen_tables();
    initialized = 1;
  }


  RK = ctx->erk;

  for ( i = 0; i < 4; i++ )
    GET_ulong( RK[i], key, i * 4 );

  /* setup encryption round keys */
  for ( i = 0; i < RCON_TABLE_ENTRIES; i++, RK += 4 )
  {
    RK[4] = RK[0] ^ RCON[i] ^
            ( FSb[ (uint8_t) ( RK[3] >> 16 ) ] << 24 ) ^
            ( FSb[ (uint8_t) ( RK[3] >>  8 ) ] << 16 ) ^
            ( FSb[ (uint8_t) ( RK[3]       ) ] <<  8 ) ^
            ( FSb[ (uint8_t) ( RK[3] >> 24 ) ]       );

     RK[5]  = RK[1] ^ RK[4];
     RK[6]  = RK[2] ^ RK[5];
     RK[7]  = RK[3] ^ RK[6];
  }

  return 0;

}  /* aes128_set_key */



/* AES 128-bit block encryption routine */
static /*HDR*/
void aes128_encrypt( aes128_context *ctx, uint8_t input[16], uint8_t output[16] )
{
  uint32_t *RK, X0, X1, X2, X3, Y0, Y1, Y2, Y3;

  RK = ctx->erk;

  GET_ulong( X0, input,  0 ); X0 ^= RK[0];
  GET_ulong( X1, input,  4 ); X1 ^= RK[1];
  GET_ulong( X2, input,  8 ); X2 ^= RK[2];
  GET_ulong( X3, input, 12 ); X3 ^= RK[3];

  #define AES_FROUND(X0,X1,X2,X3,Y0,Y1,Y2,Y3)     \
  {                                               \
    RK += 4;                                      \
                                                  \
    X0 = RK[0] ^ FT0[ (uint8_t) ( Y0 >> 24 ) ] ^  \
                 FT1[ (uint8_t) ( Y1 >> 16 ) ] ^  \
                 FT2[ (uint8_t) ( Y2 >>  8 ) ] ^  \
                 FT3[ (uint8_t) ( Y3       ) ];   \
                                                  \
    X1 = RK[1] ^ FT0[ (uint8_t) ( Y1 >> 24 ) ] ^  \
                 FT1[ (uint8_t) ( Y2 >> 16 ) ] ^  \
                 FT2[ (uint8_t) ( Y3 >>  8 ) ] ^  \
                 FT3[ (uint8_t) ( Y0       ) ];   \
                                                  \
    X2 = RK[2] ^ FT0[ (uint8_t) ( Y2 >> 24 ) ] ^  \
                 FT1[ (uint8_t) ( Y3 >> 16 ) ] ^  \
                 FT2[ (uint8_t) ( Y0 >>  8 ) ] ^  \
                 FT3[ (uint8_t) ( Y1       ) ];   \
                                                  \
    X3 = RK[3] ^ FT0[ (uint8_t) ( Y3 >> 24 ) ] ^  \
                 FT1[ (uint8_t) ( Y0 >> 16 ) ] ^  \
                 FT2[ (uint8_t) ( Y1 >>  8 ) ] ^  \
                 FT3[ (uint8_t) ( Y2       ) ];   \
  }

  AES_FROUND( Y0, Y1, Y2, Y3, X0, X1, X2, X3 );       /* round 1 */
  AES_FROUND( X0, X1, X2, X3, Y0, Y1, Y2, Y3 );       /* round 2 */
  AES_FROUND( Y0, Y1, Y2, Y3, X0, X1, X2, X3 );       /* round 3 */
  AES_FROUND( X0, X1, X2, X3, Y0, Y1, Y2, Y3 );       /* round 4 */
  AES_FROUND( Y0, Y1, Y2, Y3, X0, X1, X2, X3 );       /* round 5 */
  AES_FROUND( X0, X1, X2, X3, Y0, Y1, Y2, Y3 );       /* round 6 */
  AES_FROUND( Y0, Y1, Y2, Y3, X0, X1, X2, X3 );       /* round 7 */
  AES_FROUND( X0, X1, X2, X3, Y0, Y1, Y2, Y3 );       /* round 8 */
  AES_FROUND( Y0, Y1, Y2, Y3, X0, X1, X2, X3 );       /* round 9 */


  /* last round */

  RK += 4;

  X0 = RK[0] ^ ( FSb[ (uint8_t) ( Y0 >> 24 ) ] << 24 ) ^
               ( FSb[ (uint8_t) ( Y1 >> 16 ) ] << 16 ) ^
               ( FSb[ (uint8_t) ( Y2 >>  8 ) ] <<  8 ) ^
               ( FSb[ (uint8_t) ( Y3       ) ]       );

  X1 = RK[1] ^ ( FSb[ (uint8_t) ( Y1 >> 24 ) ] << 24 ) ^
               ( FSb[ (uint8_t) ( Y2 >> 16 ) ] << 16 ) ^
               ( FSb[ (uint8_t) ( Y3 >>  8 ) ] <<  8 ) ^
               ( FSb[ (uint8_t) ( Y0       ) ]       );

  X2 = RK[2] ^ ( FSb[ (uint8_t) ( Y2 >> 24 ) ] << 24 ) ^
               ( FSb[ (uint8_t) ( Y3 >> 16 ) ] << 16 ) ^
               ( FSb[ (uint8_t) ( Y0 >>  8 ) ] <<  8 ) ^
               ( FSb[ (uint8_t) ( Y1       ) ]       );

  X3 = RK[3] ^ ( FSb[ (uint8_t) ( Y3 >> 24 ) ] << 24 ) ^
               ( FSb[ (uint8_t) ( Y0 >> 16 ) ] << 16 ) ^
               ( FSb[ (uint8_t) ( Y1 >>  8 ) ] <<  8 ) ^
               ( FSb[ (uint8_t) ( Y2       ) ]       );

  PUT_ulong( X0, output,  0 );
  PUT_ulong( X1, output,  4 );
  PUT_ulong( X2, output,  8 );
  PUT_ulong( X3, output, 12 );

}  /* aes128_encrypt */



/* buffer size must be an integer multiple of 16 */
/*HDR*/
int aes_encrypt_buff( uint8_t *p_data,
                      uint8_t *p_key,
                      uint8_t *p_init,
                      int n )
{
  aes128_context ctx;

  uint8_t block_key[32];

  if ( n % 16 != 0 )
    return -1;

  memcpy ( block_key, p_init, 16 );

  // setup the round key
  aes128_set_key( &ctx, p_key );
  aes128_encrypt( &ctx, block_key, block_key );

  while ( n )
  {
    // cipher feedback mode
    {
      int j;
      for ( j = 0; j < 16; j++ )
        *( (char *) ( p_data + j ) ) ^= block_key[j];
    }
     // encrypt 16 byte block
    aes128_encrypt( &ctx, p_data, block_key );

    p_data += 16;
    n -= 16;
  }

  return 0;

}  /* aes_encrypt_buff */



/* buffer size must be an integer multiple of 16 */

/*HDR*/
int aes_decrypt_buff ( uint8_t *p_data,
                       uint8_t *p_key,
                       uint8_t *p_init,
                       int n )
{
  aes128_context ctx;

  uint8_t block_key[32];
  uint8_t p_tmp[32];


  if ( n % 16 != 0 )
    return -1;

  memcpy( block_key, p_init, 16 );

   // setup the 1st. block key
  aes128_set_key( &ctx, p_key );
  aes128_encrypt( &ctx, block_key, block_key );


  while ( n )
  {
    int j;

    // decrypt first block and save encrypted
    // block for new block key calculation
    for ( j = 0; j < 16; j++ )
    {
      p_tmp[j] = *( (char *) ( p_data + j ) );
      *( (char *) ( p_data + j ) ) ^= block_key[j];
    }

    // calculate new block key
    aes128_encrypt( &ctx, p_tmp, block_key );

    p_data += 16;
    n -= 16;
  }

  return 0;

}  /* aes_decrypt_buff */



