
/**************************************************************************
 *
 *  $Id: xtiocomm.c 1.2.1.5 2017/04/12 08:42:06Z martin TEST $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    Common functions for the Meinberg binary data protocol.
 *
 * -----------------------------------------------------------------------
 *  $Log: xtiocomm.c $
 *  Revision 1.2.1.5  2017/04/12 08:42:06Z  martin
 *  Fixed compiler warning.
 *  Revision 1.2.1.4  2016/12/01 13:47:49Z  philipp
 *  Moved helper function check_byte_array_bit from xdevfeat.c to xtiocomm.c and make it public
 *  Revision 1.2.1.3  2016/11/15 15:43:31  martin
 *  Account for modified mbgserio functions.
 *  Revision 1.2.1.2  2016/10/28 07:31:36  thomas-b
 *  Use calloc instead of malloc for allocation of rcv and xmt buffer to avoid uninitialized buffer warnings
 *  Revision 1.2.1.1  2016/06/29 11:59:22  philipp
 *  Extended socket API by TCP client
 *  Revision 1.2  2016/06/28 13:39:21  philipp
 *  Added missing include files if _USE_SERIAL_IO is 0
 *  Revision 1.1  2016/03/16 14:32:52  martin
 *  Initial revision.
 *
 **************************************************************************/

#define _XTIOCOMM
  #include <xtiocomm.h>
#undef _XTIOCOMM

#include <stdio.h>
#include <string.h>
#include <mbgerror.h>



/*HDR*/
/**
 * @brief Deallocate a message control structure
 *
 * Free the memory allocated for a message control structure
 * and set the pointer to NULL.
 *
 * @param[in,out] ppmctl  Address of a pointer to a message control structure
 *
 * @see ::alloc_msg_ctl
 */
void dealloc_msg_ctl( MBG_MSG_CTL **ppmctl )
{
  MBG_MSG_CTL *pmctl = *ppmctl;

  if ( pmctl )
  {
    if ( pmctl->rcv.pmb )
    {
      free( pmctl->rcv.pmb );
      pmctl->rcv.pmb = NULL;
    }

    pmctl->rcv.buf_size = 0;

    if ( pmctl->xmt.pmb )
    {
      free( pmctl->xmt.pmb );
      pmctl->xmt.pmb = NULL;
    }

    pmctl->xmt.buf_size = 0;

    free( pmctl );
    *ppmctl = NULL;
  }

}  // dealloc_msg_ctl



/*HDR*/
/**
 * @brief Allocate a message control structure
 *
 * Allocate memory for a message control structure and associated
 * message receive and transmit buffers.
 *
 * @param[out] ppmctl  Address of a pointer to a message control structure to be allocated, set to NULL on error
 *
 * @return ::MBG_SUCCESS or one of the @ref MBG_ERROR_CODES
 *
 * @see ::dealloc_msg_ctl
 */
int alloc_msg_ctl( MBG_MSG_CTL **ppmctl )
{
  int rc = MBG_ERR_UNSPEC;

  MBG_MSG_CTL *pmctl = (MBG_MSG_CTL *) malloc( sizeof( *pmctl ) );

  if ( pmctl == NULL )
  {
    #if defined( DEBUG )
      fprintf( stderr, "failed to malloc control block in %s\n", __func__ );
    #endif
    goto fail;
  }

  memset( pmctl, 0, sizeof( *pmctl ) );

  // allocate receive buffer
  pmctl->rcv.buf_size = sizeof( *(pmctl->rcv.pmb) );
  pmctl->rcv.pmb = (MBG_MSG_BUFF *) calloc( 1, pmctl->rcv.buf_size );

  if ( pmctl->rcv.pmb == NULL )
  {
    #if defined( DEBUG )
      fprintf( stderr, "failed to malloc rcv buffer in %s\n", __func__ );
    #endif
    goto fail;
  }

  // allocate transmit buffer
  pmctl->xmt.buf_size = sizeof( *(pmctl->xmt.pmb) );
  pmctl->xmt.pmb = (MBG_MSG_BUFF *) calloc( 1, pmctl->xmt.buf_size );

  if ( pmctl->xmt.pmb == NULL )
  {
    #if defined( DEBUG )
      fprintf( stderr, "failed to malloc xmt buffer in %s\n", __func__ );
    #endif
    goto fail;
  }


  // buffers allocated successfully
  rc = MBG_SUCCESS;
  goto out;


fail:  // if not all memory blocks could be allocated, clean up
  dealloc_msg_ctl( &pmctl );  // also sets pmctl to NULL
  rc = MBG_ERR_NO_MEM;

out:
  *ppmctl = pmctl;
  return rc;

}  // alloc_msg_ctl



//### TODO proper header / prototype
/*HDR*/
const char *xtiocomm_get_cmd_name( GPS_CMD cmd_code )
{
  static const MBG_CODE_NAME_TABLE_ENTRY gps_cmd_names[] = GPS_CMD_CODES_TABLE;

  const MBG_CODE_NAME_TABLE_ENTRY *p;

  cmd_code &= ~GPS_CTRL_MSK;

  for ( p = gps_cmd_names; ; )
  {
    if ( p->name == NULL )
      break;  // end of table

    if ( p->code == cmd_code )
      return p->name;

    p++;
  }

  return NULL;

}  // xtiocomm_get_cmd_name



/*HDR*/
//### TODO
int dev_open_finish( int rc, MBG_MSG_CTL *pmctl, MBG_MSG_CTL **caller_ppmctl )
{
  *caller_ppmctl = pmctl;  // valid pointer or NULL on error

  return rc;

}  // dev_open_finish



/*HDR*/
//### TODO
int dev_open_fail_free( int rc, MBG_MSG_CTL **ppmctl, MBG_MSG_CTL **caller_ppmctl )
{
  dealloc_msg_ctl( ppmctl );

  return dev_open_finish( rc, *ppmctl, caller_ppmctl );

}  // dev_open_fail_free



/*HDR*/
//### TODO
int dev_open_fail_close( int rc, MBG_MSG_CTL **ppmctl, MBG_MSG_CTL **caller_ppmctl )
{
  xtiocomm_close_connection( ppmctl );  // also frees the MBG_MSG_CTL structure

  return dev_open_finish( rc, *ppmctl, caller_ppmctl );

}  // dev_open_fail_close



/*HDR*/
//### TODO
void dev_open_init( MBG_MSG_CTL *pmctl, int conn_type, int msg_timeout_ms, int poll_timeout_ms )
{
  pmctl->conn_type = conn_type;

  xtiocomm_set_msg_rcv_timeout( pmctl, msg_timeout_ms );
  xtiocomm_set_dev_poll_timeout( pmctl, poll_timeout_ms );

}  // dev_open_init



/*HDR*/
/**
 * @brief Check if a specific bit is set in a byte array
 *
 * This API call checks if a specific bit is set in an array of bytes.
 * Bits are counted starting from the least significant bit of the least
 * significant byte.
 *
 * @param[in] bit_num    Number of the bit to be tested, 0..(8*(max_bytes-1))
 * @param[in] p          Pointer to a buffer with an array of bytes
 * @param[in] max_bytes  The number of bytes in the buffer p
 *
 * @return ::MBG_SUCCESS if the bit is set, ::MBG_ERR_NOT_SUPP_BY_DEV if not,
 *         or ::MBG_ERR_RANGE if the bit number is out of the range of the array
 */
int check_byte_array_bit( int bit_num, const uint8_t *p, int max_bytes )
{
  int byte_num = bit_num >> 3;

  if ( byte_num < max_bytes )  // the normal case
  {
    ulong bit_mask = 1UL << ( bit_num & 0x07 );

    return ( p[byte_num] & bit_mask ) ? MBG_SUCCESS : MBG_ERR_NOT_SUPP_BY_DEV;
  }

  return MBG_ERR_RANGE;

}  // check_byte_array_bit



#if _USE_SOCKET_IO

/*HDR*/
/**
 * @brief Close a socket and make the socket descriptor invalid
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure containing the socket descriptor
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
int socket_close( MBG_MSG_CTL *pmctl )
{
  int rc;

  if ( pmctl->st.sockio.sockfd == MBG_INVALID_SOCK_FD )
    return MBG_SUCCESS;

  #if defined( MBG_TGT_CVI ) || defined( MBG_TGT_WIN32 )
    rc = closesocket( pmctl->st.sockio.sockfd );  // returns 0 on success
  #elif defined( MBG_TGT_POSIX )
    rc = close( pmctl->st.sockio.sockfd );  // returns 0 on success
  #else
    #error close socket needs to be implemented for this target
  #endif

  pmctl->st.sockio.sockfd = MBG_INVALID_SOCK_FD;

  if ( rc != 0 )
  {
    rc = mbg_get_last_socket_error( "failed to close socket socket_close" );
    goto out;
  }

  rc = MBG_SUCCESS;

out:
  return rc;

}  // socket_close

#endif  // _USE_SOCKET_IO



/*HDR*/
/**
 * @brief Close a binary communication channel and release resources
 *
 * Closes a binary communication channel which has been opened by one
 * of the mbgextio_open_...() functions and releases the buffers which
 * have been allocated when the channel was opened.
 *
 * The pointer to the message control structure passed by address is set
 * to NULL after the channel has been closed and the resources have
 * been released.
 *
 * @param[in,out] ppmctl Address of a pointer to a message control structure
 *                       created when the communication channel was opened
 *                       by one of the mbgextio_open_...() calls.
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_open_socket
 * @see ::mbgextio_open_serial
 * @see ::mbgextio_open_serial_ftdi
 * @see ::mbgextio_open_usb
 * @see ::mbgextio_open_usb_direct_io
 */
_NO_MBG_API_ATTR int _MBG_API xtiocomm_close_connection( MBG_MSG_CTL **ppmctl )
{
  MBG_MSG_CTL *pmctl = *ppmctl;
  int rc = MBG_ERR_UNSPEC;

  switch ( pmctl->conn_type )
  {
    #if _USE_SERIAL_IO
      case MBG_CONN_TYPE_SERIAL:
        rc = mbgserio_close( &pmctl->st.p_serio );
        break;
    #endif  // _USE_SERIAL_IO

    #if _USE_SERIAL_IO_FTDI
      case MBG_CONN_TYPE_SERIAL_FTDI:
      {
        FT_STATUS status = FT_Close( pmctl->st.ftdi.port_handle );
        pmctl->st.ftdi.port_handle = NULL;  //### TODO FT_Handle is a PVOID
        rc = mbg_ftdi_ft_status_to_mbg( status );
      } break;
    #endif  // _USE_SERIAL_IO_FTDI

    #if _USE_SOCKET_IO
      case MBG_CONN_TYPE_SOCKET:
        rc = socket_close( pmctl );
        break;
    #endif  // _USE_SOCKET_IO

    #if _USE_USB_IO
      case MBG_CONN_TYPE_USB:
        rc = mbgusbio_close( &pmctl->st.usbio );
        break;
    #endif  // _USE_USB_IO

    #if _USE_USB_DIRECT_IO
      case MBG_CONN_TYPE_USB_DIRECT_IO:
        if ( pmctl->st.usbdio.usbdiofd == MBG_USB_DIRECT_IO_INVALID_FD )
        {
          rc = MBG_SUCCESS;
          break;
        }

        rc = close( pmctl->st.usbdio.usbdiofd );

        if ( rc < 0 )
          rc = mbg_get_last_error( "failed to close direct USB I/O device" );
        else
          rc = MBG_SUCCESS;
    #endif // _USE_USB_DIRECT_IO

    default:
      rc = MBG_ERR_CONN_TYPE;

  }  // switch

  #if _USE_MUTEX
    _mbg_mutex_destroy( &pmctl->dev_mutex );
  #endif

  dealloc_msg_ctl( ppmctl );

  return rc;

}  // xtiocomm_close_connection



/*HDR*/
/**
 * @brief Set device poll timeout
 *
 * @param[in,out] pmctl        Pointer to a valid message control structure
 * @param[in]     new_timeout  New poll timeout value [ms]
 *
 * @return Previous poll timeout value [ms]
 *
 * @see ::mbgextio_get_dev_poll_timeout
 * @see ::mbgextio_set_msg_rcv_timeout
 * @see ::mbgextio_get_msg_rcv_timeout
 */
_NO_MBG_API_ATTR ulong _MBG_API xtiocomm_set_dev_poll_timeout( MBG_MSG_CTL *pmctl, ulong new_timeout )
{
  ulong prev_poll_timeout = xtiocomm_get_dev_poll_timeout( pmctl );

  switch ( pmctl->conn_type )
  {
    #if _USE_SOCKET_IO
      case MBG_CONN_TYPE_SOCKET:
        pmctl->st.sockio.poll_timeout = new_timeout;
        break;
    #endif  // _USE_SOCKET_IO

    #if _USE_SERIAL_IO
      case MBG_CONN_TYPE_SERIAL:
        pmctl->st.p_serio->poll_timeout = new_timeout;  //### TODO call library function
        break;
    #endif  // _USE_SERIAL_IO

    #if _USE_SERIAL_IO_FTDI
      case MBG_CONN_TYPE_SERIAL_FTDI:
        pmctl->st.ftdi.poll_timeout = new_timeout;  //### TODO call FTDI library function
        break;
    #endif  // _USE_SERIAL_IO_FTDI

    #if _USE_USB_IO
      case MBG_CONN_TYPE_USB:
        pmctl->st.usbio.poll_timeout = new_timeout;  //### TODO call library function
        break;
    #endif  // _USE_USB_IO

    #if _USE_USB_DIRECT_IO
      case MBG_CONN_TYPE_USB_DIRECT_IO:
        pmctl->st.usbdio.poll_timeout = new_timeout;
        break;
    #endif

  }  // switch

  return prev_poll_timeout;

}  // xtiocomm_set_dev_poll_timeout



/*HDR*/
/**
 * @brief Get device poll timeout
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return Current poll timeout value [ms]
 *
 * @see ::mbgextio_set_dev_poll_timeout
 * @see ::mbgextio_set_msg_rcv_timeout
 * @see ::mbgextio_get_msg_rcv_timeout
 */
_NO_MBG_API_ATTR ulong _MBG_API xtiocomm_get_dev_poll_timeout( const MBG_MSG_CTL *pmctl )
{
  ulong poll_timeout = 0;

  switch ( pmctl->conn_type )
  {
    #if _USE_SOCKET_IO
      case MBG_CONN_TYPE_SOCKET:
        poll_timeout = pmctl->st.sockio.poll_timeout;
        break;
    #endif  // _USE_SOCKET_IO

    #if _USE_SERIAL_IO
      case MBG_CONN_TYPE_SERIAL:
        poll_timeout = pmctl->st.p_serio->poll_timeout;
        break;
    #endif  // _USE_SERIAL_IO

    #if _USE_SERIAL_IO_FTDI
      case MBG_CONN_TYPE_SERIAL_FTDI:
        poll_timeout = pmctl->st.ftdi.poll_timeout;
        break;
    #endif  // _USE_SERIAL_IO_FTDI

    #if _USE_USB_IO
      case MBG_CONN_TYPE_USB:
        poll_timeout = pmctl->st.usbio.poll_timeout;
        break;
    #endif  // _USE_USB_IO

    #if _USE_USB_DIRECT_IO
      case MBG_CONN_TYPE_USB_DIRECT_IO:
        poll_timeout = pmctl->st.usbdio.poll_timeout;
        break;
    #endif

    default:
      break;

  }  // switch

  return poll_timeout;

}  // xtiocomm_get_dev_poll_timeout



/*HDR*/
/**
 * @brief Set message receive timeout value
 *
 * @param[in,out] pmctl        Pointer to a valid message control structure
 * @param[in]     new_timeout  New timeout value [ms]
 *
 * @see ::mbgextio_set_dev_poll_timeout
 * @see ::mbgextio_get_dev_poll_timeout
 * @see ::mbgextio_get_msg_rcv_timeout
 */
_NO_MBG_API_ATTR void _MBG_API xtiocomm_set_msg_rcv_timeout( MBG_MSG_CTL *pmctl, ulong new_timeout )
{
  pmctl->msg_rcv_timeout = new_timeout;

}  // xtiocomm_set_msg_rcv_timeout



/*HDR*/
/**
 * @brief Get message receive timeout value
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return Current timeout value [ms]
 *
 * @see ::mbgextio_set_dev_poll_timeout
 * @see ::mbgextio_get_dev_poll_timeout
 * @see ::mbgextio_set_msg_rcv_timeout
 */
_NO_MBG_API_ATTR ulong _MBG_API xtiocomm_get_msg_rcv_timeout( const MBG_MSG_CTL *pmctl )
{
  return pmctl->msg_rcv_timeout;

}  // xtiocomm_get_msg_rcv_timeout

