
/**************************************************************************
 *
 *  $Id: mbgerror.h 1.13 2017/02/28 15:23:14Z gregoire TEST $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    Error codes used with Meinberg devices and drivers.
 *    The codes can be translated into an OS dependent error code.
 *
 * -----------------------------------------------------------------------
 *  $Log: mbgerror.h $
 *  Revision 1.13  2017/02/28 15:23:14Z  gregoire
 *  error code MBG_ERR_INV_IDX added
 *  Revision 1.12  2017/01/10 15:54:56  philipp
 *  Fixed syntax error
 *  Revision 1.11  2017/01/10 14:26:31  philipp
 *  Added error MBG_ERR_NOT_CONFIGURED
 *  Revision 1.10  2016/12/16 12:40:33  thomas-b
 *  Added MBG_ERR_NO_SPACE
 *  Revision 1.9  2016/10/31 17:41:55  martin
 *  New error code MBG_ERR_DATA_FMT.
 *  Revision 1.8  2016/08/05 12:29:20  martin
 *  Re-enabled some symbols which have been commented out.
 *  Added new codes, and initializers for code/string conversion tables.
 *  Updated doxygen comments.
 *  Updated function prorotypes.
 *  Revision 1.7  2014/05/27 13:32:47Z  martin
 *  Defined additional common error codes which can be
 *  translated from OS specific codes.
 *  Function prototypes from new module mbgerror.c.
 *  Comments in doxygen style.
 *  Revision 1.6  2012/10/02 18:42:26Z  martin
 *  New codes MBG_ERR_N_POUT_EXCEEDS_SUPP and
 *  MBG_ERR_N_UC_MSTR_EXCEEDS_SUPP.
 *  Modified comments for doxygen.
 *  Revision 1.5  2011/03/31 10:56:17  martin
 *  Added MBG_ERR_COPY_TO_USER and MBG_ERR_COPY_FROM_USER.
 *  Revision 1.4  2008/12/05 13:28:50  martin
 *  Added new code MBG_ERR_IRQ_UNSAFE.
 *  Revision 1.3  2008/02/26 14:50:14Z  daniel
 *  Added codes:
 *  MBG_ERR_NOT_SUPP_ON_OS, MBG_ERR_LIB_NOT_COMPATIBLE,
 *  MBG_ERR_N_COM_EXCEEDS_SUPP, MBG_ERR_N_STR_EXCEEDS_SUPP
 *  Added doxygen compatible comments.
 *  Revision 1.2  2007/09/27 07:26:22Z  martin
 *  Define STATUS_SUCCESS for Windows if not in kernel mode.
 *  Revision 1.1  2007/09/26 08:08:54Z  martin
 *  Initial revision.
 *
 **************************************************************************/

#ifndef _MBGERROR_H
#define _MBGERROR_H


/* Other headers to be included */

#include <mbg_tgt.h>
#include <words.h>

#ifdef _MBGERROR
 #define _ext
 #define _DO_INIT
#else
 #define _ext extern
#endif


/* Start of header body */

#if !defined( MBG_TGT_WIN32 ) || defined( MBG_TGT_KERNEL )
  //### TODO Surprisingly we need this also for Windows
  // in kernel mode. This should be fixed.
  #define DWORD  uint32_t  // just to avoid compiler errors
#endif



/**
 * @brief Error codes used with Meinberg devices and drivers
 *
 * Appropriate error strings can be retrieved via the ::mbg_strerror function.
 *
 * Under Windows, some message strings are provided as resources appended
 * to the mbgctrl DLL, but the codes specified here have to be translated
 * to Windows-specific error codes before the appropriate resource string
 * can be retrieved. Actually this is done by taking the absolute number
 * of an error code and have it or'ed with 0xE0000000 afterwards, e.g.
 * ::MBG_ERR_GENERIC (-19) will yield Windows code 0xE0000013.
 * * See ::_mbg_err_to_os
 *
 * @see ::MBG_ERR_NAMES_ENG
 *
 * @anchor MBG_RETURN_CODES @{ */

// NOTE: Some of these codes have to match codes which are defined in pcpsdefs.h
// and returned by the firmware of bus-level devices, so the definitions
// must *not* be renumbered.

#define MBG_SUCCESS           0  ///< no error, has to match ::PCPS_SUCCESS

 /** @anchor MBG_ERROR_CODES @{ */

// Other codes which have to match codes defined in pcpsdefs.h returned by bus-level devices
#define MBG_ERR_STIME        -1  ///< tried to write invalid date/time/status to device, has to match ::PCPS_ERR_STIME
#define MBG_ERR_CFG          -2  ///< tried to write invalid configuration parameters to device, has to match ::PCPS_ERR_CFG (see also ::MBG_ERR_INV_CFG)


// Codes returned by low level functions of the bus-level device driver
#define MBG_ERR_GENERIC     -19  ///< generic error
#define MBG_ERR_TIMEOUT     -20  ///< timeout accessing the device
#define MBG_ERR_FW_ID       -21  ///< invalid firmware ID
#define MBG_ERR_NBYTES      -22  ///< the number of parameter bytes passed to the device did not
                                 ///< match the number of bytes expected by the device

#define MBG_ERR_INV_TIME    -23  ///< the device doesn't have valid time
#define MBG_ERR_FIFO        -24  ///< the data FIFO of a bus-level device is empty, though it shouldn't be
#define MBG_ERR_NOT_READY   -25  ///< bus-level device is temp. unable to respond e.g. during init. after RESET
#define MBG_ERR_INV_TYPE    -26  ///< bus-level device didn't recognize data type


// Codes returned by the high level API functions
#define MBG_ERR_NO_MEM              -27  ///< failed to allocate memory
#define MBG_ERR_CLAIM_RSRC          -28  ///< failed to claim port or mem resource
#define MBG_ERR_DEV_NOT_SUPP        -29  ///< specified device type not supported by driver
#define MBG_ERR_INV_DEV_REQUEST     -30  ///< IOCTL call not supported by driver
#define MBG_ERR_NOT_SUPP_BY_DEV     -31  ///< cmd or feature not supported by device
#define MBG_ERR_USB_ACCESS          -32  ///< USB access failed
#define MBG_ERR_CYCLIC_TIMEOUT      -33  ///< cyclic event (IRQ, etc.) didn't occur
#define MBG_ERR_NOT_SUPP_ON_OS      -34  ///< function is not supported under this operating system
#define MBG_ERR_LIB_NOT_COMPATIBLE  -35  ///< installed shared lib. version not compat. with version used at build time
#define MBG_ERR_N_COM_EXCEEDS_SUPP  -36  ///< num. COM ports of the device exceeds max. supp. by driver
#define MBG_ERR_N_STR_EXCEEDS_SUPP  -37  ///< num. string formats of the device exceeds max. supp. by driver
#define MBG_ERR_IRQ_UNSAFE          -38  ///< enabled IRQ of bus-level device is unsafe with this firmware/ASIC version
#define MBG_ERR_N_POUT_EXCEEDS_SUPP -39  ///< num. prog. outputs of the device exceeds max. supp. by driver

// Legacy codes used with DOS TSRs only:
#define MBG_ERR_INV_INTNO           -40  ///< invalid interrupt number
#define MBG_ERR_NO_DRIVER           -41  ///< a driver could not be found
#define MBG_ERR_DRV_VERSION         -42  ///< the driver is too old


#define MBG_ERR_COPY_TO_USER        -43  ///< kernel driver failed to copy data from kernel to user space
#define MBG_ERR_COPY_FROM_USER      -44  ///< kernel driver failed to copy data from use to kernel space


// More codes returned by the driver's high level functions:
#define MBG_ERR_N_UC_MSTR_EXCEEDS_SUPP -45  ///< num. PTP unicast masters of the device exceeds max. supp. by driver
#define MBG_ERR_N_GNSS_EXCEEDS_SUPP    -46  ///< num. of GNSS systems supp. by device exceeds max. supp. by driver
#define MBG_ERR_N_GPIO_EXCEEDS_SUPP    -47  ///< num. of GPIO ports supp. by device exceeds max. supp. by driver
#define MBG_ERR_N_XMR_EXCEEDS_SUPP     -48  ///< num. of XMR sources supp. by device exceeds max. supp. by driver

#define MBG_ERR_UNSPEC              -60  ///< unspecified error

#define MBG_ERR_HDR_CSUM            -61  ///< binary protocol header checksum error
#define MBG_ERR_DATA_CSUM           -62  ///< binary protocol data checksum error
#define MBG_ERR_RCVD_NACK           -63  ///< binary protocol received reply msg with a NACK code
#define MBG_ERR_RCVD_NO_ACK         -64  ///< binary protocol received reply msg without expected ACK code //### TODO
#define MBG_ERR_CONN_TYPE           -65  ///< binary protocol no valid/supported connection type specified
#define MBG_ERR_BYTES_WRITTEN       -66  ///< binary protocol failed to write all bytes
#define MBG_ERR_AUTH                -67  ///< binary protocol failed authentication

#define MBG_ERR_SOCK_INIT           -68  ///< socket interface not initialized, or failed to initialize
#define MBG_ERR_INV_SOCK_FD         -69  ///< invalid socket when tried to open network socket
#define MBG_ERR_NOT_A_SOCKET        -70  ///< socket descriptor is not a socket
#define MBG_ERR_NBLOCK_WAIT_SLCT    -71  ///< select timed out when waiting for non-blocking network port to become ready
#define MBG_ERR_NBLOCK_WAIT_WR_FD   -72  ///< write fd not set after select when waiting for non-blocking network port to become ready

#define MBG_ERR_IO                  -73  ///< generic I/O error
#define MBG_ERR_INV_PARM            -74  ///< invalid parameter
#define MBG_ERR_NO_DEV              -75  ///< specified device not found
#define MBG_ERR_NOT_FOUND           -76  ///< specified item not found

#define MBG_ERR_OVERFLOW            -77  ///< range or buffer overflow
#define MBG_ERR_PIPE                -78  ///< pipe error
#define MBG_ERR_INTR                -79  ///< interrupted system call
#define MBG_ERR_ACCESS              -80  ///< access denied, e.g. when trying to access a device
#define MBG_ERR_PERM                -81  ///< operation not permitted, e.g. when trying to set the system time
#define MBG_ERR_BUSY                -82  ///< device busy
#define MBG_ERR_INV_HANDLE          -83  ///< invalid file/device handle specified

#define MBG_ERR_XBP_CASC_LVL        -84  ///< too many XBP cascading levels
#define MBG_ERR_ENCRYPT             -85  ///< encryption failed
#define MBG_ERR_DECRYPT             -86  ///< decryption failed

#define MBG_ERR_DISCONN             -87  ///< connection closed by remote site / host
#define MBG_ERR_INV_CFG             -88  ///< invalid/inconsistent configuration parameters read from device, see also ::MBG_ERR_CFG
#define MBG_ERR_RANGE               -89  ///< input parameter was out of range

#define MBG_ERR_INV_TLV_ANN_BYTES   -90  ///< number of announced TLV bytes doesn't match number of transferred bytes
#define MBG_ERR_INV_TLV_SIZE        -91  ///< ### TODO
#define MBG_ERR_INV_TLV_UID         -92  ///< ### TODO

#define MBG_ERR_EXIST               -93  ///< file exists
#define MBG_ERR_DATA_SIZE           -94  ///< the received data size toesn't match the expected data size
#define MBG_ERR_NO_ENTITY           -95  ///< no such entity  //### TODO or use MBG_ERR_NO_DEV / MBG_ERR_NOT_FOUND ?
#define MBG_ERR_ALREADY_ALLOC       -96  ///< pointer already allocated when trying to allocate memory
#define MBG_ERR_HOST_NOT_FOUND      -97  ///< host not found
#define MBG_ERR_CONN_RESET          -98  ///< connection reset by peer
#define MBG_ERR_DATA_FMT            -99  ///< invalid data format

#define MBG_ERR_NO_SPACE            -100 ///< insufficient disk space left on the device
#define MBG_ERR_NOT_CONFIGURED      -101 ///< configuration option is not active/configured
#define MBG_ERR_INV_IDX             -102 ///< invalid index value used

// NOTE: New codes have to be appended to this list, and the sequence of codes must not
// be changed. Whenever new codes have been defined, appropriate entries have to be added
// to the ::MBG_ERR_NAMES_ENG table initializer below, and the Windows-specific message
// texts specified in messages.mc/.h from which the resources appended to mbgsvctl.dll
// are generated have to be updated accordingly.

/** @} anchor MBG_ERROR_CODES */

/** @} anchor MBG_RETURN_CODES */


/**
 * @brief Strings associated with @ref MBG_RETURN_CODES
 *
 * @see @ref MBG_RETURN_CODES
 */
#define MBG_ERR_NAMES_ENG                                                                               \
{                                                                                                       \
  { MBG_SUCCESS,                     "Success, no error" },                                             \
  { MBG_ERR_STIME,                   "Invalid date/time for device" },                                  \
  { MBG_ERR_CFG,                     "Invalid configuration parameters for device" },                   \
  { MBG_ERR_GENERIC,                 "Generic error" },                                                 \
  { MBG_ERR_TIMEOUT,                 "Timeout" },                                                       \
  { MBG_ERR_FW_ID,                   "Invalid firmware ID" },                                           \
  { MBG_ERR_NBYTES,                  "Unexpected number of data bytes for this API" },                  \
  { MBG_ERR_INV_TIME,                "Invalid time passed to device" },                                 \
  { MBG_ERR_FIFO,                    "FIFO unexpectedly empty" },                                       \
  { MBG_ERR_NOT_READY,               "Device not ready" },                                              \
  { MBG_ERR_INV_TYPE,                "Unsupported data type" },                                         \
  { MBG_ERR_NO_MEM,                  "Memory allocation error" },                                       \
  { MBG_ERR_CLAIM_RSRC,              "Faild to claim resources" },                                      \
  { MBG_ERR_DEV_NOT_SUPP,            "Device not supported" },                                          \
  { MBG_ERR_INV_DEV_REQUEST,         "Request not supported" },                                         \
  { MBG_ERR_NOT_SUPP_BY_DEV,         "Not supported by device" },                                       \
  { MBG_ERR_USB_ACCESS,              "USB access failed" },                                             \
  { MBG_ERR_CYCLIC_TIMEOUT,          "Cyclic message timeout" },                                        \
  { MBG_ERR_NOT_SUPP_ON_OS,          "Not supported by OS" },                                           \
  { MBG_ERR_LIB_NOT_COMPATIBLE,      "Shared lib not compatible" },                                     \
  { MBG_ERR_N_COM_EXCEEDS_SUPP,      "Num. COM ports exceeds supported" },                              \
  { MBG_ERR_N_STR_EXCEEDS_SUPP,      "Num. string formats exceeds supported" },                         \
  { MBG_ERR_IRQ_UNSAFE,              "Unsafe IRQ support" },                                            \
  { MBG_ERR_N_POUT_EXCEEDS_SUPP,     "Num prog. outputs exceeds supported" },                           \
  { MBG_ERR_INV_INTNO,               "Invalid interrupt number" },                                      \
  { MBG_ERR_NO_DRIVER,               "Driver not found" },                                              \
  { MBG_ERR_DRV_VERSION,             "Driver too old" },                                                \
  { MBG_ERR_COPY_TO_USER,            "Error copying to user space" },                                   \
  { MBG_ERR_COPY_FROM_USER,          "Error copying from user space" },                                 \
  { MBG_ERR_N_UC_MSTR_EXCEEDS_SUPP,  "Num. PTP Unicast masters exceeds supported" },                    \
  { MBG_ERR_N_GNSS_EXCEEDS_SUPP,     "Num. GNSS systems exceeds supported" },                           \
  { MBG_ERR_N_GPIO_EXCEEDS_SUPP,     "Num. GPIO ports exceeds supported" },                             \
  { MBG_ERR_N_XMR_EXCEEDS_SUPP,      "Num. XMR sources exceeds supported" },                            \
  { MBG_ERR_UNSPEC,                  "Unspecified error" },                                             \
  { MBG_ERR_HDR_CSUM,                "Header checksum error" },                                         \
  { MBG_ERR_DATA_CSUM,               "Data checksum error" },                                           \
  { MBG_ERR_RCVD_NACK,               "Received NACK message" },                                         \
  { MBG_ERR_RCVD_NO_ACK,             "Didn't receive ACK message" },                                    \
  { MBG_ERR_CONN_TYPE,               "Invalid I/O connection type" },                                   \
  { MBG_ERR_BYTES_WRITTEN,           "Failed to write all bytes" },                                     \
  { MBG_ERR_AUTH,                    "Authentication failed" },                                         \
  { MBG_ERR_SOCK_INIT,               "Failed to initialize socket" },                                   \
  { MBG_ERR_INV_SOCK_FD,             "Invalid socket descriptor" },                                     \
  { MBG_ERR_NOT_A_SOCKET,            "Not a socket descriptor" },                                       \
  { MBG_ERR_NBLOCK_WAIT_SLCT,        "Select timed out waiting for port ready" },                       \
  { MBG_ERR_NBLOCK_WAIT_WR_FD,       "Write file descriptor not ready after waiting for port ready" },  \
  { MBG_ERR_IO,                      "Generic I/O error" },                                             \
  { MBG_ERR_INV_PARM,                "Invalid parameter" },                                             \
  { MBG_ERR_NO_DEV,                  "Specified device not found" },                                    \
  { MBG_ERR_NOT_FOUND,               "Specified item not found" },                                      \
  { MBG_ERR_OVERFLOW,                "Buffer overflow" },                                               \
  { MBG_ERR_PIPE,                    "Pipe error" },                                                    \
  { MBG_ERR_INTR,                    "Interrupted system call" },                                       \
  { MBG_ERR_ACCESS,                  "Access denied" },                                                 \
  { MBG_ERR_PERM,                    "Operation not permitted" },                                       \
  { MBG_ERR_BUSY,                    "Device busy" },                                                   \
  { MBG_ERR_INV_HANDLE,              "Invalid handle" },                                                \
  { MBG_ERR_XBP_CASC_LVL,            "Too many XBP cascading levels" },                                 \
  { MBG_ERR_ENCRYPT,                 "Encryption failed" },                                             \
  { MBG_ERR_DECRYPT,                 "Decryption failed" },                                             \
  { MBG_ERR_DISCONN,                 "Connection closed by remote site/host" },                         \
  { MBG_ERR_INV_CFG,                 "Invalid/inconsistent configuration read from device" },           \
  { MBG_ERR_RANGE,                   "Input parameter was out of range" },                              \
  { MBG_ERR_INV_TLV_ANN_BYTES,       "TLV num of transferred bytes differs from num of announced bytes" }, \
  { MBG_ERR_INV_TLV_SIZE,            "MBG_ERR_INV_TLV_SIZE" },  /* ### TODO */                          \
  { MBG_ERR_INV_TLV_UID,             "MBG_ERR_INV_TLV_UID" },   /* ### TODO */                          \
  { MBG_ERR_EXIST,                   "File exists" },                                                   \
  { MBG_ERR_DATA_SIZE,               "Received data size mismatch" },                                   \
  { MBG_ERR_NO_ENTITY,               "No such entity" },                                                \
  { MBG_ERR_ALREADY_ALLOC,           "Memory already allocated" },                                      \
  { MBG_ERR_HOST_NOT_FOUND,          "Host not found" },                                                \
  { MBG_ERR_CONN_RESET,              "Connection reset by peer" },                                      \
  { MBG_ERR_DATA_FMT,                "Invalid data format" },                                           \
  { MBG_ERR_NO_SPACE,                "Insufficient disk space" },                                       \
  { MBG_ERR_NOT_CONFIGURED,          "Configuration is not active and/or configured" },                 \
  { MBG_ERR_INV_IDX,                 "Invalid or unsupported index value used"},                        \
  { 0, NULL }  /* end of table */ \
}



#if defined( __mbg_inline )

static __mbg_inline
/**
 * @brief Check if the code returned by a function indicates an error
 *
 * @param[in] rc  One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbg_rc_is_success
 * @see @ref MBG_RETURN_CODES
 */
bool mbg_rc_is_error( int rc )
{
  // Meinberg error codes are all < 0.
  return rc < MBG_SUCCESS;

}  // mbg_rc_is_error


static __mbg_inline
/**
 * @brief Check if the code returned by a function indicates success
 *
 * @param[in] rc  One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbg_rc_is_error
 * @see @ref MBG_RETURN_CODES
 */
bool mbg_rc_is_success( int rc )
{
  // There are functions which don't only return MBG_SUCCESS
  // on success but some arbitrary positive number, e.g. the
  // number of bytes sent. So success just means "not an error".
  return !mbg_rc_is_error( rc );

}  // mbg_rc_is_success

#else

  #define mbg_rc_is_error( _rc )     ( (_rc) < MBG_SUCCESS )
  #define mbg_rc_is_success( _rc )   ( !mbg_rc_is_error( _rc ) )

#endif



#if defined( MBG_TGT_WIN32 )

  // Windows-specific codes and code conversion

  #if !defined( STATUS_SUCCESS )  // not in kernel mode
    #define STATUS_SUCCESS  0
  #endif

  #define _mbg_err_to_os( _c ) \
  ( ( (_c) == MBG_SUCCESS ) ?  STATUS_SUCCESS : ( abs( _c ) | 0xE0000000 ) )

#else

  #define _mbg_err_to_os( _c )   (_c )   //### TODO remove this

#endif



/* function prototypes: */

#ifdef __cplusplus
extern "C" {
#endif

/* ----- function prototypes begin ----- */

/* This section was generated automatically */
/* by MAKEHDR, do not remove the comments. */

 /**
 * @brief Return an error string associated with the @ref MBG_ERROR_CODES
 *
 * @param[in] mbg_errno  One of the @ref MBG_ERROR_CODES
 *
 * @return  A constant string describing the error, or NULL for unknown error codes
 */
 const char *mbg_strerror( int mbg_errno ) ;

 int mbg_ioctl_err( int rc, const char *descr ) ;
 /**
 * @brief Translate an error code from the Labwindows/CVI RS-232 library to one of the @ref MBG_ERROR_CODES
 *
 * @param[in] cvi_rc  An error code returned by a CVI RS-232 library function
 * @param[in] info    An optional informational text string, or NULL
 *
 * @return  One of the @ref MBG_ERROR_CODES
 *
 * @see http://zone.ni.com/reference/en-XX/help/370051V-01/cvi/libref/cvirs232_error_conditions/
 */
 int mbg_cvi_rs232_error_to_mbg( int cvi_rc, const char *info ) ;

 /**
 * @brief Translate a Windows non-socket API error code to one of the @ref MBG_ERROR_CODES
 *
 * @param[in] last_err  A Windows non-socket API error code as returned by GetLastError()
 * @param[in] info      An optional informational text string, or NULL
 *
 * @return  One of the @ref MBG_ERROR_CODES
 */
 int mbg_win32_last_err_to_mbg( DWORD last_err, const char *info ) ;

 /**
 * @brief Translate a Windows socket API error code to one of the @ref MBG_ERROR_CODES
 *
 * @param[in] wsa_err  A Windows socket API error code as returned by WSAGetLastError()
 * @param[in] info     An optional informational text string, or NULL
 *
 * @return  One of the @ref MBG_ERROR_CODES
 */
 int mbg_win32_wsa_err_to_mbg( DWORD wsa_err, const char *info ) ;

 /**
 * @brief Translate a POSIX errno error code to one of the @ref MBG_ERROR_CODES
 *
 * @param[in] posix_errno  A POSIX error code as usually defined in errno.h
 * @param[in] info         An optional informational text string, or NULL
 *
 * @return  One of the @ref MBG_ERROR_CODES
 */
 int mbg_posix_errno_to_mbg( int posix_errno, const char *info ) ;

 /**
 * @brief Translate a POSIX h_errno error code to one of the @ref MBG_ERROR_CODES
 *
 * This function is specific to translate error codes returned by
 * gethostbyname() and gethostbyaddr(). In case of error these functions
 * don't set errno but h_errno to a specific value.
 *
 * The functions gethostbyname() and gethostbyaddr() are obsolete,
 * and getaddressinfo() should be used preferably.
 *
 * @param[in] posix_h_errno  An error code as usually defined in netdb.h
 * @param[in] info           An optional informational text string, or NULL
 *
 * @return  One of the @ref MBG_ERROR_CODES
 */
 int mbg_posix_h_errno_to_mbg( int posix_h_errno, const char *info ) ;

 /**
 * @brief Get and translate last error after non-socket function call
 *
 * Retrieve the "last error" code after a non-socket function has been called
 * and translate to one of the @ref MBG_ERROR_CODES.
 *
 * On POSIX systems the "last error" code is always stored in errno, but
 * e.g. under Windows the "last error" code after a socket function
 * has to be retrieved by calling WSAGetLastError(), whereas the "last error"
 * code from non-socket POSIX-like functions has to be retrieved
 * by calling GetLastError().
 *
 * @param[in] info  An optional informational text string, or NULL
 *
 * @return  One of the @ref MBG_ERROR_CODES
 */
 int mbg_get_last_error( const char *info ) ;

 /**
 * @brief Get and translate last error after socket function call
 *
 * Retrieve the "last error" code after a socket function has been called
 * and translate to one of the @ref MBG_ERROR_CODES.
 *
 * On POSIX systems the "last error" code is always stored in errno, but
 * e.g. under Windows the "last error" code after a socket function
 * has to be retrieved by calling WSAGetLastError, whereas the "last error"
 * code from non-socket POSIX-like functions is stored in errno as usual.
 *
 * @param[in] info  An optional informational text string, or NULL
 *
 * @return  One of the @ref MBG_ERROR_CODES
 */
 int mbg_get_last_socket_error( const char *info ) ;

 /**
 * @brief Retrieve and convert last error after gethostbyname()
 *
 * This function is specific to retrieve and translate error codes
 * returned by gethostbyname() and gethostbyaddr(). In case of error
 * these functions don't set errno but h_errno on POSIX systems, but
 * under Windows the error code can be retrieved by WSAGetLastError()
 * as usual.
 *
 * The functions gethostbyname() and gethostbyaddr() are obsolete,
 * and getaddressinfo() should be used preferably.
 *
 * @param[in] info  An optional informational text string, or NULL
 *
 * @return  One of the @ref MBG_ERROR_CODES
 */
 int mbg_get_gethostbyname_error( const char *info ) ;

 /**
 * @brief Retrieve and convert last zlib internal error code
 *
 * @param[in] zlib_error  zlib internal error code
 * @param[in] info        An optional informational text string, or NULL
 * @param[in] msg         An optional zlib specific error msg, or NULL.
 *                        Struct z_stream contains member msg.
 *
 * @return  One of the @ref MBG_ERROR_CODES
 */
 int mbg_zlib_error_to_mbg( int zlib_error, const char *info, const char *msg ) ;


/* ----- function prototypes end ----- */

#ifdef __cplusplus
}
#endif


/* End of header body */

#undef _ext
#undef _DO_INIT

#endif  /* _MBGERROR_H */
