
/**************************************************************************
 *
 *  $Id: extiohlp.c 1.3.1.4.1.129 2017/04/12 07:59:04Z martin TEST $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    Device configuration helper functions. This is an extension to
 *    mbgextio.c providing useful functions to simplify reading/writing
 *    complex device configuration structure sets.
 *
 *  Warning:
 *    These functions should not be implemented in a DLL / shared library
 *    since the parameter sizes might vary with different versions
 *    of the API calls, which which would make different versions of
 *    precompiled libraries incompatible to each other.
 *
 * -----------------------------------------------------------------------
 *  $Log: extiohlp.c $
 *  Revision 1.3.1.4.1.129  2017/04/12 07:59:04Z  martin
 *  Fixed build under DOS.
 *  Revision 1.3.1.4.1.128  2017/04/11 15:16:26Z  martin
 *  Revision 1.3.1.4.1.127  2017/04/11 13:08:56Z  martin
 *  Use symbolic code instead of hardcoded number.
 *  Revision 1.3.1.4.1.126  2017/04/06 12:34:28Z  thomas-b
 *  Copy TLV header before and after each chunk transmission due to swab and encryption
 *  Revision 1.3.1.4.1.125  2017/04/04 10:06:03  thomas-b
 *  Added transactions for monitoring and events
 *  Revision 1.3.1.4.1.124  2017/03/29 12:33:26  philipp
 *  Extended event info and event status index structures by data pointer
 *  Revision 1.3.1.4.1.123  2017/03/27 10:37:41  thomas-b
 *  Added function mbgextio_get_all_ptp_v1_common_datasets
 *  Revision 1.3.1.4.1.122  2017/03/22 13:54:05  thomas-b
 *  Added transaction to mbgextio_get_all_io_port_status
 *  Revision 1.3.1.4.1.121  2017/03/22 09:46:05  thomas-b
 *  Added transaction to mbgextio_get_all_io_port_info and function mbgextio_save_all_io_port_info
 *  Revision 1.3.1.4.1.120  2017/03/20 13:34:49  martin
 *  Fixed clang warnings.
 *  Revision 1.3.1.4.1.119  2017/03/20 10:10:02  martin
 *  Fixed build without _PRELIMINARY_CODE.
 *  Revision 1.3.1.4.1.118  2017/03/14 12:39:26  thomas-b
 *  Do not write ref clock settings if NTP_MSK_FIXED_REFCLOCKS is set
 *  Revision 1.3.1.4.1.117  2017/03/14 10:53:56  thomas-b
 *  Fixed bug in NTP save function
 *  Revision 1.3.1.4.1.116  2017/03/08 13:35:20  thomas-b
 *  Do not read more than num_peers, instead of n_supp_peers when getting all NTP status
 *  Revision 1.3.1.4.1.115  2017/03/08 12:37:51  thomas-b
 *  Added set bit for transaction type in mbgextio_begin_transaction
 *  Revision 1.3.1.4.1.114  2017/03/03 08:29:27  thomas-b
 *  Adapted get and save all NTP functions to new ALL_NTP_CFG_INFO design
 *  Revision 1.3.1.4.1.113  2017/03/02 15:46:32  thomas-b
 *  Set empty peer settings for all supported but not configured peers for compatibility with N2x
 *  Revision 1.3.1.4.1.112  2017/03/02 15:33:10  thomas-b
 *  Do not read/write more than num_peers from NTP_CLNT_MODE_SETTINGS
 *  Revision 1.3.1.4.1.111  2017/02/28 15:21:49  gregoire
 *  peer_settings_idx in ntp_cfg_info renamed
 *  Revision 1.3.1.4.1.110  2017/02/24 09:38:25  philipp
 *  Do not use fixed buffers for monitorung event info and status
 *  Revision 1.3.1.4.1.109  2017/02/22 08:32:19  thomas-b
 *  Extended get and set of ALL_MONITORING_INFO by event structures
 *  Added get function for ALL_MONITORING_STATUS
 *  Revision 1.3.1.4.1.108  2017/02/16 12:59:51  thomas-b
 *  Renamed mbgextio_get_ptp_v2_common_datasets to mbgextio_get_all_ptp_v2_common_datasets
 *  Use ALL_PTP_V2_COMMON_DATASETS structure instead of MBG_PTP_V2_COMMON_DATASETS
 *  Revision 1.3.1.4.1.107  2017/02/16 10:19:43  martin
 *  *** empty log message ***
 *  Revision 1.3.1.4.1.106  2017/02/16 09:32:32  thomas-b
 *  Added function mbgextio_get_ptp_v2_common_datasets
 *  Revision 1.3.1.4.1.105  2017/02/15 14:09:35  martin
 *  Test, the previous version eventually contained mixed line endings.
 *  Revision 1.3.1.4.1.104  2017/02/08 13:15:43  thomas-b
 *  Added functions to get and save all monitoring, especially SNMP, configuration
 *  Revision 1.3.1.4.1.103  2016/12/08 14:39:31  martin
 *  Made XBP_ADDR * parameter const.
 *  Revision 1.3.1.4.1.102  2016/11/29 14:44:20Z  philipp
 *  Fixed memleaks
 *  Revision 1.3.1.4.1.101  2016/11/24 07:06:33  philipp
 *  Honour big endian system when swapping bytes
 *  Revision 1.3.1.4.1.100  2016/11/22 12:40:46  philipp
 *  Added I/O port helper functions
 *  Revision 1.3.1.4.1.99  2016/11/21 15:15:53  martin
 *  Define a default for UINT8_MAX if it's not defined elsewhere.
 *  Revision 1.3.1.4.1.98  2016/11/01 14:58:09Z  udo
 *  *** empty log message ***
 *  Revision 1.3.1.4.1.97  2016/11/01 09:30:48  thomas-b
 *  Fixed mbgextio subfunction calls with fix NULL XBP addr
 *  Revision 1.3.1.4.1.96  2016/11/01 09:19:29  martin
 *  Use new data types and account for structure fields which
 *  have been renamed to reduce ambiguity.
 *  Updated doxygen comments.
 *  Revision 1.3.1.4.1.95  2016/10/25 07:48:27  martin
 *  Doxygen fixes.
 *  Revision 1.3.1.4.1.94  2016/10/21 09:39:43  thomas-b
 *  Added function mbgextio_get_all_xbp_info
 *  Revision 1.3.1.4.1.93  2016/10/20 09:14:59  thomas-b
 *  Removed debug code
 *  Revision 1.3.1.4.1.92  2016/10/19 14:33:06  thomas-b
 *  Added functions mbgextio_get_all_ucap_net_info and mbgextio_save_all_ucap_net_info
 *  Revision 1.3.1.4.1.91  2016/10/06 10:10:39  thomas-b
 *  Added several security checks for older network API
 *  Revision 1.3.1.4.1.90  2016/10/06 07:12:35  thomas-b
 *  Set index before setting link, route and addr settings
 *  Revision 1.3.1.4.1.89  2016/10/04 07:54:34  thomas-b
 *  Fixed reading and settings of DNS server and search domain status and config
 *  Revision 1.3.1.4.1.88  2016/09/30 11:11:22  thomas-b
 *  If NET_CFG_API stage 2 is not supported, check if DNS server or DNS search domain is empty and decrease num
 *  Revision 1.3.1.4.1.87  2016/09/29 14:35:10  thomas-b
 *  Moved new get and set functions for NET_CFG_API stage 2 to mbgextio
 *  Revision 1.3.1.4.1.86  2016/09/29 13:38:11  thomas-b
 *  Fix for LAN_IP4 compatibility
 *  Revision 1.3.1.4.1.85  2016/09/29 12:16:36  thomas-b
 *  Added set functions for NET_CFG_API stage 2
 *  Revision 1.3.1.4.1.84  2016/09/29 09:09:02  thomas-b
 *  Added more functions for NET_CFG_API stage 2, especially status
 *  Revision 1.3.1.4.1.83  2016/09/29 06:12:38  philipp
 *  Added support for beginning / ending typed transactions
 *  Revision 1.3.1.4.1.82  2016/09/29 05:56:55  thomas-b
 *  Added first functions for NET_CFG_API stage 2
 *  Revision 1.3.1.4.1.81  2016/09/27 09:26:08  udo
 *  fixed wrong rcv- and xmt- buffer handling in mbgextio_req_generic_file()
 *  Revision 1.3.1.4.1.79  2016/09/16 08:28:50  thomas-b
 *  Fixed conversion warning by adding cast to uint8_t
 *  Revision 1.3.1.4.1.78  2016/09/15 09:20:44  thomas-b
 *  Fixed two bugs in mbgextio_get_all_net_status_info
 *  Revision 1.3.1.4.1.77  2016/08/16 12:48:30  gregoire.diehl
 *  CVI fixes
 *  Revision 1.3.1.4.1.76  2016/08/11 11:28:10Z  martin
 *  *** empty log message ***
 *  Revision 1.3.1.4.1.75  2016/08/09 07:09:55  martin
 *  *** empty log message ***
 *  Revision 1.3.1.4.1.74  2016/08/03 08:06:56  thomas-b
 *  Adapted net_cfg functions to the structure changes
 *  Revision 1.3.1.4.1.73  2016/07/07 15:04:33  martin
 *  Fixed compiler warning.
 *  Revision 1.3.1.4.1.72  2016/06/23 09:24:01  thomas-b
 *  Adapted net_cfg functions to the new structure of MBG_NET_GLB_CFG_SETTINGS
 *  Revision 1.3.1.4.1.71  2016/06/21 12:06:46  thomas-b
 *  Moved mbgextio_calloc_ucap_entry to cfg_hlp so it can be used in deviohlp as well
 *  Revision 1.3.1.4.1.70  2016/06/21 08:41:29  thomas-b
 *  Removed old mbgextio_get_all_ucap_info function and static flag from mbgextio_calloc_ucap_entry
 *  Revision 1.3.1.4.1.69  2016/06/21 07:39:22  philipp
 *  Use mbg_klist for ucap events and not an array -> Easier to handle MAX_UCAP_ENTRIES
 *  Revision 1.3.1.4.1.68  2016/06/15 14:03:56  thomas-b
 *  Added function mbgextio_get_all_ucap_info
 *  Revision 1.3.1.4.1.67  2016/06/03 11:09:43  thomas-b
 *  Fixed Windows compatibility issues in mbgextio_xmt_file
 *  Revision 1.3.1.4.1.66  2016/06/01 10:07:39  daniel
 *  Fix using _PRELIMINARY_CODE define for route_infos
 *  Revision 1.3.1.4.1.65  2016/06/01 06:04:25  philipp
 *  Added compiler warning as reminder
 *  Revision 1.3.1.4.1.64  2016/05/31 11:35:45  philipp
 *  Extended ALL_XMULTI_REF_STATUS by flags if at least one ref type supports stats and/or metrics
 *  Revision 1.3.1.4.1.63  2016/05/31 06:32:27  thomas-b
 *  Initialize rc in mbgextio_get_all_xmulti_ref_status
 *  Revision 1.3.1.4.1.62  2016/05/30 14:25:41  daniel
 *  Added missing code for HPS firmware rollback
 *  Revision 1.3.1.4.1.61  2016/05/27 05:48:08  philipp
 *  Refactoring to support XMR_METRICS properly
 *  Revision 1.3.1.4.1.60  2016/05/26 10:59:26  thomas-b
 *  Added parameter ALL_NTP_CFG_INFO to mbgextio_get_all_ntp_status
 *  Moved check functions of specific features to cfg_hlp
 *  Revision 1.3.1.4.1.59  2016/05/25 08:43:36  philipp
 *  Redesign of ALL_[xxx]_[XXX] structures and (helper) functions
 *  Revision 1.3.1.4.1.58  2016/05/24 07:26:19  philipp
 *  Use calloc instead of realloc (and only if needed) when fetching structures 'ALL_[xxx]_[xxx]'
 *  Revision 1.3.1.4.1.57  2016/05/23 09:46:52  philipp
 *  Increase performance
 *  Revision 1.3.1.4.1.56  2016/05/23 09:37:41  philipp
 *  Extended ALL_XMULTI_REF_STATUS by holdover_status
 *  Revision 1.3.1.4.1.55  2016/05/23 08:58:01  philipp
 *  New function mbgextio_get_all_gpio_state
 *  Revision 1.3.1.4.1.54  2016/05/23 08:27:51  philipp
 *  Fixed compiler warning
 *  Revision 1.3.1.4.1.53  2016/05/23 08:24:16  philipp
 *  New function mbgextio_get_all_ims_state
 *  Revision 1.3.1.4.1.52  2016/05/11 14:38:49  thomas-b
 *  Changed link status evaluation in mbgextio_get_all_net_status_info
 *  Revision 1.3.1.4.1.51  2016/05/11 13:59:37  thomas-b
 *  Adapted comments and fixed bug in mbgextio_get_all_net_status_info
 *  Revision 1.3.1.4.1.50  2016/05/11 13:21:34  thomas-b
 *  Implemented mbgextio_get_all_net_status_info
 *  Revision 1.3.1.4.1.49  2016/05/04 13:07:49  thomas-b
 *  Added function mbgextio_save_all_net_cfg_info
 *  Revision 1.3.1.4.1.48  2016/05/04 07:46:52  thomas-b
 *  Copy link mac address to route info only, if gateway is set
 *  Revision 1.3.1.4.1.47  2016/04/27 07:12:02  thomas-b
 *  Improved comments for mbgextio_get_all_net_cfg_info
 *  Revision 1.3.1.4.1.46  2016/04/26 14:42:56  thomas-b
 *  Added function mbgextio_get_all_net_cfg_info
 *  Revision 1.3.1.4.1.45  2016/04/25 14:47:43  martin
 *  *** empty log message ***
 *  Revision 1.3.1.4.1.44  2016/04/22 07:11:50  philipp
 *  Use pointer to pointer in get_all_* functions
 *  Revision 1.3.1.4.1.43  2016/04/22 05:01:48  thomas-b
 *  Added function mbgextio_save_all_ntp_cfg_info
 *  Revision 1.3.1.4.1.42  2016/04/20 14:54:33  thomas-b
 *  Added functions mbgextio_get_all_ntp_cfg_info and mbgextio_get_all_ntp_status
 *  Revision 1.3.1.4.1.41  2016/04/20 12:37:55  thomas-b
 *  Moved free functions for ALL_XMULTI_REF_INFO and ALL_XMULTI_REF_STATUS to cfg_hlp
 *  Revision 1.3.1.4.1.40  2016/04/18 14:06:19  thomas-b
 *  Changed comment
 *  Revision 1.3.1.4.1.39  2016/04/18 12:57:44  thomas-b
 *  Renamed mbgextio_set_all_xmulti_ref_settings to mbgextio_save_all_xmulti_ref_info
 *  Revision 1.3.1.4.1.38  2016/04/15 11:12:22  thomas-b
 *  Changed xmulti ref functions to alloc/free the whole struct
 *  Revision 1.3.1.4.1.37  2016/04/13 11:00:40  thomas-b
 *  Apply changes by sending dummy settings with index -1
 *  Revision 1.3.1.4.1.36  2016/04/13 07:16:48  thomas-b
 *  Added function mbgextio_set_all_xmulti_ref_settings
 *  Revision 1.3.1.4.1.35  2016/04/12 13:28:40  philipp
 *  New helper functions to get all feature related structures at once
 *  Revision 1.3.1.4.1.34  2016/04/12 08:44:03  thomas-b
 *  Fixed several errors in xmulti ref functions
 *  Revision 1.3.1.4.1.33  2016/04/12 08:26:09  thomas-b
 *  Added mbgextio_get_all_xmulti_ref_info and mbgextio_get_all_xmulti_ref_status and appropriate free functions
 *  Revision 1.3.1.4.1.32  2016/04/11 11:14:38  daniel
 *  Support incremental update
 *  Revision 1.3.1.4.1.31  2016/03/22 11:51:41  thomas-b
 *  Moved new xport functions and structure definitions to mbgextio
 *  Revision 1.3.1.4.1.30  2016/03/18 07:10:14Z  thomas-b
 *  Renamed new xport functions and structures
 *  Revision 1.3.1.4.1.29  2016/03/16 15:58:47  thomas-b
 *  Added functions mbgextio_setup_xport_list and mbgextio_free_xport_list
 *  Will have to be extended for use under Windows
 *  Revision 1.3.1.4.1.28  2016/03/16 13:47:15  martin
 *  *** empty log message ***
 *  Revision 1.3.1.4.1.27  2016/03/14 09:46:35  marvin
 *  Added function to set all ptp settings.
 *  Revision 1.3.1.4.1.26  2016/02/18 15:52:31Z  daniel
 *  Account for diag file and license upgrade
 *  Revision 1.3.1.4.1.25  2016/02/15 12:13:14  daniel
 *  Support for getting diag file
 *  Revision 1.3.1.4.1.24  2015/12/01 11:34:06  martin
 *  *** empty log message ***
 *  Revision 1.3.1.4.1.23  2015/11/30 07:06:35  philipp
 *  Access function for control message's opaque / internal USB device info structure
 *  Revision 1.3.1.4.1.22  2015/11/26 16:19:25  martin
 *  Reworked check feature API causing some other API calls to be
 *  simplified/changed since receiver info is now stored inside
 *  the message control block.
 *  Revision 1.3.1.4.1.21  2015/11/25 10:59:52  philipp
 *  Extended TLV xmt functions by uid parameter
 *  Revision 1.3.1.4.1.20  2015/11/25 09:58:13  philipp
 *  Fixed TLV receive functions to work properly and as expected
 *  Revision 1.3.1.4.1.19  2015/11/24 13:12:58  philipp
 *  Added / redesigned TLV functions
 *  Revision 1.3.1.4.1.18  2015/11/23 15:08:40  daniel
 *  More work on TLV stuff
 *  Revision 1.3.1.4.1.17  2015/11/23 14:17:02  philipp
 *  Moved TLV related sending / receiving functions to here
 *  Revision 1.3.1.4.1.16  2015/10/28 16:22:55  martin
 *  Revision 1.3.1.4.1.15  2015/10/19 16:42:16Z  martin
 *  *** empty log message ***
 *  Revision 1.3.1.4.1.14  2015/10/19 09:35:10  martin
 *  *** empty log message ***
 *  Revision 1.3.1.4.1.13  2015/10/12 13:31:23  martin
 *  *** empty log message ***
 *  Revision 1.3.1.4.1.12  2015/10/12 10:01:59  martin
 *  *** empty log message ***
 *  Revision 1.3.1.4.1.11  2015/10/09 12:20:17  martin
 *  *** empty log message ***
 *  Revision 1.3.1.4.1.10  2015/10/09 11:09:13  martin
 *  *** empty log message ***
 *  Revision 1.3.1.4.1.9  2015/10/08 10:24:30  martin
 *  *** empty log message ***
 *  Revision 1.3.1.4.1.8  2015/10/08 09:30:49  martin
 *  *** empty log message ***
 *  Revision 1.3.1.4.1.7  2015/10/08 08:04:10  martin
 *  *** empty log message ***
 *  Revision 1.3.1.4.1.6  2015/10/07 16:03:21  martin
 *  *** empty log message ***
 *  Revision 1.3.1.4.1.5  2015/10/07 09:59:01  martin
 *  More common GNSS support.
 *  Revision 1.3.1.4.1.4  2015/10/06 14:12:16  martin
 *  Account for library module chk_tstr renamed to mbg_tstr.
 *  Revision 1.3.1.4.1.3  2015/09/07 13:24:10  martin
 *  Revision 1.3.1.4.1.2  2015/01/21 13:24:44  marvin
 *  Added XBP Adress specifier to mbgextio functions.
 *  Revision 1.3.1.4.1.1  2014/10/31 12:04:45Z  martin
 *  Started to support XBP addressing.
 *  Revision 1.3.1.4  2014/10/30 14:50:00  martin
 *  Generally return Meinberg error codes only.
 *  Updated doxygen stuff.
 *  Updated function prototypes.
 *  Revision 1.3.1.3  2013/12/19 09:20:12  martin
 *  Revision 1.3.1.2  2013/12/17 14:57:48Z  martin
 *  Revision 1.3.1.1  2013/09/25 11:10:49Z  marvin
 *  Added function for all_ptp_cfg_info.
 *  Revision 1.3  2013/02/01 15:49:59Z  martin
 *  Updated doxygen comments.
 *  Cleanup.
 *  Revision 1.2  2012/03/09 08:32:58Z  martin
 *  Cleanup.
 *  Revision 1.1  2011/09/21 15:59:59  martin
 *  Initial revision.
 *
 **************************************************************************/

#define _EXTIOHLP
  #include <extiohlp.h>
#undef _EXTIOHLP

#include <cfg_hlp.h>
#include <lan_util.h>
#include <mbggeo.h>
#include <str_util.h>

#if defined( MBG_TGT_WIN32 ) && !defined( MBG_TGT_CVI )
  #include <sys/types.h>
  #include <sys/stat.h>
#elif defined( MBG_TGT_DOS )
  #include <sys/stat.h>
#endif

#define TLV_REQ_FILE_TIMEOUT   3000  // [ms]

#if !defined( UINT8_MAX )
  #define UINT8_MAX 255
#endif


#if _USE_MBG_TSTR

/*HDR*/
/**
 * @brief Set a callback function and argument used to decode a time string
 *
 * Some devices send a serial time string interleaved with some binary messages.
 * If a callback function has been specified then the binary message receiver
 * function invokes that callback function whenever some characters are received
 * which don't belong to a binary message. A callback function like ::mbg_tstr_receive
 * can then try to decode a serial time string from such spurious characters.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     fnc    Address of a ::MBG_TSTR_RCV_FNC-style callback function
 * @param[in]     arg    A prepared ::MBG_TSTR_RCV_ARG structure passed to the callback
 *                       function whenever it is invoked.
 *
 * @see ::MBG_TSTR_RCV_FNC
 * @see ::MBG_TSTR_RCV_ARG
 * @see ::mbg_tstr_set_rcv_callback
 */
void mbgextio_set_chk_tstr( MBG_MSG_CTL *pmctl, MBG_TSTR_RCV_FNC *fnc, MBG_TSTR_RCV_ARG *arg )
{
  MBG_MSG_RCV_CTL *prctl = &pmctl->rcv;

  prctl->tstr_rcv_fnc = fnc;
  prctl->tstr_rcv_arg = arg;

}  // mbgextio_set_chk_tstr

#endif  // _USE_MBG_TSTR



/*HDR*/
/**
 * @brief Read all serial port settings and supported configuration parameters
 *
 * ::mbgextio_setup_receiver_info should have been called before to set up
 * a ::RECEIVER_INFO structure to be passed to this function, so this
 * function can determine how many serial ports and associated structures
 * are supported by this device.
 *
 * The complementary function ::mbgextio_save_serial_settings should
 * be used to write a modified port configuration back to the device.
 *
 * @param[in,out] pmctl    Pointer to a valid message control structure
 * @param[in]     p_addr   Pointer to an XBP address specifier, or NULL
 * @param[out]    p_rpcfg  Pointer to a ::RECEIVER_PORT_CFG structure to be set up
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_save_serial_settings
 * @see ::mbgextio_get_receiver_info
 */
int mbgextio_get_serial_settings( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                  RECEIVER_PORT_CFG *p_rpcfg )
{
  const RECEIVER_INFO *p_ri = mbgextio_get_receiver_info_addr( pmctl );
  int rc = MBG_SUCCESS;

  memset( p_rpcfg, 0, sizeof( *p_rpcfg ) );

  if ( p_ri->model_code != GPS_MODEL_UNKNOWN )
  {
    // The device provided a RECEIVER_INFO, so we can simply read all
    // serial port configuration and supported string types directly.

    rc = mbgextio_get_all_port_info( pmctl, p_addr, p_rpcfg->pii );

    if ( mbg_rc_is_error (rc ) )
      goto out;

    rc = mbgextio_get_all_str_type_info( pmctl, p_addr, p_rpcfg->stii );
  }
  else
  {
    // The device didn't provide a RECEIVER_INFO. If the binary protocol is
    // supported then the device is old GPS receiver. Other legacy devices
    // didn't support the binary protocol.

    // We read the serial port configuration using a legacy API call
    // and set up current structures accordingly.

    rc = mbgextio_get_port_parm( pmctl, p_addr, &p_rpcfg->tmp_pp );

    if ( mbg_rc_is_error (rc ) )
      goto out;

    rc = setup_port_info_from_port_settings( p_rpcfg->pii, &p_rpcfg->tmp_pp, p_ri );

    if ( mbg_rc_is_error( rc ) )
      goto out;

    // Also set up default string type information.
    rc = setup_default_str_type_info_idx( p_rpcfg->stii, p_ri );
  }

out:
  return rc;

}  // mbgextio_get_serial_settings



/*HDR*/
/**
 * @brief Send the configuration settings for a single serial port to a device
 *
 * The function ::mbgextio_get_serial_settings must have been called before
 * to read the current settings and configuration options and fill up a
 * ::RECEIVER_PORT_CFG structure.
 *
 * After the settings for a port have been changed according to the supported
 * features this function can be used to write the new settings for a specific
 * port addressed by the port_idx parameter to the device.
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[in]     p_rpcfg   Pointer to a ::RECEIVER_PORT_CFG structure containg valid data
 * @param[in]     port_idx  Index of the serial port for which to save the settings
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_serial_settings
 */
int mbgextio_save_serial_settings( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                   const RECEIVER_PORT_CFG *p_rpcfg, uint16_t port_idx )
{
  int rc = mbgextio_set_port_settings_idx( pmctl, p_addr,
                       &p_rpcfg->pii[port_idx].port_info.port_settings, port_idx );

  return rc;

}  // mbgextio_save_serial_settings



/*HDR*/
/**
 * @brief Read all XBP information from a device into a newly or re-allocated ::ALL_XBP_INFO structure
 *
 * @note ::mbgextio_dev_has_xbp should be called to check if this API is supported.
 *
 * The allocated ::ALL_XBP_INFO needs to be freed ::free_all_ntp_cfg_info,
 * as soon as it is not needed any longer.
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[out]      p        Pointer to a pointer to ::ALL_XBP_INFO
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_xbp
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_xbp_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_XBP_INFO **p )
{
  ALL_XBP_INFO *xbp_info = *p;
  uint16_t i;
  int rc;

  if ( xbp_info == NULL )
  {
    xbp_info = (ALL_XBP_INFO *) calloc( 1, sizeof( *xbp_info ) );
    if ( xbp_info == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }
  }

  rc = mbgextio_get_xbp_limits( pmctl, p_addr, &xbp_info->limits );

  if ( mbg_rc_is_success( rc ) )
  {
    if ( mbg_rc_is_success( chk_dev_xbp_supp_nodes( xbp_info ) ) )
    {
      if ( xbp_info->node_limits == NULL )
      {
        xbp_info->node_limits = (XBP_NODE_LIMITS *) calloc( 1, sizeof( *xbp_info->node_limits ) );
        if ( xbp_info->node_limits == NULL )
        {
          rc = MBG_ERR_NO_MEM;
          goto out;
        }
      }

      rc = mbgextio_get_xbp_node_limits( pmctl, p_addr, xbp_info->node_limits );

      if ( mbg_rc_is_error( rc ) )
        goto out;

      if ( xbp_info->node_limits->node_count > 0 )
      {
        xbp_info->node_infos = ( XBP_NODE_INFO_IDX * )realloc( xbp_info->node_infos, xbp_info->node_limits->node_count * sizeof( *xbp_info->node_infos ) );
        if ( xbp_info->node_infos == NULL )
        {
          rc = MBG_ERR_NO_MEM;
          goto out;
        }

        for ( i = 0; i < xbp_info->node_limits->node_count; i++ )
        {
          xbp_info->node_infos[i].idx = i;
          rc = mbgextio_get_xbp_node_info_idx( pmctl, p_addr, &xbp_info->node_infos[i], i );
          if ( mbg_rc_is_error( rc ) )
            goto out;
        }
      }
      else
      {
        if ( xbp_info->node_infos )
          free( xbp_info->node_infos );
        xbp_info->node_infos = NULL;
      }
    }
  }

out:
  if ( mbg_rc_is_error( rc ) )
  {
    if( xbp_info != NULL )
    {
      free_all_xbp_info( xbp_info );
      xbp_info = NULL;
    }
  }

  *p = xbp_info;

  return rc;

} // mbgextio_get_all_xbp_info



/*HDR*/
/**
 * @brief Read all network configuration into an ::ALL_NET_CFG_INFO structure
 *
 * Reads the complete network configuration of a device. Depending on the
 * supported API (NET_CFG or LAN_IP4), the appropriate settings are requested and,
 * if neccessary, translated into the NET_CFG structures. The filled structures
 * can then be used as if the NET_CFG API was completely supported
 *
 * If NET_CFG or parts of it are not supported, the appropriate n_supp... members
 * of ::ALL_NET_CFG_INFO::glb_cfg_info are set to 0, whereas the num... members of
 * ::ALL_NET_CFG_INFO::glb_cfg_info::glb_settings may be 1 or bigger.
 *
 * A ::ALL_NET_CFG_INFO and the appropriate number of ::MBG_NET_INTF_LINK_INFO_IDX,
 * ::MBG_NET_INTF_ADDR_INFO_IDX, ::MBG_IP_ADDR_IDX, ::MBG_NET_NAME_IDX and
 * ::MBG_NET_INTF_ROUTE_INFO_IDX will be allocated and need to be freed later
 * by calling ::free_all_net_cfg_info
 *
 * @note ::mbgextio_dev_has_net_cfg and mbgextio_dev_has_lan_ip4 are used in the function
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[out]      p        Pointer to a pointer to ::ALL_NET_CFG_INFO
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_net_cfg
 * @see ::mbgextio_get_net_glb_cfg_info
 * @see ::mbgextio_get_net_dns_srvr_idx
 * @see ::mbgextio_get_net_dns_srch_dom_idx
 * @see ::mbgextio_dev_has_lan_ip4
 * @see ::mbgextio_get_lan_if_info
 * @see ::mbgextio_get_ip4_settings
 * @see ::free_all_net_cfg_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_net_cfg_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_NET_CFG_INFO **p )
{
  ALL_NET_CFG_INFO *net_cfg_info = *p;
  int rc;
  int rc_lan_ip4;
  int16_t i;

  if ( net_cfg_info == NULL )
  {
    net_cfg_info = (ALL_NET_CFG_INFO *) calloc( 1, sizeof( *net_cfg_info ) );
    if ( net_cfg_info == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }
  }

  rc = mbgextio_dev_has_net_cfg( pmctl );

  if ( mbg_rc_is_success( rc ) )
  {
    rc = mbgextio_get_net_glb_cfg_info( pmctl, p_addr, &net_cfg_info->glb_cfg_info );

    if ( mbg_rc_is_error( rc ) )
      goto out;

    if ( mbg_rc_is_success( chk_dev_net_cfg_supp_stage_2( net_cfg_info ) ) &&
       ( net_cfg_info->glb_cfg_info.glb_settings.num_intf_link > 0 ) )
    {
      net_cfg_info->link_infos = ( MBG_NET_INTF_LINK_INFO_IDX * )realloc( net_cfg_info->link_infos, net_cfg_info->glb_cfg_info.glb_settings.num_intf_link * sizeof( *net_cfg_info->link_infos ) );
      if ( net_cfg_info->link_infos == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }

      for ( i = 0; i < net_cfg_info->glb_cfg_info.glb_settings.num_intf_link; ++i )
      {
        rc = mbgextio_get_net_intf_link_info_idx( pmctl, p_addr, &net_cfg_info->link_infos[i], i );
        if ( mbg_rc_is_error( rc ) )
          goto out;
      }
    }
    else
    {
      if ( net_cfg_info->link_infos )
        free( net_cfg_info->link_infos );
      net_cfg_info->link_infos = NULL;
    }

    if ( mbg_rc_is_success( chk_dev_net_cfg_supp_stage_2( net_cfg_info ) ) &&
       ( net_cfg_info->glb_cfg_info.glb_settings.num_intf_addr > 0 ) )
    {
      net_cfg_info->addr_infos = ( MBG_NET_INTF_ADDR_INFO_IDX * )realloc( net_cfg_info->addr_infos, net_cfg_info->glb_cfg_info.glb_settings.num_intf_addr * sizeof( *net_cfg_info->addr_infos ) );
      if ( net_cfg_info->addr_infos == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }

      for ( i = 0; i < net_cfg_info->glb_cfg_info.glb_settings.num_intf_addr; i++ )
      {
        rc = mbgextio_get_net_intf_addr_info_idx( pmctl, p_addr, &net_cfg_info->addr_infos[i], i );
        if ( mbg_rc_is_error( rc ) )
          goto out;
      }
    }
    else
    {
      if ( net_cfg_info->addr_infos )
        free( net_cfg_info->addr_infos );
      net_cfg_info->addr_infos = NULL;
    }

    if ( mbg_rc_is_success( chk_dev_net_cfg_supp_stage_2( net_cfg_info ) ) &&
       ( net_cfg_info->glb_cfg_info.glb_settings.num_intf_route > 0 ) )
    {
      net_cfg_info->route_infos = ( MBG_NET_INTF_ROUTE_INFO_IDX * )realloc( net_cfg_info->route_infos, net_cfg_info->glb_cfg_info.glb_settings.num_intf_route * sizeof( *net_cfg_info->route_infos ) );
      if ( net_cfg_info->route_infos == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }

      for ( i = 0; i < net_cfg_info->glb_cfg_info.glb_settings.num_intf_route; i++ )
      {
        rc = mbgextio_get_net_intf_route_info_idx( pmctl, p_addr, &net_cfg_info->route_infos[i], i );
        if ( mbg_rc_is_error( rc ) )
          goto out;
      }
    }
    else
    {
      if ( net_cfg_info->route_infos )
        free( net_cfg_info->route_infos );
      net_cfg_info->route_infos = NULL;
    }

    if ( net_cfg_info->glb_cfg_info.n_supp_dns_srvr > 0 )
    {
      // If the new API is only partly supported (Syncbox N2x), set the number of DNS servers to the number of supported DNS servers,
      // as net_cfg_info->glb_cfg_info.glb_settings.num_dns_srvr is newer than N2x firmware
      if ( mbg_rc_is_error( chk_dev_net_cfg_supp_stage_2( net_cfg_info ) ) )
        net_cfg_info->glb_cfg_info.glb_settings.num_dns_srvr = (uint8_t)net_cfg_info->glb_cfg_info.n_supp_dns_srvr;
    }

    if ( ( net_cfg_info->glb_cfg_info.n_supp_dns_srvr > 0 ) &&
         ( net_cfg_info->glb_cfg_info.glb_settings.num_dns_srvr > 0 ) )
    {
      net_cfg_info->dns_srvrs = ( MBG_IP_ADDR_IDX * )realloc( net_cfg_info->dns_srvrs, net_cfg_info->glb_cfg_info.glb_settings.num_dns_srvr * sizeof( *net_cfg_info->dns_srvrs ) );
      if ( net_cfg_info->dns_srvrs == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }

      for ( i = 0; i < net_cfg_info->glb_cfg_info.glb_settings.num_dns_srvr; ++i )
      {
        net_cfg_info->dns_srvrs[i].idx = i;
        rc = mbgextio_get_net_dns_srvr_idx( pmctl, p_addr, &net_cfg_info->dns_srvrs[i], i );
        if ( mbg_rc_is_error( rc ) )
          goto out;
        else
        {
          if((net_cfg_info->dns_srvrs[i].addr.type == MBG_IP_ADDR_TYPE_UNKNOWN) ||
            ((net_cfg_info->dns_srvrs[i].addr.type == MBG_IP_ADDR_TYPE_IP4) &&
             (net_cfg_info->dns_srvrs[i].addr.u_addr.ip4_addr == 0)))
          {
            net_cfg_info->glb_cfg_info.glb_settings.num_dns_srvr--;
            if( net_cfg_info->glb_cfg_info.glb_settings.num_dns_srvr == 0 )
            {
              free( net_cfg_info->dns_srvrs );
              net_cfg_info->dns_srvrs = NULL;
            }
            else
            {
              net_cfg_info->dns_srvrs = ( MBG_IP_ADDR_IDX * )realloc( net_cfg_info->dns_srvrs, net_cfg_info->glb_cfg_info.glb_settings.num_dns_srvr * sizeof( *net_cfg_info->dns_srvrs ) );
              if ( net_cfg_info->dns_srvrs == NULL )
              {
                rc = MBG_ERR_NO_MEM;
                goto out;
              }
            }
            i--;
          }
        }
      }
    }
    else
    {
      if ( net_cfg_info->dns_srvrs )
        free( net_cfg_info->dns_srvrs );
      net_cfg_info->dns_srvrs = NULL;
    }

    if ( net_cfg_info->glb_cfg_info.n_supp_dns_srch_dom > 0 )
    {
      // If the new API is only partly supported (Syncbox N2x), set the number of DNS search domains to the number of supported DNS search domains,
      // as net_cfg_info->glb_cfg_info.glb_settings.num_dns_srch_dom is newer than N2x firmware
      if ( mbg_rc_is_error( chk_dev_net_cfg_supp_stage_2( net_cfg_info ) ) )
        net_cfg_info->glb_cfg_info.glb_settings.num_dns_srch_dom = (uint8_t)net_cfg_info->glb_cfg_info.n_supp_dns_srch_dom;
    }

    if ( ( net_cfg_info->glb_cfg_info.n_supp_dns_srch_dom > 0 )  &&
         ( net_cfg_info->glb_cfg_info.glb_settings.num_dns_srch_dom > 0 ) )
    {
      net_cfg_info->dns_srch_doms = ( MBG_NET_NAME_IDX * )realloc( net_cfg_info->dns_srch_doms, net_cfg_info->glb_cfg_info.glb_settings.num_dns_srch_dom * sizeof( *net_cfg_info->dns_srch_doms ) );
      if ( net_cfg_info->dns_srch_doms == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }

      for ( i = 0; i < net_cfg_info->glb_cfg_info.glb_settings.num_dns_srch_dom; ++i )
      {
        net_cfg_info->dns_srch_doms[i].idx = i;
        rc = mbgextio_get_net_dns_srch_dom_idx( pmctl, p_addr, &net_cfg_info->dns_srch_doms[i], i );
        if ( mbg_rc_is_error( rc ) )
          goto out;
        else
        {
          if( strcmp( net_cfg_info->dns_srch_doms[i].net_name.name, "" ) == 0 )
          {
            net_cfg_info->glb_cfg_info.glb_settings.num_dns_srch_dom--;
            if( net_cfg_info->glb_cfg_info.glb_settings.num_dns_srch_dom == 0 )
            {
              free( net_cfg_info->dns_srch_doms );
              net_cfg_info->dns_srch_doms = NULL;
            }
            else
            {
              net_cfg_info->dns_srch_doms = ( MBG_NET_NAME_IDX * )realloc( net_cfg_info->dns_srch_doms, net_cfg_info->glb_cfg_info.glb_settings.num_dns_srch_dom * sizeof( *net_cfg_info->dns_srch_doms ) );
              if ( net_cfg_info->dns_srch_doms == NULL )
              {
                rc = MBG_ERR_NO_MEM;
                goto out;
              }
            }
            i--;
          }
        }
      }
    }
    else
    {
      if ( net_cfg_info->dns_srch_doms )
        free( net_cfg_info->dns_srch_doms );
      net_cfg_info->dns_srch_doms = NULL;
    }

  }

  if ( mbg_rc_is_success( chk_dev_net_cfg_supp_stage_2( net_cfg_info ) ) )
    goto out;

  rc_lan_ip4 = mbgextio_dev_has_lan_ip4( pmctl );

  if ( mbg_rc_is_success( rc_lan_ip4 ) )
  {
    LAN_IF_INFO lan_if_info;
    IP4_SETTINGS ip4_settings;

    rc = mbgextio_get_lan_if_info( pmctl, p_addr, &lan_if_info );

    if ( mbg_rc_is_error( rc ) )
      goto out;

    if ( net_cfg_info->link_infos == NULL )
    {
      net_cfg_info->link_infos = (MBG_NET_INTF_LINK_INFO_IDX *) calloc( 1, sizeof( *net_cfg_info->link_infos ) );
      if ( net_cfg_info->link_infos == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }
    }

    net_cfg_info->glb_cfg_info.n_supp_intf_link = UINT8_MAX;
    net_cfg_info->glb_cfg_info.glb_settings.num_intf_link = 1;

    memset( &net_cfg_info->link_infos[0], 0, sizeof( net_cfg_info->link_infos[0] ) );

    net_cfg_info->link_infos[0].info.supp_states = MBG_NET_INTF_LINK_STATE_MASK_UP;
    net_cfg_info->link_infos[0].info.supp_types = MBG_NET_INTF_LINK_TYPE_MASK_PHYS | MBG_NET_INTF_LINK_TYPE_MASK_VLAN;
    net_cfg_info->link_infos[0].info.supp_speed_modes = MBG_NET_INTF_LINK_SPEED_MODE_MASK_10_T_HALF | MBG_NET_INTF_LINK_SPEED_MODE_MASK_10_T_FULL | MBG_NET_INTF_LINK_SPEED_MODE_MASK_100_T_HALF | MBG_NET_INTF_LINK_SPEED_MODE_MASK_100_T_FULL;
    net_cfg_info->link_infos[0].info.supp_port_types = MBG_NET_INTF_LINK_PORT_TYPE_MASK_TP;

    snprintf_safe( net_cfg_info->link_infos[0].info.link_settings.name, sizeof( net_cfg_info->link_infos[0].info.link_settings.name ), "lan0" );
    net_cfg_info->link_infos[0].info.link_settings.mac_addr = lan_if_info.mac_addr;
    net_cfg_info->link_infos[0].info.link_settings.if_index = 0;
    net_cfg_info->link_infos[0].info.link_settings.hw_type = MBG_ARPHRD_ETHER;
    net_cfg_info->link_infos[0].info.link_settings.type = MBG_NET_INTF_LINK_TYPE_PHYS;
    net_cfg_info->link_infos[0].info.link_settings.port_type = MBG_NET_INTF_LINK_PORT_TYPE_TP;

    rc = mbgextio_get_ip4_settings( pmctl, p_addr, &ip4_settings );

    if ( mbg_rc_is_error( rc ) )
      goto out;

    if( ip4_settings.flags & IP4_MSK_LINK )
    {
      net_cfg_info->link_infos[0].info.link_settings.states |= MBG_NET_INTF_LINK_STATE_MASK_UP;
      net_cfg_info->link_infos[0].info.link_settings.states |= MBG_NET_INTF_LINK_STATE_MASK_LOWER_UP;
    }

    if ( net_cfg_info->addr_infos == NULL )
    {
      net_cfg_info->addr_infos = (MBG_NET_INTF_ADDR_INFO_IDX *) calloc( 1, sizeof( *net_cfg_info->addr_infos ) );
      if ( net_cfg_info->addr_infos == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }
    }

    net_cfg_info->glb_cfg_info.n_supp_intf_addr = 1;
    net_cfg_info->glb_cfg_info.glb_settings.num_intf_addr = 1;

    memset( &net_cfg_info->addr_infos[0], 0, sizeof( net_cfg_info->addr_infos[0] ) );

    net_cfg_info->addr_infos[0].info.supp_flags = MBG_NET_INTF_ADDR_MASK_DHCP4;

    snprintf_safe( net_cfg_info->addr_infos[0].info.addr_settings.label, sizeof( net_cfg_info->addr_infos[0].info.addr_settings.label ), "lan0:0" );

    if ( ip4_settings.flags & IP4_MSK_VLAN )
    {
      net_cfg_info->link_infos = ( MBG_NET_INTF_LINK_INFO_IDX * )realloc( net_cfg_info->link_infos, 2 * sizeof( *net_cfg_info->link_infos ) );
      if ( net_cfg_info->link_infos == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }

      net_cfg_info->glb_cfg_info.glb_settings.num_intf_link = 2;

      memset(&net_cfg_info->link_infos[1], 0, sizeof(net_cfg_info->link_infos[1]));

      net_cfg_info->link_infos[1].idx = 1;
      net_cfg_info->link_infos[1].info.supp_states = MBG_NET_INTF_LINK_STATE_MASK_UP;
      net_cfg_info->link_infos[1].info.supp_types = MBG_NET_INTF_LINK_TYPE_MASK_VLAN;
      net_cfg_info->link_infos[1].info.supp_speed_modes = MBG_NET_INTF_LINK_SPEED_MODE_MASK_10_T_HALF | MBG_NET_INTF_LINK_SPEED_MODE_MASK_10_T_FULL | MBG_NET_INTF_LINK_SPEED_MODE_MASK_100_T_HALF | MBG_NET_INTF_LINK_SPEED_MODE_MASK_100_T_FULL;
      net_cfg_info->link_infos[1].info.supp_port_types = MBG_NET_INTF_LINK_PORT_TYPE_MASK_TP;

      snprintf_safe( net_cfg_info->link_infos[1].info.link_settings.name, sizeof( net_cfg_info->link_infos[1].info.link_settings.name ), "vlan0" );
      net_cfg_info->link_infos[1].info.link_settings.mac_addr = net_cfg_info->link_infos[0].info.link_settings.mac_addr;
      net_cfg_info->link_infos[1].info.link_settings.if_index = 1;
      net_cfg_info->link_infos[1].info.link_settings.ass_if_index = 0;
      net_cfg_info->link_infos[1].info.link_settings.states = net_cfg_info->link_infos[0].info.link_settings.states;
      net_cfg_info->link_infos[1].info.link_settings.type = MBG_NET_INTF_LINK_TYPE_VLAN;
      net_cfg_info->link_infos[1].info.link_settings.vlan_cfg = ip4_settings.vlan_cfg;

      net_cfg_info->addr_infos[0].info.addr_settings.ass_if_index = 1;
    }

    if ( ip4_settings.flags & IP4_MSK_DHCP )
      net_cfg_info->addr_infos[0].info.addr_settings.flags |= MBG_NET_INTF_ADDR_MASK_DHCP4;

    net_cfg_info->addr_infos[0].info.addr_settings.ip.type = MBG_IP_ADDR_TYPE_IP4;
    net_cfg_info->addr_infos[0].info.addr_settings.ip.u_addr.ip4_addr = ip4_settings.ip_addr;

    net_cfg_info->addr_infos[0].info.addr_settings.broadcast.type = MBG_IP_ADDR_TYPE_IP4;
    net_cfg_info->addr_infos[0].info.addr_settings.broadcast.u_addr.ip4_addr = ip4_settings.broad_addr;

    net_cfg_info->addr_infos[0].info.addr_settings.prefix_bits = get_ip4_net_mask_bits(&ip4_settings.netmask);

    if ( net_cfg_info->route_infos == NULL )
    {
      net_cfg_info->route_infos = (MBG_NET_INTF_ROUTE_INFO_IDX *) calloc( 1, sizeof( *net_cfg_info->route_infos ) );
      if ( net_cfg_info->route_infos == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }
    }

    net_cfg_info->glb_cfg_info.n_supp_intf_route = 1;
    net_cfg_info->glb_cfg_info.glb_settings.num_intf_route = 1;

    memset( &net_cfg_info->route_infos[0], 0, sizeof( net_cfg_info->route_infos[0] ) );

    net_cfg_info->route_infos[0].info.route_settings.type = MBG_NET_INTF_ROUTE_TYPE_DEFAULT_GATEWAY;
    net_cfg_info->route_infos[0].info.route_settings.gateway.type = MBG_IP_ADDR_TYPE_IP4;
    net_cfg_info->route_infos[0].info.route_settings.gateway.u_addr.ip4_addr = ip4_settings.gateway;
    if ( ip4_settings.gateway != 0 )
    {
      if(net_cfg_info->glb_cfg_info.glb_settings.num_intf_link == 2)
        net_cfg_info->route_infos[0].info.route_settings.ass_if_index = 1;
    }
    else net_cfg_info->route_infos[0].info.route_settings.ass_if_index = (uint32_t)-1;

  }

  if ( mbg_rc_is_error( rc ) && mbg_rc_is_error( rc_lan_ip4 ) )
    rc = rc_lan_ip4;

out:
  if ( mbg_rc_is_error( rc ) )
  {
    free_all_net_cfg_info( net_cfg_info );
    net_cfg_info = NULL;
  }

  *p = net_cfg_info;

  return rc;

} // mbgextio_get_all_net_cfg_info



/*HDR*/
/**
 * @brief Write all network settings to a device
 *
 * The complementary function ::mbgextio_get_all_net_cfg_info should
 * have been used to read the original network settings and supported
 * configuration parameters.
 *
 * Depending on the supported API (NET_CFG or LAN_IP4 or both partly),
 * the appropriate settings are, if neccessary, translated into LAN_IP4 structures
 * and send to the device using the appropriate API functions.
 *
 * @param[in,out]  pmctl   Pointer to a valid message control structure
 * @param[in]      p_addr  Pointer to an XBP address specifier, or NULL
 * @param[in]      p       Pointer to the ::ALL_NET_CFG_INFO
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_save_all_net_cfg_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_NET_CFG_INFO *p )
{
  int rc = MBG_ERR_NOT_SUPP_BY_DEV;
  int16_t i;

  // Check if stage 2 of NET_CFG_API is supported
  // If it isn't, the current configuration has to be set using the LAN_IP4 API.
  if ( mbg_rc_is_error( chk_dev_net_cfg_supp_stage_2( p ) ) )
  {
    rc = mbgextio_dev_has_lan_ip4( pmctl );

    if ( mbg_rc_is_success( rc ) )
    {
      IP4_SETTINGS ip4_settings;
      memset( &ip4_settings, 0, sizeof( ip4_settings ) );

      if (p->glb_cfg_info.glb_settings.num_intf_addr > 0)
      {
        if ( p->addr_infos[0].info.addr_settings.ip.type == MBG_IP_ADDR_TYPE_IP4 )
          ip4_settings.ip_addr = p->addr_infos[0].info.addr_settings.ip.u_addr.ip4_addr;

        ip4_settings.netmask = ip4_net_mask_from_cidr( p->addr_infos[0].info.addr_settings.prefix_bits );
        ip4_settings.broad_addr = ip4_broad_addr_from_addr( &ip4_settings.ip_addr, &ip4_settings.netmask );

        if ( ( p->addr_infos[0].info.addr_settings.flags & MBG_NET_INTF_ADDR_MASK_DHCP4 ) == MBG_NET_INTF_ADDR_MASK_DHCP4 )
          ip4_settings.flags |= IP4_MSK_DHCP;
      }

      if (p->glb_cfg_info.glb_settings.num_intf_route > 0)
      {
        if ( p->route_infos[0].info.route_settings.gateway.type == MBG_IP_ADDR_TYPE_IP4 )
          ip4_settings.gateway = p->route_infos[0].info.route_settings.gateway.u_addr.ip4_addr;
      }

      if ( ( p->glb_cfg_info.glb_settings.num_intf_link > 1 ) && ( p->addr_infos[0].info.addr_settings.ass_if_index == 1 ) )
      {
        ip4_settings.flags |= IP4_MSK_VLAN;
        ip4_settings.vlan_cfg = p->link_infos[1].info.link_settings.vlan_cfg;
      }

      rc = mbgextio_set_ip4_settings( pmctl, p_addr, &ip4_settings );
    }
  }

  if ( mbg_rc_is_success( mbgextio_dev_has_net_cfg( pmctl ) ) )
  {
    if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
      mbgextio_begin_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_NETWORK, 1 );

    if ( mbg_rc_is_success( chk_dev_net_cfg_supp_stage_2( p ) ) )
    {
      // If stage 2 is supported, write ::MBG_NET_GLB_CFG_SETTINGS first
      rc = mbgextio_set_net_glb_cfg_settings( pmctl, p_addr, &p->glb_cfg_info.glb_settings );

      for ( i = 0; ( i < p->glb_cfg_info.glb_settings.num_intf_link ) && mbg_rc_is_success( rc ) ; ++i )
      {
        p->link_infos[i].idx = i;
        rc = mbgextio_set_net_intf_link_settings_idx( pmctl, p_addr, &p->link_infos[i].info.link_settings, i );

        if ( mbg_rc_is_error( rc ) )
          goto out;
      }

      for ( i = 0; ( i < p->glb_cfg_info.glb_settings.num_intf_addr ) && mbg_rc_is_success( rc ) ; ++i )
      {
        p->addr_infos[i].idx = i;
        rc = mbgextio_set_net_intf_addr_settings_idx( pmctl, p_addr, &p->addr_infos[i].info.addr_settings, i );

        if ( mbg_rc_is_error( rc ) )
          goto out;
      }

      for ( i = 0; ( i < p->glb_cfg_info.glb_settings.num_intf_route ) && mbg_rc_is_success( rc ) ; ++i )
      {
        p->route_infos[i].idx = i;
        rc = mbgextio_set_net_intf_route_settings_idx( pmctl, p_addr, &p->route_infos[i].info.route_settings, i );

        if ( mbg_rc_is_error( rc ) )
          goto out;
      }

      for ( i = 0; ( i < p->glb_cfg_info.glb_settings.num_dns_srch_dom ) && mbg_rc_is_success( rc ) ; ++i )
      {
        p->dns_srch_doms[i].idx = i;
        rc = mbgextio_set_net_dns_srch_dom_idx( pmctl, p_addr, &p->dns_srch_doms[i].net_name, i );

        if ( mbg_rc_is_error( rc ) )
          goto out;
      }

      for ( i = 0; ( i < p->glb_cfg_info.glb_settings.num_dns_srvr ) && mbg_rc_is_success( rc ) ; ++i )
      {
        p->dns_srvrs[i].idx = i;
        rc = mbgextio_set_net_dns_srvr_idx( pmctl, p_addr, &p->dns_srvrs[i].addr, i );

        if ( mbg_rc_is_error( rc ) )
          goto out;
      }

    }
    else
    {
      MBG_NET_NAME_IDX no_dns_srch_dom;
      MBG_IP_ADDR_IDX no_dns_srvr;

      memset(&no_dns_srch_dom, 0, sizeof(no_dns_srch_dom));
      memset(&no_dns_srvr, 0, sizeof(no_dns_srvr));
      no_dns_srvr.addr.type = MBG_IP_ADDR_TYPE_IP4;

      for ( i = 0; ( i < p->glb_cfg_info.n_supp_dns_srch_dom ) && mbg_rc_is_success( rc ) ; ++i )
      {
        if( i >= p->glb_cfg_info.glb_settings.num_dns_srch_dom )
        {
          no_dns_srch_dom.idx = i;
          rc = mbgextio_set_net_dns_srch_dom_idx( pmctl, p_addr, &no_dns_srch_dom.net_name, i );
        }
        else
        {
          p->dns_srch_doms[i].idx = i;
          rc = mbgextio_set_net_dns_srch_dom_idx( pmctl, p_addr, &p->dns_srch_doms[i].net_name, i );
        }

        if ( mbg_rc_is_error( rc ) )
          goto out;
      }

      for ( i = 0; ( i < p->glb_cfg_info.n_supp_dns_srvr ) && mbg_rc_is_success( rc ) ; ++i )
      {
        if( i >= p->glb_cfg_info.glb_settings.num_dns_srvr )
        {
          no_dns_srvr.idx = i;
          rc = mbgextio_set_net_dns_srvr_idx( pmctl, p_addr, &no_dns_srvr.addr, i );
        }
        else
        {
          p->dns_srvrs[i].idx = i;
          rc = mbgextio_set_net_dns_srvr_idx( pmctl, p_addr, &p->dns_srvrs[i].addr, i );
        }

        if ( mbg_rc_is_error( rc ) )
          goto out;
      }

      // If stage 2 is not supported, write ::MBG_NET_GLB_CFG_SETTINGS at the end
      rc = mbgextio_set_net_glb_cfg_settings( pmctl, p_addr, &p->glb_cfg_info.glb_settings );
    }

    if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
      mbgextio_end_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_NETWORK );

  }

out:
  return rc;

} // mbgextio_save_all_net_cfg_info



/*HDR*/
/**
 * @brief Read current network status into an ::ALL_NET_STATUS_INFO structure
 *
 * Depending on the supported API (NET_CFG or LAN_IP4), the appropriate status
 * is requested and, if neccessary, translated into the NET_CFG structures.
 * The filled structures can then be used as if the NET_CFG API was completely supported.
 *
 * A ::ALL_NET_STATUS_INFO and the appropriate number of ::MBG_NET_INTF_LINK_INFO_IDX,
 * ::MBG_NET_INTF_ADDR_INFO_IDX, ::MBG_IP_ADDR_IDX, ::MBG_NET_NAME_IDX and
 * ::MBG_NET_INTF_ROUTE_INFO_IDX will be allocated and need to be freed later
 * by calling ::free_all_net_status_info
 *
 * @note ::mbgextio_dev_has_net_cfg and mbgextio_dev_has_lan_ip4 are used in the function
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[out]      p        Pointer to a pointer to ::ALL_NET_STATUS_INFO
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_net_cfg
 * @see ::mbgextio_get_net_glb_cfg_info
 * @see ::mbgextio_get_net_stat_dns_srvr_idx
 * @see ::mbgextio_get_net_stat_dns_srch_dom_stat_idx
 * @see ::mbgextio_dev_has_lan_ip4
 * @see ::mbgextio_get_lan_if_info
 * @see ::mbgextio_get_ip4_state
 * @see ::free_all_net_status_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_net_status_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_NET_STATUS_INFO **p )
{
  ALL_NET_STATUS_INFO *net_status_info = *p;
  int rc;
  int rc_lan_ip4;
  int16_t i;

  if ( net_status_info == NULL )
  {
    net_status_info = (ALL_NET_STATUS_INFO *) calloc( 1, sizeof( *net_status_info ) );
    if ( net_status_info == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }
  }

  rc = mbgextio_dev_has_net_cfg( pmctl );

  if ( mbg_rc_is_success( rc ) )
  {
    rc = mbgextio_get_net_glb_cfg_info( pmctl, p_addr, &net_status_info->glb_cfg_info );

    if( mbg_rc_is_success( rc ) && mbg_rc_is_success( chk_dev_net_cfg_supp_stage_2( net_status_info ) ) )
      rc = mbgextio_get_net_stat_glb_cfg_info( pmctl, p_addr, &net_status_info->glb_cfg_info );

    if ( mbg_rc_is_error( rc ) )
      goto out;

    if ( mbg_rc_is_success( chk_dev_net_cfg_supp_stage_2( net_status_info ) ) &&
       ( net_status_info->glb_cfg_info.glb_settings.num_intf_link > 0 ) )
    {
      net_status_info->link_infos = ( MBG_NET_INTF_LINK_INFO_IDX * )realloc( net_status_info->link_infos, net_status_info->glb_cfg_info.glb_settings.num_intf_link * sizeof( *net_status_info->link_infos ) );
      if ( net_status_info->link_infos == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }

      for ( i = 0; i < net_status_info->glb_cfg_info.glb_settings.num_intf_link; ++i )
      {
        rc = mbgextio_get_net_stat_intf_link_info_idx( pmctl, p_addr, &net_status_info->link_infos[i], i );
        if ( mbg_rc_is_error( rc ) )
          goto out;
      }
    }
    else
    {
      if ( net_status_info->link_infos )
        free( net_status_info->link_infos );
      net_status_info->link_infos = NULL;
    }

    if ( mbg_rc_is_success( chk_dev_net_cfg_supp_stage_2( net_status_info ) ) &&
       ( net_status_info->glb_cfg_info.glb_settings.num_intf_addr > 0 ) )
    {
      net_status_info->addr_infos = ( MBG_NET_INTF_ADDR_INFO_IDX * )realloc( net_status_info->addr_infos, net_status_info->glb_cfg_info.glb_settings.num_intf_addr * sizeof( *net_status_info->addr_infos ) );
      if ( net_status_info->addr_infos == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }

      for ( i = 0; i < net_status_info->glb_cfg_info.glb_settings.num_intf_addr; ++i )
      {
        rc = mbgextio_get_net_stat_intf_addr_info_idx( pmctl, p_addr, &net_status_info->addr_infos[i], i );
        if ( mbg_rc_is_error( rc ) )
          goto out;
      }
    }
    else
    {
      if ( net_status_info->addr_infos )
        free( net_status_info->addr_infos );
      net_status_info->addr_infos = NULL;
    }

    if ( mbg_rc_is_success( chk_dev_net_cfg_supp_stage_2( net_status_info ) ) &&
       ( net_status_info->glb_cfg_info.glb_settings.num_intf_route > 0 ) )
    {
      net_status_info->route_infos = ( MBG_NET_INTF_ROUTE_INFO_IDX * )realloc( net_status_info->route_infos, net_status_info->glb_cfg_info.glb_settings.num_intf_route * sizeof( *net_status_info->route_infos ) );
      if ( net_status_info->route_infos == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }

      for ( i = 0; i < net_status_info->glb_cfg_info.glb_settings.num_intf_route; ++i )
      {
        rc = mbgextio_get_net_stat_intf_route_info_idx( pmctl, p_addr, &net_status_info->route_infos[i], i );
        if ( mbg_rc_is_error( rc ) )
          goto out;
      }
    }
    else
    {
      if ( net_status_info->route_infos )
        free( net_status_info->route_infos );
      net_status_info->route_infos = NULL;
    }

    if ( net_status_info->glb_cfg_info.n_supp_dns_srvr > 0 )
    {
      // If the new API is only partly supported (Syncbox N2x), set the number of DNS servers to the number of supported DNS servers,
      // as net_status_info->glb_cfg_info.glb_settings.num_dns_srvr is newer than N2x firmware
      if ( mbg_rc_is_error( chk_dev_net_cfg_supp_stage_2( net_status_info ) ) )
        net_status_info->glb_cfg_info.glb_settings.num_dns_srvr = (uint8_t)net_status_info->glb_cfg_info.n_supp_dns_srvr;
    }

    if ( ( net_status_info->glb_cfg_info.n_supp_dns_srvr > 0 ) &&
         ( net_status_info->glb_cfg_info.glb_settings.num_dns_srvr > 0 ) )
    {
      net_status_info->dns_srvrs = ( MBG_IP_ADDR_IDX * )realloc( net_status_info->dns_srvrs, net_status_info->glb_cfg_info.glb_settings.num_dns_srvr * sizeof( *net_status_info->dns_srvrs ) );
      if ( net_status_info->dns_srvrs == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }

      for ( i = 0; i < net_status_info->glb_cfg_info.glb_settings.num_dns_srvr; i++ )
      {
        net_status_info->dns_srvrs[i].idx = i;
        rc = mbgextio_get_net_stat_dns_srvr_idx( pmctl, p_addr, &net_status_info->dns_srvrs[i], i );
        if ( mbg_rc_is_error( rc ) )
          goto out;
        else
        {
          if((net_status_info->dns_srvrs[i].addr.type == MBG_IP_ADDR_TYPE_UNKNOWN) ||
            ((net_status_info->dns_srvrs[i].addr.type == MBG_IP_ADDR_TYPE_IP4) &&
             (net_status_info->dns_srvrs[i].addr.u_addr.ip4_addr == 0)))
          {
            net_status_info->glb_cfg_info.glb_settings.num_dns_srvr--;
            if( net_status_info->glb_cfg_info.glb_settings.num_dns_srvr == 0 )
            {
              free( net_status_info->dns_srvrs );
              net_status_info->dns_srvrs = NULL;
            }
            else
            {
              net_status_info->dns_srvrs = ( MBG_IP_ADDR_IDX * )realloc( net_status_info->dns_srvrs, net_status_info->glb_cfg_info.glb_settings.num_dns_srvr * sizeof( *net_status_info->dns_srvrs ) );
              if ( net_status_info->dns_srvrs == NULL )
              {
                rc = MBG_ERR_NO_MEM;
                goto out;
              }
            }
            i--;
          }
        }
      }
    }
    else
    {
      if ( net_status_info->dns_srvrs )
        free( net_status_info->dns_srvrs );
      net_status_info->dns_srvrs = NULL;
    }

    if ( net_status_info->glb_cfg_info.n_supp_dns_srch_dom > 0 )
    {
      // If the new API is only partly supported (Syncbox N2x), set the number of DNS search domains to the number of supported DNS search domains,
      // as net_status_info->glb_cfg_info.glb_settings.num_dns_srch_dom is newer than N2x firmware
      if ( mbg_rc_is_error( chk_dev_net_cfg_supp_stage_2( net_status_info ) ) )
        net_status_info->glb_cfg_info.glb_settings.num_dns_srch_dom = (uint8_t)net_status_info->glb_cfg_info.n_supp_dns_srch_dom;
    }

    if ( ( net_status_info->glb_cfg_info.n_supp_dns_srch_dom > 0 ) &&
         ( net_status_info->glb_cfg_info.glb_settings.num_dns_srch_dom > 0 ) )
    {
      net_status_info->dns_srch_doms = ( MBG_NET_NAME_IDX * )realloc( net_status_info->dns_srch_doms, net_status_info->glb_cfg_info.glb_settings.num_dns_srch_dom * sizeof( *net_status_info->dns_srch_doms ) );
      if ( net_status_info->dns_srch_doms == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }

      for ( i = 0; i < net_status_info->glb_cfg_info.glb_settings.num_dns_srch_dom; i++ )
      {
        net_status_info->dns_srch_doms[i].idx = i;
        rc = mbgextio_get_net_stat_dns_srch_dom_stat_idx( pmctl, p_addr, &net_status_info->dns_srch_doms[i], i );
        if ( mbg_rc_is_error( rc ) )
          goto out;
        else
        {
          if( strcmp( net_status_info->dns_srch_doms[i].net_name.name, "" ) == 0 )
          {
            net_status_info->glb_cfg_info.glb_settings.num_dns_srch_dom--;
            if( net_status_info->glb_cfg_info.glb_settings.num_dns_srch_dom == 0 )
            {
              free( net_status_info->dns_srch_doms );
              net_status_info->dns_srch_doms = NULL;
            }
            else
            {
              net_status_info->dns_srch_doms = ( MBG_NET_NAME_IDX * )realloc( net_status_info->dns_srch_doms, net_status_info->glb_cfg_info.glb_settings.num_dns_srch_dom * sizeof( *net_status_info->dns_srch_doms ) );
              if ( net_status_info->dns_srch_doms == NULL )
              {
                rc = MBG_ERR_NO_MEM;
                goto out;
              }
            }
            i--;
          }
        }
      }
    }
    else
    {
      if ( net_status_info->dns_srch_doms )
        free( net_status_info->dns_srch_doms );
      net_status_info->dns_srch_doms = NULL;
    }

  }

  if ( mbg_rc_is_success( chk_dev_net_cfg_supp_stage_2( net_status_info ) ) )
    goto out;

  rc_lan_ip4 = mbgextio_dev_has_lan_ip4( pmctl );

  if ( mbg_rc_is_success( rc_lan_ip4 ) )
  {
    LAN_IF_INFO lan_if_info;
    IP4_SETTINGS ip4_state;

    if ( net_status_info->glb_cfg_info.n_supp_intf_link != 0 )
      goto out;

    rc = mbgextio_get_lan_if_info( pmctl, p_addr, &lan_if_info );

    if ( mbg_rc_is_error( rc ) )
      goto out;

    if ( net_status_info->link_infos == NULL )
    {
      net_status_info->link_infos = (MBG_NET_INTF_LINK_INFO_IDX *) calloc( 1, sizeof( *net_status_info->link_infos ) );
      if ( net_status_info->link_infos == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }
    }

    net_status_info->glb_cfg_info.n_supp_intf_link = UINT8_MAX;
    net_status_info->glb_cfg_info.glb_settings.num_intf_link = 1;

    memset( &net_status_info->link_infos[0], 0, sizeof( net_status_info->link_infos[0] ) );

    net_status_info->link_infos[0].info.supp_states = MBG_NET_INTF_LINK_STATE_MASK_UP;
    net_status_info->link_infos[0].info.supp_types = MBG_NET_INTF_LINK_TYPE_MASK_PHYS | MBG_NET_INTF_LINK_TYPE_MASK_VLAN;
    net_status_info->link_infos[0].info.supp_speed_modes = MBG_NET_INTF_LINK_SPEED_MODE_MASK_10_T_HALF | MBG_NET_INTF_LINK_SPEED_MODE_MASK_10_T_FULL | MBG_NET_INTF_LINK_SPEED_MODE_MASK_100_T_HALF | MBG_NET_INTF_LINK_SPEED_MODE_MASK_100_T_FULL;
    net_status_info->link_infos[0].info.supp_port_types = MBG_NET_INTF_LINK_PORT_TYPE_MASK_TP;

    snprintf_safe( net_status_info->link_infos[0].info.link_settings.name, sizeof( net_status_info->link_infos[0].info.link_settings.name ), "lan0" );
    net_status_info->link_infos[0].info.link_settings.mac_addr = lan_if_info.mac_addr;
    net_status_info->link_infos[0].info.link_settings.if_index = 0;
    net_status_info->link_infos[0].info.link_settings.hw_type = MBG_ARPHRD_ETHER;
    net_status_info->link_infos[0].info.link_settings.type = MBG_NET_INTF_LINK_TYPE_PHYS;
    net_status_info->link_infos[0].info.link_settings.port_type = MBG_NET_INTF_LINK_PORT_TYPE_TP;

    if ( net_status_info->glb_cfg_info.n_supp_intf_addr != 0 )
      goto out;

    rc = mbgextio_get_ip4_state( pmctl, p_addr, &ip4_state );

    if ( mbg_rc_is_error( rc ) )
          goto out;

    if( ip4_state.flags & IP4_MSK_LINK )
    {
      net_status_info->link_infos[0].info.link_settings.states |= MBG_NET_INTF_LINK_STATE_MASK_UP;
      net_status_info->link_infos[0].info.link_settings.states |= MBG_NET_INTF_LINK_STATE_MASK_LOWER_UP;
    }

    if ( net_status_info->addr_infos == NULL )
    {
      net_status_info->addr_infos = (MBG_NET_INTF_ADDR_INFO_IDX *) calloc( 1, sizeof( *net_status_info->addr_infos ) );
      if ( net_status_info->addr_infos == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }
    }

    net_status_info->glb_cfg_info.n_supp_intf_addr = 1;
    net_status_info->glb_cfg_info.glb_settings.num_intf_addr = 1;

    memset( &net_status_info->addr_infos[0], 0, sizeof( net_status_info->addr_infos[0] ) );

    net_status_info->addr_infos[0].info.supp_flags = MBG_NET_INTF_ADDR_MASK_DHCP4;

    snprintf_safe( net_status_info->addr_infos[0].info.addr_settings.label, sizeof( net_status_info->addr_infos[0].info.addr_settings.label ), "lan0:0" );

    if ( ip4_state.flags & IP4_MSK_VLAN )
    {
      net_status_info->link_infos = ( MBG_NET_INTF_LINK_INFO_IDX * )realloc( net_status_info->link_infos, 2 * sizeof( *net_status_info->link_infos ) );
      if ( net_status_info->link_infos == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }

      net_status_info->glb_cfg_info.glb_settings.num_intf_link = 2;

      memset(&net_status_info->link_infos[1], 0, sizeof(net_status_info->link_infos[1]));

      net_status_info->link_infos[1].idx = 1;
      net_status_info->link_infos[1].info.supp_states = MBG_NET_INTF_LINK_STATE_MASK_UP;
      net_status_info->link_infos[1].info.supp_types = MBG_NET_INTF_LINK_TYPE_MASK_VLAN;
      net_status_info->link_infos[1].info.supp_speed_modes = MBG_NET_INTF_LINK_SPEED_MODE_MASK_10_T_HALF | MBG_NET_INTF_LINK_SPEED_MODE_MASK_10_T_FULL | MBG_NET_INTF_LINK_SPEED_MODE_MASK_100_T_HALF | MBG_NET_INTF_LINK_SPEED_MODE_MASK_100_T_FULL;
      net_status_info->link_infos[1].info.supp_port_types = MBG_NET_INTF_LINK_PORT_TYPE_MASK_TP;

      snprintf_safe( net_status_info->link_infos[1].info.link_settings.name, sizeof( net_status_info->link_infos[1].info.link_settings.name ), "vlan0" );
      net_status_info->link_infos[1].info.link_settings.mac_addr = net_status_info->link_infos[0].info.link_settings.mac_addr;
      net_status_info->link_infos[1].info.link_settings.if_index = 1;
      net_status_info->link_infos[1].info.link_settings.ass_if_index = 0;
      net_status_info->link_infos[1].info.link_settings.states = net_status_info->link_infos[0].info.link_settings.states;
      net_status_info->link_infos[1].info.link_settings.type = MBG_NET_INTF_LINK_TYPE_VLAN;
      net_status_info->link_infos[1].info.link_settings.vlan_cfg = ip4_state.vlan_cfg;

      net_status_info->addr_infos[0].info.addr_settings.ass_if_index = 1;
    }

    if ( ip4_state.flags & IP4_MSK_DHCP )
      net_status_info->addr_infos[0].info.addr_settings.flags |= MBG_NET_INTF_ADDR_MASK_DHCP4;

    net_status_info->addr_infos[0].info.addr_settings.ip.type = MBG_IP_ADDR_TYPE_IP4;
    net_status_info->addr_infos[0].info.addr_settings.ip.u_addr.ip4_addr = ip4_state.ip_addr;

    net_status_info->addr_infos[0].info.addr_settings.broadcast.type = MBG_IP_ADDR_TYPE_IP4;
    net_status_info->addr_infos[0].info.addr_settings.broadcast.u_addr.ip4_addr = ip4_state.broad_addr;

    net_status_info->addr_infos[0].info.addr_settings.prefix_bits = get_ip4_net_mask_bits(&ip4_state.netmask);

    if ( net_status_info->glb_cfg_info.n_supp_intf_route == 0 )
    {
      if ( net_status_info->route_infos == NULL )
      {
        net_status_info->route_infos = (MBG_NET_INTF_ROUTE_INFO_IDX *) calloc( 1, sizeof( *net_status_info->route_infos ) );
        if ( net_status_info->route_infos == NULL )
        {
          rc = MBG_ERR_NO_MEM;
          goto out;
        }
      }

      net_status_info->glb_cfg_info.n_supp_intf_route = 1;
      net_status_info->glb_cfg_info.glb_settings.num_intf_route = 1;

      memset( &net_status_info->route_infos[0], 0, sizeof( net_status_info->route_infos[0] ) );

      net_status_info->route_infos[0].info.route_settings.type = MBG_NET_INTF_ROUTE_TYPE_DEFAULT_GATEWAY;
      net_status_info->route_infos[0].info.route_settings.gateway.type = MBG_IP_ADDR_TYPE_IP4;
      net_status_info->route_infos[0].info.route_settings.gateway.u_addr.ip4_addr = ip4_state.gateway;
      if ( ip4_state.gateway != 0 )
      {
        if(net_status_info->glb_cfg_info.glb_settings.num_intf_link == 2)
          net_status_info->route_infos[0].info.route_settings.ass_if_index = 1;
      }
      else net_status_info->route_infos[0].info.route_settings.ass_if_index = (uint32_t)-1;

    }

  }

  if ( mbg_rc_is_error( rc ) && mbg_rc_is_error( rc_lan_ip4 ) )
    rc = rc_lan_ip4;

out:
  if ( mbg_rc_is_error( rc ) )
  {
    free_all_net_status_info( net_status_info );
    net_status_info = NULL;
  }

  *p = net_status_info;

  return rc;

} // mbgextio_get_all_net_status_info



/*HDR*/
/**
 * @brief Read all monitoring configuration into an ::ALL_MONITORING_INFO structure
 *
 * A ::ALL_MONITORING_INFO will be allocated and needs
 * to be freed later by calling ::free_all_monitoring_info.
 * A ::ALL_SNMP_INFO may be allocated, if SNMP is supported.
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[out]      p        Pointer to a pointer to ::ALL_MONITORING_INFO
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 * @see ::mbgextio_get_all_snmp_info
 * @see ::free_all_monitoring_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_monitoring_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_MONITORING_INFO **p )
{
  ALL_MONITORING_INFO *monitoring_info = *p;
  int rc;
  int16_t i;

  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_begin_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_MONITORING, 0 );

  if ( monitoring_info == NULL )
  {
    monitoring_info = (ALL_MONITORING_INFO *) calloc( 1, sizeof( *monitoring_info ) );
    if ( monitoring_info == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }
  }

  rc = mbgextio_get_monitoring_limits( pmctl, p_addr, &monitoring_info->limits );
  if ( mbg_rc_is_error( rc ) )
    goto out;

  if ( monitoring_info->event_infos == NULL )
  {
    monitoring_info->event_infos = (MBG_EVENT_INFO_IDX_DATA *) calloc( monitoring_info->limits.supp_num_events, sizeof( *monitoring_info->event_infos ) );
    if ( monitoring_info->event_infos == NULL )
    {
        rc = MBG_ERR_NO_MEM;
        goto out;
    }
  }

  if ( ( monitoring_info->limits.supp_types & MBG_MONITORING_TYPE_MSK_SNMP ) == MBG_MONITORING_TYPE_MSK_SNMP )
    rc = mbgextio_get_all_snmp_info( pmctl, p_addr, &monitoring_info->all_snmp_info );

  if ( monitoring_info->limits.supp_num_events > 0 )
  {
    if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
      mbgextio_begin_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_EVENTS, 0 );

    for ( i = 0; (i < monitoring_info->limits.supp_num_events) && mbg_rc_is_success( rc ) ; ++i )
    {
      monitoring_info->event_infos[i].info.idx = i;
      rc = mbgextio_get_event_info_idx( pmctl, p_addr, &monitoring_info->event_infos[i].info, i );
    }

    if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
      mbgextio_end_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_EVENTS );
  }

out:
  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_end_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_MONITORING );

  if ( mbg_rc_is_error( rc ) )
  {
    free_all_monitoring_info( monitoring_info );
    monitoring_info = NULL;
  }

  *p = monitoring_info;

  return rc;

} // mbgextio_get_all_monitoring_info



/*HDR*/
/**
 * @brief Write all monitoring configuration to a device
 *
 * The complementary function ::mbgextio_get_all_monitoring_info should have been used
 * to read the original monitoring settings and supported configuration parameters.
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[in]       p        Pointer to a ::ALL_MONITORING_INFO structure with all monitoring parameters
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 * @see ::mbgextio_get_all_monitoring_info
 * @see ::mbgextio_save_all_snmp_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_save_all_monitoring_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_MONITORING_INFO *p )
{
  int rc = MBG_SUCCESS;
  int16_t i;

  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_begin_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_MONITORING, 1 );

  if ( ( p->limits.supp_types & MBG_MONITORING_TYPE_MSK_SNMP ) == MBG_MONITORING_TYPE_MSK_SNMP )
    rc = mbgextio_save_all_snmp_info( pmctl, p_addr, p->all_snmp_info );

  if ( p->limits.supp_num_events > 0 )
  {
    if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
      mbgextio_begin_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_EVENTS, 1 );

    for ( i = 0; (i < p->limits.supp_num_events) && mbg_rc_is_success( rc ) ; ++i )
      rc = mbgextio_set_event_settings_idx( pmctl, p_addr, &p->event_infos[i].info.info.settings, i );

    if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
      mbgextio_end_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_EVENTS );
  }

  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_end_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_MONITORING );

  return rc;

} // mbgextio_save_all_monitoring_info



/*HDR*/
/**
 * @brief Read all monitoring status into an ::ALL_MONITORING_STATUS structure
 *
 * A ::ALL_MONITORING_STATUS will be allocated and needs
 * to be freed later by calling ::free_all_monitoring_status.
 * The ::ALL_MONITORING_INFO::limits is used to check which events are supported
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[in]       info     Pointer to a ::ALL_MONITORING_INFO structure which has been read before
 * @param[out]      p        Pointer to a pointer to ::ALL_MONITORING_STATUS
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 * @see ::free_all_monitoring_status
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_monitoring_status( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_MONITORING_INFO *info, ALL_MONITORING_STATUS **p )
{
  ALL_MONITORING_STATUS *monitoring_status = *p;
  int rc;
  int16_t i;

  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_begin_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_MONITORING, 0 );

  if ( monitoring_status == NULL )
  {
    monitoring_status = (ALL_MONITORING_STATUS *) calloc( 1, sizeof( *monitoring_status ) );
    if ( monitoring_status == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }
  }

  rc = mbgextio_get_monitoring_status( pmctl, p_addr, &monitoring_status->status );
  if ( mbg_rc_is_error( rc ) )
    goto out;

  if ( monitoring_status->event_stati == NULL )
  {
    monitoring_status->event_stati = (MBG_EVENT_STATUS_IDX_DATA *) calloc( info->limits.supp_num_events, sizeof( *monitoring_status->event_stati ) );
    if ( monitoring_status->event_stati == NULL )
    {
        rc = MBG_ERR_NO_MEM;
        goto out;
    }
  }

  if ( info->limits.supp_num_events > 0 )
  {
    if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
      mbgextio_begin_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_EVENTS, 0 );

    for ( i = 0; (i < info->limits.supp_num_events) && mbg_rc_is_success( rc ) ; ++i )
    {
      monitoring_status->event_stati[i].status.idx = i;
      rc = mbgextio_get_event_status_idx( pmctl, p_addr, &monitoring_status->event_stati[i].status, i );
    }

    if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
      mbgextio_end_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_EVENTS );
  }

out:
  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_end_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_MONITORING );

  if ( mbg_rc_is_error( rc ) )
  {
    free_all_monitoring_status( monitoring_status );
    monitoring_status = NULL;
  }

  *p = monitoring_status;

  return rc;

} // mbgextio_get_all_monitoring_status



/*HDR*/
/**
 * @brief Read all SNMP configuration into an ::ALL_SNMP_INFO structure
 *
 * A ::ALL_SNMP_INFO and the appropriate number of ::MBG_SNMP_V12_INFO_IDX,
 * ::MBG_SNMP_V12_TRAP_INFO_IDX, ::MBG_SNMP_V3_INFO_IDX, and ::MBG_SNMP_V3_TRAP_INFO_IDX
 * will be allocated and need to be freed later by calling ::free_all_snmp_info
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[out]      p        Pointer to a pointer to ::ALL_SNMP_INFO
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_monitoring
 * @see ::mbgextio_get_snmp_glb_info
 * @see ::mbgextio_get_snmp_v12_info_idx
 * @see ::mbgextio_get_snmp_v12_trap_info_idx
 * @see ::mbgextio_get_snmp_v3_info_idx
 * @see ::mbgextio_get_snmp_v3_trap_info_idx
 * @see ::free_all_snmp_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_snmp_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_SNMP_INFO **p )
{
  ALL_SNMP_INFO *snmp_info = *p;
  int rc;
  int16_t i;

  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_begin_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_MONITORING_SNMP, 0 );

  if ( snmp_info == NULL )
  {
    snmp_info = (ALL_SNMP_INFO *) calloc( 1, sizeof( *snmp_info ) );
    if ( snmp_info == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }
  }

  rc = mbgextio_get_snmp_glb_info( pmctl, p_addr, &snmp_info->glb_info );

  if ( mbg_rc_is_error( rc ) )
    goto out;

  if ( ( ( snmp_info->glb_info.supp_versions & MBG_SNMP_VERSION_MSK_V1 ) == MBG_SNMP_VERSION_MSK_V1 ) ||
       ( ( snmp_info->glb_info.supp_versions & MBG_SNMP_VERSION_MSK_V2c ) == MBG_SNMP_VERSION_MSK_V2c ) )
  {
    snmp_info->v12_infos = ( MBG_SNMP_V12_INFO_IDX * )realloc( snmp_info->v12_infos, snmp_info->glb_info.settings.num_v12_settings * sizeof( *snmp_info->v12_infos ) );
    if ( snmp_info->v12_infos == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }

    for ( i = 0; ( i < snmp_info->glb_info.settings.num_v12_settings ) && mbg_rc_is_success( rc ) ; ++i )
    {
      snmp_info->v12_infos[i].idx = i;
      rc = mbgextio_get_snmp_v12_info_idx( pmctl, p_addr, &snmp_info->v12_infos[i], i );
    }
    if ( mbg_rc_is_error( rc ) )
      goto out;

    snmp_info->v12_trap_infos = ( MBG_SNMP_V12_TRAP_INFO_IDX * )realloc( snmp_info->v12_trap_infos, snmp_info->glb_info.settings.num_v12_trap_receivers * sizeof( *snmp_info->v12_trap_infos ) );
    if ( snmp_info->v12_trap_infos == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }

    for ( i = 0; ( i < snmp_info->glb_info.settings.num_v12_trap_receivers ) && mbg_rc_is_success( rc ) ; ++i )
    {
      snmp_info->v12_trap_infos[i].idx = i;
      rc = mbgextio_get_snmp_v12_trap_info_idx( pmctl, p_addr, &snmp_info->v12_trap_infos[i], i );
    }
    if ( mbg_rc_is_error( rc ) )
      goto out;

  }

  if ( ( snmp_info->glb_info.supp_versions & MBG_SNMP_VERSION_MSK_V3 ) == MBG_SNMP_VERSION_MSK_V3 )
  {
    snmp_info->v3_infos = ( MBG_SNMP_V3_INFO_IDX * )realloc( snmp_info->v3_infos, snmp_info->glb_info.settings.num_v3_settings * sizeof( *snmp_info->v3_infos ) );
    if ( snmp_info->v3_infos == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }

    for ( i = 0; ( i < snmp_info->glb_info.settings.num_v3_settings ) && mbg_rc_is_success( rc ) ; ++i )
    {
      snmp_info->v3_infos[i].idx = i;
      rc = mbgextio_get_snmp_v3_info_idx( pmctl, p_addr, &snmp_info->v3_infos[i], i );
    }
    if ( mbg_rc_is_error( rc ) )
      goto out;

    snmp_info->v3_trap_infos = ( MBG_SNMP_V3_TRAP_INFO_IDX * )realloc( snmp_info->v3_trap_infos, snmp_info->glb_info.settings.num_v3_trap_receivers * sizeof( *snmp_info->v3_trap_infos ) );
    if ( snmp_info->v3_trap_infos == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }

    for ( i = 0; ( i < snmp_info->glb_info.settings.num_v3_trap_receivers ) && mbg_rc_is_success( rc ) ; ++i )
    {
      snmp_info->v3_trap_infos[i].idx = i;
      rc = mbgextio_get_snmp_v3_trap_info_idx( pmctl, p_addr, &snmp_info->v3_trap_infos[i], i );
    }
    if ( mbg_rc_is_error( rc ) )
      goto out;

  }

out:
  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_end_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_MONITORING_SNMP );

  if ( mbg_rc_is_error( rc ) )
  {
    free_all_snmp_info( snmp_info );
    snmp_info = NULL;
  }

  *p = snmp_info;

  return rc;

} // mbgextio_get_all_snmp_info



/*HDR*/
/**
 * @brief Write all SNMP settings to a device
 *
 * The complementary function ::mbgextio_get_all_snmp_info should have been used
 * to read the original SNMP settings and supported configuration parameters.
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[in]       p        Pointer to a ::ALL_SNMP_INFO structure with all SNMP parameters
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_all_snmp_info
 * @see ::mbgextio_set_snmp_glb_settings
 * @see ::mbgextio_set_snmp_v12_settings_idx
 * @see ::mbgextio_set_snmp_v12_trap_settings_idx
 * @see ::mbgextio_set_snmp_v3_settings_idx
 * @see ::mbgextio_set_snmp_v3_trap_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_save_all_snmp_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_SNMP_INFO *p )
{
  int rc;
  int16_t i;

  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_begin_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_MONITORING_SNMP, 1 );

  rc = mbgextio_set_snmp_glb_settings( pmctl, p_addr, &p->glb_info.settings );
  if ( mbg_rc_is_error( rc ) )
    goto out;

  if ( ( ( p->glb_info.supp_versions & MBG_SNMP_VERSION_MSK_V1 ) == MBG_SNMP_VERSION_MSK_V1 ) ||
       ( ( p->glb_info.supp_versions & MBG_SNMP_VERSION_MSK_V2c ) == MBG_SNMP_VERSION_MSK_V2c ) )
  {
    for ( i = 0; ( i < p->glb_info.settings.num_v12_settings ) && mbg_rc_is_success( rc ); ++i )
      rc = mbgextio_set_snmp_v12_settings_idx( pmctl, p_addr, &p->v12_infos[i].info.settings, i );
    if ( mbg_rc_is_error( rc ) )
      goto out;

    for ( i = 0; ( i < p->glb_info.settings.num_v12_trap_receivers ) && mbg_rc_is_success( rc ); ++i )
      rc = mbgextio_set_snmp_v12_trap_settings_idx( pmctl, p_addr, &p->v12_trap_infos[i].info.settings, i );
    if ( mbg_rc_is_error( rc ) )
      goto out;
  }

  if ( ( p->glb_info.supp_versions & MBG_SNMP_VERSION_MSK_V3 ) == MBG_SNMP_VERSION_MSK_V3 )
  {
    for ( i = 0; ( i < p->glb_info.settings.num_v3_settings ) && mbg_rc_is_success( rc ); ++i )
      rc = mbgextio_set_snmp_v3_settings_idx( pmctl, p_addr, &p->v3_infos[i].info.settings, i );
    if ( mbg_rc_is_error( rc ) )
      goto out;

    for ( i = 0; ( i < p->glb_info.settings.num_v3_trap_receivers ) && mbg_rc_is_success( rc ); ++i )
      rc = mbgextio_set_snmp_v3_trap_settings_idx( pmctl, p_addr, &p->v3_trap_infos[i].info.settings, i );
    if ( mbg_rc_is_error( rc ) )
      goto out;
  }

out:
  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_end_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_MONITORING_SNMP );

  return rc;

} // mbgextio_save_all_snmp_info



/*HDR*/
/**
 * @brief Read all PTP configuration into an ::ALL_PTP_CFG_INFO structure
 *
 * @note ::mbgextio_dev_has_ptp should be called to check if this API is supported.
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[out]      p        Pointer to a ::ALL_PTP_CFG_INFO structure to be filled up
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_ptp
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_ptp_cfg_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                             ALL_PTP_CFG_INFO *p )
{
  int rc;

  memset( p, 0, sizeof( *p ) );

  // First check if the device supports PTP.
  rc = mbgextio_dev_has_ptp( pmctl );

  if ( mbg_rc_is_error( rc ) )
    goto out;


  // First get general PTP configuration information
  rc = mbgextio_get_ptp_cfg_info( pmctl, p_addr, &p->ptp_cfg_info );

  if ( mbg_rc_is_error( rc ) )
    goto out;


  // PTP unicast may not be supported by all devices, so check if it is.
  if ( p->ptp_cfg_info.supp_flags & PTP_CFG_MSK_SUPPORT_PTP_UNICAST )
  {
    // Read PTP parameters specific to unicast.
    rc = mbgextio_get_ptp_uc_master_cfg_limits( pmctl, p_addr,
                                           &p->ptp_uc_master_cfg_limits );

    if ( mbg_rc_is_error( rc ) )
      goto out;


    if ( p->ptp_uc_master_cfg_limits.n_supp_master > MAX_PARM_PTP_UC_MASTER )
    {
      // The number of PTP unicast masters supported by this device
      // exceeds the number of unicast masters supported by this API,
      // so the driver library needs to be updated.
      rc = MBG_ERR_N_UC_MSTR_EXCEEDS_SUPP;
      goto out;
    }

    // Read all PTP unicas master configuration.
    rc = mbgextio_get_all_ptp_uc_master_info( pmctl, p_addr,
                  p->all_ptp_uc_master_info_idx, &p->ptp_uc_master_cfg_limits );
  }

out:
  return rc;

}  // mbgextio_get_all_ptp_cfg_info



/*HDR*/
/**
 * @brief Write all PTP settings to a device
 *
 * The complementary function ::mbgextio_get_all_ptp_cfg_info should
 * have been used to read the original PTP settings and supported
 * configuration parameters.
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[in]       p        Pointer to a ::ALL_PTP_CFG_INFO structure with all ptp parameters
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_ptp_cfg_settings
 * @see ::mbgextio_set_ptp_uc_master_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_save_all_ptp_cfg_info(MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, const ALL_PTP_CFG_INFO *p )
{
  int rc = mbgextio_set_ptp_cfg_settings( pmctl, p_addr, &p->ptp_cfg_info.settings );

  if ( mbg_rc_is_error( rc ) )
    goto out;

  if ( p->ptp_cfg_info.supp_flags & PTP_CFG_MSK_SUPPORT_PTP_UNICAST )
  {
    int16_t i;

    for ( i = 0; i < p->ptp_uc_master_cfg_limits.n_supp_master && i < MAX_PARM_PTP_UC_MASTER; i++ )
    {
      rc = mbgextio_set_ptp_uc_master_settings_idx( pmctl, p_addr, &p->all_ptp_uc_master_info_idx[i].info.settings, i );

      if ( mbg_rc_is_error( rc ) )
        goto out;
    }
  }

out:
  return rc;

}  // mbgextio_save_all_ptp_cfg_info



/*HDR*/
/**
 * @brief Read all PTPv1 common datasets into a newly or re-allocated ::ALL_PTP_V1_COMMON_DATASETS structure
 *
 * @note This is only supported if ::PTP_CFG_MSK_HAS_V1_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 * A number of ::MBG_PTP_V1_DEFAULT_DATASET::number_ports port datasets will be allocated.
 * To free the allocated ::ALL_PTP_V1_COMMON_DATASETS, ::free_all_ptp_v1_common_datasets should be called.
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[out]      p        Pointer to a pointer of ::ALL_PTP_V1_COMMON_DATASETS structure to be allocated and filled
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_get_ptp_v1_default_dataset
 * @see ::mbgextio_get_ptp_v1_current_dataset
 * @see ::mbgextio_get_ptp_v1_parent_dataset
 * @see ::mbgextio_get_ptp_v1_time_properties_dataset
 * @see ::mbgextio_get_ptp_v1_port_dataset_idx
 * @see ::free_all_ptp_v1_common_datasets
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_ptp_v1_common_datasets( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                       ALL_PTP_V1_COMMON_DATASETS **p )
{
  ALL_PTP_V1_COMMON_DATASETS *ptp_v1_common_datasets = *p;
  int rc;
  int16_t i;

  if ( ptp_v1_common_datasets == NULL )
  {
    ptp_v1_common_datasets = (ALL_PTP_V1_COMMON_DATASETS *) calloc( 1, sizeof( *ptp_v1_common_datasets ) );
    if ( ptp_v1_common_datasets == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }
  }

  rc = mbgextio_get_ptp_v1_default_dataset( pmctl, p_addr, &ptp_v1_common_datasets->default_dataset );
  if ( mbg_rc_is_error( rc ) )
    goto out;

  rc = mbgextio_get_ptp_v1_current_dataset( pmctl, p_addr, &ptp_v1_common_datasets->current_dataset );
  if ( mbg_rc_is_error( rc ) )
    goto out;

  rc = mbgextio_get_ptp_v1_parent_dataset( pmctl, p_addr, &ptp_v1_common_datasets->parent_dataset );
  if ( mbg_rc_is_error( rc ) )
    goto out;

  rc = mbgextio_get_ptp_v1_time_properties_dataset( pmctl, p_addr, &ptp_v1_common_datasets->time_properties_dataset );
  if ( mbg_rc_is_error( rc ) )
    goto out;

  if ( ptp_v1_common_datasets->default_dataset.number_ports > 0 )
  {
    ptp_v1_common_datasets->port_datasets = ( MBG_PTP_V1_PORT_DATASET_IDX* )realloc( ptp_v1_common_datasets->port_datasets, ptp_v1_common_datasets->default_dataset.number_ports * sizeof( *ptp_v1_common_datasets->port_datasets ) );
    if ( ptp_v1_common_datasets->port_datasets == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }

    for ( i = 0; ( i < ptp_v1_common_datasets->default_dataset.number_ports ) && mbg_rc_is_success( rc ) ; ++i )
    {
      ptp_v1_common_datasets->port_datasets[i].idx = i;
      rc = mbgextio_get_ptp_v1_port_dataset_idx( pmctl, p_addr, &ptp_v1_common_datasets->port_datasets[i], i );
    }
  }
  else
  {
    if ( ptp_v1_common_datasets->port_datasets )
      free( ptp_v1_common_datasets->port_datasets );
    ptp_v1_common_datasets->port_datasets = NULL;
  }

out:
  if ( mbg_rc_is_error( rc ) )
  {
    free_all_ptp_v1_common_datasets( ptp_v1_common_datasets );
    ptp_v1_common_datasets = NULL;
  }

  *p = ptp_v1_common_datasets;

  return rc;

} // mbgextio_get_ptp_v1_common_datasets



/*HDR*/
/**
 * @brief Read all PTPv2 common datasets into a newly or re-allocated ::ALL_PTP_V2_COMMON_DATASETS structure
 *
 * @note This is only supported if ::PTP_CFG_MSK_HAS_V2_COMMON_DATASETS is set in ::PTP_CFG_INFO::supp_flags
 * A number of ::MBG_PTP_V2_DEFAULT_DATASET::number_ports port datasets will be allocated.
 * To free the allocated ::ALL_PTP_V2_COMMON_DATASETS, ::free_all_ptp_v2_common_datasets should be called.
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[out]      p        Pointer to a pointer of ::ALL_PTP_V2_COMMON_DATASETS structure to be allocated and filled
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_ptp_cfg_info
 * @see ::mbgextio_get_ptp_v2_default_dataset
 * @see ::mbgextio_get_ptp_v2_current_dataset
 * @see ::mbgextio_get_ptp_v2_parent_dataset
 * @see ::mbgextio_get_ptp_v2_time_properties_dataset
 * @see ::mbgextio_get_ptp_v2_port_dataset_idx
 * @see ::free_all_ptp_v2_common_datasets
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_ptp_v2_common_datasets( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                                                       ALL_PTP_V2_COMMON_DATASETS **p )
{
  ALL_PTP_V2_COMMON_DATASETS *ptp_v2_common_datasets = *p;
  int rc;
  int16_t i;

  if ( ptp_v2_common_datasets == NULL )
  {
    ptp_v2_common_datasets = (ALL_PTP_V2_COMMON_DATASETS *) calloc( 1, sizeof( *ptp_v2_common_datasets ) );
    if ( ptp_v2_common_datasets == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }
  }

  rc = mbgextio_get_ptp_v2_default_dataset( pmctl, p_addr, &ptp_v2_common_datasets->default_dataset );
  if ( mbg_rc_is_error( rc ) )
    goto out;

  rc = mbgextio_get_ptp_v2_current_dataset( pmctl, p_addr, &ptp_v2_common_datasets->current_dataset );
  if ( mbg_rc_is_error( rc ) )
    goto out;

  rc = mbgextio_get_ptp_v2_parent_dataset( pmctl, p_addr, &ptp_v2_common_datasets->parent_dataset );
  if ( mbg_rc_is_error( rc ) )
    goto out;

  rc = mbgextio_get_ptp_v2_time_properties_dataset( pmctl, p_addr, &ptp_v2_common_datasets->time_properties_dataset );
  if ( mbg_rc_is_error( rc ) )
    goto out;

  if ( ptp_v2_common_datasets->default_dataset.number_ports > 0 )
  {
    ptp_v2_common_datasets->port_datasets = ( MBG_PTP_V2_PORT_DATASET_IDX* )realloc( ptp_v2_common_datasets->port_datasets, ptp_v2_common_datasets->default_dataset.number_ports * sizeof( *ptp_v2_common_datasets->port_datasets ) );
    if ( ptp_v2_common_datasets->port_datasets == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }

    for ( i = 0; ( i < ptp_v2_common_datasets->default_dataset.number_ports ) && mbg_rc_is_success( rc ) ; ++i )
    {
      ptp_v2_common_datasets->port_datasets[i].idx = i;
      rc = mbgextio_get_ptp_v2_port_dataset_idx( pmctl, p_addr, &ptp_v2_common_datasets->port_datasets[i], i );
    }
  }
  else
  {
    if ( ptp_v2_common_datasets->port_datasets )
      free( ptp_v2_common_datasets->port_datasets );
    ptp_v2_common_datasets->port_datasets = NULL;
  }

out:
  if ( mbg_rc_is_error( rc ) )
  {
    free_all_ptp_v2_common_datasets( ptp_v2_common_datasets );
    ptp_v2_common_datasets = NULL;
  }

  *p = ptp_v2_common_datasets;

  return rc;

} // mbgextio_get_ptp_v2_common_datasets



#if defined( _PRELIMINARY_CODE )

/*HDR*/
/**
 * @brief Read all NTP configuration into a newly or re-allocated ::ALL_NTP_CFG_INFO structure
 *
 * @note ::mbgextio_dev_has_ntp should be called to check if this API is supported.
 *
 * A ::ALL_NTP_CFG_INFO and ::NTP_SRV_MODE_INFO, ::NTP_CLNT_MODE_INFO and a number of
 * ::NTP_CLNT_MODE_INFO::settings::num_peers of ::NTP_PEER_SETTINGS_IDX will be allocated and
 * needs to be freed by calling ::free_all_ntp_cfg_info
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[out]      p        Pointer to a pointer to ::ALL_NTP_CFG_INFO
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_ntp
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_ntp_cfg_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_NTP_CFG_INFO **p )
{
  ALL_NTP_CFG_INFO *ntp_cfg_info = *p;
  int rc, i;

  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_begin_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_NTP, 0 );

  if ( ntp_cfg_info == NULL )
  {
    ntp_cfg_info = (ALL_NTP_CFG_INFO *) calloc( 1, sizeof( *ntp_cfg_info ) );
    if ( ntp_cfg_info == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }
  }

  // read NTP_GLB_INFO
  rc = mbgextio_get_ntp_glb_info( pmctl, p_addr, &ntp_cfg_info->glb_info );

  if ( mbg_rc_is_error( rc ) )
    goto out;

  // read NTP_SYMM_KEY_LIMITS and NTP_SYMM_KEY_INFO_IDX
  if ( ( ntp_cfg_info->glb_info.supp_flags & NTP_MSK_SYMM_KEYS ) == NTP_MSK_SYMM_KEYS )
  {
    if ( ntp_cfg_info->symm_key_limits == NULL )
    {
      ntp_cfg_info->symm_key_limits = (NTP_SYMM_KEY_LIMITS *) calloc( 1, sizeof( *ntp_cfg_info->symm_key_limits ) );
      if ( ntp_cfg_info->symm_key_limits == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }
    }

    rc = mbgextio_get_ntp_symm_key_limits( pmctl, p_addr, ntp_cfg_info->symm_key_limits );

    if ( mbg_rc_is_error ( rc ) )
      goto out;

    if ( ntp_cfg_info->glb_info.settings.num_symm_keys > 0 )
    {
      ntp_cfg_info->symm_key_info_idx = ( NTP_SYMM_KEY_INFO_IDX * )realloc( ntp_cfg_info->symm_key_info_idx, ntp_cfg_info->glb_info.settings.num_symm_keys * sizeof( *ntp_cfg_info->symm_key_info_idx ) );
      if ( ntp_cfg_info->symm_key_info_idx == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }

      for ( i = 0; ( i < ntp_cfg_info->glb_info.settings.num_symm_keys ) && mbg_rc_is_success( rc ); ++i )
      {
        ntp_cfg_info->symm_key_info_idx[i].idx = i;
        rc = mbgextio_get_ntp_symm_key_info_idx( pmctl, p_addr, &ntp_cfg_info->symm_key_info_idx[i], i );
      }

      if ( mbg_rc_is_error ( rc ) )
        goto out;
    }
    else
    {
      if ( ntp_cfg_info->symm_key_info_idx  != NULL )
      {
        free( ntp_cfg_info->symm_key_info_idx );
        ntp_cfg_info->symm_key_info_idx = NULL;
      }
    }
  }

  // read NTP_TRUSTED_KEY_INFO_IDX
  if ( ( ntp_cfg_info->glb_info.supp_flags & NTP_MSK_TRUSTED_KEYS ) == NTP_MSK_TRUSTED_KEYS )
  {
    if ( ntp_cfg_info->glb_info.settings.num_trusted_keys > 0 )
    {
      ntp_cfg_info->trusted_key_info_idx = ( NTP_TRUSTED_KEY_INFO_IDX * )realloc( ntp_cfg_info->trusted_key_info_idx, ntp_cfg_info->glb_info.settings.num_trusted_keys * sizeof( *ntp_cfg_info->trusted_key_info_idx ) );
      if ( ntp_cfg_info->trusted_key_info_idx == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }

      for ( i = 0; ( i < ntp_cfg_info->glb_info.settings.num_trusted_keys ) && mbg_rc_is_success( rc ); ++i )
      {
        ntp_cfg_info->trusted_key_info_idx[i].idx = i;
        rc = mbgextio_get_ntp_trusted_key_info_idx( pmctl, p_addr, &ntp_cfg_info->trusted_key_info_idx[i], i );
      }

      if ( mbg_rc_is_error ( rc ) )
        goto out;
    }
    else
    {
      if ( ntp_cfg_info->trusted_key_info_idx  != NULL )
      {
        free( ntp_cfg_info->trusted_key_info_idx );
        ntp_cfg_info->trusted_key_info_idx = NULL;
      }
    }
  }

  // read NTP_CLNT_MODE_INFO and NTP_PEER_SETTINGS_IDX
  if ( ( ( ntp_cfg_info->glb_info.supp_ntp_roles & NTP_MSK_ROLE_CLIENT ) == NTP_MSK_ROLE_CLIENT ) ||
       ( ( ntp_cfg_info->glb_info.supp_ntp_roles & NTP_MSK_ROLE_CLIENT_SERVER ) == NTP_MSK_ROLE_CLIENT_SERVER ) )
  {
    if ( ntp_cfg_info->clnt_info == NULL )
    {
      ntp_cfg_info->clnt_info = (NTP_CLNT_MODE_INFO *) calloc( 1, sizeof( *ntp_cfg_info->clnt_info ) );
      if ( ntp_cfg_info->clnt_info == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }
    }

    rc = mbgextio_get_ntp_clnt_mode_info( pmctl, p_addr, ntp_cfg_info->clnt_info );

    if ( mbg_rc_is_error ( rc ) )
      goto out;

    if ( ntp_cfg_info->clnt_info->settings.num_peers > 0 )
    {
      ntp_cfg_info->peer_settings_idx = ( NTP_PEER_SETTINGS_IDX * )realloc( ntp_cfg_info->peer_settings_idx, ntp_cfg_info->clnt_info->settings.num_peers * sizeof( *ntp_cfg_info->peer_settings_idx ) );
      if ( ntp_cfg_info->peer_settings_idx == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }

      for ( i = 0; ( i < ntp_cfg_info->clnt_info->settings.num_peers ) && mbg_rc_is_success( rc ); ++i )
      {
        ntp_cfg_info->peer_settings_idx[i].idx = i;
        rc = mbgextio_get_ntp_peer_settings_idx( pmctl, p_addr, &ntp_cfg_info->peer_settings_idx[i], i );
      }

      if ( mbg_rc_is_error ( rc ) )
        goto out;
    }
    else
    {
      if ( ntp_cfg_info->peer_settings_idx != NULL )
      {
        free( ntp_cfg_info->peer_settings_idx );
        ntp_cfg_info->peer_settings_idx = NULL;
      }
    }
  }

  // read NTP_SRV_MODE_INFO and NTP_REFCLK_CFG_INFO_IDX
  if ( ( ( ntp_cfg_info->glb_info.supp_ntp_roles & NTP_MSK_ROLE_SERVER ) == NTP_MSK_ROLE_SERVER ) ||
       ( ( ntp_cfg_info->glb_info.supp_ntp_roles & NTP_MSK_ROLE_CLIENT_SERVER ) == NTP_MSK_ROLE_CLIENT_SERVER ) )
  {
    if ( ntp_cfg_info->srv_info == NULL )
    {
      ntp_cfg_info->srv_info = (NTP_SRV_MODE_INFO *) calloc( 1, sizeof( *ntp_cfg_info->srv_info ) );
      if ( ntp_cfg_info->srv_info == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }
    }

    rc = mbgextio_get_ntp_srv_mode_info( pmctl, p_addr, ntp_cfg_info->srv_info );

    if ( mbg_rc_is_error ( rc ) )
      goto out;

    if ( ntp_cfg_info->srv_info->settings.num_refclks > 0 )
    {
      ntp_cfg_info->refclk_info_idx = ( NTP_REFCLK_CFG_INFO_IDX * )realloc( ntp_cfg_info->refclk_info_idx, ntp_cfg_info->srv_info->settings.num_refclks * sizeof( *ntp_cfg_info->refclk_info_idx ) );
      if ( ntp_cfg_info->refclk_info_idx == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }

      for ( i = 0; ( i < ntp_cfg_info->srv_info->settings.num_refclks ) && mbg_rc_is_success( rc ); ++i )
      {
        ntp_cfg_info->refclk_info_idx[i].idx = i;
        rc = mbgextio_get_ntp_refclk_cfg_info_idx( pmctl, p_addr, &ntp_cfg_info->refclk_info_idx[i], i );
      }

      if ( mbg_rc_is_error ( rc ) )
        goto out;
    }
    else
    {
      if ( ntp_cfg_info->refclk_info_idx != NULL )
      {
        free( ntp_cfg_info->refclk_info_idx );
        ntp_cfg_info->refclk_info_idx = NULL;
      }
    }
  }

  // read NTP_MISC_LIMITS
  if ( ( ntp_cfg_info->glb_info.supp_flags & NTP_MSK_MISCELLANEOUS ) == NTP_MSK_MISCELLANEOUS )
  {
    if ( ntp_cfg_info->misc_limits == NULL )
    {
      ntp_cfg_info->misc_limits = (NTP_MISC_LIMITS *) calloc( 1, sizeof( *ntp_cfg_info->misc_limits ) );
      if ( ntp_cfg_info->misc_limits == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }
    }

    rc = mbgextio_get_ntp_misc_limits( pmctl, p_addr, ntp_cfg_info->misc_limits );

    if ( mbg_rc_is_error ( rc ) )
      goto out;

    // read NTP_MISC_ORPHAN_MODE_INFO
    if ( ( ntp_cfg_info->misc_limits->supp_flags & NTP_MISC_MSK_ORPHAN_MODE ) == NTP_MISC_MSK_ORPHAN_MODE )
    {
      if ( ntp_cfg_info->orphan_mode_info == NULL )
      {
        ntp_cfg_info->orphan_mode_info = (NTP_MISC_ORPHAN_MODE_INFO *) calloc( 1, sizeof( *ntp_cfg_info->orphan_mode_info ) );
        if ( ntp_cfg_info->orphan_mode_info == NULL )
        {
          rc = MBG_ERR_NO_MEM;
          goto out;
        }
      }

      rc = mbgextio_get_ntp_misc_orphan_mode_info( pmctl, p_addr, ntp_cfg_info->orphan_mode_info );
    }
  }

out:
  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_end_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_NTP );

  if ( mbg_rc_is_error( rc ) )
  {
    free_all_ntp_cfg_info( ntp_cfg_info );
    ntp_cfg_info = NULL;
  }

  *p = ntp_cfg_info;

  return rc;

} // mbgextio_get_all_ntp_cfg_info

#endif  // defined( _PRELIMINARY_CODE )



#if defined( _PRELIMINARY_CODE )

/*HDR*/
/**
 * @brief Write all NTP settings to a device
 *
 * The complementary function ::mbgextio_get_all_ntp_cfg_info should
 * have been used to read the original NTP settings and supported
 * configuration parameters.
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[in]       p        Pointer to a ::ALL_NTP_CFG_INFO structure with all ntp parameters
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_ptp_cfg_settings
 * @see ::mbgextio_set_ptp_uc_master_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_save_all_ntp_cfg_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_NTP_CFG_INFO *p )
{
  int rc, i;

  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_begin_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_NTP, 1 );

  // set NTP_GLB_SETTINGS
  rc =  mbgextio_set_ntp_glb_info( pmctl, p_addr, &p->glb_info.settings );

  if ( mbg_rc_is_error( rc ) )
    goto out;

  // set NTP_SYMM_KEY_SETTINGS_IDX
  if ( ( p->glb_info.supp_flags & NTP_MSK_SYMM_KEYS ) == NTP_MSK_SYMM_KEYS )
  {
    for ( i = 0; ( i < p->glb_info.settings.num_symm_keys ) && mbg_rc_is_success( rc ); ++i )
      rc = mbgextio_set_ntp_symm_key_settings_idx( pmctl, p_addr, &p->symm_key_info_idx[i].info.settings, i );

    if ( mbg_rc_is_error( rc ) )
      goto out;
  }

  // set NTP_TRUSTED_KEY_SETTINGS_IDX
  if ( ( p->glb_info.supp_flags & NTP_MSK_TRUSTED_KEYS ) == NTP_MSK_TRUSTED_KEYS )
  {
    for ( i = 0; ( i < p->glb_info.settings.num_trusted_keys ) && mbg_rc_is_success( rc ); ++i )
      rc = mbgextio_set_ntp_trusted_key_settings_idx( pmctl, p_addr, &p->trusted_key_info_idx[i].info.settings, i );

    if ( mbg_rc_is_error( rc ) )
      goto out;
  }

  // set NTP_CLNT_MODE_SETTINGS and NTP_PEER_SETTINGS
  if ( ( ( p->glb_info.supp_ntp_roles & NTP_MSK_ROLE_CLIENT ) == NTP_MSK_ROLE_CLIENT ) ||
       ( ( p->glb_info.supp_ntp_roles & NTP_MSK_ROLE_CLIENT_SERVER ) == NTP_MSK_ROLE_CLIENT_SERVER ) )
  {
    NTP_PEER_SETTINGS_IDX peer_settings_idx;

    rc = mbgextio_set_ntp_clnt_mode_cfg( pmctl, p_addr, &p->clnt_info->settings );

    for ( i = 0; ( i < p->clnt_info->settings.num_peers ) && mbg_rc_is_success( rc ); ++i )
      rc = mbgextio_set_ntp_peer_settings_idx( pmctl, p_addr, &p->peer_settings_idx[i].peer_settings, i );

    if(mbg_rc_is_error(rc))
      goto out;

    // this is for compatibility with older Syncbox N2x versions
    memset(&peer_settings_idx, 0, sizeof(peer_settings_idx));
    for ( i = p->clnt_info->settings.num_peers; ( i < p->clnt_info->n_supp_peers ) && mbg_rc_is_success( rc ); ++i )
    {
      peer_settings_idx.idx = i;
      rc = mbgextio_set_ntp_peer_settings_idx( pmctl, p_addr, &peer_settings_idx.peer_settings, i );
    }
    rc = MBG_SUCCESS;
  }

  // set NTP_SRV_MODE_SETTINGS and NTP_REFCLK_CFG_SETTINGS
  if ( ( ( p->glb_info.supp_ntp_roles & NTP_MSK_ROLE_SERVER ) == NTP_MSK_ROLE_SERVER ) ||
       ( ( p->glb_info.supp_ntp_roles & NTP_MSK_ROLE_CLIENT_SERVER ) == NTP_MSK_ROLE_CLIENT_SERVER ) )
  {
    rc = mbgextio_set_ntp_srv_mode_cfg( pmctl, p_addr, &p->srv_info->settings );

    if ( ( p->glb_info.supp_flags & NTP_MSK_FIXED_REFCLOCKS) != NTP_MSK_FIXED_REFCLOCKS )
    {
      for ( i = 0; ( i < p->srv_info->settings.num_refclks ) && mbg_rc_is_success( rc ); ++i )
        rc = mbgextio_set_ntp_refclk_cfg_settings_idx( pmctl, p_addr, &p->refclk_info_idx[i].info.settings, i );
    }

    if(mbg_rc_is_error(rc))
      goto out;
  }

  if ( ( p->glb_info.supp_flags & NTP_MSK_MISCELLANEOUS ) == NTP_MSK_MISCELLANEOUS )
  {
    // set NTP_MISC_ORPHAN_MODE_SETTINGS
    if ( ( p->misc_limits->supp_flags & NTP_MISC_MSK_ORPHAN_MODE ) == NTP_MISC_MSK_ORPHAN_MODE )
      rc = mbgextio_set_ntp_misc_orphan_mode_settings( pmctl, p_addr, &p->orphan_mode_info->settings );
  }

out:
  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_end_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_NTP );

  return rc;

} // mbgextio_save_all_ntp_cfg_info

#endif  // defined( _PRELIMINARY_CODE )



#if defined( _PRELIMINARY_CODE )

/*HDR*/
/**
 * @brief Read all NTP status info into a newly or re-allocated ::ALL_NTP_STATUS structure
 *
 * @note ::mbgextio_dev_has_ntp should be called to check if this API is supported.
 *
 * A ::ALL_NTP_STATUS and ::NTP_SRV_MODE_INFO, ::NTP_CLNT_MODE_INFO and a number of
 * ::NTP_CLNT_MODE_INFO::settings::num_peers of ::NTP_PEER_STATE_IDX will be allocated and
 * needs to be freed by calling ::free_all_ntp_status
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[in]       info     Pointer to the appropriate info structure
 * @param[out]      p        Pointer to a pointer to ::ALL_NTP_STATUS
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_ntp
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_ntp_status( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, const ALL_NTP_CFG_INFO *info, ALL_NTP_STATUS **p )
{
  ALL_NTP_STATUS *ntp_status = *p;
  int rc, i;

  if ( info == NULL )
    return MBG_ERR_INV_PARM;

  if ( ntp_status == NULL )
  {
    ntp_status = (ALL_NTP_STATUS *) calloc( 1, sizeof( *ntp_status ) );
    if ( ntp_status == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }
  }

  rc = mbgextio_get_ntp_sys_state( pmctl, p_addr, &ntp_status->sys_state );

  if ( mbg_rc_is_error( rc ) )
    goto out;

  if ( mbg_rc_is_success( chk_dev_ntp_supp_client( info ) ) )
  {
    if ( mbg_rc_is_error ( rc ) )
      goto out;

    if ( ntp_status->peer_states == NULL )
    {
      ntp_status->peer_states = (NTP_PEER_STATE_IDX *) calloc(info->clnt_info->settings.num_peers, sizeof( *ntp_status->peer_states ) );
      if ( ntp_status->peer_states == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }
    }

    for ( i = 0; i < info->clnt_info->settings.num_peers; ++i )
    {
      ntp_status->peer_states[i].idx = i;
      rc = mbgextio_get_ntp_peer_state_idx( pmctl, p_addr, &ntp_status->peer_states[i], i );
    }
  }

  if ( mbg_rc_is_success( chk_dev_ntp_supp_server(info) ) )
  {
    // TODO: At this point, server status has to bread as soon as its implemented
  }

out:
  if ( mbg_rc_is_error( rc ) )
  {
    free_all_ntp_status( ntp_status );
    ntp_status = NULL;
  }

  *p = ntp_status;

  return rc;

} // mbgextio_get_all_ntp_status

#endif  // defined( _PRELIMINARY_CODE )



/*HDR*/
/**
 * @brief Read all XMR info into a newly or re-allocated ::ALL_XMULTI_REF_INFO
 *
 * @note ::mbgextio_dev_has_xmulti_ref should be called before using this function
 *
 * A ::ALL_XMULTI_REF_INFO and a number of ::XMULTI_REF_INSTANCES::n_xmr_settings
 * of ::XMULTI_REF_INFO_IDX and ::XMR_EXT_SRC_INFO_IDX will be allocated and needs
 * to be freed by calling ::free_all_xmulti_ref_info
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[out]      p        Pointer to a pointer of ::ALL_XMULTI_REF_INFO
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::free_all_xmulti_ref_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_xmulti_ref_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_XMULTI_REF_INFO **p )
{
  ALL_XMULTI_REF_INFO *xmr_info = *p;
  int rc;
  int16_t i;

  if ( xmr_info == NULL )
  {
    xmr_info = (ALL_XMULTI_REF_INFO *) calloc( 1, sizeof( *xmr_info ) );
    if ( xmr_info == NULL )
      return MBG_ERR_NO_MEM;
  }

  // First, get the XMULTI_REF_INSTANCES to check how many sources are supported
  rc = mbgextio_get_xmr_instances( pmctl, p_addr, &xmr_info->instances );

  if ( mbg_rc_is_error( rc ) )
    goto out;

  if ( xmr_info->infos == NULL )
  {
    xmr_info->infos = (XMULTI_REF_INFO_IDX *) calloc( xmr_info->instances.n_xmr_settings, sizeof( *xmr_info->infos ) );

    if ( xmr_info->infos == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }
  }

  for ( i = 0; i < xmr_info->instances.n_xmr_settings; i++ )
  {
    xmr_info->infos[i].idx = i;
    rc = mbgextio_get_xmr_info_idx( pmctl, p_addr, &xmr_info->infos[i], i );
    if ( mbg_rc_is_error( rc ) )
      goto out;
  }

  if ( mbg_rc_is_success( chk_dev_xmulti_ref_supp_ext_src_info( xmr_info ) ) )
  {
    if ( xmr_info->ext_src_infos == NULL )
    {
      xmr_info->ext_src_infos = (XMR_EXT_SRC_INFO_IDX *) calloc( N_MULTI_REF, sizeof( *xmr_info->ext_src_infos ) );

      if ( xmr_info->ext_src_infos == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }
    }

    for ( i = 0; i < N_MULTI_REF; i++ )
    {
      xmr_info->ext_src_infos[i].idx = i;
      rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_XMR_EXT_SRC_INFO_IDX, i, &xmr_info->ext_src_infos[i], sizeof( xmr_info->ext_src_infos[i] ) );
      if ( ( mbg_rc_is_error( rc ) ) )
        goto out;
    }
  }

out:
  if ( mbg_rc_is_error( rc ) )
  {
    free_all_xmulti_ref_info( xmr_info );
    xmr_info = NULL;
  }

  *p = xmr_info;

  return rc;

} // mbgextio_get_all_xmulti_ref_info


/*HDR*/
/**
 * @brief Set all extended multi ref settings to a device
 *
 * The complementary function ::mbgextio_get_all_xmulti_ref_info should
 * have been used to read the original extended multi ref settings.
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[in]       p        Pointer to a ::ALL_XMULTI_REF_INFO structure with all settings
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_set_xmr_settings_idx
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_save_all_xmulti_ref_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_XMULTI_REF_INFO *p )
{
  int rc = MBG_SUCCESS;
  int16_t i;

  for (i = 0; ( i < p->instances.n_xmr_settings ) && mbg_rc_is_success( rc ); i++ )
    rc = mbgextio_set_xmr_settings_idx( pmctl, p_addr, &p->infos[i].info.settings, i );

  // if all settings have been successully set, send dummy structure with index -1 to apply settings
  if( mbg_rc_is_success( rc ) )
  {
    XMULTI_REF_SETTINGS dummy_settings;
    memset( &dummy_settings, 0, sizeof( dummy_settings ) );
    dummy_settings.id.type = MULTI_REF_NONE;
    rc = mbgextio_set_xmr_settings_idx( pmctl, p_addr, &dummy_settings, -1 );
  }

  return rc;

} // mbgextio_save_all_xmulti_ref_info


/*HDR*/
/**
 * @brief Read all XMR status info into a newly or re-allocated ::ALL_XMULTI_REF_STATUS
 *
 * @note ::mbgextio_dev_has_xmulti_ref should be called before using this function
 *
 * A ::ALL_XMULTI_REF_STATUS and a number of ::XMULTI_REF_INSTANCES::n_xmr_settings
 * of ::XMULTI_REF_STATUS_IDX will be allocated and needs to be freed by calling
 * ::free_all_xmulti_ref_status
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[in]       info     Pointer to an ::ALL_XMULTI_REF_INFO list which has been read before
 * @param[out]      p        Pointer to a pointer of ::ALL_XMULTI_REF_STATUS
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::free_all_xmulti_ref_status
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_xmulti_ref_status( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, const ALL_XMULTI_REF_INFO* info, ALL_XMULTI_REF_STATUS **p )
{
  ALL_XMULTI_REF_STATUS *xmr_status = *p;
  int rc = MBG_SUCCESS;
  int16_t i;

  if ( info == NULL )
    return MBG_ERR_INV_PARM;

  if ( xmr_status == NULL )
  {
    xmr_status = (ALL_XMULTI_REF_STATUS *) calloc( 1, sizeof( *xmr_status ) );
    if ( xmr_status == NULL )
      return MBG_ERR_NO_MEM;
  }

  if ( xmr_status->status == NULL )
  {
    xmr_status->status = (XMULTI_REF_STATUS_IDX *) calloc( info->instances.n_xmr_settings, sizeof( *xmr_status->status ) );

    if ( xmr_status->status == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }
  }

  for ( i = 0; i < info->instances.n_xmr_settings; i++ )
  {
    xmr_status->status[i].idx = i;
    rc = mbgextio_get_xmr_status_idx( pmctl, p_addr, &xmr_status->status[i], i );
    if ( mbg_rc_is_error( rc ) )
      goto out;
  }

  if ( mbg_rc_is_success( chk_dev_xmulti_ref_supp_holdover_status( info ) ) )
  {
    if ( xmr_status->holdover_status == NULL )
    {
      xmr_status->holdover_status = (XMR_HOLDOVER_STATUS *) calloc( 1, sizeof( *xmr_status->holdover_status ) );

      if ( xmr_status->holdover_status == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }
    }

    rc = mbgextio_get_xmr_holdover_status(pmctl, p_addr, xmr_status->holdover_status);
    if ( mbg_rc_is_error( rc ) )
      goto out;
  }

  if ( mbg_rc_is_success( chk_dev_xmulti_ref_supp_ext_src_info( info ) ) )
  {
    uint8_t type;

    if ( xmr_status->stats_idx == NULL )
    {
      xmr_status->stats_idx = (XMR_STATS_IDX *) calloc( info->instances.n_xmr_settings, sizeof( *xmr_status->stats_idx ) );

      if ( xmr_status->stats_idx == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }
    }

    xmr_status->has_stats = 0;

    for ( i = 0;  i < info->instances.n_xmr_settings; ++i )
    {
      type = info->infos[i].info.settings.id.type;

      /* We only malloced N_MULTI_REF ext source info idxs... */
      if ( mbg_rc_is_success( chk_dev_xmulti_ref_supp_ext_source_stats( info, type ) ) )
      {
        rc = mbgextio_get_xmr_ext_source_stats_idx( pmctl, p_addr, &xmr_status->stats_idx[i], i );
        if ( mbg_rc_is_error( rc ) )
          goto out;
      }

      xmr_status->has_stats = 1;
    }

    if ( xmr_status->metrics_idx == NULL )
    {
      xmr_status->metrics_idx = (XMR_METRICS_IDX *) calloc( info->instances.n_xmr_settings, sizeof( *xmr_status->metrics_idx ) );

      if ( xmr_status->metrics_idx == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }
    }

    xmr_status->has_metrics = 0;

    for ( i = 0;  i < info->instances.n_xmr_settings; ++i )
    {
      type = info->infos[i].info.settings.id.type;

      /* We only malloced N_MULTI_REF ext source info idxs... */
      if ( mbg_rc_is_success( chk_dev_xmulti_ref_supp_ext_source_metrics( info, type ) ) )
      {
        rc = mbgextio_get_xmr_ext_source_metrics_idx( pmctl, p_addr, &xmr_status->metrics_idx[i], i );
        if ( mbg_rc_is_error( rc ) )
          goto out;

        xmr_status->has_metrics = 1;
      }
    }

  }

out:
  if ( mbg_rc_is_error( rc ) )
  {
    free_all_xmulti_ref_status( xmr_status );
    xmr_status = NULL;
  }

  *p = xmr_status;

  return rc;

}  //  mbgextio_get_all_xmulti_ref_status



#if defined( _PRELIMINARY_CODE )

/*HDR*/
/**
 * @brief Read all IMS related information / configuration into an ::ALL_IMS_INFO structure
 *
 * @note ::mbgextio_dev_has_ims should be called before using this function
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in,out] pp     Pointer to a pointer of ::ALL_IMS_INFO structure which will be malloced
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_ims
 * @see ::mbgextio_dev_ims_has_fdm
 * @see ::free_all_ims_info
 * @see ::mbgextio_get_all_ims_state
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_ims_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_IMS_INFO **pp )
{
  int rc;
  uint16_t i;
  ALL_IMS_INFO *p = *pp;

  if ( p == NULL )
  {
    p = (ALL_IMS_INFO *) calloc( 1, sizeof( *p ) );
    if ( p == NULL )
      return MBG_ERR_NO_MEM;
  }

  rc = mbgextio_get_ims_state(pmctl, p_addr, &p->state);
  if ( mbg_rc_is_error( rc ) )
    goto err_out;

  if ( mbg_rc_is_error( chk_dev_ims_has_fdm( p ) ) )
    goto success_out;

  if ( p->fdm_info == NULL )
  {
    p->fdm_info = (MBG_IMS_FDM_INFO *) calloc( 1, sizeof( *p->fdm_info ) );
    if ( p->fdm_info == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto err_out;
    }
  }

  rc = mbgextio_get_ims_fdm_info(pmctl, p_addr, p->fdm_info);
  if ( mbg_rc_is_error( rc ) )
    goto err_out;


  if ( p->fdm_limits == NULL )
  {
    p->fdm_limits = (MBG_IMS_FDM_LIMITS *) calloc( 1, sizeof( *p->fdm_limits ) );
    if ( p->fdm_limits == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto err_out;
    }
  }

  rc = mbgextio_get_ims_fdm_limits(pmctl, p_addr, p->fdm_limits);
  if ( mbg_rc_is_error( rc ) )
    goto err_out;


  if ( p->fdm_outinfo_idx == NULL )
  {
    p->fdm_outinfo_idx = (MBG_IMS_FDM_OUTPUT_INFO_IDX *) calloc( p->fdm_limits->n_outputs, sizeof( *p->fdm_outinfo_idx ) );
    if ( p->fdm_outinfo_idx == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto err_out;
    }
  }

  for ( i = 0; i < p->fdm_limits->n_outputs; ++i )
  {
    rc = mbgextio_get_ims_fdm_output_info_idx( pmctl, p_addr, &p->fdm_outinfo_idx[i], i );
    if ( mbg_rc_is_error( rc ) )
      goto err_out;
  }

success_out:

  *pp = p;

  return MBG_SUCCESS;

err_out:
  free_all_ims_info( p );

  *pp = NULL;

  return rc;

} // mbgextio_get_all_ims_info



/*HDR*/
/**
 * @brief Read all IMS related states into an ::ALL_IMS_STATE structure
 *
 * @note ::mbgextio_dev_has_ims should be called before using this function
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     info   Pointer to a ::ALL_GPIO_INFO structure which has been read before
 * @param[in,out] pp     Pointer to a pointer of ::ALL_IMS_STATE structure which will be malloced
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_ims
 * @see ::chk_dev_ims_has_fdm
 * @see ::free_all_ims_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_ims_state( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, const ALL_IMS_INFO *info, ALL_IMS_STATE **pp )
{
  int rc;
  uint16_t i;
  ALL_IMS_STATE *p = *pp;

  if ( info == NULL )
    return MBG_ERR_INV_PARM;

  if ( p == NULL )
  {
    p = (ALL_IMS_STATE *) calloc( 1, sizeof( *p ) );
    if ( p == NULL )
      return MBG_ERR_NO_MEM;
  }

  if ( p->sensor_state_idx == NULL )
  {
    p->sensor_state_idx = (MBG_IMS_SENSOR_STATE_IDX *) calloc( info->state.num_sensors, sizeof( *p->sensor_state_idx ) );
    if ( p->sensor_state_idx == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto err_out;
    }
  }

  for ( i = 0; i < info->state.num_sensors; ++i )
  {
    rc = mbgextio_get_ims_sensor_state_idx( pmctl, p_addr, &p->sensor_state_idx[i], i );
    if ( mbg_rc_is_error( rc ) )
      goto err_out;
  }


  if ( mbg_rc_is_error( chk_dev_ims_has_fdm( info ) ) )
    goto success_out;


  if ( p->fdm_state == NULL )
  {
    p->fdm_state = (MBG_IMS_FDM_STATE *) calloc( 1, sizeof( *p->fdm_state ) );
    if ( p->fdm_state == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto err_out;
    }
  }

  rc = mbgextio_get_ims_fdm_state( pmctl, p_addr, p->fdm_state );
  if ( mbg_rc_is_error( rc ) )
    goto err_out;


  if ( p->fdm_output_state_idx == NULL )
  {
    p->fdm_output_state_idx = (MBG_IMS_FDM_OUTPUT_STATE_IDX *) calloc( info->fdm_limits->n_outputs, sizeof( *p->fdm_output_state_idx ) );
    if ( p->fdm_output_state_idx == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto err_out;
    }
  }

  for ( i = 0; i < info->fdm_limits->n_outputs; ++i )
  {
    rc = mbgextio_get_ims_fdm_output_state_idx( pmctl, p_addr, &p->fdm_output_state_idx[i], i );
    if ( mbg_rc_is_error( rc ) )
      goto err_out;
  }


success_out:

  *pp = p;

  return MBG_SUCCESS;

err_out:

  free_all_ims_state( p );

  *pp = NULL;

  return rc;

}  //  mbgextio_get_all_ims_state

#endif  // defined( _PRELIMINARY_CODE )


/*HDR*/
/**
 * @brief Read all GPIO related information / configuration into an ::ALL_GPIO_INFO structure
 *
 * @note ::mbgextio_dev_has_gpio should be called before using this function
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in,out] pp     Pointer to a pointer of ::ALL_GPIO_INFO structure which will be malloced
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_gpio
 * @see ::chk_dev_gpio_has_status
 * @see ::free_all_gpio_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_gpio_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_GPIO_INFO **pp )
{
  int rc;
  uint16_t i;
  ALL_GPIO_INFO *p = *pp;

  if ( p == NULL )
  {
    p = (ALL_GPIO_INFO *) calloc( 1, sizeof( *p ) );
    if ( p == NULL )
      return MBG_ERR_NO_MEM;
  }

  rc = mbgextio_req_data(pmctl, p_addr, GPS_GPIO_CFG_LIMITS, &p->cfg_limits, sizeof( p->cfg_limits ) );
  if ( mbg_rc_is_error( rc ) )
    goto err_out;
  _mbg_swab_mbg_gpio_cfg_limits( &p->cfg_limits );

  if ( p->infos == NULL )
  {
    p->infos = (MBG_GPIO_INFO_IDX *) calloc( p->cfg_limits.num_io, sizeof( *p->infos ) );
    if ( p == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto err_out;
    }
  }

  for ( i = 0; i < p->cfg_limits.num_io; ++i )
  {
    rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_GPIO_INFO_IDX, i, &p->infos[i], sizeof( *p->infos ) );
    if ( mbg_rc_is_error( rc ) )
      goto err_out;
    _mbg_swab_mbg_gpio_info_idx( &p->infos[i], 1 );
  }

  *pp = p;

  return MBG_SUCCESS;

err_out:
  free_all_gpio_info( p );

  *pp = NULL;

  return rc;
}



/*HDR*/
/**
 * @brief Read all GPIO related state information to an ::ALL_GPIO_STATE structure
 *
 * @note ::mbgextio_dev_has_gpio should be called before using this function
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     info   Pointer to a ::ALL_GPIO_INFO structure which has been read before
 * @param[in,out] pp     Pointer to a pointer of ::ALL_GPIO_STATE structure which will be malloced
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_gpio
 * @see ::chk_dev_gpio_has_status
 * @see ::free_all_gpio_info
 * @see ::free_all_gpio_state
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_gpio_state( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, const ALL_GPIO_INFO *info, ALL_GPIO_STATE **pp )
{
  int rc;
  uint16_t i;
  ALL_GPIO_STATE *p = *pp;

  if ( info == NULL )
    return MBG_ERR_INV_PARM;

  rc = chk_dev_gpio_has_status( info );
  if ( mbg_rc_is_error( rc ) )
    return rc;

  if ( p == NULL )
  {
    p = (ALL_GPIO_STATE *) calloc( 1, sizeof( *p ) );
    if ( p == NULL )
      return MBG_ERR_NO_MEM;
  }

  if ( p->states == NULL )
  {
    p->states = (MBG_GPIO_STATUS_IDX *) calloc( info->cfg_limits.num_io, sizeof( *p->states ) );
    if ( p == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto err_out;
    }
  }

  for ( i = 0; i < info->cfg_limits.num_io; ++i )
  {
    rc = mbgextio_req_data_idx( pmctl, p_addr, GPS_GPIO_STATUS_IDX, i, &p->states[i], sizeof( *p->states ) );
    if ( mbg_rc_is_error( rc ) )
      goto err_out;
    _mbg_swab_mbg_gpio_status_idx( &p->states[i] );
  }

  *pp = p;

  return MBG_SUCCESS;

err_out:
  free_all_gpio_state( p );

  *pp = NULL;

  return rc;

}  //  mbgextio_get_all_gpio_state


/*HDR*/
/**
 * @brief Read all I/O port related information to an ::ALL_IO_PORT_INFO structure

 *
 * @note ::mbgextio_dev_has_io_ports should be called before using this function
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in,out] pp     Pointer to a pointer of ::ALL_IO_PORT_INFO structure which will be malloced
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_io_ports
 * @see ::free_all_io_port_info
 * @see ::free_all_io_port_state
 * @see ::mbgextio_get_all_io_port_status
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_io_port_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_IO_PORT_INFO **pp )
{
  int rc;
  uint8_t i, k;
  ALL_IO_PORT_INFO *p = *pp;
  MBG_IO_PORT_TYPE_INFO_IDX *pti_idx;

  if ( p == NULL )
  {
    p = (ALL_IO_PORT_INFO *) calloc( 1, sizeof( *p ) );
    if ( p == NULL )
      return MBG_ERR_NO_MEM;
  }

  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_begin_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_IO_PORT, 0 );

  rc = mbgextio_get_io_port_limits( pmctl, p_addr, &p->limits );
  if ( mbg_rc_is_error( rc ) )
    goto out;

  if ( p->p_infos == NULL )
  {
    p->p_infos = (MBG_IO_PORT_INFO_IDX *) calloc( p->limits.num_ports, sizeof( *p->p_infos ) );
    if ( p->p_infos == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }
  }

  if ( p->pt_infos == NULL )
  {
    p->pt_infos = ( MBG_IO_PORT_TYPE_INFO_IDX **) calloc( p->limits.num_ports, sizeof( p->pt_infos ) );
    if ( p->pt_infos == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }
  }

  for ( i = 0; i < p->limits.num_ports; ++i )
  {
    rc = mbgextio_get_io_port_info_idx( pmctl, p_addr, &p->p_infos[i], i );
    if ( mbg_rc_is_error( rc ) )
      goto out;

    if ( p->pt_infos[i] == NULL )
    {
      pti_idx = (MBG_IO_PORT_TYPE_INFO_IDX *) calloc( p->p_infos[i].info.num_types, sizeof( *pti_idx ) );
      if ( pti_idx == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto out;
      }

      p->pt_infos[i] = pti_idx;
    }
    else
    {
      pti_idx = p->pt_infos[i];
    }

    for ( k = 0; k < p->p_infos[i].info.num_types; ++k )
    {
      rc = mbgextio_get_io_port_type_info_idx( pmctl, p_addr, &pti_idx[k], i ,k );
      if ( mbg_rc_is_error( rc ) )
        goto out;
    }
  }

out:
  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_end_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_IO_PORT );

  if( mbg_rc_is_error( rc ) )
  {
    free_all_io_port_info( p );
    p = NULL;
  }

  *pp = p;

  return rc;

}  // mbgextio_get_all_io_port_info



/*HDR*/
/**
 * @brief Write all I/O port related configuration to a device
 *
 * The complementary function ::mbgextio_get_all_io_port_info should
 * have been used to read the original I/O port settings and supported
 * configuration parameters.
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[in]       p        Pointer to a ::ALL_IO_PORT_INFO structure with all I/O port settings
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_all_io_port_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_save_all_io_port_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_IO_PORT_INFO *p )
{
  int rc = MBG_SUCCESS;
  uint8_t i;

  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_begin_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_IO_PORT, 1 );

  for ( i = 0; ( i < p->limits.num_ports ) && mbg_rc_is_success( rc ); ++i )
  {
    if(p->p_infos[i].info.settings.type != (uint16_t)MBG_IO_PORT_TYPE_NONE)
      rc = mbgextio_set_io_port_settings_idx( pmctl, p_addr, &p->p_infos[i].info.settings, i );
  }

  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_end_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_IO_PORT );

  return rc;

}



/*HDR*/
/**
 * @brief Read all I/O port related status information to an ::ALL_IO_PORT_STATUS structure
 *
 * @note ::mbgextio_dev_has_io_ports should be called before using this function
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     info   Pointer to ::ALL_IO_PORT_INFO
 * @param[in,out] pp     Pointer to a pointer of ::ALL_IO_PORT_STATUS structure which will be malloced
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_io_ports
 * @see ::free_all_io_port_info
 * @see ::free_all_io_port_state
 * @see ::mbgextio_get_all_io_port_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_io_port_status( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, const ALL_IO_PORT_INFO *info, ALL_IO_PORT_STATUS **pp )
{
  ALL_IO_PORT_STATUS *p = *pp;
  int rc = MBG_SUCCESS;
  uint8_t i;

  if ( p == NULL )
  {
    p = (ALL_IO_PORT_STATUS *) calloc( 1, sizeof( *p ) );
    if ( p == NULL )
      return MBG_ERR_NO_MEM;
  }

  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_begin_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_IO_PORT, 0 );

  if ( p->status == NULL )
  {
    p->status = (MBG_IO_PORT_STATUS_IDX *) calloc( info->limits.num_ports, sizeof( *p->status ) );
    if ( p->status == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }
  }

  for ( i = 0; i < info->limits.num_ports; ++i )
  {
    rc = mbgextio_get_io_port_status_idx( pmctl, p_addr, &p->status[i], i );
    if ( mbg_rc_is_error( rc ) )
      goto out;
  }

out:
  if ( mbg_rc_is_success( mbgextio_dev_has_transactions( pmctl ) ) )
    mbgextio_end_transaction( pmctl, p_addr, MBG_TRANSACTION_TYPE_IO_PORT );

  if ( mbg_rc_is_error( rc ) )
  {
    free_all_io_port_status( p );
    p = NULL;
  }

  *pp = p;

  return rc;
}


/*HDR*/
/**
 * @brief Read all user capture information and store it into a newly allocated or reused ::ALL_UCAP_INFO
 *
 * @note ::mbgextio_dev_has_ucap should be called to check if this API is supported.
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[out]      p        Pointer to a pointer to ::ALL_UCAP_INFO
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_ucap
 * @see ::free_all_ucap_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_ucap_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_UCAP_INFO **p )
{
  int rc;
  ALL_UCAP_INFO *ucap_info = *p;
  UCAP_ENTRY *new_entry, *old_entry;

  // if this function is called for the first time, allocate a new ::ALL_UCAP_INFO structure
  if ( ucap_info == NULL )
  {
    ucap_info = (ALL_UCAP_INFO *) calloc( 1, sizeof( *ucap_info ) );
    if ( ucap_info == NULL )
    {
      *p = NULL;
      return MBG_ERR_NO_MEM;
    }
    mbg_klist_init(&ucap_info->list);
  }

  do
  {
    new_entry = calloc_ucap_entry();
    if ( !new_entry )
    {
      rc = MBG_ERR_NO_MEM;
      goto err_out;
    }

    rc = mbgextio_get_ucap( pmctl, p_addr, &new_entry->ttm );
    if ( mbg_rc_is_error( rc ) )
      goto err_out;

    if ( !_ttm_time_is_avail( &new_entry->ttm ) )
    {
      free ( new_entry );
      new_entry = NULL;
      break;
    }

    if ( ucap_info->num_ucaps < MAX_UCAP_ENTRIES )
    {
      mbg_klist_append_item(&ucap_info->list, &new_entry->head);
      ++ucap_info->num_ucaps;
    }
    else
    {
      old_entry = mbg_klist_first_entry(&ucap_info->list, UCAP_ENTRY, head);
      mbg_klist_delete_item(&old_entry->head);
      free(old_entry);
      mbg_klist_append_item(&ucap_info->list, &new_entry->head);
    }

  } while (1);

  rc = MBG_SUCCESS;

  goto success_out;

err_out:
  free_all_ucap_info( ucap_info );
  ucap_info = NULL;
  if ( new_entry )
    free( new_entry );

success_out:
  *p = ucap_info;

  return rc;
}


/*HDR*/
/**
 * @brief Read all user capture network info and store it into a newly allocated or reused ::ALL_UCAP_NET_INFO
 *
 * @note ::mbgextio_dev_has_ucap_net should be called to check if this API is supported.
 *
 * @param[in,out]   pmctl    Pointer to a valid message control structure
 * @param[in]       p_addr   Pointer to an XBP address specifier, or NULL
 * @param[out]      p        Pointer to a pointer to ::ALL_UCAP_NET_INFO
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_ucap_net
 * @see ::free_all_ucap_net_info
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_get_all_ucap_net_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_UCAP_NET_INFO **p )
{
  ALL_UCAP_NET_INFO *ucap_net_info = *p;
  uint16_t i;
  int rc;

  if ( ucap_net_info == NULL )
  {
    ucap_net_info = (ALL_UCAP_NET_INFO *) calloc( 1, sizeof( *ucap_net_info ) );
    if ( ucap_net_info == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }
  }

  rc = mbgextio_get_ucap_net_glb_info( pmctl, p_addr, &ucap_net_info->glb_info );
  if ( mbg_rc_is_error( rc ) )
    goto out;

  if ( ucap_net_info->glb_info.settings.num_recvs > 0 )
  {
    ucap_net_info->recv_infos = ( MBG_UCAP_NET_RECV_INFO_IDX* )realloc( ucap_net_info->recv_infos, ucap_net_info->glb_info.settings.num_recvs * sizeof( *ucap_net_info->recv_infos ) );
    if ( ucap_net_info->recv_infos == NULL )
    {
      rc = MBG_ERR_NO_MEM;
      goto out;
    }

    for ( i = 0; i < ucap_net_info->glb_info.settings.num_recvs; i++ )
    {
      rc = mbgextio_get_ucap_net_recv_info_idx( pmctl, p_addr, &ucap_net_info->recv_infos[i], i );
      if ( mbg_rc_is_error( rc ) )
        goto out;
    }
  }
  else
  {
    if ( ucap_net_info->recv_infos != NULL )
      free( ucap_net_info->recv_infos );
    ucap_net_info->recv_infos = NULL;
  }

out:
  if ( mbg_rc_is_error( rc ) )
  {
    free_all_ucap_net_info( ucap_net_info );
    ucap_net_info = NULL;
  }

  *p = ucap_net_info;

  return rc;

} // mbgextio_get_all_ucap_net_info



/*HDR*/
/**
 * @brief Write all user capture network settings to a device
 *
 * The complementary function ::mbgextio_get_all_ucap_net_info should
 * have been used to read the original user capture settings and supported
 * configuration parameters.
 *
 * @param[in,out]  pmctl   Pointer to a valid message control structure
 * @param[in]      p_addr  Pointer to an XBP address specifier, or NULL
 * @param[in]      p       Pointer to the ::ALL_UCAP_NET_INFO
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_save_all_ucap_net_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, ALL_UCAP_NET_INFO *p )
{
  MBG_UCAP_NET_RECV_SETTINGS_IDX settings_idx;
  uint16_t i;
  int rc;

  rc = mbgextio_set_ucap_net_glb_settings( pmctl, p_addr, &p->glb_info.settings );

  if ( mbg_rc_is_success( rc ) )
  {
    for ( i = 0; i < p->glb_info.settings.num_recvs; i++ )
    {
      settings_idx.idx = i;
      settings_idx.settings = p->recv_infos[i].info.settings;

      rc = mbgextio_set_ucap_net_recv_settings_idx( pmctl, p_addr, &settings_idx, i );

      if ( mbg_rc_is_error( rc ) )
        goto out;
    }
  }

out:
  return rc;

} // mbgextio_save_all_ucap_net_info



/*HDR*/
/**
 * @brief Read or setup all GNSS status information
 *
 * This function should be called preferably to get a GNSS status summary
 * from GNSS receivers (GPS, GLONASS, ...).
 *
 * The function ::mbgextio_setup_receiver_info must have been called before, and
 * the returned ::RECEIVER_INFO structure has to be passed to this function.
 *
 * If the device supports this then the low level GNSS API functions
 * are called directly to collect the status information. If the device
 * doesn't support the GNSS API but is a pure GPS receiver then the GPS
 * API functions are called and the GNSS data structures are filled up
 * accordingly, so the calling application can always evaluate the
 * GNSS data structures in ::ALL_GNSS_INFO.
 *
 * If neither GPS nor another GNSS system is supported then this function
 * returns the ::MBG_ERR_NOT_SUPP_BY_DEV error.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[out]    p_agi  Pointer to a ::ALL_GNSS_INFO to be filled
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_get_gps_stat_info
 * @see ::mbgextio_get_gnss_mode_info
 * @see ::mbgextio_get_all_gnss_sat_info
 */
int mbgextio_chk_get_all_gnss_info( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                                    ALL_GNSS_INFO *p_agi )
{
  GNSS_SAT_INFO_IDX *p_gsii;
  int rc;

  memset( p_agi, 0, sizeof( *p_agi ) );

  // First we check if the device is a GPS receiver,
  // which includes GNSS receivers.
  rc = mbgextio_dev_is_gps( pmctl );

  if ( mbg_rc_is_error( rc ) )
    goto out;


  // Read the STAT_INFO structure which is supported by all GPS
  // and GNSS receivers.
  rc = mbgextio_get_gps_stat_info( pmctl, p_addr, &p_agi->stat_info );

  if ( mbg_rc_is_error( rc ) )
    goto out;


  // Check if the device is a GNSS receiver, i.e. can track satellites
  // from different systems (GPS, GLONASS, Beidou, ...).
  rc = mbgextio_dev_is_gnss( pmctl );

  if ( mbg_rc_is_success( rc ) )
  {
    // The device is a GNSS receiver, i.e. it can track satellites
    // from different systems (GPS, GLONASS, Beidou, ...).
    // Read some specific GNSS information.
    rc = mbgextio_get_gnss_mode_info( pmctl, p_addr, &p_agi->gnss_mode_info );

    if ( mbg_rc_is_error( rc ) )
      goto out;

    rc = chk_set_n_gnss_supp( p_agi );

    if ( mbg_rc_is_error( rc ) )
      goto out;


    // If the the device supports the current API we can simply
    // retrieve status information for each individual GNSS system.
    if ( p_agi->gnss_mode_info.flags & MBG_GNSS_FLAG_MSK_SAT_INFO_IDX_SUPP_SER )
    {
      rc = mbgextio_get_all_gnss_sat_info( pmctl, p_addr, p_agi->gnss_sat_info_idx,
                                           &p_agi->gnss_mode_info );
      goto out;
    }


    // The device is a GNSS receiver but only supports a preliminary API
    // which is now obsolete. We expect that this receiver can (only)
    // track GLONASS satellites in addition to GPS.
    if ( p_agi->gnss_mode_info.supp_gnss_types != ( MBG_GNSS_TYPE_MSK_GPS | MBG_GNSS_TYPE_MSK_GLONASS ) )
    {
      rc = MBG_ERR_INV_CFG;
      goto out;
    }


    // First we set up GNSS info for GPS from STAT_INFO.
    setup_gps_only_sat_info_idx_from_statinfo( p_agi );

    // Then we read the GLONASS status information from the device.
    p_gsii = &p_agi->gnss_sat_info_idx[GNSS_TYPE_GLONASS];

    rc = mbgextio_get_gnss_sat_info( pmctl, p_addr, &p_gsii->gnss_sat_info );

    if ( mbg_rc_is_error( rc ) )
      goto out;


    p_gsii->idx = GNSS_TYPE_GLONASS;

    #if 1 // ### TODO
    {
      // This type of receiver provides the numbers of GPS satellites in
      // STAT_INFO from which we have set up gnss_sat_info[GPS], and the
      // GLONASS-only numbers of satellites info has been read to
      // gnss_sat_info[GLONASS]. We want the summary numbers of GPS and
      // GLONASS satellites in STAT_INFO, though, so we add the numbers
      // of GLONASS satellites to the numbers in STAT_INFO.
      STAT_INFO *p_si = &p_agi->stat_info;
      GNSS_SAT_INFO *p_gsi_gln = &p_agi->gnss_sat_info_idx[GNSS_TYPE_GLONASS].gnss_sat_info;

      p_si->svs_in_view += p_gsi_gln->svs_in_view;
      p_si->good_svs += p_gsi_gln->good_svs;
    }
    #else
    {
      // This type of receiver provides the summary numbers of GPS and GLONASS
      // satellites in STAT_INFO from which we have set up gnss_sat_info[GPS],
      // and the GLONASS-only numbers of satellites info has been read to
      // gnss_sat_info[GLONASS].
      // We want the GPS-only numbers of satellites in gnss_sat_info[GPS],
      // so we have to subtract the numbers of GLONASS satellites.
      GNSS_SAT_INFO *p_gsi_gps = &p_agi->gnss_sat_info_idx[GNSS_TYPE_GPS].gnss_sat_info;
      GNSS_SAT_INFO *p_gsi_gln = &p_agi->gnss_sat_info_idx[GNSS_TYPE_GLONASS].gnss_sat_info;

      p_gsi_gps->svs_in_view -= p_gsi_gln->svs_in_view;

      if ( p_gsi_gps->svs_in_view < 0 )  // this should never happen
        p_gsi_gps->svs_in_view = 0;

      p_gsi_gps->good_svs -= p_gsi_gln->good_svs;

      if ( p_gsi_gps->good_svs < 0 )  // this should never happen
        p_gsi_gps->good_svs = 0;
    }
    #endif

    goto out;
  }


  // If we get here then the device is a pure GPS receiver
  // which neither supports additional GNSS systems nor the
  // associated data structures, so we set up all GNSS
  // data structures for GPS only.
  rc = setup_gps_only_gnss_info_from_statinfo( p_agi );

out:
  return rc;

}  // mbgextio_chk_get_all_gnss_info



/*HDR*/
/**
 * @brief Convert a mathematical angle [rad] to a geographic angle [degree, minute, second]
 *
 * @note This has been copied from mbggeo.c, so mbggeo.h should
 * provide the same prototype.
 */
void rad_to_dms( const double *rad, DMS *dms, const char prefix )
{
  static const double r2d = 180.0 / PI;

  double r;

  if ( prefix == 'E' )
    dms->prefix = ( *rad < 0.0 ) ? 'W' : 'E';
  else
    dms->prefix = ( *rad < 0.0 ) ? 'S' : 'N';

  r = *rad * r2d;

  if ( r < 0 )
    r = -r;

  dms->deg = (int) r;

  r -= dms->deg;
  r *= 60;

  dms->min = (int) r;

  r -= dms->min;
  r *= 60;

  dms->sec = r;

}  /* rad_to_dms */



/*HDR*/
/**
 * @brief Convert mathematic coords to to geographic coords
 *
 * Mathematic coords are (longitude, latitude in [rad]), geographic coords
 * are (longitude, latitude in [degree, minute, second])
 *
 * @note This has been copied from mbggeo.c, so mbggeo.h should
 * provide the same prototype.
 */
void lla_to_dms( POS *pos )
{
  rad_to_dms( &pos->lla[LON], &pos->longitude, 'E' );
  rad_to_dms( &pos->lla[LAT], &pos->latitude, 'N' );

}  /* lla_to_dms */



/*HDR*/
/**
 * @brief Transmit an ACK response to the sender.
 *
 * A device should call this function if it has accepted a parameter set,
 * and an acknowledgement has been requested by the sender by or'ing
 * the ::GPS_CMD code with ::GPS_REQACK. If a parameter set has not been
 * accepted then ::mbgextio_xmt_nack should be called instead.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     cmd    One of the command codes enumerated in ::GPS_CMD_CODES
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_xmt_nack
 * @see ::mbgextio_xmt_cmd
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_xmt_ack( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, GPS_CMD cmd )
{
  if ( cmd & GPS_REQACK )
    return mbgextio_xmt_cmd( pmctl, p_addr, (GPS_CMD) ( ( cmd & ~GPS_CTRL_MSK ) | GPS_ACK ) );

  return MBG_SUCCESS;

}  // mbgextio_xmt_ack



/*HDR*/
/**
 * @brief Transmit a NACK response to the sender.
 *
 * A device should call this function if a request has been received
 * which is not supported, or if a data set has been received which
 * contains invalid/unsupported data and thus is ignored. Otherwise
 * ::mbgextio_xmt_ack should be called to return an acknowledge code
 * to the sender, if requested.
 *
 * @param[in,out] pmctl   Pointer to a valid message control structure
 * @param[in]     p_addr  Pointer to an XBP address specifier, or NULL
 * @param[in]     cmd     One of the command codes enumerated in ::GPS_CMD_CODES, including the ::GPS_CMD_CTRL_CODES flags
 * @param[in]     error   One of the @ref MBG_ERROR_CODES to be returned to the caller
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_xmt_ack
 * @see ::mbgextio_xmt_cmd
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_xmt_nack( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, GPS_CMD cmd, int error )
{
  if ( cmd & GPS_REQACK )
    return mbgextio_xmt_cmd_us( pmctl, p_addr, (GPS_CMD) ( ( cmd & ~GPS_CTRL_MSK ) | GPS_NACK ), (uint16_t ) error );  //### TODO check error cast

  return MBG_SUCCESS;

}  // mbgextio_xmt_nack


/*HDR*/
/**
 * @brief Begin a transaction
 *
 * @param[in,out] pmctl   Pointer to a valid message control structure
 * @param[in]     p_addr  Pointer to an XBP address specifier, or NULL
 * @param[in]     type    One of ::MBG_TRANSACTION_TYPES
 * @param[in]     set     Indicates, that this is a set transaction
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_end_transaction
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_begin_transaction( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, uint16_t type, uint8_t set )
{
  GPS_CMD cmd = GPS_BEGIN_TRANSACTION | OPT_GPS_ACK_CODE;

  if ( set )
    _mbg_transaction_type_set( type );

  return mbgextio_xmt_cmd_us( pmctl, p_addr, cmd, type );
}


/*HDR*/
/**
 * @brief End a transaction
 *
 * @param[in,out] pmctl   Pointer to a valid message control structure
 * @param[in]     p_addr  Pointer to an XBP address specifier, or NULL
 * @param[in]     type    One of ::MBG_TRANSACTION_TYPES
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_begin_transaction
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_end_transaction( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr, uint16_t type )
{
  GPS_CMD cmd = GPS_END_TRANSACTION | OPT_GPS_ACK_CODE;

  return mbgextio_xmt_cmd_us( pmctl, p_addr, cmd, type );
}


#if defined( _PRELIMINARY_CODE )

/*HDR*/
/**
 * @brief Get control message's internal TLV state
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return Pointer to internal ::MBG_TLV_RCV_STATE structure
 */
_NO_MBG_API_ATTR MBG_TLV_RCV_STATE* _MBG_API mbgextio_get_tlv_rcv_state( MBG_MSG_CTL *pmctl )
{
  return &pmctl->tlv_rcv_state;

}  // mbgextio_get_tlv_rcv_state



#if _USE_USB_IO

/*HDR*/
/**
 * @brief Get control message's internal USB device info structure
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 *
 * @return Pointer to internal ::MBG_USB_DEV_INFO structure
 */
_NO_MBG_API_ATTR MBG_USB_DEV_INFO * _MBG_API mbgextio_get_usb_dev_info( MBG_MSG_CTL *pmctl )
{
  return &pmctl->st.usbio.mdev_info;

}  // mbgextio_get_usb_dev_info

#endif // _USE_USB_IO



/*HDR*/
/**
 * @brief Send a ::MBG_TLV_ANNOUNCE structure to a host.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     p      Pointer to the data structure to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_xmt_tlv_announce( MBG_MSG_CTL *pmctl,
                                                         const XBP_ADDR *p_addr,
                                                         const MBG_TLV_ANNOUNCE *p )
{
  GPS_CMD cmd = GPS_TLV_ANNOUNCE | GPS_REQACK;
  MBG_TLV_ANNOUNCE *tlv = &pmctl->xmt.pmb->u.msg_data.tlv_announce;

  #if _USE_MUTEX
    _mbg_mutex_acquire( &pmctl->dev_mutex );
  #endif

  *tlv = *p;
  _mbg_swab_tlv_announce( tlv );

  return mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( *tlv ) );

}  // mbgextio_xmt_tlv_announce



/*HDR*/
/**
 * @brief Send a buffer with variable length to a host.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     buf    Pointer to data to be sent to the device
 * @param[in]     buflen Length / size of data
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_xmt_tlv_chunk( MBG_MSG_CTL *pmctl,
                                                      const XBP_ADDR *p_addr,
                                                      void *buf, size_t buflen )
{
  int rc;
  size_t chunk_size;
  uint8_t *pos;
  uint8_t *end;
  GPS_CMD cmd = GPS_TLV_DATA | GPS_REQACK;
  MBG_MSG_BUFF *mbuf = mbgextio_get_xmt_buffer_addr( pmctl );
  MBG_TLV *tlv = &mbuf->u.msg_data.tlv;
  MBG_TLV_HDR tlv_hdr;

  if ( buf )
    pos = (uint8_t*) buf;
  else
    pos = (uint8_t*) tlv->value;

  end = pos + buflen;

  while ( pos < end )
  {
    /* Calculate max bytes to be sent */
    chunk_size = end - pos;
    if ( chunk_size > sizeof( tlv->value ) )
      chunk_size = sizeof( tlv->value );
    tlv->hdr.cur_bytes = chunk_size;
    tlv->hdr.trans_bytes += chunk_size;

    /* Copy and send chunk. If NULL, data is already inside TLV's value */
    if ( buf )
      memcpy( tlv->value, pos, chunk_size );

    tlv_hdr = tlv->hdr;
    _mbg_swab_tlv( tlv );
    rc = mbgextio_xmt_msg( pmctl, p_addr, cmd, NULL, sizeof( tlv->hdr ) + chunk_size );
    if ( mbg_rc_is_error( rc ) )
      return rc;
    tlv->hdr = tlv_hdr;

    pos += chunk_size;
  }

  return MBG_SUCCESS;

}  // mbgextio_xmt_tlv_chunk



/*HDR*/
/**
 * @brief Send a file to a host.
 *
 * @param[in,out] pmctl  Pointer to a valid message control structure
 * @param[in]     p_addr Pointer to an XBP address specifier, or NULL
 * @param[in]     f      Pointer to an opened file to be sent to the device
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_do_xmt_file( MBG_MSG_CTL *pmctl,
                                                    const XBP_ADDR *p_addr,
                                                    FILE *f )
{
  int rc;
  size_t num;
  unsigned char buf[1024];

  do
  {
    num = fread( buf, 1, sizeof( buf ), f );

    /* Error? */
    if ( ferror ( f ) )
        return MBG_ERR_UNSPEC;

    /* There are bytes to send */
    if ( num > 0 )
    {
      rc = mbgextio_xmt_tlv_chunk( pmctl, p_addr, buf, num );
      if ( mbg_rc_is_error( rc ) )
        return rc;
    }


    /* End of file? */
    if ( feof ( f ) )
      break;

  } while ( 1 );

  return MBG_SUCCESS;

}  // mbgextio_do_xmt_file



/*HDR*/
/**
 * @brief Send a file with a specific command to a host.
 *
 * @param[in,out] pmctl          Pointer to a valid message control structure
 * @param[in]     p_addr         Pointer to an XBP address specifier, or NULL
 * @param[in]     file           Name of file to be sent
 * @param[in]     rev            Name of software revision
 * @param[in]     dest_path      Name of destination path on target
 * @param[in]     tlv_feat_type  One of the ::MBG_TLV_FEAT_TYPES
 * @param[in]     uid            If uid == 0 create random uid, otherwise use the specified one
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_xmt_file( MBG_MSG_CTL *pmctl,
                                                 const XBP_ADDR *p_addr,
                                                 const char *file,
                                                 const char *rev,
                                                 const char *dest_path,
                                                 MBG_TLV_TYPE tlv_feat_type,
                                                 MBG_TLV_UID uid )
{
  int rc;
  FILE *f;
  MBG_TLV_ANNOUNCE tlv_announce;
  MBG_MSG_BUFF *buf = mbgextio_get_xmt_buffer_addr( pmctl );
  MBG_TLV *tlv = &buf->u.msg_data.tlv;
  uint32_t announce_size;
  size_t rev_len = 0;
  size_t dest_len = 0;

#if defined( MBG_TGT_WIN32 )
  struct _stat st;
  if ( _stat( file, &st ) < 0 )
#else
  struct stat st;
  if ( stat( file, &st ) < 0 )
#endif
    return mbg_get_last_error( NULL );

  f = fopen( file, "rb" );

  if ( f == NULL )
    return mbg_get_last_error( NULL );

  if ( uid == 0 )
    uid = mbg_tlv_create_id();

  announce_size = (uint32_t) st.st_size + sizeof( tlv->hdr );

  if ( rev )
  {
    rev_len = strlen( rev ) + 1; // Transfer terminating 0 ;)

    if ( rev_len > sizeof( tlv->value ) )
      rev_len = sizeof( tlv->value );

    announce_size += ( rev_len + sizeof( tlv->hdr ) );
  }

  if ( dest_path )
  {
    dest_len = strlen( dest_path ) + 1; // Transfer terminating 0 ;)

    if ( dest_len > sizeof( tlv->value ) )
      dest_len = sizeof( tlv->value );

    announce_size += ( dest_len + sizeof( tlv->hdr ) );
  }

  mbg_tlv_announce_init( &tlv_announce, uid, tlv_feat_type, announce_size );

  rc = mbgextio_xmt_tlv_announce( pmctl, p_addr, &tlv_announce );

  if ( mbg_rc_is_error( rc ) )
    goto out;

  if ( rev )
  {
    mbg_tlv_init( tlv, uid, MBG_TLV_TYPE_STR, rev_len );
    strncpy_safe( (char*) tlv->value, rev,  sizeof( tlv->value ) );
    rc = mbgextio_xmt_tlv_chunk( pmctl, p_addr, NULL,  rev_len );

    if ( mbg_rc_is_error( rc ) )
      goto out;
  }

  if ( dest_path )
  {
    mbg_tlv_init( tlv, uid, MBG_TLV_TYPE_STR, dest_len );
    strncpy_safe( (char*) tlv->value, dest_path,  sizeof( tlv->value ) );

    rc = mbgextio_xmt_tlv_chunk( pmctl, p_addr, NULL,  dest_len );

    if ( mbg_rc_is_error( rc ) )
      goto out;
  }

  mbg_tlv_init( tlv, uid, MBG_TLV_TYPE_FILE, ( uint32_t ) st.st_size );

  rc = mbgextio_do_xmt_file( pmctl, p_addr, f );

out:
  fclose( f );

  return rc;

}  // mbgextio_xmt_file



/*HDR*/
/**
 * @brief Send a cmd line to a host.
 *
 * @param[in,out] pmctl     Pointer to a valid message control structure
 * @param[in]     p_addr    Pointer to an XBP address specifier, or NULL
 * @param[in]     cmd_line  Full command line to be executed as system call on target
 * @param[in]     uid       If uid == 0, create random uid. Otherwise use given one
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_xmt_cmd_line( MBG_MSG_CTL *pmctl,
                                                 const XBP_ADDR *p_addr,
                                                 const char *cmd_line,
                                                 MBG_TLV_UID uid )
{
  int rc;
  MBG_TLV_ANNOUNCE tlv_announce;
  MBG_MSG_BUFF *buf = mbgextio_get_xmt_buffer_addr( pmctl );
  MBG_TLV *tlv = &buf->u.msg_data.tlv;
  uint32_t announce_size;
  size_t cmd_line_len = 0;

  if ( uid == 0 )
    uid = mbg_tlv_create_id();

  announce_size = sizeof( tlv->hdr );

  if ( cmd_line )
  {
    cmd_line_len = strlen( cmd_line ) + 1; // Transfer terminating 0 ;)

    if ( cmd_line_len > sizeof( tlv->value ) )
      cmd_line_len = sizeof( tlv->value );

    announce_size += ( cmd_line_len + sizeof( tlv->hdr ) );
  }

  mbg_tlv_announce_init( &tlv_announce, uid, MBG_TLV_FEAT_TYPE_EXEC_CMD, announce_size );

  rc = mbgextio_xmt_tlv_announce( pmctl, p_addr, &tlv_announce );

  if ( mbg_rc_is_error( rc ) )
    goto out;

  if ( cmd_line )
  {
    mbg_tlv_init( tlv, uid, MBG_TLV_TYPE_STR, cmd_line_len );
    strncpy_safe( (char*) tlv->value, cmd_line,  sizeof( tlv->value ) );

    rc = mbgextio_xmt_tlv_chunk( pmctl, p_addr, NULL,  cmd_line_len );

    if ( mbg_rc_is_error( rc ) )
      goto out;
  }

out:
  return rc;

}  // mbgextio_xmt_cmd_line


/*HDR*/
/**
 * @brief Request a generic file of a target.
 *
 * @param[in,out] pmctl          Pointer to a valid message control structure
 * @param[in]     p_addr         Pointer to an XBP address specifier, or NULL
 * @param[in]     filename       Name of the source file
 * @param[in]     dest_path      Name of destination path and filename on target
 * @param[in]     tlv_feat_type  One of the ::MBG_TLV_FEAT_TYPES
 * @param[in]     uid            If uid == 0, create random uid. Otherwise use given one
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_req_generic_file( MBG_MSG_CTL *pmctl,
                                                         const XBP_ADDR *p_addr,
                                                         const char *filename,
                                                         const char *dest_path,
                                                         MBG_TLV_TYPE tlv_feat_type,
                                                         MBG_TLV_UID uid )
{
  int rc;
  MBG_TLV_ANNOUNCE tlv_announce;
  MBG_MSG_BUFF *xmt_buf = mbgextio_get_xmt_buffer_addr( pmctl );
  MBG_MSG_BUFF *rcv_buf = mbgextio_get_rcv_buffer_addr( pmctl );
  MBG_TLV_ANNOUNCE *rcv_ann = &rcv_buf->u.msg_data.tlv_announce;
  MBG_TLV_RCV_STATE *state = mbgextio_get_tlv_rcv_state( pmctl );
  MBG_TLV *tlv = &xmt_buf->u.msg_data.tlv;
  ulong tmo = mbgextio_get_msg_rcv_timeout( pmctl );
  uint32_t announce_size;
  size_t filename_len = 0;

  if ( uid == 0 )
    uid = mbg_tlv_create_id();

  if ( filename )
  {
    filename_len = strlen( filename ) + 1; // Transfer terminating 0 ;)

    if ( filename_len > sizeof( tlv->value ) )
      filename_len = sizeof( tlv->value );

    announce_size = ( filename_len + sizeof( tlv->hdr ) );
  }
  else
  {
    rc = MBG_ERR_INV_PARM;
    goto out;
  }

  mbg_tlv_announce_init( &tlv_announce, uid, tlv_feat_type, announce_size );

  rc = mbgextio_xmt_tlv_announce( pmctl, p_addr, &tlv_announce );

  if ( mbg_rc_is_error( rc ) )
    goto out;

  mbg_tlv_init( tlv, uid, MBG_TLV_TYPE_STR, filename_len );
  strncpy_safe( (char *) tlv->value, filename,  sizeof( tlv->value ) );

  rc = mbgextio_xmt_tlv_chunk( pmctl, p_addr, NULL,  filename_len );

  if ( mbg_rc_is_error( rc ) )
    goto out;

  /* Wait for announce */
  mbgextio_set_msg_rcv_timeout( pmctl, TLV_REQ_FILE_TIMEOUT );

  rc = mbgextio_rcv_msg_unlocked( pmctl, p_addr, GPS_TLV_ANNOUNCE, NULL, 0 );

  if ( mbg_rc_is_error( rc ) )
  {
    if ( rc != MBG_ERR_TIMEOUT )
      mbgextio_xmt_nack( pmctl, p_addr, rcv_buf->hdr.cmd, rc );

    goto out;
  }

  if ( tlv_announce.data.uid != rcv_ann->data.uid )
  {
    rc = MBG_ERR_INV_TLV_UID;
    mbgextio_xmt_nack( pmctl, p_addr, rcv_buf->hdr.cmd, rc );
    goto out;
  }

  if ( rcv_ann->data.type != tlv_feat_type )
  {
    rc = MBG_ERR_INV_TYPE;
    mbgextio_xmt_nack( pmctl, p_addr, rcv_buf->hdr.cmd, rc );
    goto out;
  }

  mbgextio_xmt_ack( pmctl, p_addr, rcv_buf->hdr.cmd );

  mbg_tlv_rcv_state_init( state, tlv_announce.data.uid,  rcv_ann->data.total_bytes );

  state->data.type = MBG_TLV_TYPE_FILE;

  rc = mbgextio_tlv_to_file( pmctl, p_addr, filename );

  if ( mbg_rc_is_error( rc ) )
    goto out;

out:
  mbgextio_set_msg_rcv_timeout( pmctl, tmo );

  return rc;

} // mbgextio_req_generic_file



/*HDR*/
/**
 * @brief Request a file with a specific command from a host.
 *
 * @param[in,out] pmctl          Pointer to a valid message control structure
 * @param[in]     p_addr         Pointer to an XBP address specifier, or NULL
 * @param[in]     file           Name of file to be sent
 * @param[in]     dest_path      Name of destination path on target
 * @param[in]     tlv_feat_type  One of the ::MBG_TLV_FEAT_TYPES
 * @param[in]     uid            If uid == 0, create random uid. Otherwise use given one
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_req_file_with_cmd( MBG_MSG_CTL *pmctl,
                                                          const XBP_ADDR *p_addr,
                                                          const char *file,
                                                          const char *dest_path,
                                                          MBG_TLV_TYPE tlv_feat_type,
                                                          MBG_TLV_UID uid )
{
  int rc;
  MBG_TLV_ANNOUNCE tlv_announce;
  MBG_MSG_BUFF *buf = mbgextio_get_rcv_buffer_addr( pmctl );
  MBG_TLV_ANNOUNCE *rcv_ann = &buf->u.msg_data.tlv_announce;
  MBG_TLV_RCV_STATE *state = mbgextio_get_tlv_rcv_state( pmctl );
  ulong tmo = mbgextio_get_msg_rcv_timeout( pmctl );

  if ( uid == 0 )
    uid = mbg_tlv_create_id();

  mbg_tlv_announce_init( &tlv_announce, uid, tlv_feat_type, 0 );

  rc = mbgextio_xmt_tlv_announce( pmctl, p_addr, &tlv_announce );

  if ( mbg_rc_is_error( rc ) )
    goto out;

  /* Wait for announce */
  mbgextio_set_msg_rcv_timeout( pmctl, TLV_REQ_FILE_TIMEOUT );

  rc = mbgextio_rcv_msg_unlocked( pmctl, p_addr, GPS_TLV_ANNOUNCE, NULL, 0 );

  if ( mbg_rc_is_error( rc ) )
  {
    if (rc != MBG_ERR_TIMEOUT )
      mbgextio_xmt_nack( pmctl, p_addr, buf->hdr.cmd, rc );

    goto out;
  }

  if ( tlv_announce.data.uid != rcv_ann->data.uid )
  {
    rc = MBG_ERR_INV_TLV_UID;
    mbgextio_xmt_nack( pmctl, p_addr, buf->hdr.cmd, rc );
    goto out;
  }

  if ( rcv_ann->data.type != tlv_feat_type )
  {
    rc = MBG_ERR_INV_TYPE;
    mbgextio_xmt_nack( pmctl, p_addr, buf->hdr.cmd, rc );
    goto out;
  }

  mbgextio_xmt_ack( pmctl, p_addr, buf->hdr.cmd );

  mbg_tlv_rcv_state_init( state, tlv_announce.data.uid,  rcv_ann->data.total_bytes );

  /* Expect a string that contains firmware version first */
  /* Expect firmware update as data blob */
  state->data.type = MBG_TLV_TYPE_FILE;

  if ( dest_path )
  {
    rc = mbgextio_tlv_to_file( pmctl, p_addr, dest_path );

    if ( mbg_rc_is_error( rc ) )
      goto out;
  }

out:
  mbgextio_set_msg_rcv_timeout( pmctl, tmo );

  return rc;

}



/*HDR*/
/**
 * @brief Send a tlv cmd to a host.
 *
 * @param[in,out] pmctl          Pointer to a valid message control structure
 * @param[in]     p_addr         Pointer to an XBP address specifier, or NULL
 * @param[in]     tlv_feat_type  One of the ::MBG_TLV_FEAT_TYPES
 * @param[in]     uid            If uid == 0, create random uid. Otherwise use given one
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_xmt_tlv_cmd( MBG_MSG_CTL *pmctl,
                                                    const XBP_ADDR *p_addr,
                                                    MBG_TLV_TYPE tlv_feat_type,
                                                    MBG_TLV_UID uid )
{
  int rc;
  MBG_TLV_ANNOUNCE tlv_announce;

  if ( uid == 0 )
    uid = mbg_tlv_create_id();

  mbg_tlv_announce_init( &tlv_announce, uid, tlv_feat_type, 0 );

  rc = mbgextio_xmt_tlv_announce( pmctl, p_addr, &tlv_announce );

  return rc;

}



/*HDR*/
/**
 * @brief Wrapper function for ::mbgextio_xmt_file ::TODO
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_xmt_fw_update( MBG_MSG_CTL *pmctl,
                                                      const XBP_ADDR *p_addr,
                                                      const char *file,
                                                      const char *rev,
                                                      const char *dest_path,
                                                      MBG_TLV_UID uid )
{
  return mbgextio_xmt_file( pmctl, p_addr, file, rev, dest_path,
                            MBG_TLV_FEAT_TYPE_FW_UPDATE, uid );

}  // mbgextio_xmt_fw_update



/*HDR*/
/**
 * @brief Wrapper function for ::mbgextio_xmt_tlv_cmd
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_xmt_fw_rollback( MBG_MSG_CTL *pmctl,
                                                        const XBP_ADDR *p_addr,
                                                        MBG_TLV_UID uid )
{
  return mbgextio_xmt_tlv_cmd( pmctl, p_addr, MBG_TLV_FEAT_TYPE_FW_ROLLBACK, uid );

}  // mbgextio_xmt_fw_rollback



/*HDR*/
/**
 * @brief Wrapper function for ::mbgextio_xmt_file ::TODO
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_xmt_binary_file( MBG_MSG_CTL *pmctl,
                                                        const XBP_ADDR *p_addr,
                                                        const char *file,
                                                        const char *dest_path,
                                                        MBG_TLV_UID uid )
{
  return mbgextio_xmt_file( pmctl, p_addr, file, NULL, dest_path,
                            MBG_TLV_FEAT_TYPE_FILE_TRANSFER, uid );

}  // mbgextio_xmt_binary_file



/*HDR*/
/**
 * @brief Wrapper function for ::mbgextio_xmt_file ::TODO
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_apply_license_file( MBG_MSG_CTL *pmctl,
                                                           const XBP_ADDR *p_addr,
                                                           const char *file,
                                                           MBG_TLV_UID uid )
{
  return mbgextio_xmt_file( pmctl, p_addr, file, NULL, NULL,
                            MBG_TLV_FEAT_TYPE_LICENSE_UPGRADE, uid );

}  // mbgextio_apply_license_file



/*HDR*/
/**
 * @brief Wrapper function for ::mbgextio_xmt_file ::TODO move to firmware-specific module
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_xmt_diag_file( MBG_MSG_CTL *pmctl,
                                                      const XBP_ADDR *p_addr,
                                                      const char *file,
                                                      MBG_TLV_UID uid )
{
  return mbgextio_xmt_file( pmctl, p_addr, file, NULL, NULL,
                            MBG_TLV_FEAT_TYPE_DIAG_FILE, uid );

} // mbgextio_xmt_diag_file



/*HDR*/
/**
 * @brief Wrapper function
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_req_diag_file( MBG_MSG_CTL *pmctl,
                                                      const XBP_ADDR *p_addr,
                                                      const char *dest_path,
                                                      MBG_TLV_UID uid )
{
  return mbgextio_req_file_with_cmd( pmctl, p_addr, NULL, dest_path,
                            MBG_TLV_FEAT_TYPE_DIAG_FILE, uid );

} // mbgextio_xmt_diag_file

#define HPS_TEMP_UPDATE_DEST_FILE "/tmp/update.tgz"

_NO_MBG_API_ATTR int _MBG_API mbgextio_xmt_incremental_fw_update( MBG_MSG_CTL *pmctl,
                                                      const XBP_ADDR *p_addr,
                                                      const char *update_src_file,
                                                      MBG_TLV_UID uid )
{
  int rc;
  static const char *update_cmd = "mkdir -p /mnt/data;mount /dev/mmcblk0p3 /mnt/data; mkdir -p /mnt/data/usr/bin; tar xzf /tmp/update.tgz -C /mnt/data;sync;umount /mnt/data; reboot ";

  rc = mbgextio_xmt_file( pmctl, p_addr, update_src_file, NULL, HPS_TEMP_UPDATE_DEST_FILE, MBG_TLV_FEAT_TYPE_FILE_TRANSFER, uid );

  if ( rc == MBG_SUCCESS )
  {
    return mbgextio_xmt_cmd_line( pmctl, p_addr, update_cmd, uid );
  }
  else
    return rc;
}



static int
mbgextio_tlv_type_checks( MBG_TLV *tlv, MBG_TLV_RCV_STATE *state )
{
  if ( tlv->hdr.uid != state->data.uid )
    return MBG_ERR_INV_TLV_UID;

  if ( tlv->hdr.tlv_type != state->data.type )
    return MBG_ERR_INV_TYPE;

  return MBG_SUCCESS;

} // mbgextio_type_checks



static int
mbgextio_tlv_size_checks( MBG_TLV *tlv, MBG_TLV_RCV_STATE *state, uint32_t bytes )
{
  /* TLV has been announced incorrectly? */
  if ( ( bytes + tlv->hdr.cur_bytes ) > tlv->hdr.total_bytes )
    return MBG_ERR_INV_TLV_ANN_BYTES;

  /* Overall size has been announced incorrectly? */
  if ( ( tlv->hdr.cur_bytes + state->read_bytes ) > state->data.total_bytes )
    return MBG_ERR_INV_TLV_ANN_BYTES;

  return MBG_SUCCESS;

}  // mbgextio_tlv_size_checks



static int
mbgextio_tlv_do_checks( MBG_TLV *tlv, MBG_TLV_RCV_STATE *state, uint32_t bytes )
{
  int rc;

  rc = mbgextio_tlv_type_checks( tlv, state );

  if ( mbg_rc_is_error( rc ) )
    return rc;

  rc = mbgextio_tlv_size_checks( tlv, state, bytes );

  if ( mbg_rc_is_error( rc ) )
    return rc;

  return MBG_SUCCESS;

}  // mbgextio_tlv_do_checks



/*HDR*/
/**
 * @brief Read TLV data into a single structure
 *
 * @param[in,out]  pmctl   Pointer to a valid message control structure
 * @param[in]      p_addr  Pointer to an XBP address specifier, or NULL
 * @param[out]     buf     Pointer to a buffer / data structure to be filled
 * @param[in]      buflen  Size of buffer
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_tlv_to_struct( MBG_MSG_CTL *pmctl,
                                                      const XBP_ADDR *p_addr,
                                                      void *buf, size_t buflen )
{
  int rc;
  uint32_t bytes = 0;
  unsigned char *buf8 = ( unsigned char* )buf;
  MBG_MSG_BUFF *msg_buf = mbgextio_get_rcv_buffer_addr( pmctl );
  MBG_TLV_RCV_STATE *state = mbgextio_get_tlv_rcv_state( pmctl );
  MBG_TLV *tlv = &msg_buf->u.msg_data.tlv;

  do
  {
    rc = mbgextio_rcv_msg_unlocked( pmctl, p_addr, GPS_TLV_DATA, NULL, 0 );
    /*
     * Timeout error needs special handling: Sender already stopped
     * sending TLVS, so it does not respond to (n)acks anymore.
     * Thus, we simply step out here. Any other error should be
     * (n)acknowledged
     */
    if ( rc == MBG_ERR_TIMEOUT )
      return rc;

    if ( mbg_rc_is_error( rc ) )
      goto err;

    /*
     * We need to check this here as we have to receive a TLV first
     * before we can decide whether our expected structure size
     * fits to the size of the structure that is going to be sent.
     */
    //### TODO Pretty sure tlv->hdr.cur_bytes needs to be used below as tlv->hdr.total_bytes contains TLV header size....

    if ( buflen != tlv->hdr.total_bytes )
    {
      rc = MBG_ERR_INV_TLV_SIZE;
      goto err;
    }

    rc = mbgextio_tlv_do_checks( tlv, state, bytes );

    if ( mbg_rc_is_error( rc ) )
      goto err;

    /* Send ack if requested */
    mbgextio_xmt_ack( pmctl, p_addr, msg_buf->hdr.cmd );

    memcpy( &buf8[ bytes ], tlv->value, tlv->hdr.cur_bytes );
    bytes += tlv->hdr.cur_bytes;
    state->read_bytes += tlv->hdr.cur_bytes;

  } while ( bytes < tlv->hdr.total_bytes );

  return MBG_SUCCESS;

err:
  mbgextio_xmt_nack( pmctl, p_addr, msg_buf->hdr.cmd, rc );

  return rc;

}  // mbgextio_tlv_to_struct



static int
mbgextio_tlv_xmt_early_nack( MBG_MSG_CTL *pmctl, const XBP_ADDR *p_addr,
                             MBG_MSG_BUFF *msg_buf, int err )
{
  int rc;

  rc = mbgextio_rcv_msg_unlocked( pmctl, p_addr, GPS_TLV_DATA, NULL, 0 );

  if ( rc == MBG_ERR_TIMEOUT )
    return rc;

  mbgextio_xmt_nack( pmctl, p_addr, msg_buf->hdr.cmd, err );

  return err;

}  // mbgextio_tlv_xmt_early_nack



/*HDR*/
/**
 * @brief Read TLV into memory
 *
 * If this function succeeds then an array of memory has been allocated
 * using a malloc() call, and the pointer returned via the buf parameter
 * has to be freed afterwards by the caller.
 *
 * @param[in,out]  pmctl   Pointer to a valid message control structure
 * @param[in]      p_addr  Pointer to an XBP address specifier, or NULL
 * @param[out]     buf     Address of a pointer to a buffer to be allocated and filled
 *
 * @return One of the @ref MBG_RETURN_CODES
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_tlv_to_mem( MBG_MSG_CTL *pmctl,
                                                   const XBP_ADDR *p_addr,
                                                   void **buf )
{
  int rc;
  unsigned char *buf8 = NULL;
  uint32_t bytes;
  MBG_MSG_BUFF *msg_buf = mbgextio_get_rcv_buffer_addr( pmctl );
  MBG_TLV_RCV_STATE *state = mbgextio_get_tlv_rcv_state( pmctl );
  MBG_TLV *tlv = &msg_buf->u.msg_data.tlv;

  *buf = NULL;
  bytes = 0;

  do
  {
    rc = mbgextio_rcv_msg_unlocked( pmctl, p_addr, GPS_TLV_DATA, NULL, 0 );
    /*
     * Timeout error needs special handling: Sender already stopped
     * sending TLVS, so it does not respond to (n)acks anymore.
     * Thus, we simply step out here. Any other error should be
     * (n)acknowledged
     */
    if ( rc == MBG_ERR_TIMEOUT )
    {
      if ( *buf )
      {
        free( *buf );
        *buf = NULL;
      }
      return rc;
    }

    if ( mbg_rc_is_error( rc ) )
      goto err;

    rc = mbgextio_tlv_do_checks( tlv, state, bytes );

    if ( mbg_rc_is_error( rc ) )
      goto err;

    /*
     * We need to malloc the buffer here as we have to receive a TLV first
     * before we know which amount of data is going to be sent.
     */
    if ( *buf == NULL )
    {
      *buf = malloc( tlv->hdr.total_bytes );

      if ( *buf == NULL )
      {
        rc = MBG_ERR_NO_MEM;
        goto err;
      }
      buf8 = ( unsigned char* ) *buf;
    }

    /* Send ack if requested */
    mbgextio_xmt_ack( pmctl, p_addr, msg_buf->hdr.cmd );

    memcpy( &buf8[ bytes ], tlv->value, tlv->hdr.cur_bytes );
    bytes += tlv->hdr.cur_bytes;
    state->read_bytes += tlv->hdr.cur_bytes;

  } while ( bytes < tlv->hdr.total_bytes );

  return MBG_SUCCESS;

err:
  if ( *buf )
  {
    free( *buf );
    *buf = NULL;
  }

  mbgextio_xmt_nack( pmctl, p_addr, msg_buf->hdr.cmd, rc );

  return rc;

}  // mbgextio_tlv_to_mem



/*HDR*/
/**
 * @brief Read TLV into a single file. See ::mbgextio_tlv_to_mem ::TODO
 */
_NO_MBG_API_ATTR int _MBG_API mbgextio_tlv_to_file( MBG_MSG_CTL *pmctl,
                                                    const XBP_ADDR *p_addr,
                                                    const char *file )
{
  int rc;
  uint32_t bytes;
  FILE *f;
  MBG_MSG_BUFF *msg_buf = mbgextio_get_rcv_buffer_addr( pmctl );
  MBG_TLV_RCV_STATE *state = mbgextio_get_tlv_rcv_state( pmctl );
  MBG_TLV *tlv = &msg_buf->u.msg_data.tlv;

  f = fopen( file, "wb" );

  if ( f == NULL )
    return mbgextio_tlv_xmt_early_nack( pmctl, p_addr, msg_buf, MBG_ERR_IO );

  bytes = 0;

  do
  {
    rc = mbgextio_rcv_msg_unlocked( pmctl, p_addr, GPS_TLV_DATA, NULL, 0 );
    /*
     * Timeout error needs special handling: Sender already stopped
     * sending TLVS, so it does not respond to (n)acks anymore.
     * Thus, we simply step out here. Any other error should be
     * (n)acknowledged
     */
    if ( rc == MBG_ERR_TIMEOUT )
        goto err;

    if ( mbg_rc_is_error( rc ) )
      goto err;

    rc = mbgextio_tlv_do_checks( tlv, state, bytes );

    if ( mbg_rc_is_error( rc ) )
      goto err;

    if ( fwrite( tlv->value, tlv->hdr.cur_bytes, 1, f ) != 1 )
    {
      if ( ferror( f ) )
      {
        rc = mbg_get_last_error( NULL );
        goto err;
      }

      if ( !feof( f ) )
      {
        rc = MBG_ERR_IO;
        goto err;
      }
    }

    /* Send ack if requested */
    mbgextio_xmt_ack( pmctl, p_addr, msg_buf->hdr.cmd );

    bytes += tlv->hdr.cur_bytes;
    state->read_bytes += tlv->hdr.cur_bytes;

  } while ( bytes < tlv->hdr.total_bytes );

  rc = MBG_SUCCESS;

err:
  fflush( f );
  fclose( f );

  if ( mbg_rc_is_error( rc ) )
  {
    mbgextio_xmt_nack( pmctl, p_addr, msg_buf->hdr.cmd, rc );
    remove( file );
  }

  return rc;

}  // mbgextio_tlv_to_file

#endif // _PRELIMINARY_CODE

