
/**************************************************************************
 *
 *  $Id: cfg_hlp.c 1.12 2024/09/13 10:13:31Z gregoire REL_M $
 *
 *  Copyright (c) Meinberg Funkuhren, Bad Pyrmont, Germany
 *
 *  Description:
 *    Meinberg device configuration helper functions.
 *
 * -----------------------------------------------------------------------
 *  $Log: cfg_hlp.c $
 *  Revision 1.12  2024/09/13 10:13:31Z  gregoire
 *  Added missing free in free_all_user_info
 *  Revision 1.11  2024/09/05 10:14:33Z  kai.heine
 *  Add missing HDR comments and some more doxygen documentation.
 *  Revision 1.10  2024/09/04 09:00:59Z  martin.burnicki
 *  Removed obsolete _int_from_size_t() usage.
 *  Revision 1.9  2024/09/03 15:53:26Z  martin.burnicki
 *  Merged branch 1.8.1.x
 *  Struct ALL_RECEIVER_CONFIG within ALL_RECEIVER_INFO by thomas-b.
 *  external_users added to to ALL_USER_INFO by thomas-b.
 *  Pointer for ca_certs_list added to ALL_SYSTEM_STATUS by thomas-b.
 *  Various new functions and improvements by thomas-b.
 *  Moved some funtions to more appropriate modules.
 *  Removed conditional _PRELIMINARY_CODE.
 *  Doxygen fixes and cleanup.
 *  Revision 1.8  2023/03/13 11:21:57Z  gregoire.diehl
 *  Merged 1.7.1.x and 1.7.2.x
 *  New helper functions from Philipp for MBG_SYS_REF_DEV_STATE
 *  temp. changed Oregano cfg file parameters for PSX210
 *  Disable some stuff on CVI (merged from 1.7.2.1).
 *  Fixed a potential compiler warning.
 *  Account for fixed quality config in oregano config file
 *  Fixed possible memory leak in free function of ALL_USER_INFO
 *  %i will interpret decimals with leading zeros as octal values (use %d, instead)
 *  Fixed conversion of SW_REV (hex) to u32 (dec)
 *  Added free_all functions for ALL_RECEIVER_INFO and ALL_RECEIVER_STATUS
 *  Added clock_class_to_str
 *  Add MBG_TGT_CVI defines for non supported stuff
 *  Revision 1.7.1.6.1.2  2023/02/22 15:07:07Z  daniel
 *  New helper functions from Philipp for MBG_SYS_REF_DEV_STATE
 *  Revision 1.7.1.6.1.1  2022/11/14 08:40:42  daniel
 *  temp. changed Oregano cfg file parameters for PSX210
 *  Revision 1.7.1.6  2022/09/07 11:09:55  martin.burnicki
 *  Disable some stuff on CVI (merged from 1.7.2.1).
 *  Fixed a potential compiler warning.
 *  Revision 1.7.1.5  2022/05/18 09:05:30Z  daniel
 *  Account for fixed quality config in oregano config file
 *  Revision 1.7.1.4  2021/12/09 09:39:18  thomas-b
 *  Fixed possible memory leak in free function of ALL_USER_INFO
 *  Revision 1.7.1.3  2021/11/17 10:42:33  thomas-b
 *  %i will interpret decimals with leading zeros as octal values (use %d, instead)
 *  Revision 1.7.1.2  2021/11/17 07:27:12  thomas-b
 *  Fixed conversion of SW_REV (hex) to u32 (dec)
 *  Revision 1.7.1.1  2021/10/20 05:07:11  thomas-b
 *  Added free_all functions for ALL_RECEIVER_INFO and ALL_RECEIVER_STATUS
 *  Revision 1.7  2021/09/09 15:01:12  daniel
 *  Account for neighbor_prop_delay_threshold parameter in oregano cfg file
 *  Revision 1.6  2021/07/30 07:28:59  thomas-b
 *  Added support for TLV file API, XMR dict, new model NIMBRA100 and several helper functions (Merged branch 1.5.x)
 *  Revision 1.5  2019/09/27 14:16:40  martin
 *  Merged changes from 1.3.1.33:
 *  Support for next gen PTP config API added by thomas-b.
 *  XMR improvements by philipp.
 *  Sys ref API definitions by thomas-b.
 *  Network address conversion functions added by gregoire.diehl.
 *  New functions were added, and existing functions were improved.
 *  Use new types MBG_MSG_IDX and MBG_MSG_IDX_32.
 *  Lots of doxygen updates and fixes.
 *  Revision 1.4  2018/09/19 16:55:08Z  martin
 *  Account for renamed global variables.
 *  Revision 1.3  2018/07/05 10:28:07Z  martin
 *  Renamed setup_port_info_from_port_settings()
 *  to setup_port_info_from_port_parm().
 *  Added function chk_dev_receiver_info().
 *  Function free_all_firmware_info() added by thomas-b.
 *  Syslog option added to monitoring feature by philipp.
 *  I/O helper structures and API refuctored by philipp.
 *  Service feature and API added by philipp.
 *  Database feature support added by philipp.
 *  Function to free ALL_USER_INFO structure added by thomas-b.
 *  MBG_EVENT stuff integrated with common/general monitoring
 *  stuff by philipp.
 *  Monitoring events refactored and tainted config support
 *  added by philipp.
 *  Account for renamed library symbol NSEC_PER_SEC.
 *  Documented some conversion functions for legacy parameters.
 *  Doxygen fixes.
 *  Revision 1.2  2017/07/05 12:17:38  martin
 *  Support functions for TLV, IMS, GPIO, and
 *  mbg_snprint_revision() provided by philipp.
 *  Support functions for network configuration,
 *  NTP configuration, ALL_UCAP stuff, SNMP and
 *  MONITORING, as well as ALL_PTP_V2_COMMON_DATASETS
 *  and ALL_PTP_V1_COMMON_DATASETS provided by thomas-b.
 *  Support functions for xmulti_ref and IO PORT stuff
 *  provided by philipp and thomas-b.
 *  More common GNSS support.
 *  New functions alloc_dev_hw_id() and chk_free_dev_hw_id().
 *  Tried portable printing of int64_t types.
 *  Account for frac_sec_from_bin() obsoleted by
 *  bin_frac_32_to_dec_frac().
 *  Revision 1.1  2014/04/25 09:14:49  martin
 *  Initial revision.
 *
 **************************************************************************/

#define _CFG_HLP
 #include <cfg_hlp.h>
#undef _CFG_HLP

#if defined( MBG_TGT_UNIX )
  #include <arpa/inet.h>
  #include <str_util.h>
  #include <math.h>
#endif

#include <mbgerror.h>
#include <timeutil.h>
#include <str_util.h>
#include <lan_util.h>
#include <myutil.h>
#include <mbgtime.h>
#include <mbgmktm.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#if defined( MBG_TGT_WIN32 ) && !defined( MBG_TGT_CVI )
  #include <sys/types.h>
  #include <sys/stat.h>
#endif


/*HDR*/
/**
 * @brief Check if a software revision name should be displayed.
 *
 * The software revision name is usually empty, except if the
 * firmware is a customized version, in which case the field
 * contains an identifier string.
 *
 * There are some standard firmware versions where this string
 * is not empty but padded with spaces, etc., so we try to
 * clean this up, so it can be displayed appropriately.
 *
 * @param[in,out]  p        The ::SW_REV name to check.
 * @param[in]      verbose  The app's verbosity level.
 *
 * @return  != 0 if SW name should be displayed.
 */
int chk_sw_rev_name( SW_REV *p, int verbose )
{
  if ( verbose > 1 )  // more than just verbose
    return 1;         // just return raw string

  trim_whitespace( p->name );

  // Some firmware versions have "CC_STANDARD" in their standard version,
  // which doesn't provide any valuable information.
  // We discard this by default.
  if ( strstr( p->name, "CC_STANDARD" ) )
    p->name[0] = 0;

  if ( verbose )
    return 1;  // calling app should display string, even if empty

  // calling app should display string only if not empty
  return strlen( p->name ) != 0;

}  // chk_sw_rev_name



/*HDR*/
/**
 * @brief Search a string table for a specific string.
 *
 * @param[in]  search     The string to be searched for in table @a str_table.
 * @param[in]  str_table  A table of strings like 'const char *strs[N_STRS]'.
 * @param[in]  n_entries  The number of strings in table @a str_table.
 *
 * @return  The index number of the string table entry matching the @a search string,
 *          or -1 if the @a search string could not be found.
 */
int get_str_idx( const char *search,
                 const char *str_table[],
                 int n_entries )
{
  int i;

  for ( i = 0; i < n_entries; i++ )
    if ( strcmp( search, str_table[i] ) == 0 )
      return i;

  return -1;

}  // get_str_idx



/*HDR*/
/**
 * @brief Search the ::mbg_baud_rates table for a specific baud rate.
 *
 * @param[in]  baud_rate  The baud_rate to be searched for.
 *
 * @return  The index number of the specific @a baud_rate in table ::mbg_baud_rates,
 *          or -1 if the specific @a baud_rate could not be found.
 */
int get_baud_rate_idx( BAUD_RATE baud_rate )
{
  int i;

  for ( i = 0; i < N_MBG_BAUD_RATES; i++ )
    if ( baud_rate == mbg_baud_rates[i] )
      return i;

  return -1;

}  // get_baud_rate_idx



/*HDR*/
/**
 * @brief Search the ::mbg_framing_strs table for a specific framing string.
 *
 * @param[in]  framing  The framing string to be searched for, e.g. "8N1".
 *
 * @return  The index number of the specific @a framing in table ::mbg_framing_strs,
 *          or -1 if the specific @a framing could not be found.
 */
int get_framing_idx( const char *framing )
{
  return get_str_idx( framing, mbg_framing_strs, N_MBG_FRAMINGS );

}  // get_framing_idx



/*HDR*/
/**
 * @brief Convert ::PORT_PARM::mode to ::PORT_SETTINGS::mode.
 *
 * This function is used to evaluate the code from a @a mode field of a legacy
 * ::PORT_PARM structure and set up the appropriate fields in a ::PORT_SETTINGS
 * structure.
 *
 * @param[out]  p_ps         Pointer to a ::PORT_SETTINGS structure to be updated.
 * @param[in]   pp_mode      The mode code from a ::PORT_PARM structure, see ::LGCY_STR_MODES.
 * @param[in]   cap_str_idx  The capture string index number for the case that @a pp_mode is
 *                           a capture string mode. Usually 1 for legacy devices.
 *
 * @ingroup cfg_hlp_com_parm_cnv_fncs
 * @see ::port_parm_mode_from_port_settings (the complementary function)
 * @see ::port_settings_from_port_parm
 */
void port_settings_from_port_parm_mode( PORT_SETTINGS *p_ps, uint8_t pp_mode,
                                        int cap_str_idx )
{
  if ( pp_mode >= LGCY_STR_UCAP )
  {
    // If @a pp_mode is ::LGCY_STR_UCAP or greater
    // then the string is a capture string which
    // can have the mode ::STR_AUTO or ::STR_ON_REQ,
    // and the string type is a capture string, which
    // usually has index 1 for legacy devices.
    p_ps->str_type = cap_str_idx;
    p_ps->mode = ( pp_mode == LGCY_STR_UCAP ) ? STR_AUTO : STR_ON_REQ;
  }
  else
  {
    // If @a pp_mode is less than ::LGCY_STR_UCAP
    // then the string is the default time string,
    // the format of which depends on the firmware
    // version if the device is legacy.
    // The mode numbers are compatible in this case.
    p_ps->str_type = 0;
    p_ps->mode = pp_mode;
  }

}  // port_settings_from_port_parm_mode



/*HDR*/
/**
 * @brief Convert a ::PORT_SETTINGS::mode to a legacy ::PORT_PARM::mode.
 *
 * This function is used to derive a a legacy ::PORT_PARM::mode value
 * from a ::PORT_SETTINGS structure.
 *
 * @param[out]  pp_mode      Pointer to a ::PORT_PARM::mode variable to be updated.
 * @param[in]   p_ps         Pointer to a ::PORT_SETTINGS structure to be evaluated.
 * @param[in]   cap_str_idx  The capture string index number for legacy devices, which is usually 1.
 *
 * @ingroup cfg_hlp_com_parm_cnv_fncs
 * @see ::port_settings_from_port_parm_mode (the complementary function)
 * @see ::port_parm_from_port_settings
 */
void port_parm_mode_from_port_settings( uint8_t *pp_mode, const PORT_SETTINGS *p_ps,
                                        int cap_str_idx )
{
  if ( p_ps->str_type == cap_str_idx )
  {
    // If the string type is a capture string
    // then convert the mode accordingly.
    *pp_mode = ( p_ps->mode == STR_ON_REQ ) ? LGCY_STR_UCAP_REQ : LGCY_STR_UCAP;
  }
  else
  {
    // Other string modes are compatible, so just copy the mode.
    *pp_mode = p_ps->mode;
  }

}  // port_parm_mode_from_port_settings



/*HDR*/
/**
 * @brief Set up a ::PORT_SETTINGS structure from a legacy ::PORT_PARM structure.
 *
 * @param[out]  p_ps         Pointer to a ::PORT_SETTINGS structure to be updated.
 * @param[in]   port_idx     Index number of the port settings to be converted.
 * @param[in]   p_pp         The ::PORT_PARM structure to be evaluated, contains settings for 2 ports.
 * @param[in]   cap_str_idx  The capture string index number for legacy devices, which is usually 1.
 *
 * @ingroup cfg_hlp_com_parm_cnv_fncs
 * @see ::port_parm_from_port_settings (the complementary function)
 * @see ::port_settings_from_port_parm_mode
 */
void port_settings_from_port_parm( PORT_SETTINGS *p_ps, int port_idx,
                                   const PORT_PARM *p_pp, int cap_str_idx )
{
  // The basic serial settings can simply be copied.
  p_ps->parm = p_pp->com[port_idx];

  // The string mode may need to be translated.
  port_settings_from_port_parm_mode( p_ps, p_pp->mode[port_idx],
                                     cap_str_idx );

}  // port_info_from_port_parm



/*HDR*/
/**
 * @brief Set up a a legacy ::PORT_PARM structure from a ::PORT_SETTINGS structure.
 *
 * @param[out]  p_pp         Pointer to a ::PORT_PARM structure to be updated.
 * @param[in]   port_idx     Index number of the port settings to be converted.
 * @param[out]  p_ps         Pointer to a ::PORT_SETTINGS structure to be updated.
 * @param[in]   cap_str_idx  The capture string index number for legacy devices, which is usually 1.
 *
 * @ingroup cfg_hlp_com_parm_cnv_fncs
 * @see ::port_settings_from_port_parm (the complementary function)
 * @see ::port_parm_mode_from_port_settings
 */
void port_parm_from_port_settings( PORT_PARM *p_pp, int port_idx,
                                   const PORT_SETTINGS *p_ps, int cap_str_idx )
{
  // The basic serial settings can simply be copied.
  p_pp->com[port_idx] = p_ps->parm;

  // The string mode may need to be translated.
  port_parm_mode_from_port_settings( &p_pp->mode[port_idx],
                                     p_ps, cap_str_idx );

}  // port_parm_from_port_settings



/*HDR*/
/**
 * @brief Check if all serial port capabilities reported by a device are supported by the current driver.
 *
 * @param[in]   p_pi               Address of a ::PORT_INFO structure to be checked.
 * @param[out]  str_type_info_idx  An array of ::STR_TYPE_INFO_IDX structures supported by the device.
 * @param[in]   n_str_type         The number of entries in the @a str_type_info_idx array.
 *
 * @return  A bit mask of @ref MBG_COM_CFG_STATUS_MASKS flags indicating which type of parameter
 *          may not be fully supported. Should be 0 if everything is OK.
 *
 * @see ::is_valid_port_info
 * @see @ref MBG_COM_CFG_STATUS_MASKS
 */
uint32_t check_valid_port_info( const PORT_INFO *p_pi,
                                const STR_TYPE_INFO_IDX str_type_info_idx[],
                                int n_str_type )
{
  const PORT_SETTINGS *p_ps = &p_pi->port_settings;
  int idx;
  uint32_t flags = 0;


  if ( p_pi->supp_baud_rates & ~_mask( N_MBG_BAUD_RATES ) )
    flags |= MBG_PS_MSK_BAUD_RATE_OVR_SW;  // dev. supports more baud rates than driver

  idx = get_baud_rate_idx( p_ps->parm.baud_rate );

  if ( !_inrange( idx, 0, N_MBG_BAUD_RATES ) ||
       !_is_supported( idx, p_pi->supp_baud_rates ) )
    flags |= MBG_PS_MSK_BAUD_RATE;


  if ( p_pi->supp_framings & ~_mask( N_MBG_FRAMINGS ) )
    flags |= MBG_PS_MSK_FRAMING_OVR_SW;    // dev. supports more framings than driver

  idx = get_framing_idx( p_ps->parm.framing );

  if ( !_inrange( idx, 0, N_MBG_FRAMINGS ) ||
       !_is_supported( idx, p_pi->supp_framings ) )
    flags |= MBG_PS_MSK_FRAMING;


  if ( p_ps->parm.handshake >= N_COM_HS )
    flags |= MBG_PS_MSK_HS_OVR_SW;         // handshake index exceeds max.

  if ( p_ps->parm.handshake != HS_NONE )   // currently no device supports any handshake
    flags |= MBG_PS_MSK_HS;                // handshake mode not supp. by dev.


  if ( p_pi->supp_str_types & ~_mask( n_str_type ) )
    flags |= MBG_PS_MSK_STR_TYPE_OVR_SW;   // firmware error: more string types supported than reported

  idx = p_ps->str_type;

  if ( idx >= n_str_type )
    flags |= MBG_PS_MSK_STR_TYPE_OVR_DEV;  // string type index exceeds max.
  else
  {
    if ( !_is_supported( idx, p_pi->supp_str_types ) )
      flags |= MBG_PS_MSK_STR_TYPE;        // string type not supported by this port
    else
    {
      // Use the str_type index to get the supported output mode mask
      // from the string type info table. This is required to check
      // whether the selected mode is supported by the selected
      // string type.
      ulong supp_modes = str_type_info_idx[idx].str_type_info.supp_modes;

      if ( supp_modes & ~_mask( N_STR_MODE ) )
        flags |= MBG_PS_MSK_STR_MODE_OVR_SW;  // dev. supports more string modes than driver

      idx = p_ps->mode;

      if ( idx >= N_STR_MODE )                // mode is always >= 0
        flags |= MBG_PS_MSK_STR_MODE_OVR_SW;  // string mode index exceeds max.
      else
        if ( !_is_supported( idx, supp_modes ) )
          flags |= MBG_PS_MSK_STR_MODE;       // string mode not supp. by this string type and port
    }
  }


  if ( p_ps->flags != 0 )            /* currently always 0 */
    flags |= MBG_PS_MSK_FLAGS_OVR_SW | MBG_PS_MSK_FLAGS;


  return flags;

}  // check_valid_port_info



/*HDR*/
/**
 * @brief Check if the content of a ::PORT_INFO structure is known and valid.
 *
 * @param[in]  p_pi               Address of a ::PORT_INFO structure to be checked.
 * @param[in]  str_type_info_idx  An array of string type information.
 * @param[in]  n_str_type         The number of entries in the @a str_type_info_idx array.
 *
 * @return  @a true if the ::PORT_INFO structure contains valid data,
 *          else @a false.
 *
 * @see ::check_valid_port_info
 */
bool is_valid_port_info( const PORT_INFO *p_pi,
                         const STR_TYPE_INFO_IDX str_type_info_idx[],
                         int n_str_type )
{
  return check_valid_port_info( p_pi, str_type_info_idx, n_str_type ) == 0;

}  // valid_port_info



/*HDR*/
/**
 * @brief Setup an array of ::PORT_INFO_IDX structures from a ::PORT_PARM.
 *
 * Some legacy GPS receivers that don't provide a ::RECEIVER_INFO structure
 * also provide a ::PORT_PARM structure with the current serial port settings
 * only.
 *
 * This function sets up an array of ::PORT_INFO_IDX structures with the
 * associated settings, and fills up the remaining ::PORT_INFO fields with
 * the well-known supported settings, so applications can simply deal with
 * the current API structures.
 *
 * @param[out]  pii   Array with @a p_ri->n_com_ports ::PORT_INFO_IDX elements to be filled up.
 * @param[in]   p_pp  Pointer to a ::PORT_PARM structure providing the current COM port settings.
 * @param[in]   p_ri  Pointer to a ::RECEIVER_INFO structure providing the number of supported COM ports.
 *
 * @return  One of the @ref MBG_RETURN_CODES, but actually always ::MBG_SUCCESS.
 *
 * @ingroup cfg_hlp_com_parm_cnv_fncs
 * @see setup_default_str_type_info_idx
 */
int setup_port_info_from_port_parm( PORT_INFO_IDX pii[], const PORT_PARM *p_pp,
                                    const RECEIVER_INFO *p_ri )
{
  int i;

  for ( i = 0; i < p_ri->n_com_ports; i++ )
  {
    PORT_INFO_IDX *p_pii = &pii[i];
    PORT_INFO *p_pi = &p_pii->port_info;

    p_pii->idx = i;
    port_settings_from_port_parm( &p_pi->port_settings, i, p_pp, 1 );

    p_pi->supp_baud_rates = DEFAULT_GPS_BAUD_RATES_C166;
    p_pi->supp_framings = DEFAULT_GPS_FRAMINGS_C166;
    p_pi->supp_str_types = DEFAULT_SUPP_STR_TYPES_GPS;
  }

  return MBG_SUCCESS;

}  // setup_port_info_from_port_parm



/*HDR*/
/**
 * @brief Setup an array of ::STR_TYPE_INFO_IDX for well-known string types.
 *
 * Some legacy GPS receivers that don't provide a ::RECEIVER_INFO structure
 * also don't support the ::STR_TYPE_INFO structure to indicate which serial
 * string formats are supported.
 *
 * This function sets up an array of ::STR_TYPE_INFO_IDX structures with the
 * well-known supported string types, so applications can simply deal with
 * the current API structures.
 *
 * @param[out]  stii  Array with at least @a p_ri->n_str_type ::STR_TYPE_INFO_IDX elements to be filled up.
 * @param[in]   p_ri  Pointer to a ::RECEIVER_INFO structure providing the number of supported COM ports.
 *
 * @return  One of the @ref MBG_RETURN_CODES, but actually always ::MBG_SUCCESS.
 *
 * @ingroup cfg_hlp_com_parm_cnv_fncs
 * @see ::setup_port_info_from_port_parm
 */
int setup_default_str_type_info_idx( STR_TYPE_INFO_IDX stii[], const RECEIVER_INFO *p_ri )
{
  int i;

  for ( i = 0; i < p_ri->n_str_type; i++ )
  {
    STR_TYPE_INFO_IDX *stip = &stii[i];
    stip->idx = i;
    stip->str_type_info = default_str_type_info[i];
  }

  return MBG_SUCCESS;

}  // setup_default_str_type_info_idx



/*HDR*/
/**
 * @brief Determine how many GNSS systems are marked to be supported.
 *
 * This function counts the number of bits that are set to indicate
 * that specific GNSS types (GPS, GLONASS, Galileo, ...) are supported
 * by the device, and sets up the ::ALL_GNSS_INFO::n_gnss_supp field
 * accordingly.
 *
 * @param[in,out]  p_agi  Address of an ::ALL_GNSS_INFO structure to be updated.
 *
 * @return  ::MBG_SUCCESS on success, or ::MBG_ERR_N_GNSS_EXCEEDS_SUPP if the
 *          number of bits that are set exceeds the number of known GNSS types.
 */
int chk_set_n_gnss_supp( ALL_GNSS_INFO *p_agi )
{
  p_agi->n_gnss_supp = num_bits_set( p_agi->gnss_mode_info.supp_gnss_types );

  if ( p_agi->n_gnss_supp > N_GNSS_TYPES )
    return MBG_ERR_N_GNSS_EXCEEDS_SUPP;

  return MBG_SUCCESS;

}  // chk_set_n_gnss_supp



/*HDR*/
/**
 * @brief Setup GNSS sat info from the legacy GPS ::STAT_INFO.
 *
 * This simplyifies further evaluation since e.g. a common
 * display routine can be used.
 *
 * @param[in,out]  p_agi  Address of an ::ALL_GNSS_INFO structure to be updated.
 */
void setup_gps_only_gnss_sat_info_idx_from_stat_info( ALL_GNSS_INFO *p_agi )
{
  STAT_INFO *p_si = &p_agi->stat_info;
  GNSS_SAT_INFO_IDX *p_gsii = &p_agi->gnss_sat_info_idx[GNSS_TYPE_GPS];
  GNSS_SAT_INFO *p_gsi = &p_gsii->gnss_sat_info;

  memset( p_gsii, 0, sizeof( *p_gsii ) );
  p_gsii->idx = GNSS_TYPE_GPS;

  p_gsi->gnss_type = GNSS_TYPE_GPS;
  p_gsi->svs_in_view = p_si->svs_in_view;
  p_gsi->good_svs = p_si->good_svs;

}  // setup_gps_only_gnss_sat_info_idx_from_stat_info



/*HDR*/
/**
 * @brief Setup GNSS mode info from the legacy GPS ::STAT_INFO.
 *
 * This simplyifies further evaluation since e.g. a common
 * display routine can be used.
 *
 * @param[in,out]  p_agi  Address of an ::ALL_GNSS_INFO structure to be updated.
 */
int setup_gps_only_gnss_mode_info_from_stat_info( ALL_GNSS_INFO *p_agi )
{
  MBG_GNSS_MODE_INFO *p_gmi = &p_agi->gnss_mode_info;

  memset( p_gmi, 0, sizeof( *p_gmi ) );

  p_gmi->supp_gnss_types = MBG_GNSS_TYPE_MSK_GPS;
  p_gmi->settings.gnss_set = p_gmi->supp_gnss_types;

  memset( p_agi->gnss_sat_info_idx, 0, sizeof( p_agi->gnss_sat_info_idx ) );

  setup_gps_only_gnss_sat_info_idx_from_stat_info( p_agi );

  return chk_set_n_gnss_supp( p_agi );

}  // setup_gps_only_gnss_mode_info_from_stat_info



/*HDR*/
void chk_free_dev_hw_id( DEVICE_INFO *p )
{
  if ( p->hw_id )
  {
    free( p->hw_id );
    p->hw_id = NULL;
  }

}  // chk_free_dev_hw_id



/*HDR*/
int alloc_dev_hw_id( DEVICE_INFO *p, size_t len )
{
  if ( p->hw_id )
    return MBG_ERR_ALREADY_ALLOC;


  p->hw_id = (char *) malloc( len );

  if ( p->hw_id == NULL )
    return MBG_ERR_NO_MEM;

  return MBG_SUCCESS;

}  // alloc_dev_hw_id



/*HDR*/
const char *get_fw_id_from_hw_id( const char *hw_id )
{
  int i;

  for ( i = 0; i < N_SUPP_DEV_TOTAL; i++ )
  {
    DEVICE_INFO *p = &device_list[i];

    if ( strlen( p->fw_id ) && p->hw_id )  //### TODO check if this still works as expected
    {
      if ( strcmp( hw_id, p->hw_id ) == 0 )
      {
        if ( strlen( p->fw_id ) > 0 )
          return p->fw_id;

        #if defined( DEBUG )
          fprintf( stderr, "Length of fw_id is 0 in %s for device %i (%s)\n",
                   __func__, i, p->hw_id );
        #endif
      }
    }
  }

  return NULL;

}  // get_fw_id_from_hw_id



/*HDR*/
const char *get_hw_id_from_fw_id( const char *fw_id )
{
  int i;

  for ( i = 0; i < N_SUPP_DEV_TOTAL; i++ )
  {
    DEVICE_INFO *p = &device_list[i];

    if ( strlen( p->fw_id ) && p->hw_id )  //### TODO check if this still works as expected
    {
      if ( strcmp( fw_id, p->fw_id ) == 0 )
      {
        if ( strlen( p->hw_id ) > 0 )
          return p->hw_id;

        #if defined( DEBUG )
          fprintf( stderr, "Length of hw_id is 0 in %s for device %i (%s)\n",
                   __func__, i, p->fw_id );
        #endif
      }
    }
  }

  return NULL;

}  // get_hw_id_from_fw_id



/*HDR*/
/**
 * @brief Return the currently used ::MBG_IO_PORT_TYPE_INFO for the appropriate ::MBG_IO_PORT.
 *
 * @param[in]  port       Pointer to the ::MBG_IO_PORT to search for port_type.
 * @param[in]  port_type  Enum value of ::MBG_IO_PORT_TYPES to search for in @a port (::MBG_IO_PORT).
 *
 * @return  Pointer to the ::MBG_IO_PORT_TYPE_INFO_IDX found, or NULL.
 */
MBG_IO_PORT_TYPE_INFO *get_io_port_type_info( const MBG_IO_PORT *port,
                                              uint16_t port_type )
{
  unsigned i;
  MBG_IO_PORT_TYPE_INFO *pti;

  if ( !port )
    return NULL;

  for ( i = 0; i < port->p_info.num_types; ++i )
  {
    pti = &port->pt_infos[i];

    if ( pti->port_type == port_type )
    {
      if ( pti->port_type == MBG_IO_PORT_TYPE_GPIO )
      {
        if ( pti->data.gpio_limits.type == port->setts->data.gpio_settings.type )
          return pti;
      }
      else
        return pti;
    }
  }

  return NULL;

}  // get_io_port_type_info



/*HDR*/
/**
 * @brief Initialize a ::MBG_TLV_ANNOUNCE structure.
 *
 * @param[out]  tlv            Pointer to a ::MBG_TLV_ANNOUNCE structure.
 * @param[in]   uid            Unique sender ID used as identifier with all
 *                             subsequent messages related to this transaction.
 * @param[in]   tlv_feat_type  One of the ::MBG_TLV_FEAT_TYPES.
 * @param[in]   total_bytes    Total number of bytes of all upcoming TLVs.
 */
void mbg_tlv_announce_init( MBG_TLV_ANNOUNCE *tlv, MBG_TLV_UID uid,
                            MBG_TLV_TYPE tlv_feat_type, uint32_t total_bytes )
{
  memset( tlv, 0, sizeof( *tlv ) );
  tlv->data.uid = uid;
  tlv->data.type = tlv_feat_type;
  tlv->data.total_bytes = total_bytes;
  tlv->data.reserved_1 = 0;
  tlv->reserved_1 = 0;
  tlv->reserved_2 = 0;

}  // mbg_tlv_announce_init



/*HDR*/
/**
 * @brief Initialize a ::MBG_TLV structure.
 *
 * @param[out]  tlv          Pointer to a valid ::MBG_TLV structure.
 * @param[in]   uid          Unique sender ID used as identifier for each further
 *                           TLV message related to this type.
 * @param[in]   tlv_type     Type identifier, see ::MBG_TLV_TYPES.
 * @param[in]   total_bytes  Total number of bytes belonging to this TLV transaction
 *                           (which is very likely split into several TLVs).
 */
void mbg_tlv_init( MBG_TLV *tlv, MBG_TLV_UID uid,
                   MBG_TLV_TYPE tlv_type, uint32_t total_bytes )
{
  memset( tlv, 0, sizeof( *tlv ) );
  tlv->hdr.uid = uid;
  tlv->hdr.tlv_type = tlv_type;
  tlv->hdr.cur_bytes = 0;
  tlv->hdr.trans_bytes = 0;
  tlv->hdr.total_bytes = total_bytes;
  tlv->hdr.reserved_1 = 0;
  tlv->hdr.reserved_2 = 0;
  tlv->hdr.reserved_3 = 0;

}  // mbg_tlv_init



/*HDR*/
/**
 * @brief Initialize ::MBG_TLV_RCV_STATE structure.
 *
 * @param[out]  state        Pointer to ::MBG_TLV_RCV_STATE structure to be initialized.
 * @param[in]   uid          Unique sender ID used as identifier for each further
 *                           TLV message related to this type.
 * @param[in]   total_bytes  Total number of bytes belonging to this TLV transaction
 *                           (which is very likely split into several TLVS).
 */
void mbg_tlv_rcv_state_init( MBG_TLV_RCV_STATE *state, MBG_TLV_UID uid, uint32_t total_bytes )
{
  state->data.uid = uid;
  state->data.type = 0;
  state->data.total_bytes = total_bytes;
  state->data.reserved_1 = 0;
  state->read_bytes = 0;
  state->reserved_1 = 0;

}  // mbg_tlv_state_init



/*HDR*/
/**
 * @brief Print some Meinberg OS revision information into a string buffer.
 *
 * @param[out] s        The string buffer to be filled.
 * @param[in]  max_len  Size of the output buffer for 0-terminated string.
 * @param[in]  prefix   Pointer to an optional prefix string, or NULL.
 * @param[in]  suffix   Pointer to an optional suff string, or NULL.
 * @param[in]  rev      The revision code which will be split into major, minor, and patch version.
 *
 * @return Length of the string in the buffer.
 *
 * @see @ref group_ext_sys_info
 */
int mbg_snprint_revision( char *s, size_t max_len,
                          const char *prefix, const char *suffix,
                          uint32_t rev )
{
  int n = 0;
  unsigned major, minor, patch;

  if ( prefix )
    n += snprintf_safe( &s[n], max_len - n, "%s", prefix );

  _mbg_decode_revision( rev, major, minor, patch );

  n += snprintf_safe( &s[n], max_len - n, "%u.%u.%u",
                      major, minor, patch);

  if ( suffix )
    n += snprintf_safe( &s[n], max_len - n, "%s", suffix );

  return n;

}  // mbg_snprint_revision



/*HDR*/
/**
 * @brief Check if all ::RECEIVER_INFO capabilities reported by a device are supported by the current driver.
 *
 * @param[in]  p  Address of a ::RECEIVER_INFO structure to be checked.
 *
 * @return  ::MBG_SUCCESS if everything is OK, one of the @ref MBG_ERROR_CODES, as appropriate.
 */
_NO_MBG_API_ATTR int _MBG_API chk_dev_receiver_info( const RECEIVER_INFO *p )
{
  // Make sure this program supports at least as many serial ports
  // as the current device.
  if ( p->n_com_ports > MAX_PARM_PORT )
    return MBG_ERR_N_COM_EXCEEDS_SUPP;

  // Make sure this program supports at least as many string types
  // as the current device.
  if ( p->n_str_type > MAX_PARM_STR_TYPE )
    return MBG_ERR_N_STR_EXCEEDS_SUPP;

  // Make sure this program supports at least as many programmable
  // outputs as the current device.
  if ( p->n_prg_out > MAX_PARM_POUT )
    return MBG_ERR_N_STR_EXCEEDS_SUPP;

  return MBG_SUCCESS;

}  // chk_dev_receiver_info



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_xbp_supp_nodes( const ALL_XBP_INFO *info )
{
  if ( info )
  {
    if ( ( info->limits.features & XBP_FEAT_MASK_NODES ) == XBP_FEAT_MASK_NODES )
      return MBG_SUCCESS;

    return MBG_ERR_NOT_SUPP_BY_DEV;
  }

  return MBG_ERR_INV_PARM;

}  // chk_dev_xbp_supp_nodes



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_net_cfg_supp_stage_2( const ALL_NET_CFG_INFO *info )
{
  if ( info->glb_cfg_info.feat_flags & MBG_NET_GLB_SUPP_STAGE_2_MASK )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // chk_dev_net_cfg_supp_stage_2



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_ntp_supp_client( const ALL_NTP_CFG_INFO *info )
{
  if ( ( ( info->glb_info.supp_ntp_roles & NTP_MSK_ROLE_CLIENT ) == NTP_MSK_ROLE_CLIENT ) ||
       ( ( info->glb_info.supp_ntp_roles & NTP_MSK_ROLE_CLIENT_SERVER ) == NTP_MSK_ROLE_CLIENT_SERVER ) )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // chk_dev_ntp_supp_client



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_ntp_supp_server( const ALL_NTP_CFG_INFO *info )
{
  if ( ( ( info->glb_info.supp_ntp_roles & NTP_MSK_ROLE_SERVER ) == NTP_MSK_ROLE_SERVER ) ||
       ( ( info->glb_info.supp_ntp_roles & NTP_MSK_ROLE_CLIENT_SERVER ) == NTP_MSK_ROLE_CLIENT_SERVER ) )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // chk_dev_ntp_supp_server



static __mbg_inline /*HDR*/
/**
 * @brief Check if a specific bit is set in ::XMULTI_REF_INSTANCES::flags.
 *
 * This indicates if a specific feature is supported.
 * The ::XMULTI_REF_INSTANCES structure is part of an ::ALL_XMULTI_REF_INFO
 * structure, namely ::ALL_XMULTI_REF_INFO::instances.
 *
 * @param[in]  info  Address of an ::ALL_XMULTI_REF_INFO structure
 *                   of which to check the ::XMULTI_REF_INSTANCES::flags.
 *
 * @param[in]  mask  One of the ::XMR_INST_FLAG_BIT_MASKS.
 *
 * @return  ::MBG_SUCCESS if the flag specified in @a mask is set,
 *          else ::MBG_ERR_NOT_SUPP_BY_DEV.
 *
 * @see ::XMR_INST_FLAG_BIT_MASKS
 */
int chk_mrs_instances_flags( const ALL_XMULTI_REF_INFO *info, int mask )
{
  return ( info->instances.flags & mask ) ? MBG_SUCCESS : MBG_ERR_NOT_SUPP_BY_DEV;

}  // chk_mrs_instances_flags



/*HDR*/
/**
 * @brief Check if the ::XMULTI_EXT_REF_INSTANCES structure is supported.
 *
 * @param[in]  info  Address of an ::ALL_XMULTI_REF_INFO structure to be checked.
 *
 * @return  ::MBG_SUCCESS or ::MBG_ERR_NOT_SUPP_BY_DEV,
 *          as returned by ::chk_mrs_instances_flags.
 */
_NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_supp_ext_instances( const ALL_XMULTI_REF_INFO *info )
{
  return chk_mrs_instances_flags( info, XMRIF_MSK_EXT_REF_INSTANCES_SUPP );

}  // chk_dev_xmulti_supp_ext_instances



/*HDR*/
/**
 * @brief Check if the ::XMULTI_REF_DICT structure is supported.
 *
 * @param[in]  info  Address of an ::ALL_XMULTI_REF_INFO structure to be checked.
 *
 * @return  ::MBG_SUCCESS or ::MBG_ERR_NOT_SUPP_BY_DEV,
 *          as returned by ::chk_mrs_instances_flags.
 */
_NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_supp_dict( const ALL_XMULTI_REF_INFO *info )
{
  return chk_mrs_instances_flags( info, XMRIF_MSK_REF_DICT );

}  // chk_dev_xmulti_supp_dict



/*HDR*/
/**
 * @brief Check if the ::XMULTI_EXT_REF_INSTANCES structure is supported.
 *
 * @param[in]  info  Address of an ::ALL_XMULTI_REF_INFO structure to be checked.
 *
 * @return  ::MBG_SUCCESS or ::MBG_ERR_NOT_SUPP_BY_DEV,
 *          as returned by ::chk_mrs_instances_flags.
 */
_NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_not_configurable( const ALL_XMULTI_REF_INFO *info )
{
  return chk_mrs_instances_flags( info, XMRIF_MSK_NOT_CONFIGURABLE );

}  // chk_dev_xmulti_not_configurable



/*HDR*/
/**
 * @brief Check if the ::XMRIF_BIT_NO_STATUS bit is set.
 *
 * If this feature bit is set, ***no*** status, stats, etc., are provided.
 *
 * @param[in]  info  Address of an ::ALL_XMULTI_REF_INFO structure to be checked.
 *
 * @return  ::MBG_SUCCESS if the bit is set, or ::MBG_ERR_NOT_SUPP_BY_DEV,
 *          as returned by ::chk_mrs_instances_flags.
 */
_NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_no_status( const ALL_XMULTI_REF_INFO *info )
{
  return chk_mrs_instances_flags( info, XMRIF_MSK_NO_STATUS );

}  // chk_dev_xmulti_no_status



/*HDR*/
/**
 * @brief Check if the ::XMRIF_BIT_MRF_NONE_SUPP bit is set.
 *
 * If this feature bit is set, the MRS pseudo-type ::MULTI_REF_NONE is supported.
 *
 * @param[in]  info  Address of an ::ALL_XMULTI_REF_INFO structure to be checked.
 *
 * @return  ::MBG_SUCCESS if the bit is set, or ::MBG_ERR_NOT_SUPP_BY_DEV,
 *          as returned by ::chk_mrs_instances_flags.
 */
_NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_ref_supp_mrf_none( const ALL_XMULTI_REF_INFO *info )
{
  return chk_mrs_instances_flags( info, XMRIF_MSK_MRF_NONE_SUPP );

}  // chk_dev_xmulti_ref_supp_mrf_none



/*HDR*/
/**
 * @brief Check if the ::XMR_EXT_SRC_INFO structure is supported.
 *
 * @param[in]  info  Address of an ::ALL_XMULTI_REF_INFO structure to be checked.
 *
 * @return  ::MBG_SUCCESS or ::MBG_ERR_NOT_SUPP_BY_DEV,
 *          as returned by ::chk_mrs_instances_flags.
 */
_NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_ref_supp_ext_src_info( const ALL_XMULTI_REF_INFO *info )
{
  return chk_mrs_instances_flags( info, XMRIF_MSK_EXT_SRC_INFO_SUPP );

}  // chk_dev_xmulti_ref_supp_ext_src_info



/*HDR*/
/**
 * @brief Check if the ::::XMR_EXT_SRC_INFO structure is supported.
 *
 * @param[in]  info  Address of an ::ALL_XMULTI_REF_INFO structure to be checked.
 *
 * @return  ::MBG_SUCCESS or ::MBG_ERR_NOT_SUPP_BY_DEV,
 *          as returned by ::chk_mrs_instances_flags.
 */
_NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_ref_supp_holdover_status( const ALL_XMULTI_REF_INFO *info )
{
  return chk_mrs_instances_flags( info, XMRIF_MSK_HOLDOVER_STATUS_SUPP );

}  // chk_dev_xmulti_ref_supp_holdover_status



static __mbg_inline /*HDR*/
/**
 * @brief Check if an MRS type code is valid.
 *
 * A valid type must be one of the currently defined ::MULTI_REF_TYPES,
 * except ::MULTI_REF_NONE, which is < 0, or any other value which is
 * less than ::MAX_N_MULTI_REF_TYPES.
 *
 * @param[in]  type  The MRS type code to be checked, see ::MULTI_REF_TYPES.
 *
 * @return  @a true if the MRS type code is valid, else @a false.
 *
 * @see ::MULTI_REF_TYPES
 */
bool is_valid_mrs_type( int type )
{
  return ( type >= 0 ) && ( type < MAX_N_MULTI_REF_TYPES );

}  // is_valid_mrs_type



static __mbg_inline /*HDR*/
/**
 * @brief Check if an extended MRS source feature is supported.
 *
 * Whether a feature is supported, or not, depends on whether a specific
 * bit is set in the @a feat_flags field of the @a info field of the
 * @a ext_src_infos array entry for the specified @a type.
 *
 * The bit masks of the features that can be checked are defined
 * in ::XMR_EXT_SRC_FEAT_FLAG_MSKS.
 *
 * @param[in]  info  Address of an ::ALL_XMULTI_REF_INFO structure
 *                   of which to check the @a ext_src_infos flags.
 *
 * @param[in]  type  A valid code of ::MULTI_REF_TYPES, see ::is_valid_mrs_type.
 *
 * @param[in]  mask  One of the ::XMR_EXT_SRC_FEAT_FLAG_MSKS.
 *
 * @return  ::MBG_SUCCESS if the flag specified in @a mask is set,
 *          ::MBG_ERR_NOT_SUPP_BY_DEV if the bit is not set, or
 *          ::MBG_ERR_INV_PARM if the @a type is out of range.
 *
 * @see ::XMR_EXT_SRC_FEAT_FLAG_MSKS
 */
int chk_mrs_ext_src_feat( const ALL_XMULTI_REF_INFO *info, int type, int mask )
{
  if ( !is_valid_mrs_type( type ) )
    return MBG_ERR_INV_PARM;

  return ( info->ext_src_infos[type].info.feat_flags & mask ) ?
         MBG_SUCCESS : MBG_ERR_NOT_SUPP_BY_DEV;

}  // chk_mrs_ext_src_feat



/*HDR*/
/**
 * @brief Check if an XMR source provides ::XMR_STATS.
 *
 * @param[in]  info  Address of an ::ALL_XMULTI_REF_INFO structure to be checked.
 * @param[in]  type  One of the ::MULTI_REF_TYPES, except ::MULTI_REF_NONE which is -1.
 *
 * @see ::chk_dev_xmulti_ref_supp_ext_src_info
 */
_NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_ref_supp_ext_source_stats( const ALL_XMULTI_REF_INFO *info, int type )
{
  return chk_mrs_ext_src_feat( info, type, XMR_EXT_SRC_FEAT_FLAG_MSK_STATS );

}  // chk_dev_xmulti_ref_supp_ext_source_stats



/*HDR*/
/**
 * @brief Check if an XMR source provides ::XMR_METRICS.
 *
 * @param[in]  info  Address of an ::ALL_XMULTI_REF_INFO structure to be checked.
 * @param[in]  type  One of the ::MULTI_REF_TYPES, except ::MULTI_REF_NONE which is -1.
 *
 * @see ::chk_dev_xmulti_ref_supp_ext_source_adv_metrics
 * @see ::chk_dev_xmulti_ref_supp_ext_src_info
 */
_NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_ref_supp_ext_source_metrics( const ALL_XMULTI_REF_INFO *info, int type )
{
  return chk_mrs_ext_src_feat( info, type, XMR_EXT_SRC_FEAT_FLAG_MSK_METRICS );

}  // chk_dev_xmulti_ref_supp_ext_source_metrics



/*HDR*/
/**
 * @brief Check if an XMR source supports advanced XMR QL metrics configuration.
 *
 * If this feature is not available, ::XMR_METRICS can not be configured.
 *
 * @param[in]  info  Address of an ::ALL_XMULTI_REF_INFO structure to be checked.
 * @param[in]  type  One of the ::MULTI_REF_TYPES, except ::MULTI_REF_NONE which is -1.
 *
 * @see ::XMR_QL_LIMITS
 * @see ::chk_dev_xmulti_ref_supp_ext_source_metrics
 * @see ::chk_dev_xmulti_ref_supp_ext_src_info
 */
_NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_ref_supp_ext_source_adv_metrics( const ALL_XMULTI_REF_INFO *info, int type )
{
  return chk_mrs_ext_src_feat( info, type, XMR_EXT_SRC_FEAT_FLAG_MSK_ADV_METRICS );

}  // chk_dev_xmulti_ref_supp_ext_source_adv_metrics



/*HDR*/
/**
 * @brief Check if an XMR source XMR source supports coasting mode.
 *
 * If this feature is not available, ::XMR_METRICS can not be configured.
 *
 * @param[in]  info  Address of an ::ALL_XMULTI_REF_INFO structure to be checked.
 * @param[in]  type  One of the ::MULTI_REF_TYPES, except ::MULTI_REF_NONE which is -1.
 *
 * @see ::XMR_QL_LIMITS
 * @see ::chk_dev_xmulti_ref_supp_ext_src_info
 */
_NO_MBG_API_ATTR int _MBG_API chk_dev_xmulti_ref_supp_ext_source_coasting( const ALL_XMULTI_REF_INFO *info, int type )
{
  return chk_mrs_ext_src_feat( info, type, XMR_EXT_SRC_FEAT_FLAG_MSK_COASTING );

}  // chk_dev_xmulti_ref_supp_ext_source_coasting



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_ims_has_fdm( const ALL_IMS_INFO *info )
{
  if ( info->state.flags & MBG_IMS_STATE_FLAG_MSK_HAS_FDM )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // chk_dev_ims_has_fdm



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_ims_is_volt_out_enabled( const ALL_IMS_STATE *ims_state, unsigned idx )
{
  const MBG_IMS_SENSOR_STATE *sstate = &ims_state->sensor_state_idx[idx].state;

  if ( ( sstate->type == MBG_IMS_SENSOR_VOLTAGE ) && ( sstate->flags & MBG_IMS_SENSOR_VOLTAGE_OUT_ENB ) )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // chk_dev_ims_is_volt_out_enabled



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_ims_is_volt_out_overload( const ALL_IMS_STATE *ims_state, unsigned idx )
{
  const MBG_IMS_SENSOR_STATE *sstate = &ims_state->sensor_state_idx[idx].state;

  if ( ( sstate->type == MBG_IMS_SENSOR_VOLTAGE ) && ( sstate->flags & MBG_IMS_SENSOR_VOLTAGE_OUT_OVR ) )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // chk_dev_ims_is_volt_out_overload



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_ims_is_pll_locked( const ALL_IMS_STATE *ims_state, unsigned idx )
{
  const MBG_IMS_SENSOR_STATE *sstate = &ims_state->sensor_state_idx[idx].state;

  if ( ( sstate->type == MBG_IMS_SENSOR_PLL ) && ( sstate->flags & MBG_IMS_SENSOR_PLL_LOCKED ) )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // chk_dev_ims_is_pll_locked



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_gpio_supp_ass_idx( const ALL_GPIO_INFO *gpio_info, unsigned idx )
{
  const MBG_GPIO_LIMITS *limits = &gpio_info->infos[idx].info.limits;

  if ( limits->supp_flags & MSK_MBG_GPIO_DEPENDS_ON_ASS_IO_IDX )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // chk_dev_gpio_supp_ass_idx



/*HDR*/
_NO_MBG_API_ATTR int _MBG_API chk_dev_gpio_dep_on_ass_idx( const ALL_GPIO_INFO *gpio_info, unsigned idx )
{
  const MBG_GPIO_SETTINGS *settings = &gpio_info->infos[idx].info.settings;

  if ( settings->flags & MSK_MBG_GPIO_DEPENDS_ON_ASS_IO_IDX )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // chk_dev_gpio_dep_on_ass_idx



/*HDR*/
/**
 * @brief Check whether GPIO provides a status.
 *
 * @param[in]  info  Address of an ::ALL_GPIO_INFO structure to be checked.
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_gpio
 * @see ::mbg_chk_dev_has_gpio
 * @see ::free_all_gpio_info
 */
_NO_MBG_API_ATTR int _MBG_API chk_dev_gpio_has_status( const ALL_GPIO_INFO *info )
{
  if ( info->cfg_limits.flags & MBG_GPIO_CFG_LIMIT_FLAG_MASK_STATUS_SUPP )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // chk_dev_gpio_has_status



/*HDR*/
/**
 * @brief Convert one ot the ::MBG_LED_MODES to the appropriate pendant
 * in ::MBG_IO_PORT_STATUS_BITS.
 *
 * @param[in]  mode  One of ::MBG_LED_MODES to be converted
 *
 * @return One of the @ref MBG_IO_PORT_STATUS_BITS
 *
 */
unsigned mbg_led_mode_to_io_port_status_bit( uint8_t mode )
{
  switch ( mode )
  {
    case MBG_LED_MODE_FLASH:
      return MBG_IO_PORT_STATUS_BIT_LIGHT_FLASHING;
    case MBG_LED_MODE_FLASH_5S:
      return MBG_IO_PORT_STATUS_BIT_LIGHT_FLASHING_5S;
    case MBG_LED_MODE_FLASH_GREEN:
      return MBG_IO_PORT_STATUS_BIT_LIGHT_FLASHING_GREEN;
    case MBG_LED_MODE_FLASH_RED:
      return MBG_IO_PORT_STATUS_BIT_LIGHT_FLASHING_RED;
    default: break;
  }

  return N_MBG_IO_PORT_STATUS_BITS;

} // mbg_led_mode_to_io_port_status_bit



/*HDR*/
/**
 * @brief Convert one ot the ::MBG_LED_COLORS to the appropriate pendant
 * in ::MBG_IO_PORT_STATUS_BITS.
 *
 * @param[in]  color  One of ::MBG_LED_COLORS to be converted
 *
 * @return One of the @ref MBG_IO_PORT_STATUS_BITS
 */
unsigned mbg_led_color_to_io_port_status_bit( uint8_t color )
{
  switch ( color )
  {
    case MBG_LED_COLOR_RED:
      return MBG_IO_PORT_STATUS_BIT_LIGHT_RED;
    case MBG_LED_COLOR_GREEN:
      return MBG_IO_PORT_STATUS_BIT_LIGHT_GREEN;
    case MBG_LED_COLOR_YELLOW:
      return MBG_IO_PORT_STATUS_BIT_LIGHT_YELLOW;
    case MBG_LED_COLOR_BLUE:
      return MBG_IO_PORT_STATUS_BIT_LIGHT_BLUE;
    default: break;
  }

  return N_MBG_IO_PORT_STATUS_BITS;

} // mbg_led_color_to_io_port_status_bit


/*HDR*/
/**
 * @brief Free an ::ALL_XBP_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_XBP_INFO structure to be freed.
 *
 * @see ::mbgextio_get_all_xbp_info
 */
void free_all_xbp_info( ALL_XBP_INFO *p )
{
  if ( p )
  {
    if ( p->node_limits )
      free( p->node_limits );

    if ( p->node_infos )
      free( p->node_infos );

    free( p );
  }

}  // free_all_xbp_info



/*HDR*/
/**
 * @brief Free an ::ALL_SYSTEM_CONFIG structure.
 *
 * @param[in]  p  Pointer to the ::ALL_SYSTEM_CONFIG structure to be freed.
 */
void free_all_system_config( ALL_SYSTEM_CONFIG *p )
{
  if ( p )
  {
    if ( p->config_sets_glb_info )
    {
      if ( p->config_sets_glb_info->num_supp_config_sets > 0 )
        free( p->config_sets );

      free( p->config_sets_glb_info );
    }

    free (p);
  }

}  // free_all_system_config



/*HDR*/
/**
 * @brief Free an ::ALL_SYSTEM_STATUS structure.
 *
 * @param[in]  p  Pointer to the ::ALL_SYSTEM_STATUS structure to be freed.
 */
void free_all_system_status( ALL_SYSTEM_STATUS *p )
{
  if ( p )
  {
    if ( p->ext_sys_info )
      free( p->ext_sys_info );

    if ( p->ext_sys_status )
      free( p->ext_sys_status );

    if ( p->ca_certs_list )
      free( p->ca_certs_list );

    free( p );
  }

} // free_all_system_status



/*HDR*/
/**
 * @brief Free an ::ALL_SYSTEM_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_SYSTEM_INFO structure to be freed.
 */
void free_all_system_info( ALL_SYSTEM_INFO *p )
{
  if ( p )
  {
    if ( p->config )
      free_all_system_config( p->config );

    if ( p->status )
      free_all_system_status( p->status );

    free( p );
  }

} // free_all_system_info


/*HDR*/
/**
 * @brief Free an ::ALL_RECEIVER_CONFIG structure.
 *
 * @param[in]  p  Pointer to the ::ALL_RECEIVER_CONFIG structure to be freed.
 */
void free_all_receiver_config( ALL_RECEIVER_CONFIG *p )
{
  if ( p )
  {
    if ( p->ant_cable_len )
        free( p->ant_cable_len );

    if ( p->gnss_mode_info )
        free( p->gnss_mode_info );

    if ( p->ignore_lock )
        free( p->ignore_lock );

    if ( p->xmr_cntl )
        free( p->xmr_cntl );

    free( p );
  }

}  // free_all_receiver_config


/*HDR*/
/**
 * @brief Free an ::ALL_RECEIVER_STATUS structure.
 *
 * @param[in]  p  Pointer to the ::ALL_RECEIVER_STATUS structure to be freed.
 */
void free_all_receiver_status( ALL_RECEIVER_STATUS *p )
{
  if ( p )
  {
    if ( p->position )
      free( p->position );

    if ( p->ttm )
      free( p->ttm );

    if ( p->stat_info )
      free( p->stat_info );

    if ( p->sat_infos )
      free( p->sat_infos );

    free( p );
  }

} // free_all_receiver_status



/*HDR*/
/**
 * @brief Free an ::ALL_RECEIVER_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_RECEIVER_INFO structure to be freed.
 */
void free_all_receiver_info( ALL_RECEIVER_INFO *p )
{
  if ( p )
  {
    if ( p->config )
      free_all_receiver_config( p->config );

    if ( p->status )
      free_all_receiver_status( p->status );

    free( p );
  }

} // free_all_receiver_info



/*HDR*/
/**
 * @brief Free an ::ALL_TIME_ZONE_CONFIG structure.
 *
 * @param[in]  p  Pointer to the ::ALL_TIME_ZONE_CONFIG structure to be freed.
 */
void free_all_time_zone_config( ALL_TIME_ZONE_CONFIG *p )
{
  if ( p )
  {
    if ( p->time_scale )
      free( p->time_scale );

    if ( p->tzdl )
      free( p->tzdl );

    free( p );

  }

} // free_all_time_zone_config


/*HDR*/
/**
* @brief Free an ::ALL_TIME_ZONE_INFO structure.
*
* @param[in]  p  Pointer to the ::ALL_TIME_ZONE_INFO structure to be freed.
*/
void free_all_time_zone_info( ALL_TIME_ZONE_INFO *p )
{
  if ( p )
  {
    if ( p->config )
      free_all_time_zone_config( p->config );

    free( p );
  }

} // free_all_time_zone_info


/*HDR*/
/**
 * @brief Free an ::ALL_NET_CFG_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_NET_CFG_INFO structure to be freed.
 *
 * @see ::mbgextio_get_all_net_cfg_info
 */
void free_all_net_cfg_info( ALL_NET_CFG_INFO *p )
{
  if ( p )
  {
    if ( p->link_infos )
      free( p->link_infos );

    if ( p->addr_infos )
      free( p->addr_infos );

    if ( p->dns_srvrs )
      free( p->dns_srvrs );

    if ( p->dns_srch_doms )
      free( p->dns_srch_doms );

    if ( p->route_infos )
      free( p->route_infos );

    free( p );
  }

}  // free_all_net_cfg_info



/*HDR*/
/**
 * @brief Free an ::ALL_NET_STATUS_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_NET_STATUS_INFO structure to be freed.
 *
 * @see ::mbgextio_get_all_net_status_info
 */
void free_all_net_status_info( ALL_NET_STATUS_INFO *p )
{
  if ( p )
  {
    if ( p->link_infos )
      free( p->link_infos );

    if ( p->addr_infos )
      free( p->addr_infos );

    if ( p->dns_srvrs )
      free( p->dns_srvrs );

    if ( p->dns_srch_doms )
      free( p->dns_srch_doms );

    if ( p->route_infos )
      free( p->route_infos );

    free( p );
  }

}  // free_all_net_status_info


/*HDR*/
/**
 * @brief Free an ::ALL_NETWORK_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_NETWORK_INFO structure to be freed.
 */
void free_all_network_info( ALL_NETWORK_INFO *p )
{
  if ( p )
  {
    if ( p->config )
      free_all_net_cfg_info( p->config );
    if ( p->ext_config )
      free( p->ext_config );
    if ( p->status )
      free_all_net_status_info( p->status );
    free( p );
  }
}


/*HDR*/
/**
 * @brief Free an ::ALL_SNMP_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_SNMP_INFO structure to be freed.
 */
void free_all_snmp_info ( ALL_SNMP_INFO *p )
{
  if ( p )
  {
    if ( p->v12_infos )
      free( p->v12_infos );

    if ( p->v12_trap_infos )
      free( p->v12_trap_infos );

    if ( p->v3_infos )
      free( p->v3_infos );

    if ( p->v3_trap_infos )
      free( p->v3_trap_infos );

    if ( p->ext_config )
      free( p->ext_config );

    free( p );
  }

}  // free_all_snmp_info



/*HDR*/
/**
 * @brief Free an ::ALL_SYSLOG_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_SYSLOG_INFO structure to be freed.
 */
void free_all_syslog_info ( ALL_SYSLOG_INFO *p )
{
  if ( p )
  {
    if ( p->servers )
    {
      unsigned i;
      MBG_SYSLOG_SERVER *srv;

      for (i = 0; i < p->glb_info.settings.num_servers; ++i)
      {
        srv = &p->servers[i];

        if ( srv->release_priv )
          srv->release_priv( srv );
      }

      free( p->servers );
    }

    free( p );
  }

}  // free_all_syslog_info



/*HDR*/
/**
 * @brief Free an ::ALL_MONITORING_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_MONITORING_INFO structure to be freed.
 */
void free_all_monitoring_info ( ALL_MONITORING_INFO *p )
{
  if ( p )
  {
    if ( p->all_snmp_info )
      free_all_snmp_info( p->all_snmp_info );

    if ( p->all_events )
      free_all_events( p->all_events );

    if ( p->all_syslog_info )
      free_all_syslog_info( p->all_syslog_info );

    free ( p );
  }

}  // free_all_monitoring_info



/*HDR*/
/**
 * @brief Free an ::ALL_EVENTS structure.
 *
 * @param[in]  p  Pointer to the ::ALL_EVENTS structure to be freed.
 */
void free_all_events( ALL_EVENTS *p )
{
  if ( p )
  {
    if ( p->events )
    {
      unsigned i;
      MBG_EVENT *evt;

      for (i = 0; i < p->num_events; ++i)
      {
        evt = &p->events[i];

        if ( evt->release_priv )
          evt->release_priv( evt );

        if ( evt->release_backref )
          evt->release_backref( evt );
      }

      free( p->events );
    }

    free ( p );
  }

}  // free_all_events



/*HDR*/
/**
 * @brief Free an ::ALL_XMULTI_REF_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_XMULTI_REF_INFO structure to be freed.
 *
 * @see ::mbgextio_get_all_xmulti_ref_info
 * @see ::mbg_get_all_xmulti_ref_info
 */
void free_all_xmulti_ref_info( ALL_XMULTI_REF_INFO *p )
{
  if ( p )
  {
    if ( p->infos )
      free( p->infos );

    if ( p->ext_src_infos )
      free( p->ext_src_infos );

    if ( p->ql_settings )
      free( p->ql_settings );

    free ( p );
  }

}  // free_all_xmulti_ref_info



/*HDR*/
/**
 * @brief Free an ::ALL_XMULTI_REF_STATUS structure.
 *
 * @param[in]  p  Pointer to the ::ALL_XMULTI_REF_STATUS structure to be freed.
 *
 * @see ::mbgextio_get_all_xmulti_ref_status
 * @see ::mbg_get_all_xmulti_ref_status
 */
void free_all_xmulti_ref_status( ALL_XMULTI_REF_STATUS *p )
{
  if ( p )
  {
    if ( p->status )
      free( p->status );

    if ( p->holdover_status )
      free( p->holdover_status );

    if ( p->stats_idx )
      free( p->stats_idx );

    if ( p->metrics_idx )
      free( p->metrics_idx );

    free( p );
  }

}  // free_all_xmulti_ref_status



/*HDR*/
/**
 * @brief Free an ::ALL_PTP_V1_COMMON_DATASETS structure.
 *
 * The structure is allocated by ::mbgextio_get_all_ptp_v1_common_datasets.
 *
 * @param[in]  p  Pointer to the ::ALL_PTP_V1_COMMON_DATASETS structure to be freed.
 *
 * @see ::mbgextio_get_all_ptp_v1_common_datasets
 */
void free_all_ptp_v1_common_datasets( ALL_PTP_V1_COMMON_DATASETS *p )
{
  if ( p )
  {
    if ( p->port_datasets )
      free( p->port_datasets );

    free( p );
  }

}  // free_all_ptp_v1_common_datasets



/*HDR*/
/**
 * @brief Free an ::ALL_PTP_V2_COMMON_DATASETS structure.
 *
 * The structure is allocated by ::mbgextio_get_all_ptp_v2_common_datasets.
 *
 * @param[in]  p  Pointer to the ::ALL_PTP_V2_COMMON_DATASETS structure to be freed.
 *
 * @see ::mbgextio_get_all_ptp_v2_common_datasets
 */
void free_all_ptp_v2_common_datasets( ALL_PTP_V2_COMMON_DATASETS *p )
{
  if ( p )
  {
    if ( p->port_datasets )
      free( p->port_datasets );

    free( p );
  }

}  // free_all_ptp_v2_common_datasets



/*HDR*/
/**
 * @brief Free an ::ALL_NTP_CFG_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_NTP_CFG_INFO structure to be freed.
 *
 * @see ::mbgextio_get_all_ntp_cfg_info
 */
void free_all_ntp_cfg_info( ALL_NTP_CFG_INFO *p )
{
  if ( p )
  {
    if ( p->symm_key_limits )
      free( p->symm_key_limits );

    if ( p->symm_key_info_idx )
      free( p->symm_key_info_idx );

    if ( p->trusted_key_info_idx )
      free( p->trusted_key_info_idx );

    if ( p->clnt_info )
      free( p->clnt_info );

    if ( p->peer_settings_idx )
      free( p->peer_settings_idx );

    if ( p->srv_info )
      free( p->srv_info );

    if ( p->refclk_info_idx )
      free( p->refclk_info_idx );

    if ( p->misc_limits )
      free( p->misc_limits );

    if ( p->orphan_mode_info )
      free( p->orphan_mode_info );

    free( p );
  }

}  // free_all_ntp_cfg_info



/*HDR*/
/**
 * @brief Free an ::ALL_NTP_STATUS structure.
 *
 * @param[in]  p  Pointer to the ::ALL_NTP_STATUS structure to be freed.
 *
 * @see ::mbgextio_get_all_ntp_status
 */
void free_all_ntp_status( ALL_NTP_STATUS *p )
{
  if ( p )
  {
    if ( p->refclk_states )
      free( p->refclk_states );

    if ( p->peer_states )
      free( p->peer_states );

    free( p );
  }

}  // free_all_ntp_status


/*HDR*/
/**
 * @brief Free an ::ALL_NTP_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_NTP_INFO structure to be freed.
 */
void free_all_ntp_info( ALL_NTP_INFO *p )
{
  if ( p )
  {
    if ( p->config )
      free_all_ntp_cfg_info( p->config );
    if ( p->ext_config )
      free( p->ext_config );
    if ( p->status )
      free_all_ntp_status( p->status );
    free( p );
  }
}


/*HDR*/
/**
 * @brief Free an ::ALL_PTP_NG_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_PTP_NG_INFO structure to be freed.
 *
 * @see ::mbgextio_get_all_ptp_ng_info
 */
void free_all_ptp_ng_info( ALL_PTP_NG_INFO *p )
{
  if ( p )
  {
    MBG_PTP_NG_UC_MASTER *uc_master;
    MBG_PTP_NG_TSTAMPER *tstamper;
    MBG_PTP_NG_INSTC *instc;
    unsigned i;

    if ( p->timestampers )
    {
      for ( i = 0; i < p->glb_info.num_timestampers; ++i )
      {
        tstamper = &p->timestampers[i];

        if ( tstamper->release_priv )
          tstamper->release_priv( tstamper );
      }

      free( p->timestampers );
    }

    if ( p->instances )
    {
      for (i = 0; i < p->glb_info.settings.num_instances; ++i)
      {
        instc = &p->instances[i];

        if ( instc->uc_slave_list )
          free( instc->uc_slave_list );

        if ( instc->release_priv )
          instc->release_priv( instc );
      }

      free( p->instances );
    }

    if ( p->uc_masters )
    {
      for (i = 0; i < p->glb_info.settings.num_uc_masters; ++i)
      {
        uc_master = &p->uc_masters[i];

        if ( uc_master->release_priv )
          uc_master->release_priv( uc_master );
      }

      free( p->uc_masters );
    }

    free( p );
  }

}  // free_all_ptp_ng_info



/*HDR*/
/**
 * @brief Free a list of ::MBG_SYS_REF_SRC structures.
 *
 * @param[in]  p            Pointer to the ::MBG_SYS_REF_SRC list to be freed.
 * @param[in]  free_source  Address of a function to be called to free resources.
 *
 * @see ::mbgextio_get_all_sys_ref_info
 * @see ::free_all_sys_ref_info
 */
void free_all_sys_ref_srcs( struct mbg_klist_head *p,
                            void (*free_source)( MBG_SYS_REF_SRC * ) )
{
  struct mbg_klist_head *pos, *next;
  MBG_SYS_REF_SRC *src;

  if ( !p || mbg_klist_is_empty( p ) )
    return;

  mbg_klist_for_each_safe( p, pos, next )
  {
    src = mbg_container_of( pos, MBG_SYS_REF_SRC, head );

    mbg_klist_delete_item( &src->head );
    if ( src->release_priv )
      src->release_priv( src );
    if ( free_source )
      free_source( src );
    else
      free( src );
  }

}  // free_all_sys_ref_srcs



/*HDR*/
/**
 * @brief Free an ::ALL_SYS_REF_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_SYS_REF_INFO structure to be freed.
 *
 * @see ::mbgextio_get_all_sys_ref_info
 * @see ::free_all_sys_ref_srcs
 */
void free_all_sys_ref_info( ALL_SYS_REF_INFO *p )
{
  if ( !p )
    return;

  free_all_sys_ref_srcs( &p->sources, p->free_source );
  free( p );

}  // free_all_sys_ref_info



/*HDR*/
/**
 * @brief Free memory allocated by ::mbgextio_get_all_ims_info.
 *
 * @param[in]  p  Pointer to the ::ALL_IMS_INFO structure to be freed.
 *
 * @see ::mbgextio_dev_has_ims
 * @see ::mbgextio_get_all_ims_info
 * @see ::mbgextio_get_all_ims_state
 */
void free_all_ims_info( ALL_IMS_INFO *p )
{
  if ( p )
  {
    if ( p->fdm_info )
      free( p->fdm_info );

    if ( p->fdm_limits )
      free( p->fdm_limits );

    if ( p->fdm_outinfo_idx )
      free( p->fdm_outinfo_idx );

    free( p );
  }

}  // free_all_ims_info



/*HDR*/
/**
 * @brief Free memory allocated by ::mbgextio_get_all_ims_state.
 *
 * @param[in]  p  Pointer to the ::ALL_IMS_STATE structure to be freed.
 *
 * @see ::mbgextio_dev_has_ims
 * @see ::mbgextio_get_all_ims_info
 * @see ::mbgextio_get_all_ims_state
 */
void free_all_ims_state( ALL_IMS_STATE *p )
{
  if ( p )
  {
    if ( p->sensor_state_idx )
      free( p->sensor_state_idx );

    if ( p->fdm_state )
      free( p->fdm_state );

    if ( p->fdm_output_state_idx )
      free( p->fdm_output_state_idx );

    free( p );
  }

}  // free_all_ims_state



/*HDR*/
/**
 * @brief Free memory allocated by ::mbgextio_get_all_gpio_info.
 *
 * @param[in]  p  Pointer to the ::ALL_GPIO_INFO structure to be freed.
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_gpio
 * @see ::mbgextio_get_all_gpio_info
 */
void free_all_gpio_info( ALL_GPIO_INFO *p )
{
  if ( p )
  {
    if ( p->infos )
      free( p->infos );

    free( p );
  }

}  // free_all_gpio_info



/*HDR*/
/**
 * @brief Free memory allocated by ::mbgextio_get_all_io_ports.
 *
 * @param[in]  p  Pointer to the ::ALL_MBG_IO_PORTS structure to be freed.
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_io_ports
 * @see ::mbgextio_get_all_io_ports
 * @see ::mbgextio_get_all_io_port_status
 */
void free_all_io_ports( ALL_MBG_IO_PORTS *p )
{
  if ( p )
  {
    if ( p->ports )
    {
      unsigned i;
      MBG_IO_PORT *port;

      for ( i = 0; i < p->num_ports; ++i )
      {
        port = &p->ports[i];

        if ( port->free_priv_data )
          port->free_priv_data( port, port->priv_data );

        if ( port->pt_infos )
            free( port->pt_infos );
      }

      free( p->ports );
    }

    free( p );
  }

}  // free_all_io_ports



/*HDR*/
/**
 * @brief Free memory allocated by ::mbgextio_get_all_gpio_state.
 *
 * @param[in]  p  Pointer to the ::ALL_GPIO_STATE structure to be freed.
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_gpio
 * @see ::mbgextio_get_all_gpio_state
 */
void free_all_gpio_state( ALL_GPIO_STATE *p )
{
  if ( p )
  {
    if ( p->states )
      free( p->states );

    free( p );
  }

}  // free_all_gpio_state



/*HDR*/
/**
* @brief Free an ::ALL_SERIAL_PORTS structure.
*
* @param[in]  p  Pointer to the ::ALL_SERIAL_PORTS structure to be freed.
*/
void free_all_serial_ports( ALL_SERIAL_PORTS *p )
{
  if ( p )
  {
    if ( p->ports )
      free( p->ports );

    if ( p->str_types )
      free( p->str_types );

    free( p );
  }

} // free_all_serial_ports



/*HDR*/
/**
* @brief Free an ::ALL_PROG_OUTS structure.
*
* @param[in]  p  Pointer to the ::ALL_PROG_OUTS structure to be freed.
*/
void free_all_prog_outs( ALL_PROG_OUTS *p )
{
  if ( p )
  {
    if ( p->ports )
      free( p->ports );

    free( p );
  }

} // free_all_prog_outs



/*HDR*/
/**
* @brief Free an ::ALL_GPIOS structure.
*
* @param[in]  p  Pointer to the ::ALL_GPIOS structure to be freed.
*/
void free_all_gpios( ALL_GPIOS *p )
{
  if ( p )
  {
    if ( p->ports )
      free( p->ports );

    if ( p->states )
      free( p->states );

    free( p );
  }

} // free_all_gpios



/*HDR*/
/**
* @brief Free an ::ALL_TIMECODE structure.
*
* @param[in]  p  Pointer to the ::ALL_TIMECODE structure to be freed.
*/
void free_all_timecode( ALL_TIMECODE *p )
{
  if ( p )
  {
    if ( p->irig )
      free( p->irig );

    if ( p->ref_offset )
      free( p->ref_offset );

    if ( p->havequick )
      free( p->havequick );

    free( p );
  }

} // free_all_timecode



/*HDR*/
/**
* @brief Free an ::ALL_PORTS_INFO structure.
*
* @param[in]  p  Pointer to the ::ALL_PORTS_INFO structure to be freed.
*/
void free_all_ports_info( ALL_PORTS_INFO *p )
{
  if ( p )
  {
    if ( p->io_ports )
      free_all_io_ports( p->io_ports );

    if ( p->serial_ports )
      free_all_serial_ports( p->serial_ports );

    if ( p->prog_outs )
      free_all_prog_outs( p->prog_outs );

    if ( p->gpios )
      free_all_gpios( p->gpios );

    if ( p->timecode_rx )
      free_all_timecode( p->timecode_rx );

    if ( p->timecode_tx )
      free_all_timecode( p->timecode_tx );

    if ( p->synthesizer )
      free( p->synthesizer );

    if ( p->enable_flags )
      free( p->enable_flags );

    free( p );
  }

} // free_all_ports_info


/*HDR*/
/**
 * @brief Free an ::ALL_LED_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_LED_INFO structure to be freed.
 */
void free_all_led_info( ALL_LED_INFO *p )
{
  if ( p )
  {
    if ( p->leds )
      free( p->leds );

    free( p );
  }

} // free_all_led_info


/*HDR*/
/**
 * @brief Allocate memory for a new ::UCAP_ENTRY structure.
 *
 * @return The new allocated ::UCAP_ENTRY or NULL if the calloc call was not successful.
 */
UCAP_ENTRY *calloc_ucap_entry( void )
{
  UCAP_ENTRY *entry = (UCAP_ENTRY *) calloc( 1, sizeof( *entry ) );

  if ( entry )
    mbg_klist_init( &entry->head );

  return entry;

}  // calloc_ucap_entry



/*HDR*/
/**
 * @brief Free memory allocated by ::mbgextio_get_all_ucap_info.
 *
 * @param[in]  p  Pointer to the ::ALL_UCAP_INFO structure to freed.
 *
 * @return One of the @ref MBG_RETURN_CODES
 *
 * @see ::mbgextio_dev_has_ucap
 * @see ::mbg_chk_dev_has_ucap
 * @see ::mbgextio_get_all_ucap_info
 * @see ::mbg_get_all_ucap_info
 */
void free_all_ucap_info( ALL_UCAP_INFO *p )
{
  if ( p )
  {
    UCAP_ENTRY *entry;

    while ( !mbg_klist_is_empty( &p->list ) )
    {
      entry = mbg_klist_first_entry( &p->list, UCAP_ENTRY, head );
      mbg_klist_delete_item( &entry->head );
      free( entry );
    }

    free( p );
  }

}  // free_all_ucap_info



/*HDR*/
/**
 * @brief Free an ::ALL_UCAP_NET_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_UCAP_NET_INFO structure to be freed.
 *
 * @see ::mbgextio_get_all_ucap_net_info
 */
void free_all_ucap_net_info( ALL_UCAP_NET_INFO *p )
{
  if ( p )
  {
    if ( p->recv_infos )
      free( p->recv_infos );

    free( p );
  }

}  // free_all_ucap_net_info



/*HDR*/
/**
 * @brief Free an ::ALL_USER_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_USER_INFO structure to be freed.
 *
 * @see ::mbgextio_get_all_user_info
 */
void free_all_user_info( ALL_USER_INFO *p )
{
  if ( p )
  {
    unsigned i;
    MBG_USER *usr;

    if ( p->users )
    {
      for ( i = 0; i < p->user_mngmnt_info.settings.num_users; ++i )
      {
        usr = &p->users[i];

        if ( usr->release_priv )
          usr->release_priv( usr );

        if ( usr->release_backref )
          usr->release_backref( usr );
      }

      free( p->users );
    }

    if ( p->external_users )
    {
      for ( i = 0; i < p->user_mngmnt_info.settings.num_external; ++i )
      {
        usr = &p->external_users[i];

        if ( usr->release_priv )
          usr->release_priv( usr );

        if ( usr->release_backref )
          usr->release_backref( usr );
      }

      free( p->external_users );
    }

    if ( p->user_levels )
    {
      for ( i = 0; i < p->user_mngmnt_info.settings.num_levels; ++i )
      {
        usr = &p->user_levels[i];

        if ( usr->release_priv )
          usr->release_priv( usr );

        if ( usr->release_backref )
          usr->release_backref( usr );
      }

      free( p->user_levels );
    }

    if(p->pw_rules) {
        free(p->pw_rules);
    }

    free( p );
  }

}  // free_all_user_info



/*HDR*/
/**
 * @brief Free an ::ALL_SERVICE_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_SERVICE_INFO structure to be freed.
 *
 * @see ::mbgextio_get_all_svc_info
 */
void free_all_svc_info( ALL_SERVICE_INFO *p )
{
  if ( p )
  {
    if ( p->svcs )
    {
      unsigned i;
      MBG_SERVICE *svc;

      for ( i = 0; i < p->mgmt_info.num_services; ++i )
      {
          svc = &p->svcs[i];

        if ( svc->release_priv )
          svc->release_priv( svc );

        if ( svc->release_backref )
          svc->release_backref( svc );
      }

      free( p->svcs );
    }

    free( p );
  }

}  // free_all_svc_info



/*HDR*/
/**
 * @brief Free an ::ALL_FIRMWARE_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_FIRMWARE_INFO structure to be freed.
 *
 * @see ::mbgextio_get_all_firmware_info
 */
void free_all_firmware_info( ALL_FIRMWARE_INFO *p )
{
  if ( p )
  {
    if ( p->firmwares )
    {
      MBG_FIRMWARE_UFU *ufu;
      MBG_FIRMWARE *fw;
      unsigned i, j;

      for ( i = 0; i < p->glb_info.installed_fws; ++i )
      {
        fw = &p->firmwares[i];

        for ( j = 0; j < fw->info.num_ufus; ++j )
        {
          ufu = &fw->ufus[j];

          if ( ufu->release_priv )
            ufu->release_priv( ufu );
        }
        free( fw->ufus );

        if ( fw->release_priv )
          fw->release_priv( fw );
      }

      free( p->firmwares );
    }

    free( p );
  }

}  // free_all_firmware_info



/*HDR*/
/**
 * @brief Free an ::ALL_DATABASE_INFO structure.
 *
 * @param[in]  p  Pointer to the ::ALL_DATABASE_INFO structure to be freed.
 *
 * @see ::mbgextio_get_all_database_info
 */
void free_all_database_info( ALL_DATABASE_INFO *p )
{
  if ( !p )
    return;

  if ( p->dbs )
  {
    uint8_t i;
    MBG_DATABASE *db;

    for ( i = 0; i < p->glb_info.num_dbs; ++i )
    {
        db = &p->dbs[i];

        if ( db->release_priv )
            db->release_priv( db );
    }

    free( p->dbs );
  }

  free( p );

}  // free_all_database_info



/*HDR*/
/**
 * @brief Set up a ::NTP_TSTAMP structure from a hex string with a time in seconds and binary fractions.
 *
 * @param[in]   s  A string with a time in seconds since NTP epoch 1900-01-01,
 *                 with binary fractions separated by decimal point,
 *                 e.g. 'dc763e43.73bd5a8f' as printed by the ntpq utility.
 * @param[out]  p  Address of a ::NTP_TSTAMP structure to be set up.
 *
 * @see ::str_ntp_hex_to_nano_time_64
 */
void str_ntp_hex_to_ntp_tstamp( const char *s, NTP_TSTAMP *p )
{
  char *cp = NULL;

  p->seconds = strtoul( s, &cp, 16 );

  if ( *cp == '.' )  // fractions may follow
  {
    cp++;  // skip '.'

    p->fractions = strtoul( cp, NULL, 16 );
  }
  else
    p->fractions = 0;

}  // str_ntp_hex_to_ntp_tstamp



#if !defined( MBG_TGT_MISSING_64_BIT_TYPES )

/*HDR*/
/**
 * @brief Convert a ::NTP_TSTAMP structure to a ::NANO_TIME_64 structure.
 *
 * @param[in]   p_nts   The ::NTP_TSTAMP structure to be converted.
 * @param[out]  p_nt64  The ::NANO_TIME_64 structure to be filled up.
 */
void ntp_tstamp_to_nanotime_64( const NTP_TSTAMP *p_nts, NANO_TIME_64 *p_nt64 )
{
  p_nt64->secs = p_nts->seconds - NTP_SEC_BIAS;
  p_nt64->nano_secs = bin_frac_32_to_dec_frac( p_nts->fractions, NSEC_PER_SEC );

}  // ntp_tstamp_to_nanotime_64



/*HDR*/
/**
 * @brief Set up a ::NANO_TIME_64 structure from a hex string with a time in seconds and binary fractions.
 *
 * @param[in]   s  A string with a time in seconds since epoch 1900-01-01,
 *                 with binary fractions separated by decimal point,
 *                 e.g. 'dc763e43.73bd5a8f' as printed by the ntpq utility.
 * @param[out]  p  Address of a ::NANO_TIME_64 structure to be set up.
 *
 * @see ::str_ntp_hex_to_ntp_tstamp
 */
void str_ntp_hex_to_nano_time_64( const char *s, NANO_TIME_64 *p )
{
  NTP_TSTAMP nts = { 0 };

  str_ntp_hex_to_ntp_tstamp( s, &nts );
  ntp_tstamp_to_nanotime_64( &nts, p );

}  // str_ntp_hex_to_nano_time_64

#endif  // !defined( MBG_TGT_MISSING_64_BIT_TYPES )



/*HDR*/
/**
 * @brief Convert an array of bytes to uint64_t.
 *
 * @param[in]  byte_array  Address of the byte array to be converted.
 * @param[in]  num_bytes   The number of bytes in @p byte_array.
 *
 * @return  The converted number.
 */
uint64_t cnvrt_bytes_to_long( const uint8_t byte_array[], int num_bytes )
{
  int i;
  uint64_t val = 0;

  for ( i = 0; i < num_bytes; i++ )
    val = (val << 8) | byte_array[i];

  return val;

}  // cnvrt_bytes_to_long



/*HDR*/
/**
 * @brief Convert an uint64_t to an array of bytes.
 *
 * @param[out] byte_array  Address of the byte array to be filled.
 * @param[in]  num_bytes   The number of bytes in @p byte_array.
 * @param[in]  val         The value to be converted.
 */
void cnvrt_long_to_bytes( uint8_t byte_array[], int num_bytes, uint64_t val )
{
  int i;

  for ( i = num_bytes - 1; i >= 0; i-- )
  {
    byte_array[i] = (uint8_t) val;  // masked by 0xFF anyway
    val >>= 8;
  }

  return;

}  // cnvrt_long_to_bytes



/*HDR*/
int nw_str_mac_addr_to_str( MBG_MAC_ADDR *mac_addr, char *buf, size_t buflen )
{
  int len = 0;

  if ( !mac_addr || !buf )
    return len;

  len += snprintf_safe( &buf[len], buflen - len, "%02X:%02X:%02X:%02X:%02X:%02X",
                        mac_addr->b[0], mac_addr->b[1], mac_addr->b[2], mac_addr->b[3], mac_addr->b[4], mac_addr->b[5] );

  return len;

}  // nw_str_mac_addr_to_str



/*HDR*/
int nw_str_to_mac_addr( MBG_MAC_ADDR *mac_addr, const char *str )
{
  int a, b, c, d, e, f;
  memset( mac_addr, 0, sizeof( *mac_addr ) );

  if ( (sscanf(str, "%02x:%02x:%02x:%02x:%02x:%02x", &a, &b, &c, &d, &e, &f) == 6 ) ||
     ( sscanf(str, "%02x-%02x-%02x-%02x-%02x-%02x", &a, &b, &c, &d, &e, &f) == 6 ) )
  {
    mac_addr->b[0] = a;
    mac_addr->b[1] = b;
    mac_addr->b[2] = c;
    mac_addr->b[3] = d;
    mac_addr->b[4] = e;
    mac_addr->b[5] = f;
    return MBG_SUCCESS;
  }

  return MBG_ERR_GENERIC;

}  // nw_str_to_mac_addr



/*HDR*/
int nw_ip6_addr_bytes_to_str( const uint8_t *addr, uint8_t protocol, char *str, size_t buflen )
{
  switch(protocol)
  {
    case PTP_NW_PROT_UDP_IPV4 :
      snprint_ip4_addr( str, buflen, (IP4_ADDR *) addr, NULL ); break;

    case PTP_NW_PROT_UDP_IPV6 :
      snprint_ip6_addr( str, buflen, (IP6_ADDR *) addr, NULL ); break;

    case PTP_NW_PROT_IEEE_802_3 :
      nw_str_mac_addr_to_str( (MBG_MAC_ADDR*) addr, str, buflen ); break;

    default :
      return MBG_ERR_INV_PARM;
  }

  return MBG_SUCCESS;

}  // nw_ip6_addr_bytes_to_str



/*HDR*/
int nw_str_to_ip6_addr_bytes( uint8_t *addr, const uint8_t protocol, char *str, size_t buflen )
{
  int rc = 0;

  switch ( protocol )
  {
    case PTP_NW_PROT_UDP_IPV4 :
      rc = str_to_ip4_addr( (IP4_ADDR *) addr, str ); break;

    case PTP_NW_PROT_UDP_IPV6 :
      rc = str_to_ip6_addr( (IP6_ADDR *) addr, str ); break;

    case PTP_NW_PROT_IEEE_802_3 :
      rc = nw_str_mac_addr_to_str( (MBG_MAC_ADDR *) addr, str, buflen ); break;

    default :
      return MBG_ERR_INV_PARM;
  }

  return rc;

}  // nw_str_to_ip6_addr_bytes


/*HDR*/
/**
 * @brief Convert ::SW_REV::code (uint16_t) to extended revision (uint32_t).
 *
 * The extended revision code is used e.g. in ::MBG_EXT_SYS_INFO::sw_rev
 * or ::MBG_FW_UFU_INFO::revision.
 *
 * @param[in]  sw_rev  Address of a ::SW_REV structure.
 *
 * @return
 */
uint32_t mbg_sw_rev_to_u32_revision( const SW_REV *sw_rev )
{
  uint32_t revision;
  int major;
  int minor;
  char buf[6];

  // TODO OMG, this can be greatly simplified.
  // TODO @author of the above comment: OMG, please tell us how!
  snprintf_safe( buf, sizeof( buf ), "%X.%02X", ( sw_rev->code >> 8 ) & 0xff,
                 sw_rev->code & 0xff );
  sscanf(buf, "%d.%d", &major, &minor);

  revision = _mbg_encode_revision( major, minor, 0 );

  return revision;

}  // mbg_sw_rev_to_u32_revision


/*HDR*/
int mbg_os_version_to_str( const MBG_EXT_SYS_INFO *info, char *buf, size_t buflen, int long_version )
{
  ssize_t bytes;
  const char* os_strs[] = MBG_EXT_SYS_INFO_OS_SHORT_STRS;

  if ( !( info->supp_members & MBG_EXT_SYS_INFO_MSK_OS_TYPE ) )
    return MBG_ERR_NOT_SUPP_BY_DEV;

  if ( info->mbg_os_type >= N_MBG_EXT_SYS_INFO_OS_TYPES )
    return MBG_ERR_INV_PARM;

  bytes = 0;

  if ( ( info->supp_members & MBG_EXT_SYS_INFO_MSK_OS_NAME ) && long_version )
    bytes += snprintf_safe(&buf[bytes], buflen - bytes, "%s ", info->os_name);

  if ( info->supp_members & MBG_EXT_SYS_INFO_MSK_SW_REV )
  {
    unsigned year, month, patch;

    _mbg_decode_revision( info->sw_rev, year, month, patch );
    year += MBG_OS_YEAR_CONSTANT;

    if ( ( info->supp_members & MBG_EXT_SYS_INFO_MSK_RELEASE_CANDIDATE ) &&
           info->release_candidate )
    {
      if ( info->release_candidate != MBG_REVISION_RC_DEVEL )
        bytes += snprintf_safe(&buf[bytes], buflen - bytes,
                               "%04u.%02u.%u-rc%i-",
                               year, month, patch, info->release_candidate);
      else
        bytes += snprintf_safe(&buf[bytes], buflen - bytes,
                               "%04u.%02u.%u-%s-",
                               year, month, patch, MBG_REVISION_RC_DEVEL_STR);
    }
    else
    {
      bytes += snprintf_safe(&buf[bytes], buflen - bytes,
                             "%04u.%02u.%u-",
                             year, month, patch);
    }
  }

  bytes += snprintf_safe(&buf[bytes], buflen - bytes,
                         "%s ",
                         os_strs[info->mbg_os_type]);


  if ( ( info->supp_members & MBG_EXT_SYS_INFO_MSK_COMMIT_HASH ) && long_version )
    bytes += snprintf_safe(&buf[bytes], buflen - bytes,
                           "%08x ", info->commit_hash);

  if ( bytes >= (ssize_t)buflen )
    return MBG_ERR_OVERFLOW;

  buf[--bytes] = 0;

  return MBG_SUCCESS;

}  // mbg_os_version_to_str



/*HDR*/
int mbg_fw_can_set_osv(const MBG_FW_GLB_INFO *info)
{
  if ( info && ( info->flags & MBG_FW_GLB_MSK_CAN_SET_OSV ) )
    return MBG_SUCCESS;

  return MBG_ERR_NOT_SUPP_BY_DEV;

}  // mbg_fw_can_set_osv


/*HDR*/
/**
 * @brief Convert line and horizontal delay into phase offset.
 *
 * @param[in]  line    Number of lines delay.
 * @param[in]  hdly    Horizonal (sub line) delay in seconds.
 * @param[in]  t_line  Seconds per line, see ::GPIO_VIDEO_FORMAT_INFO.
 *
 * @return  The phase offset in seconds
 *
 * @see ::phase_to_line
 * @see ::phase_to_hdly
 */
double dly_to_phase( int line, double hdly, double t_line )
{
  return ( ( line * t_line ) + hdly );
}


/*HDR*/
/**
 * @brief Convert phase offset to vertical line offset.
 *
 * Complementary function to ::dly_to_phase.
 *
 * @param[in]  phase   Delay in seconds.
 * @param[in]  t_line  Seconds per line, see ::GPIO_VIDEO_FORMAT_INFO.
 *
 * @return  The vertical line offset
 *
 * @see ::dly_to_phase
 * @see ::phase_to_hdly
 */
int phase_to_line( double phase, double t_line )
{
  return ( (int) ( phase / t_line ) );
}


/*HDR*/
/**
 * @brief Convert phase offset to horizontal delay.
 *
 * Complementary function to dly_to_phase.
 *
 * @param[in]  phase   Delay in seconds.
 * @param[in]  t_line  Seconds per line, see ::GPIO_VIDEO_FORMAT_INFO
 *
 * @return  The horizonal offset in nano seconds.
 *
 * @see ::dly_to_phase
 * @see ::phase_to_line
 */
double phase_to_hdly( double phase, double t_line )
{
  return ( t_line * ( ( phase / t_line ) - phase_to_line( phase, t_line ) ) ) * (double) NSEC_PER_SEC;
}


/*HDR*/
/**
 * @brief Initialize a ::MBG_SYS_REF_DEV_STATE structure
 *
 * @param[out] devst The ::MBG_SYS_REF_DEV_STATE structure to be initialized.
 */
void mbg_sys_ref_dev_state_init(MBG_SYS_REF_DEV_STATE *devst)
{
    memset(devst, 0, sizeof(*devst));
    devst->meta.version = MBG_SYS_REF_DEV_STATE_CURRENT_VERSION;
}


/*HDR*/
/**
 * @brief Set the global status of a ::MBG_SYS_REF_DEV_STATE structure.
 *
 * @param[out] devst The structure to be updated.
 * @param[in]  glbst The new global status data.
 */
void mbg_sys_ref_dev_state_set_glb_status(MBG_SYS_REF_DEV_STATE *devst,
                                          const MBG_SYS_REF_GLB_STATUS *glbst)
{
    devst->glb_status = *glbst;
    devst->supp_members |= MBG_SYS_REF_DEV_STATE_FLAG_GLB_STATUS;
}


/*HDR*/
/**
 * @brief Set the TZDL of a ::MBG_SYS_REF_DEV_STATE structure.
 *
 * @param[out] devst The structure to be updated.
 * @param[in]  tzdl  The new TZDL data.
 */
void mbg_sys_ref_dev_state_set_tzdl(MBG_SYS_REF_DEV_STATE *devst,
                                    const TZDL *tzdl)
{
    devst->tzdl = *tzdl;
    devst->supp_members |= MBG_SYS_REF_DEV_STATE_FLAG_TZDL;
}


/*HDR*/
/**
 * @brief Set the telecom quality of a ::MBG_SYS_REF_DEV_STATE structure.
 *
 * @param[out] devst The structure to be updated.
 * @param[in]  ssm   The new ssm output value.
 * @param[in]  boc   The new boc output value.
 */
void mbg_sys_ref_dev_state_set_telecom_qlt(MBG_SYS_REF_DEV_STATE *devst,
                                           uint16_t ssm, uint16_t boc)
{
    devst->glb_ssm_output = ssm;
    devst->glb_boc_output = boc;
    devst->supp_members |= MBG_SYS_REF_DEV_STATE_FLAGS_TELECOM_QLT;
}


/*HDR*/
/**
 * @brief Set the PTP V2 datasets of a ::MBG_SYS_REF_DEV_STATE structure.
 *
 * @param[out] devst   The structure to be updated.
 * @param[in]  parent  The new parent dataset.
 * @param[in]  current The new current dataset.
 * @param[in]  tm_prop The new time properties dataset.
 */
void mbg_sys_ref_dev_state_set_ptp_ds(MBG_SYS_REF_DEV_STATE *devst,
                                      const MBG_PTP_V2_NG_PARENT_DATASET *parent,
                                      const MBG_PTP_V2_NG_CURRENT_DATASET *current,
                                      const MBG_PTP_V2_NG_TIME_PROPERTIES_DATASET *tm_prop)
{
    if (parent) {
        devst->ptp_parent_ds = *parent;
        devst->supp_members |= MBG_SYS_REF_DEV_STATE_FLAG_PTP_PARENT_DS;
    }

    if (current) {
        devst->ptp_current_ds = *current;
        devst->supp_members |= MBG_SYS_REF_DEV_STATE_FLAG_PTP_CUR_DS;
    }

    if (tm_prop) {
        devst->ptp_tm_prop_ds = *tm_prop;
        devst->supp_members |= MBG_SYS_REF_DEV_STATE_FLAG_PTP_TM_PROP_DS;
    }
}


/*HDR*/
const char *ptp_clock_class_to_str( int clock_class )
{
    static const MBG_CODE_NAME_TABLE_ENTRY tbl[] = PTP_CLOCK_CLASS_STR_TBL;

    const MBG_CODE_NAME_TABLE_ENTRY *tbl_entry;

    for ( tbl_entry = tbl; tbl_entry->name; tbl_entry++ )
    {
        if ( clock_class == tbl_entry->code )
            return tbl_entry->name;
    }

    return "Unknown Clock Class";
}
