#if !defined( _PCPSDRVR_H )
#define _PCPSDRVR_H

#include "compat.h"

// #include <linux/init.h>
// #include <linux/module.h>
// #include <linux/pci.h>

#include <linux/kernel.h>
#include <linux/spinlock.h>
#include <linux/ioport.h>
#include <linux/param.h>    // HZ
#include <linux/jiffies.h>  // timeafter()
#include <linux/delay.h>    // msleep


#define DEBUG_IO_PROBE       1

#define DEBUG_ACCESS_TIMING  0
#define DEBUG_IO_TIMING      0
#define DEBUG_IO             0


#define PCPS_ST_BUSY       0x01  ///< The device is busy filling the output buffer.

#define PCPS_FIFO_SZ       16
#define PCPS_FW_ID_SZ      ( 2 * PCPS_FIFO_SZ + 1 )
#define PCPS_GIVE_FW_ID_1  0x40  ///< (r-) Read first ::PCPS_FIFO_SIZE chars of firmware ID.
#define PCPS_GIVE_FW_ID_2  0x41  ///< (r-) Read last ::PCPS_FIFO_SIZE chars of firmware ID.


#define MBG_SUCCESS  0
#define MBG_ERR_TIMEOUT     -20  ///< Timeout accessing the device.

#define mbg_rc_is_error( _rc )  ( _rc < 0 )


#define __mbg_inline __inline__ __attribute__((always_inline))


enum MBG_LOG_LEVELS
{
  MBG_LOG_ERR,
  MBG_LOG_WARN,
  MBG_LOG_NOTICE,
  MBG_LOG_INFO,
  MBG_LOG_DEBUG,
  N_MBG_LOG_LVL
};


#define _pcps_time_after( _curr, _tmo ) \
  time_after( (unsigned long) _curr, (unsigned long) _tmo )


#define _MBG_IOMEM  __iomem
#define FAR


#define _pcps_irq_flags \
  unsigned long irq_flags;

#define _pcps_disb_local_irq_save() \
  local_irq_save( irq_flags )

#define _pcps_local_irq_restore() \
  local_irq_restore( irq_flags )

#define _pcps_ddev_timeout_clk( pddev ) \
  ( (ulong)( 200 * HZ ) / 1000 )


#define uint8_t   u8
#define uint16_t  u16
#define uint32_t  u32



/**
 * @brief Set of PCI ASIC registers which are writeable once after power-up.
 **/
typedef struct
{
  uint32_t cfg_class_rev_id;
  uint16_t cfg_badr_0;
  uint16_t cfg_dev_id;

} PCI_ASIC_CFG;


/**
 * @brief A PCI ASIC register as 32, 16, or 8 bit accessible union.
 */
typedef union
{
  uint32_t ul;
  uint16_t us[2];
  uint8_t b[4];

} PCI_ASIC_REG;



/**
 * @brief A data type to hold the PCI ASIC version code.
 */
typedef uint32_t PCI_ASIC_VERSION;



/**
 * @brief A data type to hold the PCI ASIC feature flags mask.
 *
 * @see @ref PCI_ASIC_FEATURE_MASKS
 */
typedef uint32_t PCI_ASIC_FEATURES;



/**
 * @brief The addon-data part of a PCI ASIC.
 */
typedef union
{
  uint32_t ul[4];
  uint16_t us[8];
  uint8_t b[16];

} PCI_ASIC_ADDON_DATA;



/**
 * @brief Register layout of a PCI ASIC.
 */
typedef struct
{
  PCI_ASIC_CFG cfg;                ///< Registers that are writeable from add-on once after power-up.
  PCI_ASIC_VERSION raw_version;    ///< Raw version code.
  PCI_ASIC_FEATURES features;      ///< PCI ASIC feature mask, see @ref PCI_ASIC_FEATURE_MASKS.
  PCI_ASIC_REG status_port;        ///< The status port register.
  PCI_ASIC_REG control_status;     ///< See @ref PCI_ASIC_CONTROL_STATUS_MASKS.
  PCI_ASIC_REG pci_data;           ///< Register used to pass byte from PCI bus to add-on side.
  PCI_ASIC_REG reserved_1;         ///< Currently not implemented / used.

  PCI_ASIC_ADDON_DATA addon_data;  ///< Register set used to return data from add-on to PCI bus.
  PCI_ASIC_ADDON_DATA reserved_2;  ///< Currently not implemented / used.

} PCI_ASIC;



typedef struct
{
  int bar_mask;
  int mem_bar_idx;
  unsigned long mmio_start;
  unsigned long mmio_len;
  void _MBG_IOMEM *mapped_mem;    // Pointer for I/O operations.

  uint16_t status_port_offs;

  /// Base address of the ASIC part of the register block
  /// if it has been mapped to memory, else @a NULL.
  PCI_ASIC _MBG_IOMEM *mm_asic_addr;
  PCI_ASIC_VERSION raw_asic_version;   ///< Raw ASIC version.
  PCI_ASIC_FEATURES asic_features;     ///< ASIC feature mask.

  char fw_id[PCPS_FW_ID_SZ];

} PCPS_DDEV;



void pcps_read_asic_version( PCPS_DDEV *pddev );

void pcps_read_asic_features( PCPS_DDEV *pddev );

int pcps_read_asic_mm( PCPS_DDEV *pddev, uint8_t cmd,
                       void FAR *buffer, uint16_t count );

int pcps_probe_device( PCPS_DDEV *pddev );

#endif  // !defined( _PCPSDRVR_H )
