
#include "pcpsdrvr.h"

// #include <asm/uaccess.h>
#include <asm/io.h>


#define USE_CMD_PTR      1


// Definitions for clock() support in kernel drivers.
extern unsigned long volatile jiffies;
#define clock() jiffies


#define mbg_get_pc_cycles( _p ) \
  do {} while ( 0 )


int msg_sleep;



static __mbg_inline /*HDR*/
__attribute__( ( format( printf, 3, 0 ) ) )
int mbg_kdd_vsnprintf( char *s, size_t max_len, const char *fmt, va_list args )
{
  return vscnprintf( s, max_len, fmt, args );

}  // mbg_kdd_vsnprintf



/*HDR*/
__attribute__( ( format( printf, 2, 3 ) ) )
void mbg_kdd_msg( int lvl, const char *fmt, ... )
{
  va_list args;
  char s[256];
  const char *cp;

  va_start( args, fmt );
  mbg_kdd_vsnprintf( s, sizeof( s ), fmt, args );
  va_end( args );

  switch ( lvl )  // See linux/kern_levels.h.
  {
    case MBG_LOG_ERR:
      cp = KERN_ERR;
      break;

    case MBG_LOG_WARN:
      cp = KERN_WARNING;
      break;

    case MBG_LOG_INFO:
      cp = KERN_INFO;
      break;

    case MBG_LOG_DEBUG:
      cp = KERN_DEBUG;
      break;

    default:
      #if defined( KERN_DEFAULT )
        cp = KERN_DEFAULT;
      #else
        cp = KERN_WARNING;
      #endif
      break;
  }

  printk( "%s" MBG_DRVR_NAME ": %s\n", cp, s );

  if ( msg_sleep )
    msleep( msg_sleep );

}  // mbg_kdd_msg



static /*HDR*/
void pcps_dump_data( const void *buffer, size_t count, const char *info )
{
  const uint8_t *p = (const uint8_t *) buffer;
  size_t i = 0;

  while ( i < count )
  {
    char ws[80];
    const char *cp = (const char *) p;
    int j;
    int n = 0;
    ws[0] = 0;

    for ( j = 0; j < 4; )
    {
      n += snprintf( &ws[n], sizeof( ws ) - n, " %02X", *p++ );

      j++;

      if ( ++i >= count )
        break;
    }

    n += snprintf( &ws[n], sizeof( ws ) - n, "  \"" );

    while ( j > 0 )
    {
      n += snprintf( &ws[n], sizeof( ws ) - n, "%c",
             ( ( *cp >= ' ' ) && ( *cp < 0x7F ) ) ? *cp : '#' );
      cp++;
      j--;
    }

    n += snprintf( &ws[n], sizeof( ws ) - n, "\"" );

    mbg_kdd_msg( MBG_LOG_INFO,"%s:%s", info, ws );
  }

}  // pcps_dump_data



/*HDR*/
void pcps_dump_asic_info( void )
{
  mbg_kdd_msg( MBG_LOG_INFO, "PCI_ASIC size:    0x%04lx", (long) sizeof( PCI_ASIC ) );
  mbg_kdd_msg( MBG_LOG_INFO, "  cfg:            0x%04lx", (long) offsetof( PCI_ASIC, cfg ) );
  mbg_kdd_msg( MBG_LOG_INFO, "  raw_version:    0x%04lx", (long) offsetof( PCI_ASIC, raw_version ) );
  mbg_kdd_msg( MBG_LOG_INFO, "  features:       0x%04lx", (long) offsetof( PCI_ASIC, features ) );
  mbg_kdd_msg( MBG_LOG_INFO, "  status_port:    0x%04lx", (long) offsetof( PCI_ASIC, status_port ) );
  mbg_kdd_msg( MBG_LOG_INFO, "  control_status: 0x%04lx", (long) offsetof( PCI_ASIC, control_status ) );
  mbg_kdd_msg( MBG_LOG_INFO, "  pci_data:       0x%04lx", (long) offsetof( PCI_ASIC, pci_data ) );
  mbg_kdd_msg( MBG_LOG_INFO, "  reserved_1:     0x%04lx", (long) offsetof( PCI_ASIC, reserved_1 ) );
  mbg_kdd_msg( MBG_LOG_INFO, "  addon_data:     0x%04lx", (long) offsetof( PCI_ASIC, addon_data ) );
  mbg_kdd_msg( MBG_LOG_INFO, "  reserved_2:     0x%04lx", (long) offsetof( PCI_ASIC, reserved_2 ) );

}  // pcps_dump_asic_info



/*HDR*/
void pcps_dump_regs_32( PCPS_DDEV *pddev )
{
  uint32_t _MBG_IOMEM *p = (uint32_t _MBG_IOMEM *) pddev->mm_asic_addr;
  int i;

  for ( i = 0; i < 8; i++ )
  {
    uint32_t u32 = ioread32( p );

    mbg_kdd_msg( MBG_LOG_INFO,"reg val at %px: 0x%08X", p, u32 );

    p++;
  }

}  // pcps_dump_regs_32



/*HDR*/
void pcps_dump_regs_16( PCPS_DDEV *pddev )
{
  uint16_t _MBG_IOMEM *p = (uint16_t _MBG_IOMEM *) pddev->mm_asic_addr;
  int i;

  for ( i = 0; i < 16; i++ )
  {
    uint16_t u16 = ioread16( p );

    mbg_kdd_msg( MBG_LOG_INFO,"reg val at %px: 0x%04X", p, u16 );

    p++;
  }

}  // pcps_dump_regs_16



/*HDR*/
void pcps_read_asic_version( PCPS_DDEV *pddev )
{
  PCI_ASIC_VERSION _MBG_IOMEM *p = &pddev->mm_asic_addr->raw_version;

  pddev->raw_asic_version = ioread32( p );

  mbg_kdd_msg( MBG_LOG_INFO,"ASIC version  0x%04X read from 0x%px",
               pddev->raw_asic_version, p );

}  // pcps_read_asic_version



/*HDR*/
void pcps_read_asic_features( PCPS_DDEV *pddev )
{
  PCI_ASIC_FEATURES _MBG_IOMEM *p = &pddev->mm_asic_addr->features;
  pddev->asic_features = ioread32( p );

  mbg_kdd_msg( MBG_LOG_INFO,"ASIC features 0x%04X read from 0x%px",
               pddev->asic_features, p );

}  // pcps_read_asic_features



static __mbg_inline /*HDR*/
uint8_t pcps_ddev_read_status_port_mm_dbg( const PCPS_DDEV *pddev )
{
  #if 0
    uint8_t status = _mbg_mmrd8_offs( pddev, 0, pddev->status_port_offs );
  #elif 0
    uint8_t status = ioread8( (uint8_t _MBG_IOMEM *) pddev->mm_asic_addr + pddev->status_port_offs );
  #else
    uint32_t status = ioread32( (uint8_t _MBG_IOMEM *) pddev->mm_asic_addr + pddev->status_port_offs );
  #endif

  return (uint8_t) status;

}  // pcps_ddev_read_status_port_mm_dbg



/*HDR*/
void pcps_test_status_port( const PCPS_DDEV *pddev )
{
  uint8_t _MBG_IOMEM *p = (uint8_t _MBG_IOMEM *) pddev->mm_asic_addr + pddev->status_port_offs;
  uint32_t st32;
  uint8_t st8;

  mbg_kdd_msg( MBG_LOG_INFO,"Going to read 32 bit status at 0x%px", p );

  st32 = ioread32( p );

  mbg_kdd_msg( MBG_LOG_INFO,"  32 bit status: 0x%08x", st32 );

  // Caution: This may cause a trap on ARM64!
  mbg_kdd_msg( MBG_LOG_INFO,"Going to read 8 bit status at 0x%px", p );

  st8 = ioread8( p );

  mbg_kdd_msg( MBG_LOG_INFO,"  8 bit status: 0x%02x", st8 );

}  // pcps_test_status_port



static __mbg_inline /*HDR*/
int pcps_wait_busy_mm_dbg( PCPS_DDEV *pddev, uint8_t cmd )
{
  int rc = MBG_SUCCESS;
  clock_t curr_clk = 0;
  clock_t start_clk = 0;
  clock_t timeout_clk = 0;

  uint8_t status = pcps_ddev_read_status_port_mm_dbg( pddev );

  #if DEBUG_IO_PROBE
    mbg_kdd_msg( MBG_LOG_INFO,"  wait_busy: %i, base 0x%px, offs 0x%x",
                 ( status & PCPS_ST_BUSY ) != 0,
                 pddev->mm_asic_addr, pddev->status_port_offs );
  #endif

  if ( status & PCPS_ST_BUSY )
  {
    curr_clk = clock();
    start_clk = curr_clk;
    timeout_clk = start_clk + _pcps_ddev_timeout_clk( pddev );

    for (;;)
    {
      status = pcps_ddev_read_status_port_mm_dbg( pddev );

      if ( ! ( status & PCPS_ST_BUSY ) )
        break;

      curr_clk = clock();

      if ( _pcps_time_after( curr_clk, timeout_clk ) )
      {
        rc = MBG_ERR_TIMEOUT;
        goto out;
      }
    }
  }

out:
  #if DEBUG_IO_PROBE
    mbg_kdd_msg( MBG_LOG_INFO,"  wait_busy: %i, start %li, tmo %li, curr %li%s",
                 ( status & PCPS_ST_BUSY ) != 0,
                 start_clk, timeout_clk, curr_clk,
                 ( rc == MBG_ERR_TIMEOUT ) ? ": timeout! #####" : "" );
  #endif

  return rc;

}  // pcps_wait_busy_mm_dbg



/*HDR*/
/**
 * @brief Read function for devices with Meinberg PCI interface ASIC.
 *
 * Unlike ::pcps_read_asic and most of the other read functions,
 * this function accesses the memory mapped registers rather than
 * the I/O ports, so this is significantly faster.
 *
 * @param[in]  pddev   Pointer to the device structure.
 * @param[in]  cmd     The command code for the device, see @ref PCPS_CMD_CODES.
 * @param[out] buffer  A buffer to take the bytes to be read.
 * @param[in]  count   The number of bytes to be read into the buffer.
 *
 * @return ::MBG_SUCCESS on success, else one of the @ref MBG_ERROR_CODES.
 *
 * @ingroup pcps_read_fncs
 * @see ::pcps_read_asic
 * @see @ref pcps_read_fncs
 */
int pcps_read_asic_mm( PCPS_DDEV *pddev, uint8_t cmd,
                       void FAR *buffer, uint16_t count )
{
  uint8_t FAR *p = (uint8_t FAR *) buffer;
  uint32_t _MBG_IOMEM *p_data_reg;
  PCI_ASIC_REG ar;
  int i;
  int rc = -1;
  int dt_quot;
  int dt_rem;
  _pcps_irq_flags

  #if DEBUG_ACCESS_TIMING || DEBUG_IO_TIMING
    MBG_PC_CYCLES t_after_cmd = 0;
  #endif

  #if DEBUG_ACCESS_TIMING
    MBG_PC_CYCLES t_after_reread = 0;
  #endif

  #if DEBUG_IO_TIMING
    MBG_PC_CYCLES t_after_busy = 0;
    MBG_PC_CYCLES t_done = 0;
  #endif

  #if 0 // DEBUG_IO_PROBE
    report_io_cmd_addr( cmd, count, buffer, __func__ );
  #endif

  #if defined( MBG_ARCH_SPARC )
    // Actually there are problems on SPARC ...
    return MBG_ERR_TIMEOUT;  // FIXME TODO
  #endif

  _pcps_disb_local_irq_save();

  // Get current cycles and write the command byte.
  #if USE_CMD_PTR
  {
    uint32_t _MBG_IOMEM *p_cmd_reg = &pddev->mm_asic_addr->pci_data.ul;
    mbg_get_pc_cycles( &pddev->acc_cycles );
    iowrite32( cmd, p_cmd_reg );
  }
  #else
    mbg_get_pc_cycles( &pddev->acc_cycles );
    _mbg_mmwr32( &pddev->mm_asic_addr->pci_data.ul, cmd );
  #endif

  #if DEBUG_ACCESS_TIMING || DEBUG_IO_TIMING
    mbg_get_pc_cycles( &t_after_cmd );
  #endif

  #if DEBUG_ACCESS_TIMING
    debug_dummy_var = ioread32( &pddev->mm_asic_addr->pci_data.ul );
    mbg_get_pc_cycles( &t_after_reread );
  #endif

  _pcps_local_irq_restore();


  p_data_reg = &pddev->mm_asic_addr->addon_data.ul[0];
  dt_quot = count / 4;
  dt_rem = count % 4;


  // Wait until BUSY flag goes low, or timeout.
  rc = pcps_wait_busy_mm_dbg( pddev, cmd );

  #if DEBUG_IO_TIMING
    mbg_get_pc_cycles( &t_after_busy );
  #endif

  if ( mbg_rc_is_error( rc ) )
    goto out;


  // Success: read data, if required.

  // First read full 32 bit words.
  for ( i = 0; i < dt_quot; i++ )
  {
    ar.ul = ioread32( p_data_reg );
    #if DEBUG_IO_PROBE
      pcps_dump_data( &ar.ul, sizeof( ar.ul ), __func__ );
    #endif
    *( (uint32_t FAR *) p) = ar.ul;
    p += sizeof( ar.ul );
    p_data_reg++;
  }

  // Then read the remaining bytes, if required.
  if ( dt_rem )
  {
    ar.ul = ioread32( p_data_reg );

    for ( i = 0; i < dt_rem; i++ )
    {
      #if DEBUG_IO_PROBE
        pcps_dump_data( &ar.b[i], sizeof( ar.b[i] ), __func__ );
      #endif

      *p++ = ar.b[i];
    }
  }

out:
  #if DEBUG_IO_TIMING
    mbg_get_pc_cycles( &t_done );
    report_io_timing( pddev, "ASIC MM", cmd, count, t_after_cmd, t_after_busy, t_done );
  #endif

  #if DEBUG_ACCESS_TIMING
    report_access_timing( pddev, "ASIC MM wr/rd", t_after_cmd, t_after_reread );
  #endif

  #if defined( DEBUG )
    report_ret_val( rc, __func__ );
  #endif

  return rc;

}  // pcps_read_asic_mm



/*HDR*/
int pcps_probe_device( PCPS_DDEV *pddev )
{
  int rc = MBG_SUCCESS;

  pcps_dump_asic_info();
  pcps_dump_regs_32( pddev );
  // pcps_dump_regs_16( pddev );

  pcps_read_asic_version( pddev );
  pcps_read_asic_features( pddev );

// pcps_test_status_port( pddev );

  mbg_kdd_msg( MBG_LOG_INFO,"Going to read FW ID to addr 0x%px, sz: %lu",
               pddev->fw_id, (unsigned long) sizeof( pddev->fw_id ) );

  memset( pddev->fw_id, 0, sizeof( pddev->fw_id ) );
  memset( pddev->fw_id, '=', sizeof( pddev->fw_id ) - 1);
  mbg_kdd_msg( MBG_LOG_INFO,"FW ID i: \"%s\"", pddev->fw_id );

  rc = pcps_read_asic_mm( pddev, PCPS_GIVE_FW_ID_1, &pddev->fw_id[0], PCPS_FIFO_SZ );

  if ( rc )
  {
    mbg_kdd_msg( MBG_LOG_ERR,"Failed to read FW ID 1, rc: %i", rc );
    goto out;
  }

  mbg_kdd_msg( MBG_LOG_INFO,"FW ID 1: \"%s\"", pddev->fw_id );

  rc = pcps_read_asic_mm( pddev, PCPS_GIVE_FW_ID_2, &pddev->fw_id[PCPS_FIFO_SZ], PCPS_FIFO_SZ );

  if ( rc )
  {
    mbg_kdd_msg( MBG_LOG_ERR,"Failed to read FW ID 2, rc: %i", rc );
    goto out;
  }

  mbg_kdd_msg( MBG_LOG_INFO,"FW ID 2: \"%s\"", pddev->fw_id );

out:
  return rc;

}  // pcps_probe_device

