/*
 * A minimal Linux kernel driver to test probing
 * os some Meinberg PCI Express cards.
 */

#include "pcpsdrvr.c"  // FIXME TODO


#define MBG_DRVR_NAME  "mbgclock-probe"

module_param( msg_sleep, int, 0444 );
MODULE_PARM_DESC( msg_sleep, "milliseconds to sleep after each log message, default: 0" );


static PCPS_DDEV pcps_ddev;


static /*HDR*/
void release_resources( PCPS_DDEV *pddev, struct pci_dev *pci_dev )
{
  if ( pddev->mapped_mem )
  {
    iounmap( pddev->mapped_mem );
    mbg_kdd_msg( MBG_LOG_INFO,"Unmapped mem at 0x%px", pddev->mapped_mem );
  }

  if ( pddev->bar_mask )
  {
    pci_release_region( pci_dev, pddev->bar_mask );
    mbg_kdd_msg( MBG_LOG_INFO,"Released regions for bar mask 0x%02x", pddev->bar_mask );
  }

  memset( pddev, 0, sizeof( *pddev ) );

}  // release_resources



static /*HDR*/
int request_resources( PCPS_DDEV *pddev, struct pci_dev *pci_dev )
{
  int rc;

  // Request the I/O memory resource.
  pddev->bar_mask = pci_select_bars( pci_dev, IORESOURCE_MEM );
  mbg_kdd_msg( MBG_LOG_INFO,"Resource bar mask: 0x%02x", pddev->bar_mask );

  // Request the memory region.
  rc = pci_request_region( pci_dev, pddev->bar_mask, MBG_DRVR_NAME );

  if ( rc )
  {
    mbg_kdd_msg( MBG_LOG_ERR,"Failed to request regions, rc: %i", rc );
    goto out;
  }


  pddev->mem_bar_idx = 1;  // Should be the index with the bar for MM registers.

  // Get the start and stop memory positions.
  pddev->mmio_start = pci_resource_start( pci_dev, pddev->mem_bar_idx );
  pddev->mmio_len = pci_resource_len( pci_dev, pddev->mem_bar_idx );
  mbg_kdd_msg( MBG_LOG_INFO,"Memory rsrc: 0x%lX, %lu bytes",
               pddev->mmio_start, pddev->mmio_len );

  // Map the memory resource to a local adress.
  pddev->mapped_mem = ioremap( pddev->mmio_start, pddev->mmio_len );
  mbg_kdd_msg( MBG_LOG_INFO,"Mapped mem to 0x%px, %lu bytes",
               pddev->mapped_mem, pddev->mmio_len );

  pddev->mm_asic_addr = pddev->mapped_mem;

  pddev->status_port_offs = offsetof( PCI_ASIC, status_port );

  rc = 0;

out:
  if ( rc )
    release_resources( pddev, pci_dev );

  return rc;

}  // request_resources



static /*HDR*/
int __devinit mbgclock_probe_pci_device( struct pci_dev *pci_dev,
                                         const struct pci_device_id *ent )
{
  PCPS_DDEV *pddev = &pcps_ddev;
  int rc;

  mbg_kdd_msg( MBG_LOG_INFO,"Entering %s", __func__ );

  rc = pci_enable_device( pci_dev );

  if ( rc )
  {
    mbg_kdd_msg( MBG_LOG_ERR,"Failed to enable PCI dev %04X, I/O %04lX: rc = %i",
                 pci_dev->device, (ulong) pci_resource_start( pci_dev, 0 ), rc );
    goto fail;
  }

  mbg_kdd_msg( MBG_LOG_INFO,"Enabled PCI dev %04X:%04X",
               pci_dev->vendor, pci_dev->device );


  rc = request_resources( pddev, pci_dev );

  if ( rc )
  {
    mbg_kdd_msg( MBG_LOG_ERR,"Failed to request resources, rc = %i", rc );
    goto fail;
  }

  rc = pcps_probe_device( pddev );

  if ( mbg_rc_is_error( rc ) )
  {
    rc = -EIO;
    goto fail;
  }

  pci_set_drvdata( pci_dev, pddev );


#if 0  // FIXME TODO
  rc = mbgdrvr_create_device( pddev );

  if ( rc < 0 )
    goto fail;

#endif

  mbg_kdd_msg( MBG_LOG_INFO,"Leaving %s (success)", __func__ );
  return 0;


fail:
  mbg_kdd_msg( MBG_LOG_ERR,"Leaving %s with error %i", __func__, rc );
  return rc;

}  // mbgclock_probe_pci_device



static /*HDR*/
void __devexit mbgclock_remove_pci_device( struct pci_dev *pci_dev )
{
  PCPS_DDEV *pddev = (PCPS_DDEV *) pci_get_drvdata( pci_dev );

  mbg_kdd_msg( MBG_LOG_INFO,"Entering %s", __func__ );

  release_resources( pddev, pci_dev );

  mbg_kdd_msg( MBG_LOG_INFO,"Leaving %s", __func__ );

}  // mbgclock_remove_pci_device



static struct pci_device_id mbgclock_pci_tbl[] __devinitdata =
{
  { PCI_VENDOR_MEINBERG, PCI_ANY_ID, PCI_ANY_ID, PCI_ANY_ID },
  { 0 }
};

MODULE_DEVICE_TABLE( pci, mbgclock_pci_tbl );


static struct pci_driver mbgclock_pci_driver =
{
  name:      MBG_DRVR_NAME,
  id_table:  mbgclock_pci_tbl,
  probe:     mbgclock_probe_pci_device,
  remove:    __devexit_p( mbgclock_remove_pci_device )
};



static  /*HDR*/
void __exit mbgclock_cleanup_module( void )
{
  mbg_kdd_msg( MBG_LOG_INFO,"Entering %s", __func__ );

  pci_unregister_driver( &mbgclock_pci_driver );
  mbg_kdd_msg( MBG_LOG_INFO,"Unregistered driver" );

  mbg_kdd_msg( MBG_LOG_INFO,"Leaving %s", __func__ );

}  // mbgclock_cleanup_module

module_exit( mbgclock_cleanup_module );



static  /*HDR*/
int __init mbgclock_init_module( void )
{
  int rc;

  mbg_kdd_msg( MBG_LOG_INFO,"Entering %s", __func__ );

  rc = pci_register_driver( &mbgclock_pci_driver );

  if ( rc )  // Error!
  {
    mbg_kdd_msg( MBG_LOG_ERR,"Failed to register driver, rc: %i", rc );
    goto fail;
  }

  mbg_kdd_msg( MBG_LOG_INFO,"Successfully registered driver" );


  if ( rc == -1 )  // TODO Yet a dummy test to avoid a compiler warning.
    goto fail_with_cleanup;


  mbg_kdd_msg( MBG_LOG_INFO,"Leaving %s (success)", __func__ );
  return 0;


fail_with_cleanup:
  mbgclock_cleanup_module();

fail:
  mbg_kdd_msg( MBG_LOG_ERR,"Leaving %s with error %i", __func__, rc );
  return rc;

}  // mbgclock_init_module

module_init( mbgclock_init_module );


MODULE_AUTHOR( "Martin Burnicki <martin.burnicki@meinberg.de>" );
MODULE_LICENSE( "GPL" );
